#ifndef SAPPHIER_LISP_X_H
#define SAPPHIER_LISP_X_H

/* Copyright (c) 2020 AlaskanEmily
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from
 * the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *   claim that you wrote the original software. If you use this software in a
 *   product, an acknowledgment in the product documentation would be
 *   appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *   misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */

/* Shared components between the interpreter and compiler. */

#include "sl_s.h"

/*****************************************************************************/

#ifdef __cplusplus
extern "C"{
#endif

/*****************************************************************************/

#define SL_X_N_INTEGERS 16
const extern struct SL_S_Atom sl_x_integers[SL_X_N_INTEGERS];

/*****************************************************************************/

#define SL_X_INTEGRAL_TYPES1(X) \
    X(u8) \
    X(s8) \
    X(u16) \
    X(s16) \
    X(u32) \
    X(s32) \
    X(char)

#ifdef SL_S_ENABLE_POINTERS
# define SL_X_INTEGRAL_TYPES(X) \
    SL_X_INTEGRAL_TYPES1(X) \
    X(ptr)
#else
# define SL_X_INTEGRAL_TYPES SL_X_SL_X_INTEGRAL_TYPES1
#endif

/*****************************************************************************/

const extern struct SL_S_Atom
    sl_x_nil, sl_x_true, sl_x_false,
    sl_x_list_hint, sl_x_atom_hint,
    
#define SL_X_INTEGRAL_NAME_DECL(X) sl_x_ ## X ## _hint,
    SL_X_INTEGRAL_TYPES(SL_X_INTEGRAL_NAME_DECL)
#undef SL_X_INTEGRAL_NAME_DECL

#ifdef SL_S_ENABLE_POINTERS
# define SL_X_PTR_NAME_DECL(X) sl_x_ptr_ ## X ## _hint,
    SL_X_INTEGRAL_TYPES(SL_X_PTR_NAME_DECL)
# undef SL_X_PTR_NAME_DECL
#endif

    sl_x_defun,
    sl_x_def,
    sl_x_defrec,
    sl_x_defproto,
    sl_x_if,
    sl_x_let,
    sl_x_comment, sl_x_dot, sl_x_tick,
    sl_x_plus, sl_x_minus, sl_x_times, sl_x_divide, sl_x_mod,
    sl_x_shift_left, sl_x_shift_right,
    sl_x_bit_or, sl_x_bit_and, sl_x_bit_xor, sl_x_bit_not,
    sl_x_is_nil, sl_x_is_atom, sl_x_is_list, sl_x_is_int,
    sl_x_concat, sl_x_cons, sl_x_head, sl_x_tail, sl_x_index, sl_x_not,
    sl_x_eq, sl_x_ne, sl_x_gt, sl_x_ge, sl_x_lt, sl_x_le;

/*****************************************************************************/
/* NULL-terminated. */
const extern struct SL_S_Atom *const sl_x_all_hints[];

/*****************************************************************************/

struct SL_X_FileOps{
    void *x_stdin;
    void *x_stdout;
    void *x_stderr;
    SL_S_FUNC_PTR(void*, open_read)(const char *path);
    SL_S_FUNC_PTR(void*, open_write)(const char *path);
    SL_S_FUNC_PTR(unsigned, read)(void *file, void *to, unsigned len);
    SL_S_FUNC_PTR(unsigned, write)(void *file, const void *from, unsigned len);
    SL_S_FUNC_PTR(void, seek_cur)(void *file, int i);
    SL_S_FUNC_PTR(void, seek_set)(void *file, int i);
    SL_S_FUNC_PTR(int, is_eof)(void *file);
    SL_S_FUNC_PTR(void, close)(void *file);
};

/*****************************************************************************/

struct SL_X_Def{
    const struct SL_S_Atom *hint;
    const struct SL_S_Atom *name;
    const void *value;
};

/*****************************************************************************/

struct SL_X_Record{
    const struct SL_S_Atom *name;
    /* Each field is structured as (hint name) where hint or name may be nil */
    const struct SL_S_List *fields;
};

/*****************************************************************************/

struct SL_X_FuncArg{
    const struct SL_S_Atom *hint;
    const struct SL_S_Atom *name;
};

/*****************************************************************************/

struct SL_X_ProtocolMethod{
    struct SL_S_Atom *name;
    struct SL_S_Atom *hint;
    struct SL_X_FuncArg *args;
    sl_s_len_t arity;
};

/*****************************************************************************/

struct SL_X_ProtocolType{
    const struct SL_S_Atom *name;
    struct SL_X_ProtocolMethod *methods;
    sl_s_len_t num_methods;
};

/*****************************************************************************/

SL_S_PURE_FUNC(int) SL_X_IsRuntimeConstant(const void *value);

/*****************************************************************************/
/* Returns 0 on success. */
SL_S_PURE_FUNC(int) SL_X_IsInt(const struct SL_S_Atom *arg);

/*****************************************************************************/

SL_S_PURE_FUNC(int) SL_X_ParseInt(const struct SL_S_Atom *arg);

/*****************************************************************************/

SL_S_PURE_FUNC(int) SL_X_TryParseInt(const struct SL_S_Atom *arg, int *out);

/*****************************************************************************/
/* Returns 0 on success. */
SL_S_PURE_FUNC(int) SL_X_IsDefun(const struct SL_S_List *code);

/*****************************************************************************/
/* Returns 0 on success. */
SL_S_PURE_FUNC(int) SL_X_IsDef(const struct SL_S_List *code);

/*****************************************************************************/
/* Returns 0 on success. */
SL_S_PURE_FUNC(int) SL_X_IsDefrec(const struct SL_S_List *code);

/*****************************************************************************/
/* Returns 0 on success. */
SL_S_PURE_FUNC(int) SL_X_IsDefproto(const struct SL_S_List *code);

/*****************************************************************************/
/* Returns 0 on success. */
SL_S_PURE_FUNC(int) SL_X_IsImport(const struct SL_S_List *code);

/*****************************************************************************/
/* Returns 0 on success.
 * Does not do increfs, so you've been warned.
 */
SL_S_FUNC(int) SL_X_ParseDefun(const struct SL_S_List *code,
    const struct SL_S_Atom **out_name,
    const struct SL_S_List **out_args,
    const struct SL_S_List **out_body);

/*****************************************************************************/
/* Returns 0 on success.
 * Does not do increfs, so you've been warned.
 */
SL_S_FUNC(int) SL_X_ParseDef(const struct SL_S_List *code,
    const struct SL_S_Atom **out_hint,
    const struct SL_S_Atom **out_name,
    const void **out_value);

/*****************************************************************************/
/* Returns 0 on success.
 * This DOES incref the fields, unlike most other parsers.
 * This does no incref the name.
 */
SL_S_FUNC(int) SL_X_ParseDefrec(const struct SL_S_List *code,
    const struct SL_S_Atom **out_name,
    const struct SL_S_List **out_fields);

/*****************************************************************************/
/* Returns 0 on success. */
SL_S_FUNC(int) SL_X_ParseDefproto(const struct SL_S_List *code,
    const struct SL_S_Atom **out_name,
    const struct SL_S_List **out_methods);

/*****************************************************************************/
/* Returns 0 on success.
 * The args is a list of lists, of the structure ((hint name) (hint name) ...)
 */
SL_S_FUNC(struct SL_S_List) *SL_X_ParseArgs(const struct SL_S_List *args);

/*****************************************************************************/

SL_S_FUNC(struct SL_S_Atom) *SL_X_BufferFile(const struct SL_X_FileOps *ops,
    const struct SL_S_Atom *name);

/*****************************************************************************/

struct SL_X_MeasureResult{
    sl_s_len_t len;
    unsigned char flag;
};

/*****************************************************************************/

SL_S_FUNC(void) SL_X_MeasureAtomsCB(const void *value, void *arg);

/*****************************************************************************/
/* Arg must be an atom pre-sized to be big enough to hold all the args. */
SL_S_FUNC(void) SL_X_ConcatAtomsCB(const void *value, void *arg);

/*****************************************************************************/

#ifdef __cplusplus
} // extern "C"
#endif

#endif /* SAPPHIER_LISP_X_H */
