#ifndef SAPPHIER_LISP_S_PARSE_H
#define SAPPHIER_LISP_S_PARSE_H
/* Copyright (c) 2020 AlaskanEmily
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from
 * the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *   claim that you wrote the original software. If you use this software in a
 *   product, an acknowledgment in the product documentation would be
 *   appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *   misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */

#include "sl_s.h"

/*****************************************************************************/

#ifdef __cplusplus
extern "C"{
#endif

/*****************************************************************************/
/**
 * 1 if the library was compiled with SL_S_UNICODE, 0 otherwise.
 */
const extern int sl_s_unicode_enabled;

/*****************************************************************************/
/**
 * 0 if the library was compiled with SL_S_NO_PARSE_INFO, 1 otherwise.
 */
const extern int sl_s_parse_info_enabled;

/*****************************************************************************/
/**
 * @brief Skips ahead of any whitespace.
 *
 * Returns an updated index after consuming all whitespace immediately after
 * the index.
 */
SL_S_PURE_FUNC(unsigned) SL_S_SkipWhitespace(const char *src,
    unsigned index,
    unsigned len);

/*****************************************************************************/
/**
 * @brief Decodes a UTF8 codepoint starting at the index.
 *
 * @note This function will just fetch individual characters if the library
 *  if the library was compiled without SL_S_UNICODE
 *
 * @param out_code_unit Receives the code unit, or U+FFFD if there is an error.
 * @return Number of bytes in the codepoint. This may be zero on end-of-input.
 */
SL_S_PURE_FUNC(unsigned) SL_S_UTF8Decode(const char *src,
    unsigned index,
    unsigned len,
    unsigned *out_code_unit);

/*****************************************************************************/

SL_S_PURE_FUNC(unsigned) SL_S_UTF8Length(unsigned codepoint);

/*****************************************************************************/

SL_S_PURE_FUNC(unsigned) SL_S_UTF8Encode(unsigned codepoint, char *to);

/*****************************************************************************/
/**
 * @brief Calculates the length of a UTF8 codepoint, if it was whitespace.
 *
 * Returns the number of bytes in a single UTF8 codepoint starting at the
 * index, if that codepoint is considered whitespace.
 */
SL_S_PURE_FUNC(unsigned) SL_S_WhitespaceLength(const char *src,
    unsigned index,
    unsigned len);

/*****************************************************************************/

SL_S_FUNC(int) SL_S_ParseValue(const char *src,
    unsigned *in_out_index,
    unsigned len,
    void **out);

#define SL_S_PARSE(ATOM, I, TO) \
    SL_S_ParseValue((ATOM)->text, (I), (ATOM)->len, (TO))

/*****************************************************************************/

SL_S_FUNC(int) SL_S_ParseList(const char *src,
    unsigned *in_out_index,
    unsigned len,
    struct SL_S_List **to);
    
/*****************************************************************************/

SL_S_FUNC(int) SL_S_ParseAtom(const char *src,
    unsigned *in_out_index,
    unsigned len,
    struct SL_S_Atom *to);

/*****************************************************************************/

#ifdef __cplusplus
} // extern "C"
#endif

/*****************************************************************************/

#endif /* SAPPHIER_LISP_S_H */
