#ifndef SAPPHIER_LISP_S_H
#define SAPPHIER_LISP_S_H

/* Copyright (c) 2020 AlaskanEmily
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from
 * the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *   claim that you wrote the original software. If you use this software in a
 *   product, an acknowledgment in the product documentation would be
 *   appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *   misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */

/*****************************************************************************/

#ifdef __cplusplus
extern "C"{
#endif

/*****************************************************************************/
/* Setup function calling convention support. */
#if (defined __WATCOMC__) || (defined _MSC_VER)

# define SL_S_FUNC_PTR(RET, NAME) RET(__cdecl*NAME)
# define SL_S_FUNC(RET) RET __cdecl
# ifdef _MSC_VER
#   define SL_S_MALLOC_FUNC(RET) __declspec(restrict) RET __cdecl
#   define SL_S_PURE_FUNC(RET) __declspec(noalias) RET __cdecl
# endif

#elif (defined __CYGWIN__) || (defined _WIN32)

# define SL_S_FUNC_PTR(RET, NAME) \
    __attribute__((cdecl)) RET(*NAME)
# define SL_S_FUNC(RET) __attribute__((cdecl)) RET

# if (defined __GNUC__) || (defined __TINYC__)
#   define SL_S_MALLOC_FUNC(RET) __attribute__((malloc, cdecl)) RET
# endif
# if (defined __GNUC_) && (__GNUC__ > 4)
#   define SL_S_PURE_FUNC(RET) __attribute__((pure, cdecl)) RET
# endif

#else

# define SL_S_FUNC_PTR(RET, NAME) RET(*NAME)
# define SL_S_FUNC(RET) RET

#endif

#ifndef SL_S_MALLOC_FUNC
# define SL_S_MALLOC_FUNC SL_S_FUNC
#endif

#ifndef SL_S_PURE_FUNC
# define SL_S_PURE_FUNC SL_S_FUNC
#endif

/*****************************************************************************/
/* Notes on tagged pointers:
 *
 * Tags can be stored in either least-significant or most-significant bits of
 * pointers. This is controlled by SL_S_TAG_MSB or SL_S_TAG_LSB being set. If
 * neither is set, the current implementation will default to LSB.
 *
 * The number of tag bits can be set by SL_S_NUM_TAG_BITS which must be at
 * least 1, and will default to 1.
 *
 * If SL_S_DO_NOT_STRIP_BITS is set, then the tag bits will not be stripped
 * from the pointers on unpacking. This is useful for hardware with mirrored
 * RAM, although it is likely not useful when SL_S_TAG_LSB is set.
 *
 * Note that the settings for SL_S_ENABLE_PROTOCOLS and SL_S_ENABLE_POINTERS
 * will affect the defaults as well.
 */
/*****************************************************************************/

#if !((defined SL_S_TAG_LSB) || (defined SL_S_TAG_MSB))
# define SL_S_TAG_LSB
#elif (defined SL_S_TAG_LSB) && (defined SL_S_TAG_MSB)
# error Invalid configuration: both SL_S_TAG_LSB and SL_S_TAG_MSB are set.
#endif

#ifndef SL_S_NUM_TAG_BITS
# if (defined SL_S_ENABLE_PROTOCOLS) || (defined SL_D_ENABLE_POINTERS)
#   define SL_S_NUM_TAG_BITS 2
# else
#  define SL_S_NUM_TAG_BITS 1
# endif
#elif SL_S_NUM_TAG_BITS < 1
# error Invalid configuration: SL_S_TAG_BITS must be greater than zero.
#endif

/*****************************************************************************/
/* Tagged pointer utils */

/* List must have the 0 tag so that an empty list truly is nil */
#define SL_S_LIST_TAG 0
#define SL_S_ATOM_TAG 1

#ifdef SL_S_ENABLE_PROTOCOLS
# define SL_S_PROTOCOL_TAG 2
# define SL_S_PROTO_TAG SL_S_PROTOCOL_TAG
#endif

#ifdef SL_S_ENABLE_POINTERS
# define SL_S_POINTER_TAG 3
#endif

#define SL_S_PTR_BITS (sizeof(void*)<<3)

#ifdef SL_S_TAG_LSB

#define SL_S_PTR_TAG_MASK(NUM_BITS) ((1 << (NUM_BITS)) - 1)
#define SL_S_PTR_TAG(X, NUM_BITS) \
    ((((char*)(X)) - (char*)0) & SL_S_PTR_TAG_MASK(NUM_BITS))
#define SL_S_RAW_TAG_PTR(X, TAG, NUM_BITS) \
    ((void*)(((char*)(SL_S_RAW_PTR_FROM_TAG(X, NUM_BITS))) + (TAG)))

#else

#define SL_S_PTR_TAG_MASK(NUM_BITS) \
    ((1 << (NUM_BITS) - 1) << (SL_S_PTR_BITS - NUM_BITS))
#define SL_S_PTR_TAG(X, NUM_BITS) \
    (((((char*)(X)) - (char*)0) & SL_S_PTR_TAG_MASK(NUM_BITS)) << \
        (SL_S_PTR_BITS - NUM_BITS))
#define SL_S_RAW_TAG_PTR(X, TAG, NUM_BITS) \
    ((void*)(((char*)(SL_S_RAW_PTR_FROM_TAG(X, NUM_BITS))) + \
        ((TAG) << (SL_S_PTR_BITS - NUM_BITS))))

#endif

#define SL_S_RAW_PTR_FROM_TAG(X, NUM_BITS) \
    ((void*)(((char*)(X)) - SL_S_PTR_TAG(X, NUM_BITS)))

#ifdef SL_S_DO_NOT_STRIP_BITS

#define SL_S_PTR_FROM_TAG(X) ((X))
/* We can't just totally ignore this, since tagging a pointer which already
 * has a tag will result in a mangled tag. */
#define SL_S_TAG_PTR(X, TAG) SL_S_RAW_TAG_PTR(\
    SL_S_RAW_PTR_FROM_TAG(X, SL_S_NUM_TAG_BITS), TAG, SL_S_NUM_TAG_BITS)

#else

#define SL_S_PTR_FROM_TAG(X) SL_S_RAW_PTR_FROM_TAG(X, SL_S_NUM_TAG_BITS)
#define SL_S_TAG_PTR(X, TAG) SL_S_RAW_TAG_PTR(X, TAG, SL_S_NUM_TAG_BITS)

#endif

#define SL_S_PTR_TAG_DATA(PTR, TO_PTR, TO_TAG) do{ \
    (TO_PTR) = (void*)(((char*)(PTR)) - \
        ((TO_TAG) = SL_S_PTR_TAG((PTR), 1))); \
}while(0)

#define SL_S_IS_NIL(X) ((void*)SL_S_RAW_PTR_FROM_TAG((X), 1) == (void*)0)
#define SL_S_IS_LIST(X) (SL_S_PTR_TAG((X), SL_S_NUM_TAG_BITS) == SL_S_LIST_TAG)
#define SL_S_IS_ATOM(X) (SL_S_PTR_TAG((X), SL_S_NUM_TAG_BITS) == SL_S_ATOM_TAG)

#ifdef SL_S_ENABLE_PROTOCOLS
# define SL_S_IS_PROTOCOL(X) \
    (SL_S_PTR_TAG((X), SL_S_NUM_TAG_BITS) == SL_S_PROTOCOL_TAG)
# define SL_S_IS_PROTO SL_S_IS_PROTOCOL
#endif

#ifdef SL_S_ENABLE_POINTERS
# define SL_S_IS_POINTER(X) \
    (SL_S_PTR_TAG((X), SL_S_NUM_TAG_BITS) == SL_S_POINTER_TAG)
# define SL_S_IS_PTR SL_S_IS_POINTER
#endif

#define SL_S_MK_LIST(X) SL_S_TAG_PTR((X), SL_S_LIST_TAG)
#define SL_S_MK_ATOM(X) (SL_S_IS_NIL((X)) ? \
    SL_S_NIL : \
    SL_S_TAG_PTR((X), SL_S_ATOM_TAG))

#ifdef SL_S_ENABLE_PROTOCOLS
# define SL_S_MK_PROTOCOL(X) (SL_S_IS_NIL(X) ? \
    SL_S_NIL : \
    SL_S_TAG_PTR((X), SL_S_PROTOCOL_TAG))
# define SL_S_MK_PROTO SL_S_MK_PROTOCOL
#endif

#ifdef SL_S_ENABLE_POINTERS
# define SL_S_MK_POINTER(X) (SL_S_IS_NIL(X) ? \
    SL_S_NIL : \
    SL_S_TAG_PTR((X), SL_S_POINTER_TAG))
# define SL_S_MK_PTR SL_S_MK_POINTER
#endif

/*****************************************************************************/
/* Len/Refcount typedefs */

#if ((defined _WIN32) || (defined WIN32)) && !(defined __GNUC__)

# ifdef WIN32_LEAN_AND_MEAN
#   include <Windows.h>
# else
#   define WIN32_LEAN_AND_MEAN
#   include <Windows.h>
#   undef WIN32_LEAN_AND_MEAN
# endif

typedef DWORD sl_s_len_t;
typedef volatile LONG sl_s_ref_t;

#elif (defined __M68K__) || (defined __m68k__)

typedef unsigned short sl_s_len_t;
typedef volatile unsigned short sl_s_ref_t;

#else

typedef unsigned sl_s_len_t;
typedef unsigned sl_s_ref_t;

#endif

/*****************************************************************************/

#define SL_S_MAX_LEN (~(sl_s_len_t)0)

/*****************************************************************************/
/* Atomic integer utils */

#if (defined __GNUC__) && (__GNUC__ < 6) && (__GNUC__ > 2)

# define SL_S_ATOMIC_INC(X) __sync_fetch_and_add((X), 1)
# define SL_S_ATOMIC_DEC(X) __sync_fetch_and_sub((X), 1)

#elif (defined __GNUC__) && (__GNUC__ >= 6)

# define SL_S_ATOMIC_INC(X) __atomic_fetch_add((X), 1, __ATOMIC_RELAXED)
# define SL_S_ATOMIC_DEC(X) __atomic_fetch_sub((X), 1, __ATOMIC_RELAXED)

#elif (defined _WIN32) || (defined WIN32)

# define SL_S_ATOMIC_INC(X) (InterlockedIncrement(X)-1)
# define SL_S_ATOMIC_DEC(X) (InterlockedDecrement(X)+1)

#else

# define SL_S_ATOMIC_INC(X) ((*(X))++)
# define SL_S_ATOMIC_DEC(X) ((*(X))--)

#endif

/*****************************************************************************/

#ifndef SL_S_NIL
#define SL_S_NIL ((void*)0)
#endif

/*****************************************************************************/
/** Represents an atom. */
struct SL_S_Atom{
    sl_s_ref_t ref;
#ifndef SL_S_NO_PARSER_INFO
    sl_s_len_t line;
#endif
    sl_s_len_t len;
    char *text;
};

/*****************************************************************************/

#ifdef SL_S_NO_PARSER_INFO
#define SL_S_STATIC_ATOM(TXT) { \
	~(sl_s_ref_t)0, \
    sizeof("" TXT) - 1, \
    (TXT) \
}
#else
#define SL_S_STATIC_ATOM(TXT) { \
	~(sl_s_ref_t)0, \
    0, \
    sizeof("" TXT) - 1, \
    (TXT) \
}
#endif

/*****************************************************************************/
/** Represents a list.
 *
 * The head is a tagged pointer for a list or an atom.
 */
struct SL_S_List{
    sl_s_ref_t ref;
#ifndef SL_S_NO_PARSER_INFO
    sl_s_len_t line;
#endif
    void *head;
    struct SL_S_List *tail;
};

/*****************************************************************************/

#ifdef SL_S_ENABLE_PROTOCOLS

/*****************************************************************************/
/** Represents an protocol object. */
struct SL_S_Protocol{
    sl_s_ref_t ref;
#ifndef SL_S_NO_PARSER_INFO
    sl_s_len_t line;
#endif
    SL_S_FUNC_PTR(void, destroy)(struct SL_S_Protocol *that);
};

/*****************************************************************************/

#endif

/*****************************************************************************/

#ifdef SL_S_ENABLE_POINTERS

/*****************************************************************************/
/** Holds a boxed pointer.
 *
 * Sadly, for use in Sapphire Lisp, all pointers must be boxed to work properly
 * with tag bits.
 *
 * NULL (in C) can still be represented as an unboxed NIL, and SL_S_MK_PTR will
 * actually cause this to happen when possible.
 *
 * The ref-count only applies to the pointer's box.
 */
struct SL_S_Pointer{
    sl_s_ref_t ref;
#ifndef SL_S_NO_PARSER_INFO
    sl_s_len_t line;
#endif
    void *data;
};

/*****************************************************************************/

#endif

/*****************************************************************************/
/** Shared struct to allow access to refcount without conditional casts */
struct SL_S_Ref{
    sl_s_ref_t ref;
};

/*****************************************************************************/
/**
 * @struct SL_S_Line
 * @briefShared struct to allow access to refcount without conditional casts
 */

#ifndef SL_S_NO_PARSER_INFO
/** Shared struct to allow access to refcount without conditional casts */
struct SL_S_Line{
    sl_s_ref_t _;
    sl_s_len_t line;
};

#endif

/*****************************************************************************/

union SL_S_SizeofUnion{
    struct SL_S_Atom atom;
    struct SL_S_List list;
};

#define SL_S_VALUE_SIZE sizeof(union SL_S_SizeofUnion)

/*****************************************************************************/
/** Accesses the refcount of an item. */
#define SL_S_REF(X) (((struct SL_S_Ref*)SL_S_PTR_FROM_TAG((X)))->ref)

/*****************************************************************************/
/** 
 * @def SL_S_LINE
 * @brief Accesses the line for an item.
 */
#ifdef SL_S_NO_PARSER_INFO
#define SL_S_LINE(X) ((void)(sizeof(X)))
#define SL_S_SET_LINE(X, LN) ((void)(sizeof(X), sizeof((LN) | 0)))
#define SL_S_PTR_SET_LINE(X, LN) ((void)(sizeof(X), sizeof((LN) | 0)))
#else
#define SL_S_LINE(X) (((struct SL_S_Line*)SL_S_PTR_FROM_TAG((X)))->line)
#define SL_S_SET_LINE(X, LN) ((void)(SL_S_LINE(X) = (LN)|0))
#define SL_S_PTR_SET_LINE(X, LN) ((void)((X)->line = (LN) | 0))
#endif

/*****************************************************************************/
/** Tests if the refcounted item is interned/uncollectable */
#define SL_S_IS_INTERNED(X) (SL_S_IS_NIL(X) || (SL_S_REF(X) == ~(sl_s_ref_t)0))

/*****************************************************************************/
/** Tests if the refcounted item is interned/uncollectable */
#define SL_S_INTERNED(X) (SL_S_IS_NIL(X) || (SL_S_REF(X) == ~(sl_s_ref_t)0))

/*****************************************************************************/

#define SL_S_INCREF(X) do{ \
    if(!SL_S_IS_INTERNED(X)) \
        SL_S_ATOMIC_INC(&SL_S_REF(X)); \
}while(0)

/*****************************************************************************/

#define SL_S_DECREF(X) do{ \
    if(!SL_S_IS_INTERNED(X)) { \
        if(SL_S_ATOMIC_DEC(&SL_S_REF(X)) == 1) { \
            SL_S_FreeValue((void*)(X)); \
        } \
    } \
}while(0)

/*****************************************************************************/

SL_S_FUNC(void) SL_S_FreeValue(void *val);

/*****************************************************************************/

SL_S_FUNC(void) SL_S_FreeList(struct SL_S_List *list);
#define SL_S_FREE_LIST(X) SL_S_FreeList((void*)X)

/*****************************************************************************/

SL_S_FUNC(void) SL_S_FreeAtom(struct SL_S_Atom *atom);
#define SL_S_FREE_ATOM(ATOM) \
    SL_S_Free2((ATOM), ((struct SL_S_Atom*)(ATOM))->text)

/*****************************************************************************/

SL_S_MALLOC_FUNC(void) *SL_S_Malloc(unsigned len);

/*****************************************************************************/

SL_S_FUNC(void) SL_S_Free(void *ptr);

/*****************************************************************************/

SL_S_FUNC(void) SL_S_Free2(void *ptr1, void *ptr2);

/*****************************************************************************/

SL_S_PURE_FUNC(sl_s_len_t) SL_S_Length(const struct SL_S_List *list);

/*****************************************************************************/
/* Similar to `SL_S_Length(list) == len ? 0 : 1`, but more efficient. */
SL_S_PURE_FUNC(int) SL_S_LengthCompare(const struct SL_S_List *list,
    sl_s_len_t len);

/*****************************************************************************/

SL_S_PURE_FUNC(void) SL_S_MemCopy(void *to, const void *from, sl_s_len_t len);

/*****************************************************************************/

SL_S_PURE_FUNC(void) SL_S_MemSet(void *to, int c, sl_s_len_t len);

/*****************************************************************************/

SL_S_PURE_FUNC(int) SL_S_MemComp(const void *a, const void *b, sl_s_len_t len);

/*****************************************************************************/

SL_S_PURE_FUNC(unsigned) SL_S_CompareAtoms(const struct SL_S_Atom *a,
    const struct SL_S_Atom *b);

/*****************************************************************************/

#define SL_S_COMPARE_ATOMS(A, B) ( \
    SL_S_IS_NIL(A) ? SL_S_IS_NIL(B) :  \
    SL_S_IS_NIL(B) ? 0 : \
    ((A)->len != (B)->len) ? 0 : \
    (SL_S_MemComp((A)->text, (B)->text, (A)->len) == 0) \
)

/*****************************************************************************/

SL_S_PURE_FUNC(unsigned) SL_S_StrLen(const char *string);

/*****************************************************************************/

SL_S_FUNC(void) SL_S_ForEach(
    SL_S_FUNC_PTR(void, cb)(const void *value, void *arg),
    const struct SL_S_List *list,
    void *arg);

/*****************************************************************************/

SL_S_FUNC(struct SL_S_List) *SL_S_Map(
    SL_S_FUNC_PTR(void*, cb)(const void *value, void *arg),
    const struct SL_S_List *list,
    void *arg);

/*****************************************************************************/

SL_S_FUNC(struct SL_S_Atom) *SL_S_IntToAtom(int i, unsigned short base);

/*****************************************************************************/
/* Compares the two lexically. */
SL_S_FUNC(int) SL_S_Compare(const void *a, const void *b);

/*****************************************************************************/

#define SL_S_IN (0)
#define SL_S_OUT (0x80)
#define SL_S_IN_OUT_MASK (0x80)
#define SL_S_ANY (0x10)
#define SL_S_ANY_MASK (0x10)
#define SL_S_TEST_NIL (0x20)
#define SL_S_TEST_NIL_MASK (0x20)

#define SL_S_IGNORE 0xFF
#define SL_S_IN_LIST (SL_S_IN|SL_S_LIST_TAG)
#define SL_S_OUT_LIST (SL_S_OUT|SL_S_LIST_TAG)
#define SL_S_IN_ATOM (SL_S_IN|SL_S_ATOM_TAG)
#define SL_S_OUT_ATOM (SL_S_OUT|SL_S_ATOM_TAG)
#define SL_S_IN_ANY (SL_S_OUT|SL_S_ANY)
#define SL_S_OUT_ANY (SL_S_OUT|SL_S_ANY)

SL_S_FUNC(int) SL_S_Match(const struct SL_S_List *in,
    const void **in_out_values,
    const unsigned char *flags,
    sl_s_len_t val_min,
    sl_s_len_t val_max);

/*****************************************************************************/

#ifdef __cplusplus
} // extern "C"
#endif

/*****************************************************************************/

#endif /* SAPPHIER_LISP_S_H */
