/* Copyright (c) 2020 AlaskanEmily
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from
 * the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *   claim that you wrote the original software. If you use this software in a
 *   product, an acknowledgment in the product documentation would be
 *   appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *   misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */

#include "sl_i.h"

#include "sl_i_builtin.h"
#include "sl_x.h"

#include <stddef.h>

#define SL_I_BIND_CAP_GROWTH 64
#define SL_I_REC_CAP_GROWTH 32
#define SL_I_PROTO_CAP_GROWTH 32
#define SL_I_DEF_CAP_INIT 16
#define SL_I_DEF_CAP_DOUBLE_MAX 256

#ifdef __GNUC__
#define SL_I_UNLIKELY(X) __builtin_expect(!!(X), 0)
#define SL_I_LIKELY(X) __builtin_expect(!!(X), 1)
#else
#define SL_I_UNLIKELY(X) (!!(X))
#define SL_I_LIKELY(X) (!!(X))
#endif

/*****************************************************************************/

static const struct SL_S_Atom sl_i_stdin_name = SL_S_STATIC_ATOM("stdin");
static const struct SL_S_Atom sl_i_stdout_name = SL_S_STATIC_ATOM("stdout");
static const struct SL_S_Atom sl_i_stderr_name = SL_S_STATIC_ATOM("stderr");

/*****************************************************************************/

SL_S_FUNC(void) SL_I_InitRuntime(struct SL_I_Runtime *rt){
    unsigned i;
    
    /* Clear up to, but not including, the file ops. */
    SL_S_MemSet(rt, 0, offsetof(struct SL_I_Runtime, io));
    rt->frames = &(rt->global);
    
    rt->cap_binds = SL_I_BIND_CAP_GROWTH;
    rt->binds = SL_S_Malloc(SL_I_BIND_CAP_GROWTH * sizeof(struct SL_I_Bind));
    i = 0;

#define SL_I_SETUP_BUILTIN(FUNC, NAME, ARITY) do{ \
    rt->binds[i].name = &(NAME); \
    rt->binds[i].is_native = 1; \
    rt->binds[i].bind.native = (FUNC); \
    rt->binds[i].args = SL_S_NIL; \
    rt->binds[i].arity = (ARITY)|0; \
    i++; \
}while(0)
    
    SL_I_SETUP_BUILTIN(SL_I_Print, sl_i_print_name, 1);
    SL_I_SETUP_BUILTIN(SL_I_Open, sl_i_open_name, 2);
    SL_I_SETUP_BUILTIN(SL_I_Close, sl_i_close_name, 1);
    SL_I_SETUP_BUILTIN(SL_I_Read, sl_i_read_name, 3);
    SL_I_SETUP_BUILTIN(SL_I_Write, sl_i_write_name, 3);
    SL_I_SETUP_BUILTIN(SL_I_WriteAtom, sl_i_write_atom_name, 2);
    rt->num_binds = i;
    
#undef SL_I_SETUP_BUILTIN
    
    rt->global.cap_defs = SL_I_DEF_CAP_INIT;
    rt->global.defs = SL_S_Malloc(SL_I_DEF_CAP_INIT * sizeof(struct SL_X_Def));
    i = 0;
    
#define SL_I_SETUP_FILE(WHAT) do{ \
    (rt->WHAT ## _ptr).ref = ~(sl_s_ref_t)0; \
    SL_S_PTR_SET_LINE(&(rt->WHAT ## _ptr), 0); \
    (rt->WHAT ## _ptr).data = (rt->io.x_ ## WHAT); \
    rt->global.defs[i].value = SL_S_MK_POINTER(&(rt->WHAT ## _ptr)); \
    rt->global.defs[i].name = &(sl_i_ ## WHAT ## _name); \
    rt->global.defs[i].hint = &sl_x_ptr_hint; \
    i++; \
}while(0)
    
    SL_I_SETUP_FILE(stdin);
    SL_I_SETUP_FILE(stdout);
    SL_I_SETUP_FILE(stderr);
    
#undef SL_I_SETUP_FILE
    rt->global.num_defs = i;
}

/*****************************************************************************/

static sl_s_len_t sl_i_find_bind(const struct SL_I_Runtime *rt,
    const struct SL_S_Atom *name){
    
    register sl_s_len_t i, n;
    n = rt->num_binds;
    for(i = 0; i < n; i++){
        if(SL_S_COMPARE_ATOMS(rt->binds[i].name, name))
            return i;
    }
    return n;
}

/*****************************************************************************/

static sl_s_len_t sl_i_find_def(const struct SL_I_Runtime *rt,
    const struct SL_S_Atom *name){
    
    register sl_s_len_t i, n;
    const struct SL_X_Def *defs;
    n = rt->global.num_defs;
    defs = rt->global.defs;
    for(i = 0; i < n; i++){
        if(SL_S_COMPARE_ATOMS(defs[i].name, name))
            return i;
    }
    return n;
}

/*****************************************************************************/

static sl_s_len_t sl_i_find_rec(const struct SL_I_Runtime *rt,
    const struct SL_S_Atom *name){
    
    register sl_s_len_t i, n;
    n = rt->num_recs;
    for(i = 0; i < n; i++){
        if(SL_S_COMPARE_ATOMS(rt->recs[i].name, name))
            return i;
    }
    return n;
}

/*****************************************************************************/

static void sl_i_defun(struct SL_I_Runtime *rt,
    const struct SL_S_Atom *name,
    const struct SL_S_List *args,
    const struct SL_S_List *body){
    
    register sl_s_len_t i;
    struct SL_I_Bind *bind;
    sl_s_len_t arity;
    void *old;
    const struct SL_S_List *arg_pair, *iter;
    const struct SL_S_Atom *atom;

    arity = 0;
    iter = args;
    while(iter != SL_S_NIL){
        arg_pair = SL_S_PTR_FROM_TAG(iter->head);
        atom = SL_S_PTR_FROM_TAG(arg_pair->head);
        if(SL_I_LIKELY(SL_S_IS_NIL(atom) ||
            SL_S_COMPARE_ATOMS(atom, &sl_x_nil))){
            
            goto type_hint_ok;
        }
        /* Validate that we know what this type hint is. */
#define SL_I_TYPE_HINT_CHECK(X) \
    if(SL_S_COMPARE_ATOMS(atom, &sl_x_ ## X ## _hint)) \
        goto type_hint_ok;
        SL_X_INTEGRAL_TYPES(SL_I_TYPE_HINT_CHECK)
#undef SL_I_TYPE_HINT_CHECK

#ifdef SL_S_ENABLE_POINTERS
# define SL_I_TYPE_PTR_HINT_CHECK(X) \
    if(SL_S_COMPARE_ATOMS(atom, &sl_x_ptr_ ## X ## _hint)) \
        goto type_hint_ok;
        SL_X_INTEGRAL_TYPES(SL_I_TYPE_PTR_HINT_CHECK)
# undef SL_I_TYPE_PTR_HINT_CHECK
#endif
        /* Not a builtin, check for a defrec. */
        for(i = 0; i < rt->num_recs; i++){
            if(atom->len - 1 == rt->recs[i].name->len &&
                SL_S_MemComp(atom->text + 1,
                    rt->recs[i].name->text,
                    atom->len - 1) == 0){
                
                goto type_hint_ok;
            }
        }
        
        /* Not OK! We don't know what this hint means. */
#define SL_I_HINT_ERROR "Unknown type "
        rt->pending_error =
        rt->error_free_ptr =
            SL_S_Malloc(sizeof(SL_I_HINT_ERROR) + atom->len);
        SL_S_MemCopy(((char*)(rt->pending_error)),
            SL_I_HINT_ERROR,
            sizeof(SL_I_HINT_ERROR) - 1);
        SL_S_MemCopy(((char*)(rt->pending_error)) + sizeof(SL_I_HINT_ERROR) - 1,
            atom->text,
            atom->len);
        ((char*)(rt->pending_error))[sizeof(SL_I_HINT_ERROR) + atom->len - 1] = '\0';
        return;
#undef SL_I_HINT_ERROR
type_hint_ok:
        arity++;
        iter = iter->tail;
    }
    
    /* Search for an existing function of this name. */
    i = sl_i_find_bind(rt, name);
    
    /* Check if we found a matching bind or not. */
    if(i == rt->num_binds){
        /* Check if we need more room. */
        if(rt->cap_binds == i){
            /* Round up if the number of binds isn't a multiple of 64. */
            if(SL_I_UNLIKELY(rt->cap_binds & (SL_I_BIND_CAP_GROWTH-1)))
                rt->cap_binds +=
                    SL_I_BIND_CAP_GROWTH + SL_I_BIND_CAP_GROWTH -
                    (rt->cap_binds & (SL_I_BIND_CAP_GROWTH-1));
            else
                rt->cap_binds += SL_I_BIND_CAP_GROWTH;
            /* Try to realloc */
            old = rt->binds;
            rt->binds = SL_S_Malloc(sizeof(struct SL_I_Bind) * rt->cap_binds);
            if(SL_I_UNLIKELY(rt->binds == SL_S_NIL)){
                rt->pending_error = "Out of memory";
                rt->cap_binds = i;
                rt->binds = old;
                return;
            }
            SL_S_MemCopy(rt->binds, old, i * sizeof(struct SL_I_Bind));
            SL_S_Free(old);
        }
        bind = rt->binds + i;
        bind->args = SL_S_Malloc(sizeof(struct SL_X_FuncArg) * arity);
        rt->num_binds++;
    }
    else{
        bind = rt->binds + i;
        old = bind->args;
        
        /* Clean out the old bind. */
        SL_S_DECREF(bind->name);
        if(!(bind->is_native)){
            SL_S_DECREF(bind->bind.lisp);
            for(i = 0; i < bind->arity; i++){
                SL_S_DECREF(bind->args[i].hint);
                SL_S_DECREF(bind->args[i].name);
            }
        }
        
        /* Reuse the original args allocation, if possible. */
        if(bind->arity < arity || bind->args == SL_S_NIL){
            SL_S_Free(rt->binds[i].args);
            bind->args = SL_S_Malloc(sizeof(struct SL_X_FuncArg) * arity);
            if(SL_I_UNLIKELY(bind->args == SL_S_NIL)){
                rt->pending_error = "Out of memory";
                /* The only error recover we can do is to remove this bind, as
                 * we have already completely freed its args and body. We could
                 * do this allocation upfront, but we should have freed quite a
                 * bit leading up to this, and we aren't going to optimize for
                 * OOM situations.
                 */
                rt->num_binds--;
                if(SL_I_LIKELY(bind != rt->binds + rt->num_binds))
                    SL_S_MemCopy(bind,
                        rt->binds + rt->num_binds,
                        sizeof(struct SL_I_Bind));
                return;
            }
        }
    }
    
    bind->arity = arity;
    bind->is_native = 0;
    bind->bind.lisp = body;
    bind->name = name;
    SL_S_INCREF(bind->name);
    iter = args;
    for(i = 0; i < arity; i++){
        arg_pair = SL_S_PTR_FROM_TAG(iter->head);
        SL_S_INCREF(arg_pair->head);
        SL_S_INCREF(arg_pair->tail->head);
        bind->args[i].hint = SL_S_PTR_FROM_TAG(arg_pair->head);
        bind->args[i].name = SL_S_PTR_FROM_TAG(arg_pair->tail->head);
        iter = iter->tail;
    }
    SL_S_DECREF(args);
}

/*****************************************************************************/

static void sl_i_def(struct SL_I_Runtime *rt,
    const struct SL_S_Atom *hint,
    const struct SL_S_Atom *name,
    const void *value){
    
    register sl_s_len_t i;
    struct SL_X_Def *def;
    void *old;
    /* Check for an existing def of the same name. */
    i = sl_i_find_def(rt, name);
    
    /* Check if we found a matching def or not. */
    if(i == rt->global.num_defs){
        /* Check if we need more room. */
        if(i == rt->global.cap_defs){
            rt->global.cap_defs = SL_I_DEF_CAP_INIT;
            
            /* Low-scale doubling. */
            while(rt->global.cap_defs <= i &&
                rt->global.cap_defs < SL_I_DEF_CAP_DOUBLE_MAX){
                
                rt->global.cap_defs <<= 1;
            }
            
            /* High-scale adding */
            while(SL_I_UNLIKELY(rt->global.cap_defs <= i)){
                rt->global.cap_defs += SL_I_DEF_CAP_DOUBLE_MAX;
            }
            
            old = rt->global.defs;
            rt->global.defs =
                SL_S_Malloc(sizeof(struct SL_X_Def) * rt->global.cap_defs);
            if(SL_I_UNLIKELY(rt->global.defs == SL_S_NIL)){
                rt->pending_error = "Out of memory";
                rt->global.defs = old;
                rt->global.cap_defs = i;
                return;
            }
        }
        def = rt->global.defs + i;
        rt->global.num_defs++;
    }
    else{
        def = rt->global.defs + i;
        SL_S_DECREF(def->name);
        SL_S_DECREF(def->value);
    }
    
    def->hint = hint;
    def->name = name;
    def->value = value;
    SL_S_INCREF(hint);
    SL_S_INCREF(name);
    SL_S_INCREF(value);
}

/*****************************************************************************/

static void sl_i_defrec(struct SL_I_Runtime *rt,
    const struct SL_S_Atom *name,
    const struct SL_S_List *fields){
    
    void *old;
    register sl_s_len_t i;
    struct SL_X_Record *rec;
    
    /* Search for an existing function of this name. */
    i = sl_i_find_rec(rt, name);
    
    /* Check if we found a matching bind or not. */
    if(i == rt->num_recs){
        /* Check if we need more room. */
        if(rt->cap_recs == i){
            /* Round up if the number of recs isn't a multiple of 32. */
            if(SL_I_UNLIKELY(rt->cap_recs & (SL_I_REC_CAP_GROWTH-1)))
                rt->cap_recs +=
                    SL_I_REC_CAP_GROWTH + SL_I_REC_CAP_GROWTH -
                    (rt->cap_recs & (SL_I_REC_CAP_GROWTH-1));
            else
                rt->cap_recs += SL_I_REC_CAP_GROWTH;
            /* Try to realloc */
            old = rt->recs;
            rt->recs = SL_S_Malloc(sizeof(struct SL_X_Record) * rt->cap_recs);
            if(SL_I_UNLIKELY(rt->recs == SL_S_NIL)){
                rt->pending_error = "Out of memory";
                rt->cap_recs = i;
                rt->recs = old;
                return;
            }
            SL_S_MemCopy(rt->recs, old, i * sizeof(struct SL_X_Record));
            SL_S_Free(old);
        }
        rt->num_recs++;
        rec = rt->recs + i;
    }
    else{
        rec = rt->recs + i;
        SL_S_DECREF(rec->name);
        SL_S_DECREF(rec->fields);
    }
    
    rec->name = name;
    rec->fields = fields;
}

/*****************************************************************************/

static int sl_i_defproto_method(const struct SL_S_List *list,
    struct SL_X_ProtocolMethod *to){
    
    struct SL_S_Atom *name, *hint, *tmp, **target;
    const struct SL_S_List *args;
    int tag;
    SL_S_PTR_TAG_DATA(list->head, name, tag);
    if(SL_I_UNLIKELY(tag != SL_S_ATOM_TAG))
        return -1;
    if((list = list->tail) == SL_S_NIL)
        return -1;
    if(name->len > 1 && name->text[0] == '^'){
        hint = name;
        SL_S_PTR_TAG_DATA(list->head, name, tag);
        if(SL_I_UNLIKELY(tag != SL_S_ATOM_TAG))
            return -1;
        if(SL_I_UNLIKELY((list = list->tail) == SL_S_NIL))
            return -1;
    }
    else{
        hint = SL_S_NIL;
    }
    if(!SL_I_LIKELY(SL_S_IS_LIST(list->head)))
        return -1;
    args = list = SL_S_PTR_FROM_TAG(list->head);
    
    to->name = name;
    to->hint = hint;
    
    /* Compute the arity. */
    name = SL_S_NIL;
    hint = SL_S_NIL;
    to->arity = 0;
    while(list != SL_S_NIL){
        SL_S_PTR_TAG_DATA(list->head, tmp, tag);
        if(SL_I_UNLIKELY(tag != SL_S_ATOM_TAG))
            return -1;

        if(tmp->len > 1 && tmp->text[0] == '^')
            target = &hint;
        else
            target = &name;
        if(!SL_S_IS_NIL(*target)){
            to->arity++;
            name = SL_S_NIL;
            hint = SL_S_NIL;
        }
        *target = tmp;
        list = list->tail;
    }
    if(!SL_S_IS_NIL(name) || !SL_S_IS_NIL(hint))
        to->arity++;
    
    /* Validated, allocate and then fill. */
    if(to->arity == 0){
        to->args = SL_S_NIL;
        goto finished_args;
    }
    
    to->args = SL_S_Malloc(to->arity * sizeof(struct SL_X_FuncArg));
    if(SL_I_UNLIKELY(to->args == SL_S_NIL))
        return -1;
    
    /* Fill the args. */
    name = SL_S_NIL;
    hint = SL_S_NIL;
    to->arity = 0;
    list = args;
    while(list != SL_S_NIL){
        tmp = SL_S_PTR_FROM_TAG(list->head);

        if(tmp->len > 1 && tmp->text[0] == '^')
            target = &hint;
        else
            target = &name;
        
        if(!SL_S_IS_NIL(*target)){
            to->args[to->arity].hint = hint;
            to->args[to->arity].name = name;
            to->arity++;
            name = SL_S_NIL;
            hint = SL_S_NIL;
        }
        SL_S_INCREF(*target);
        *target = tmp;
        list = list->tail;
    }
    if(!SL_S_IS_NIL(name) || !SL_S_IS_NIL(hint)){
        to->args[to->arity].hint = hint;
        to->args[to->arity].name = name;
        to->arity++;
    }
finished_args:
    SL_S_INCREF(to->name);
    SL_S_INCREF(to->hint);
    return 0;
}

static void sl_i_defproto(struct SL_I_Runtime *rt,
    const struct SL_S_Atom *name,
    const struct SL_S_List *methods){
    
    struct SL_X_ProtocolMethod *method_array;
    void *old;
    sl_s_len_t num_methods, i, e;
    
    /* Do not allow re-defining protocols. */
    for(i = 0; i < rt->num_protocols; i++){
        if(SL_S_COMPARE_ATOMS(rt->protocols[i].name, name)){
            rt->pending_error = "Duplicate name for defproto";
            return;
        }
    }
    num_methods = SL_S_Length(methods);
    method_array =
        SL_S_Malloc(num_methods * sizeof(struct SL_X_ProtocolMethod));
    if(SL_I_UNLIKELY(SL_S_IS_NIL(methods))){
        rt->pending_error = "Out of memory";
        return;
    }
    i = 0;
    
    if(rt->num_protocols == rt->cap_protocols){
        old = rt->protocols;
        rt->cap_protocols += SL_I_PROTO_CAP_GROWTH;
        rt->protocols =
            SL_S_Malloc(rt->cap_protocols * sizeof(struct SL_X_ProtocolType));
        if(SL_I_UNLIKELY(rt->protocols == SL_S_NIL)){
            rt->protocols = old;
            rt->pending_error = "Out of memory";
            SL_S_Free(method_array);
            return;
        }
        SL_S_MemCopy(rt->protocols,
            old,
            sizeof(struct SL_X_ProtocolType) * rt->num_protocols);
    }
    
    while(!SL_S_IS_NIL(methods)){
        if(SL_I_UNLIKELY(!SL_S_IS_LIST(methods->head)) ||
            SL_I_UNLIKELY(
            sl_i_defproto_method(
                SL_S_PTR_FROM_TAG(methods->head),
                method_array + i) != 0)){
            
            goto fail_in_methods;
            methods = methods->tail;
        }
        i++;
        methods = methods->tail;
    }
    
    SL_S_INCREF(name);
    rt->protocols[rt->num_protocols].name = name;
    rt->protocols[rt->num_protocols].methods = method_array;
    rt->protocols[rt->num_protocols].num_methods = num_methods;
    rt->num_protocols++;
    return;
    
fail_in_methods:
    while(i--){
        for(e = 0; e < method_array[i].arity; e++){
            SL_S_DECREF(method_array[i].args[e].hint);
            SL_S_DECREF(method_array[i].args[e].name);
        }
        SL_S_Free(method_array[i].args);
    }
    SL_S_Free(method_array);
    rt->pending_error = "Invalid protocol method declaration";
    return;
}

/*****************************************************************************/

SL_S_FUNC(int) SL_I_Run(struct SL_I_Runtime *rt, const struct SL_S_List *code){
    
    const struct SL_S_Atom *name, *hint;
    const struct SL_S_List *data, *args, *body;
    const void *value;
    while(code){
        if(SL_S_IS_LIST(code->head)){
            data = SL_S_PTR_FROM_TAG(code->head);
            if(SL_X_IsDefun(data) == 0){
                if(SL_I_LIKELY(
                    SL_X_ParseDefun(data, &name, &args, &body) == 0)){
                    
                    data = SL_X_ParseArgs(args);
                    if(SL_I_UNLIKELY(data == SL_S_NIL && args != SL_S_NIL)){
                        rt->pending_error = "Invalid args in defun";
                        return 1;
                    }
                    sl_i_defun(rt, name, data, body);
                }
                else{
                    rt->pending_error = "Error in defun";
                    return 1;
                }
            }
            else if(SL_X_IsDef(data) == 0){
                if(SL_I_UNLIKELY(
                    SL_X_ParseDef(data, &hint, &name, &value) != 0)){
                    
                    rt->pending_error = "Error in def";
                    return 1;
                }
                sl_i_def(rt, hint, name, value);
            }
            else if(SL_X_IsDefrec(data) == 0){
                if(SL_I_UNLIKELY(SL_X_ParseDefrec(data, &name, &body) != 0)){
                    rt->pending_error = "Error in defrec";
                    return 1;
                }
                SL_S_INCREF(name);
                sl_i_defrec(rt, name, body);
            }
            else if(SL_X_IsDefproto(data) == 0){
                if(SL_I_UNLIKELY(SL_X_ParseDefproto(data, &name, &body) != 0)){
                    rt->pending_error = "Error in defproto";
                    return 1;
                }
                SL_S_INCREF(name);
                sl_i_defproto(rt, name, body);
            }
            else{
                SL_I_Execute(rt, code->head, SL_S_NIL);
            }
        }
        if(rt->pending_error)
            return -1;
        code = code->tail;
    }
    return 0;
}

/*****************************************************************************/
#define SL_I_IF_ARITY 3
static const unsigned char sl_i_if_flags[SL_I_IF_ARITY] = {
    SL_S_OUT_ANY,
    SL_S_OUT_ANY,
    SL_S_OUT_ANY
};
static void *sl_i_if(struct SL_I_Runtime *rt,
    const struct SL_S_List *in,
    const struct SL_S_Atom **opt_out_hint){
    
    const void *values[SL_I_IF_ARITY], *result;
    struct SL_S_List rebuild;
    register int i;
    
    rebuild.head = SL_I_Execute(rt, in->head, SL_S_NIL);
    if(rt->pending_error)
        return SL_S_NIL;
    
    rebuild.ref = ~0;
    rebuild.tail = in->tail;
    i = SL_S_Match(&rebuild,
        values,
        sl_i_if_flags,
        SL_I_IF_ARITY,
        SL_I_IF_ARITY);
    SL_S_DECREF(rebuild.head);
    
    if(i < 0){
        rt->pending_error = "Invalid args to if";
        return SL_S_NIL;
    }
    else if(SL_S_Compare(&sl_x_true, values[0]) == 0){
        result = values[1];
    }
    else if(SL_S_Compare(&sl_x_false, values[0]) == 0){
        result = values[2];
    }
    else{
        rt->pending_error = "First arg to if must be true or false.";
        return SL_S_NIL;
    }
    /* TODO: We should check the type hint of the un-evaluated side, too. */
    result = SL_I_Execute(rt, (void*)result, opt_out_hint);
    SL_S_INCREF(result);
    return (void*)result;
}

/*****************************************************************************/

SL_S_FUNC(void) sl_i_all_int(const void *value, void *arg){
    int *flag;
    
    flag = arg;
    if(*flag == 0  && SL_X_IsInt(value) != 0)
        *flag = 1;
}

/*****************************************************************************/

struct sl_i_to_int_arg{
    struct SL_I_Runtime *rt;
    int flag;
};

/*****************************************************************************/

SL_S_FUNC(void) *sl_i_to_int(const void *value, void *arg){
    struct sl_i_to_int_arg *data;
    data = arg;
    
    if(data->rt->pending_error || data->flag != 0)
        return SL_S_NIL;
    
    if(SL_S_IS_ATOM(value) &&
        SL_X_IsInt(SL_S_PTR_FROM_TAG(value)) == 0){
        
        SL_S_INCREF(value);
        return (void*)value;
    }
    
    value = SL_I_Execute(data->rt, (void*)value, SL_S_NIL);
    
    if(SL_S_IS_ATOM(value) &&
        SL_X_IsInt(SL_S_PTR_FROM_TAG(value)) == 0){
        
        return (void*)value;
    }
    else{
        SL_S_DECREF(value);
        data->flag = 1;
        return SL_S_NIL;
    }
}

/*****************************************************************************/

static void *sl_i_arith_setup(struct SL_I_Runtime *rt,
    const struct SL_S_List *in){
    
    struct sl_i_to_int_arg data;
    void *result;
    
    /* Check if the results are all ints already. */
    data.flag = 0;
    data.rt = rt;
    SL_S_ForEach(sl_i_all_int, in, &(data.flag));
    if(data.flag == 0){
        SL_S_INCREF(in);
        return (void*)in;
    }
    /* Needs construction */
    data.flag = 0;
    result = SL_S_Map(sl_i_to_int, in, &data);
    if(rt->pending_error){
        SL_S_DECREF(result);
        return SL_S_NIL;
    }
    
    if(data.flag != 0){
        SL_S_DECREF(result);
        return SL_S_NIL;
    }
    return SL_S_MK_LIST(result);
}

/*****************************************************************************/

#define SL_I_VARIADIC_ARITHMETIC(X) \
    X(plus, +) \
    X(minus, -) \
    X(times, *) \
    X(divide, /) \
    X(bit_or, |) \
    X(bit_and, &) \
    X(bit_xor, ^)

#define SL_I_BINARY_ARITHMETIC(X) \
    X(mod, %) \
    X(shift_left, <<) \
    X(shift_right, >>)

#define SL_I_ALL_ARITHMETIC(X) \
    SL_I_VARIADIC_ARITHMETIC(X) \
    SL_I_BINARY_ARITHMETIC(X)

/*****************************************************************************/

#define SL_I_DECL_VARIADIC_ARITHMETIC(NAME, SYMBOL) \
static const void * sl_i_ ## NAME (struct SL_I_Runtime *rt, \
    const struct SL_S_List *in){ \
     \
    struct SL_S_List *list; \
    void *value; \
    int i; \
     \
    value = sl_i_arith_setup(rt, in); \
    if(SL_S_IS_NIL(value)){ \
        rt->pending_error = "Invalid args in " #SYMBOL; \
        return SL_S_NIL; \
    } \
    list = SL_S_PTR_FROM_TAG(value); \
    i = SL_X_ParseInt(list->head); \
    while((list = list->tail) != SL_S_NIL){ \
        i SYMBOL ## = SL_X_ParseInt(list->head); \
    } \
    SL_S_DECREF(list); \
    return SL_S_MK_ATOM(SL_S_IntToAtom(i, 10)); \
}

#define SL_I_DECL_BINARY_ARITHMETIC(NAME, SYMBOL) \
static const void * sl_i_ ## NAME (struct SL_I_Runtime *rt, \
    const struct SL_S_List *in){ \
     \
    void *a, *b; \
    int i; \
     \
    if(SL_S_IS_NIL(in) || \
        SL_S_IS_NIL(in->tail) || \
        !SL_S_IS_NIL(in->tail->tail)){ \
         \
        rt->pending_error = "Wrong number of args in " #SYMBOL; \
        return SL_S_NIL; \
    } \
    a = SL_I_Execute(rt, in->head, SL_S_NIL); \
    if(rt->pending_error) \
        return SL_S_NIL; \
    b = SL_I_Execute(rt, in->tail->head, SL_S_NIL); \
    if(rt->pending_error){ \
        SL_S_DECREF(a); \
        return SL_S_NIL; \
    } \
     \
    if(!(SL_S_IS_ATOM(a) && \
        SL_S_IS_ATOM(b) && \
        SL_X_IsInt(SL_S_PTR_FROM_TAG(a)) && \
        SL_X_IsInt(SL_S_PTR_FROM_TAG(b)))){ \
         \
        SL_S_DECREF(a); \
        SL_S_DECREF(b); \
        rt->pending_error = "Invalid args in " #SYMBOL; \
        return SL_S_NIL; \
    } \
    i = SL_X_ParseInt(SL_S_PTR_FROM_TAG(a)) SYMBOL SL_X_ParseInt(SL_S_PTR_FROM_TAG(b)); \
    SL_S_DECREF(a); \
    SL_S_DECREF(b); \
    return SL_S_MK_ATOM(SL_S_IntToAtom(i, 10)); \
}

SL_I_VARIADIC_ARITHMETIC(SL_I_DECL_VARIADIC_ARITHMETIC)
SL_I_BINARY_ARITHMETIC(SL_I_DECL_BINARY_ARITHMETIC)

#undef SL_I_DECL_VARIADIC_ARITHMETIC
#undef SL_I_DECL_BINARY_ARITHMETIC

/*****************************************************************************/

static void *sl_i_compare(struct SL_I_Runtime *rt,
    const void *a,
    const void *b){
    
    const void *exe_a, *exe_b;
    int i;

    exe_a = SL_I_Execute(rt, a, SL_S_NIL);
    if(rt->pending_error)
        return SL_S_NIL;
    exe_b = SL_I_Execute(rt, b, SL_S_NIL);
    if(rt->pending_error){
        SL_S_DECREF(exe_a);
        return SL_S_NIL;
    }
    
    i = SL_S_Compare(exe_a, exe_b);
    SL_S_DECREF(exe_a);
    SL_S_DECREF(exe_b);
    return (i == 0) ? SL_S_MK_ATOM(&sl_x_true) : SL_S_MK_ATOM(&sl_x_false);
}

/*****************************************************************************/

SL_S_FUNC(void) *SL_I_Execute(struct SL_I_Runtime *rt,
    const void *value,
    const struct SL_S_Atom **opt_out_hint){
    
    struct SL_S_Atom *atom;
    struct SL_S_List *list, *ret_list;
    struct SL_X_MeasureResult measure_result;
    struct SL_I_Frame *frame;
    sl_s_len_t i;

#define SL_I_OUT_HINT(WHAT) do{ \
    if(opt_out_hint) \
        *opt_out_hint = (WHAT); \
}while(0)
    
    if(SL_S_IS_NIL(value)){
        SL_I_OUT_HINT(SL_S_NIL);
        return SL_S_NIL;
    }
    if(SL_S_IS_ATOM(value)){
        atom = SL_S_PTR_FROM_TAG(value);
        if(SL_S_COMPARE_ATOMS(&sl_x_nil, atom)){
            SL_I_OUT_HINT(SL_S_NIL);
            return SL_S_NIL;
        }
        else if(SL_S_COMPARE_ATOMS(&sl_x_true, atom)){
            SL_I_OUT_HINT(&sl_x_atom_hint);
            return SL_S_MK_ATOM(&sl_x_true);
        }
        else if(SL_S_COMPARE_ATOMS(&sl_x_false, atom)){
            SL_I_OUT_HINT(&sl_x_atom_hint);
            return SL_S_MK_ATOM(&sl_x_false);
        }
        else{
            /* Search for a matching def. */
            frame = rt->frames;
            do{
                for(i = 0; i < frame->num_defs; i++){
                    if(SL_S_COMPARE_ATOMS(atom, frame->defs[i].name)){
                        value = frame->defs[i].value;
                        SL_S_INCREF(value);
                        SL_S_INCREF(frame->defs[i].hint);
                        SL_I_OUT_HINT(frame->defs[i].hint);
                        return (void*)value;
                    }
                }
            }while((frame = frame->next) != SL_S_NIL);
            
            SL_S_INCREF(value);
            SL_I_OUT_HINT(SL_S_NIL);
            return (void*)value;
        }
    }
    else if(!SL_S_IS_LIST(value)){
        SL_I_OUT_HINT(SL_S_NIL);
        SL_S_INCREF(value);
        return (void*)value;
    }
    else{
        /* Welcome to the jungle. */
        list = SL_S_PTR_FROM_TAG(value);
        
        /* Intentionally do not execute the head. That's not supported in the
         * compiler currently, so we should keep parity in the interpreter.
         */
        if(SL_S_IS_ATOM(list->head)){
            atom = SL_S_PTR_FROM_TAG(list->head);
            
            /* Comment */
            if(SL_S_COMPARE_ATOMS(&sl_x_comment, atom)){
                SL_I_OUT_HINT(SL_S_NIL);
                return SL_S_NIL;
            }
            
            /* Tick escape */
            if(!SL_S_IS_NIL(list->tail) &&
                SL_S_COMPARE_ATOMS(&sl_x_tick, atom)){
                
                if(SL_S_IS_NIL(list->tail->tail) ||
                    !SL_S_IS_NIL(list->tail->tail->tail)){
                    rt->pending_error = "Invalid arity in tick-escape";
                    return SL_S_NIL;
                }
                
                SL_S_INCREF(list->tail->head);
                SL_I_OUT_HINT(SL_S_NIL);
                return list->tail->head;
            }
            
            /* Dot escape */
            if(SL_S_COMPARE_ATOMS(&sl_x_dot, atom)){
                if(SL_S_IS_NIL(list->tail)){
                    rt->pending_error = "Invalid arity in dot-escape";
                    return SL_S_NIL;
                }
                /* All this is safe for nil. */
                SL_S_INCREF(list->tail);
                SL_I_OUT_HINT(&sl_x_list_hint);
                return SL_S_MK_LIST(list->tail);
            }
            
            /* if */
            if(SL_S_COMPARE_ATOMS(&sl_x_if, atom)){
                return sl_i_if(rt, list->tail, opt_out_hint);
            }
            
#define SL_I_TEST_ARITHMETIC(NAME, SYMBOL) \
    if(SL_S_COMPARE_ATOMS(&sl_x_ ## NAME, atom)) \
        return (void*)sl_i_ ## NAME(rt, list->tail);
            
SL_I_ALL_ARITHMETIC(SL_I_TEST_ARITHMETIC)
            
#undef SL_I_TEST_ARITHMETIC
            
#define SL_I_TEST(NAME, TESTER) \
    if(SL_S_COMPARE_ATOMS(&sl_x_is_ ## NAME, atom)){ \
        if(SL_S_IS_NIL(list->tail) || !SL_S_IS_NIL(list->tail->tail)){ \
            rt->pending_error = "Invalid arity in " #NAME "?"; \
            return SL_S_NIL; \
        } \
        value = SL_I_Execute(rt, list->tail->head, SL_S_NIL); \
        if(rt->pending_error) \
            return SL_S_NIL; \
        SL_I_OUT_HINT(&sl_x_atom_hint); \
        if(TESTER(value)){ \
            SL_S_DECREF(value); \
            return SL_S_MK_ATOM(&sl_x_true); \
        } \
        else{ \
            SL_S_DECREF(value); \
            return SL_S_MK_ATOM(&sl_x_false); \
        } \
    }
            SL_I_TEST(nil, SL_S_IS_NIL)
            SL_I_TEST(atom, SL_S_IS_ATOM)
            SL_I_TEST(list, SL_S_IS_LIST)
#define SL_I_TEST_INT(X) \
    (SL_S_IS_ATOM(X) && SL_X_IsInt(SL_S_PTR_FROM_TAG(X)) == 0)
            
            SL_I_TEST(int, SL_I_TEST_INT)
            
#undef SL_I_TEST_INT
#undef SL_I_TEST
            
            /* concat */
            if(SL_S_COMPARE_ATOMS(&sl_x_concat, atom)){
                measure_result.flag = 0;
                measure_result.len = 0;
                ret_list = SL_S_Map(SL_I_Execute2, list->tail, rt);
                if(rt->pending_error){
                    SL_S_DECREF(ret_list);
                    return SL_S_NIL;
                }
                SL_S_ForEach(SL_X_MeasureAtomsCB, ret_list, &measure_result);
                if(measure_result.flag){
                    rt->pending_error = "Invalid arguments to concat.";
                    SL_S_DECREF(ret_list);
                    return SL_S_NIL;
                }
                atom = SL_S_Malloc(sizeof(struct SL_S_Atom));
                atom->text = SL_S_Malloc(measure_result.len + 1);
                atom->ref = 1;
                atom->len = 0;
                SL_S_ForEach(SL_X_ConcatAtomsCB, ret_list, atom);
                atom->text[measure_result.len] = '\0';
                SL_S_DECREF(ret_list);
                
                SL_I_OUT_HINT(&sl_x_atom_hint);
                return SL_S_MK_ATOM(atom);
            }
            
            /* cons */
            if(SL_S_COMPARE_ATOMS(&sl_x_cons, atom)){
                if(SL_S_LengthCompare(list->tail, 2) != 0){
                    rt->pending_error = "Invalid arity in cons.";
                    return SL_S_NIL;
                }
                value = SL_I_Execute(rt, list->tail->tail->head, SL_S_NIL);
                if(rt->pending_error)
                    return SL_S_NIL;
                if(!SL_S_IS_LIST(value)){
                    SL_S_DECREF(value);
                    rt->pending_error = "Invalid arguments to cons.";
                    return SL_S_NIL;
                }
                ret_list = SL_S_Malloc(sizeof(struct SL_S_List));
                ret_list->ref = 1;
                ret_list->head = SL_I_Execute(rt, list->tail->head, SL_S_NIL);
                ret_list->tail = SL_S_PTR_FROM_TAG(value);
                if(rt->pending_error){
                    SL_S_FREE_LIST(ret_list);
                    return SL_S_NIL;
                }
                SL_I_OUT_HINT(&sl_x_list_hint);
                return SL_S_MK_LIST(ret_list);
            }
            
            /* head */
            if(SL_S_COMPARE_ATOMS(&sl_x_head, atom)){
                if(SL_S_LengthCompare(list->tail, 1) != 0){
                    rt->pending_error = "Invalid arity in head.";
                    return SL_S_NIL;
                }
                value = SL_I_Execute(rt, list->tail->head, SL_S_NIL);
                if(rt->pending_error)
                    return SL_S_NIL;
                if(!SL_S_IS_LIST(value)){
                    SL_S_DECREF(value);
                    rt->pending_error = "Invalid arguments to head.";
                    return SL_S_NIL;
                }
                ret_list = SL_S_PTR_FROM_TAG(value);
                value = ret_list->head;
                SL_S_INCREF(value);
                SL_S_DECREF(ret_list);
                SL_I_OUT_HINT(SL_S_NIL);
                return (void*)value;
            }
            
            /* tail */
            if(SL_S_COMPARE_ATOMS(&sl_x_tail, atom)){
                if(SL_S_LengthCompare(list->tail, 1) != 0){
                    rt->pending_error = "Invalid arity in tail.";
                    return SL_S_NIL;
                }
                value = SL_S_MK_LIST(list->tail->tail);
                SL_S_INCREF(value);
                SL_I_OUT_HINT(&sl_x_list_hint);
                return (void*)value;
            }
            
            /* not */
            if(SL_S_COMPARE_ATOMS(&sl_x_not, atom)){
                if(SL_S_LengthCompare(list->tail, 1) != 0){
                    rt->pending_error = "Invalid arity in not.";
                    return SL_S_NIL;
                }
                value = SL_I_Execute(rt, list->tail->head, SL_S_NIL);
                if(rt->pending_error)
                    return SL_S_NIL;
                SL_I_OUT_HINT(&sl_x_atom_hint);
                /* Very minor optimization, check for address equivalence with
                 * the boolean primitives */
                if(SL_S_PTR_FROM_TAG(value) == &sl_x_true){
                    return SL_S_MK_ATOM(&sl_x_false);
                }
                else if(SL_S_PTR_FROM_TAG(value) == &sl_x_false){
                    return SL_S_MK_ATOM(&sl_x_true);
                }
                else if(SL_S_IS_ATOM(value) &&
                    SL_S_COMPARE_ATOMS(&sl_x_true,
                        (struct SL_S_Atom*)SL_S_PTR_FROM_TAG(value))){
                    
                    SL_S_DECREF(value);
                    return SL_S_MK_ATOM(&sl_x_false);
                }
                else if(SL_S_IS_ATOM(value) &&
                    SL_S_COMPARE_ATOMS(&sl_x_false,
                        (struct SL_S_Atom*)SL_S_PTR_FROM_TAG(value))){
                    
                    SL_S_DECREF(value);
                    return SL_S_MK_ATOM(&sl_x_true);
                }
                else{
                    SL_S_DECREF(value);
                    rt->pending_error =
                        "Argument to not must be true or false.";
                    return SL_S_NIL;
                }
            }
            
            /* = */
            if(SL_S_COMPARE_ATOMS(&sl_x_eq, atom)){
                if(SL_S_LengthCompare(list->tail, 2) != 0){
                    rt->pending_error = "Invalid arith in =";
                    return SL_S_NIL;
                }
                SL_I_OUT_HINT(&sl_x_atom_hint);
                return sl_i_compare(rt,
                    list->tail->head,
                    list->tail->tail->head);
            }
            
            /* Search for binds. */
            for(i = 0; i < rt->num_binds; i++){
                if(SL_S_COMPARE_ATOMS(rt->binds[i].name, atom)){
                    if(SL_S_LengthCompare(list->tail,
                        rt->binds[i].arity) == 0){
                        
                        SL_S_INCREF(rt->binds[i].hint);
                        SL_I_OUT_HINT(rt->binds[i].hint);
                        return SL_I_Call(rt, rt->binds + i, list->tail);
                    }
                    else{
#define BIND_ARITY_ERROR "Invalid arity in "
                        rt->pending_error = rt->error_free_ptr =
                            SL_S_Malloc(
                                rt->binds[i].name->len +
                                sizeof(BIND_ARITY_ERROR));
                        SL_S_MemCopy((char*)(rt->pending_error),
                            BIND_ARITY_ERROR,
                            sizeof(BIND_ARITY_ERROR) - 1);
                        SL_S_MemCopy((char*)(rt->pending_error) +
                                sizeof(BIND_ARITY_ERROR) - 1,
                            rt->binds[i].name->text,
                            rt->binds[i].name->len);
                        ((char*)(rt->pending_error))[
                            rt->binds[i].name->len +
                            sizeof(BIND_ARITY_ERROR) - 1] = '\0';
#undef BIND_ARITY_ERROR
                        return SL_S_NIL;
                    }
                }
            }
        }
        /* Not found, throw error. */
        rt->pending_error = "Expected bind or escape.";
        return SL_S_NIL;
    }
}

/*****************************************************************************/

SL_S_FUNC(void) *SL_I_Execute2(const void *value, void *rt){
    if(((struct SL_I_Runtime*)rt)->pending_error)
        return SL_S_NIL;
    return SL_I_Execute(rt, (void*)value, SL_S_NIL);
}

/*****************************************************************************/

SL_S_FUNC(void) *SL_I_Call(struct SL_I_Runtime *rt,
    const struct SL_I_Bind *bind,
    const struct SL_S_List *args){
    
    sl_s_len_t i, e;
    int l;
    struct SL_I_Frame frame;
    const struct SL_S_List *code;
    const struct SL_S_Atom *hint;
    void *value;
    
    if(bind->is_native){
        args = SL_S_Map(SL_I_Execute2, args, rt);
        value = (bind->bind.native)(rt, args);
        SL_S_DECREF(args);
        return value;
    }
    else{
        /* Setup the stack. */
        frame.next = rt->frames;
        rt->frames = &frame;
        frame.defs = SL_S_Malloc(bind->arity * sizeof(struct SL_X_Def));
        frame.num_defs = frame.cap_defs = bind->arity;
        for(i = 0; i < bind->arity; i++){
            frame.defs[i].name = bind->args[i].name;
            /* TODO: Use the output hint from SL_I_Execute! */
            value = SL_I_Execute(rt, args->head, SL_S_NIL);
            hint = bind->args[i].hint;
            if(!(SL_S_IS_NIL(hint) || SL_S_COMPARE_ATOMS(hint, &sl_x_nil))){
                /* Validate the type. */
#define SL_I_TYPE_ERROR "Invalid argument type for func "
#define SL_I_SET_TYPE_ERROR do{ \
    rt->pending_error = \
    rt->error_free_ptr = \
        SL_S_Malloc(sizeof(SL_I_TYPE_ERROR) + bind->name->len); \
    SL_S_MemCopy(((char*)(rt->pending_error)), \
        SL_I_TYPE_ERROR, \
        sizeof(SL_I_TYPE_ERROR) - 1); \
    SL_S_MemCopy(((char*)(rt->pending_error)) + sizeof(SL_I_TYPE_ERROR) - 1, \
        bind->name->text, \
        bind->name->len); \
    ((char*)(rt->pending_error))[ \
        sizeof(SL_I_TYPE_ERROR) + bind->name->len - 1]; \
    goto type_hint_done; \
}while(0)

#define SL_I_TEST_INTEGER do{ \
    if((!SL_S_IS_ATOM(value)) || SL_X_IsInt(SL_S_PTR_FROM_TAG(value)) != 0){ \
        SL_I_SET_TYPE_ERROR; \
    } \
    l = SL_X_ParseInt(value); \
}while(0)

#define SL_I_SET_UNSIGNED(SIZE) do{ \
    if((l & ((1 << (SIZE))-1)) != l){ \
        SL_S_DECREF(value); \
        value = SL_S_IntToAtom(l & ((1 << (SIZE))-1), 10); \
    } \
}while(0)

#define SL_I_INT_BITS (sizeof(int) << 3)
#define SL_I_SET_SIGNED(SIZE) do{ \
    if((l << (SL_I_INT_BITS - (SIZE))) >> (SL_I_INT_BITS - (SIZE)) != l) { \
        SL_S_DECREF(value); \
        value = SL_S_IntToAtom( \
            (l << (SL_I_INT_BITS - (SIZE))) >> (SL_I_INT_BITS - (SIZE)), \
            10); \
    } \
}while(0)

                if(SL_S_COMPARE_ATOMS(hint, &sl_x_atom_hint)){
                    if(!SL_S_IS_ATOM(value))
                        SL_I_SET_TYPE_ERROR;
                }
                else if(SL_S_COMPARE_ATOMS(hint, &sl_x_list_hint)){
                    if(!SL_S_IS_LIST(value))
                        SL_I_SET_TYPE_ERROR;
                }
                else if(SL_S_COMPARE_ATOMS(hint, &sl_x_u8_hint)){
                    SL_I_TEST_INTEGER;
                    SL_I_SET_UNSIGNED(8);
                }
                else if(SL_S_COMPARE_ATOMS(hint, &sl_x_s8_hint)){
                    SL_I_TEST_INTEGER;
                    SL_I_SET_SIGNED(8);
                }
                else if(SL_S_COMPARE_ATOMS(hint, &sl_x_u16_hint)){
                    SL_I_TEST_INTEGER;
                    SL_I_SET_UNSIGNED(16);
                }
                else if(SL_S_COMPARE_ATOMS(hint, &sl_x_s16_hint)){
                    SL_I_TEST_INTEGER;
                    SL_I_SET_SIGNED(16);
                }
                else if(SL_S_COMPARE_ATOMS(hint, &sl_x_u32_hint) ||
                    SL_S_COMPARE_ATOMS(hint, &sl_x_s32_hint) ||
                    SL_S_COMPARE_ATOMS(hint, &sl_x_char_hint)){
                    
                    SL_I_TEST_INTEGER;
                }
                else if(hint->len >= 4 &&
                    SL_S_MemComp(hint->text, "^ptr", 4) == 0){
                    
                    if(!SL_S_IS_NIL(value) || SL_S_IS_PTR(value))
                        SL_I_SET_TYPE_ERROR;
                }
                else{
                    /* Check for a record. */
                    for(e = 0; e < rt->num_recs; e++){
                        if(hint->len - 1 == rt->recs[e].name->len &&
                            SL_S_MemComp(hint->text + 1,
                                rt->recs[e].name->text,
                                hint->len - 1) == 0){
                            /* Done. */
                            if(!(SL_S_IS_NIL(value) || SL_S_IS_PTR(value)))
                                SL_I_SET_TYPE_ERROR;
                            
                            goto type_hint_done;
                        }
                    }
                }
            }
type_hint_done:
            frame.defs[i].value = value;
            if(rt->pending_error){
                do{
                    SL_S_DECREF(frame.defs[i].value);
                }while(i-- != 0);
                return SL_S_NIL;
            }
            args = args->tail;
        }
        value = SL_S_NIL;
        for(code = bind->bind.lisp;
            code != SL_S_NIL && !rt->pending_error;
            code = code->tail){
            
            SL_S_DECREF(value);
            /* TODO: Check that the final value matches our output hint! */
            value = SL_I_Execute(rt, code->head, SL_S_NIL);
        }
        
        /* Clean up the stack. */
        for(i = 0; i < bind->arity; i++)
            SL_S_DECREF(frame.defs[i].value);
        rt->frames = frame.next;
        SL_S_Free(frame.defs);
        return value;
    }
}

/*****************************************************************************/
