/**
 * The MIT License
 * 
 * Copyright (c) 2009 kyozi0x7c00@gmail.com
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package jp.sourceforge.hhcp.statement;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import jp.sourceforge.hhcp.statement.impl.ExpressionStatement;

import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.ConstructorInvocation;
import org.eclipse.jdt.core.dom.DoStatement;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.ForStatement;
import org.eclipse.jdt.core.dom.IMethodBinding;
import org.eclipse.jdt.core.dom.IfStatement;
import org.eclipse.jdt.core.dom.Initializer;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.MethodInvocation;
import org.eclipse.jdt.core.dom.Statement;
import org.eclipse.jdt.core.dom.SuperConstructorInvocation;
import org.eclipse.jdt.core.dom.SuperMethodInvocation;
import org.eclipse.jdt.core.dom.WhileStatement;

public class ASTVisitorStatementList extends ASTVisitor {
	private StatementFactory _factory;
	private List<IStatement> _list;
	private List<IMethodDeclaration> _methods;

	public ASTVisitorStatementList() {
		_factory = new StatementFactory();
		_list = Collections.synchronizedList(new ArrayList<IStatement>());
		_methods = Collections.synchronizedList(new ArrayList<IMethodDeclaration>());
	}

	@Override
	public boolean visit(Initializer node) {
		add(_factory.newConstructor());
		eval(node.getBody());
		add(_factory.newEndConstructor());

		List<IStatement> statement;
		int fromIndex = _list.lastIndexOf(_factory.newConstructor());
		int toIndex = _list.lastIndexOf(_factory.newEndConstructor());

		statement = Collections.synchronizedList(new ArrayList<IStatement>(_list.subList(fromIndex+1, toIndex)));
		for (int i = toIndex; i >= fromIndex; i--) {
			_list.remove(i);
		}
		if (statement.size() > 0) {
			addMethod(statement);
		}

		return false;
	}

	@Override
	public boolean visit(MethodDeclaration node) {
		add(_factory.newMethod());
		eval(node.getBody());
		add(_factory.newEndMethod());

		List<IStatement> statement;
		int fromIndex = _list.lastIndexOf(_factory.newMethod());
		int toIndex = _list.lastIndexOf(_factory.newEndMethod());

		statement = Collections.synchronizedList(new ArrayList<IStatement>(_list.subList(fromIndex + 1, toIndex)));
		for (int i = toIndex; i >= fromIndex; i--) {
			_list.remove(i);
		}
		if (statement.size() > 0) {
			addMethod(statement);
		}

		return false;
	}

	@Override
	public boolean visit(ConstructorInvocation node) {
		eval(node.resolveConstructorBinding());

		return true;
	}

	@Override
	public boolean visit(MethodInvocation node) {
		eval(node.resolveMethodBinding());

		return true;
	}

	@Override
	public boolean visit(SuperConstructorInvocation node) {
		eval(node.resolveConstructorBinding());

		return true;
	}

	@Override
	public boolean visit(SuperMethodInvocation node) {
		eval(node.resolveMethodBinding());

		return true;
	}

	@Override
	public boolean visit(ForStatement node) {
		add(_factory.newLoop());
//		for (Object expr : node.initializers()) {
//			eval((Expression) expr);
//		}
		add(_factory.newExpression());
		eval(node.getExpression());
		add(_factory.newEndExpression());
		eval(node.getBody());
//		for (Object expr : node.updaters()) {
//			eval((Expression) expr);
//		}
		add(_factory.newEndLoop());
		
		createForStatement();

		return false;
	}
	
	private void createForStatement() {
		ExpressionStatement expression = null;
		List<IStatement> statement;
		int fromIndex = _list.lastIndexOf(_factory.newLoop());
		int toIndex = _list.lastIndexOf(_factory.newEndLoop());
		int exprIndex = _list.lastIndexOf(_factory.newExpression());
		int endExprIndex = _list.lastIndexOf(_factory.newEndExpression());
		
		if (exprIndex+2 == endExprIndex) {		
			expression = (ExpressionStatement) _list.get(exprIndex+1);
		}
		statement = Collections.synchronizedList(new ArrayList<IStatement>(_list.subList(endExprIndex+1, toIndex)));
		for (int i = toIndex; i >= fromIndex; i--) {
			_list.remove(i);
		}
		
		if (expression == null) {
			return;
		}		
		if (statement.size() == 0) {
			return;
		}
		
		add(_factory.newLoopStatement(expression, statement));
	}

	@Override
	public boolean visit(WhileStatement node) {
		add(_factory.newLoop());
		add(_factory.newExpression());
		eval(node.getExpression());
		add(_factory.newEndExpression());
		eval(node.getBody());
		add(_factory.newEndLoop());

		createWhileStatement();
		
		return false;
	}
	
	private void createWhileStatement() {
		ExpressionStatement expression = null;
		List<IStatement> statement;
		int fromIndex = _list.lastIndexOf(_factory.newLoop());
		int toIndex = _list.lastIndexOf(_factory.newEndLoop());
		int exprIndex = _list.lastIndexOf(_factory.newExpression());
		int endExprIndex = _list.lastIndexOf(_factory.newEndExpression());

		if (exprIndex+2 == endExprIndex) {
			expression = (ExpressionStatement) _list.get(exprIndex+1);
		}
		statement = Collections.synchronizedList(new ArrayList<IStatement>(_list.subList(endExprIndex+1, toIndex)));
		for (int i = toIndex; i >= fromIndex; i--) {
			_list.remove(i);
		}
		
		if (expression == null) {
			return;
		}
		if (statement.size() == 0) {
			return;
		}
		
		add(_factory.newLoopStatement(expression, statement));
	}

	@Override
	public boolean visit(DoStatement node) {
		add(_factory.newLoop());
		eval(node.getBody());
		add(_factory.newExpression());
		eval(node.getExpression());
		add(_factory.newEndExpression());
		add(_factory.newEndLoop());

		createDoWhileStatement();

		return false;
	}
	
	private void createDoWhileStatement() {
		ExpressionStatement expression = null;
		List<IStatement> statement;
		int fromIndex = _list.lastIndexOf(_factory.newLoop());
		int toIndex = _list.lastIndexOf(_factory.newEndLoop());
		int exprIndex = _list.lastIndexOf(_factory.newExpression());
		int endExprIndex = _list.lastIndexOf(_factory.newEndExpression());

		if (exprIndex+2 == endExprIndex) {
			expression = (ExpressionStatement) _list.get(exprIndex+1);
		}
		statement = Collections.synchronizedList(new ArrayList<IStatement>(_list.subList(fromIndex+1, exprIndex)));
		for (int i = toIndex; i >= fromIndex; i--) {
			_list.remove(i);
		}
		if (expression == null) {
			return;
		}
		if (statement.size() == 0) {
			return;
		}
		add(_factory.newLoopStatement(expression, statement));
	}

	@Override
	public boolean visit(IfStatement node) {
		add(_factory.newIf());
		add(_factory.newExpression());
		eval(node.getExpression());
		add(_factory.newEndExpression());
		eval(node.getThenStatement());
		if (node.getElseStatement() != null) {
			add(_factory.newElse());
			eval(node.getElseStatement());
		}
		add(_factory.newEndIf());
		
		createIfStatement();

		return false;
	}

	private void createIfStatement() {
		ExpressionStatement expression = null;
		List<IStatement> thenStatement;
		List<IStatement> elseStatement;
		int ifIndex = _list.lastIndexOf(_factory.newIf());
		int elseIndex = _list.lastIndexOf(_factory.newElse());
		int endIfIndex = _list.lastIndexOf(_factory.newEndIf());
		int exprIndex = _list.lastIndexOf(_factory.newExpression());
		int endExprIndex = _list.lastIndexOf(_factory.newEndExpression());
		
		if (exprIndex+2 == endExprIndex) {
			expression = (ExpressionStatement) _list.get(exprIndex+1);
		}		
		if (elseIndex != -1 && ifIndex < elseIndex) {
			thenStatement = Collections.synchronizedList(new ArrayList<IStatement>(_list.subList(endExprIndex+1, elseIndex)));
			elseStatement = Collections.synchronizedList(new ArrayList<IStatement>(_list.subList(elseIndex+1, endIfIndex)));
		} else {
			thenStatement = Collections.synchronizedList(new ArrayList<IStatement>(_list.subList(endExprIndex+1, endIfIndex)));
			elseStatement = Collections.synchronizedList(new ArrayList<IStatement>());
		}
		for (int i = endIfIndex; i >= ifIndex; i--) {
			_list.remove(i);
		}
		
		if (expression == null) {
			return;
		}		
		if (thenStatement.size() == 0) {
			return;
		}
		if (elseStatement.size() == 0) {
			elseStatement = null;
		}
		add(_factory.newIfStatement(expression, thenStatement, elseStatement));
	}
	
//	@Override
//	public boolean visit(TryStatement node) {
//		add(_factory.newTry());
//		eval(node.getBody());
//		for (Object catchClause : node.catchClauses()) {
//			add(_factory.newCatch());
//			eval((CatchClause) catchClause);
//			add(_factory.newEndCatch());
//		}
//		if (node.getFinally() != null) {
//			add(_factory.newFinally());
//			eval(node.getFinally());
//		}
//		add(_factory.newEndTry());
//		
//		List<IStatement> statement;
//		int fromIndex = _list.lastIndexOf(_factory.newTry());
//		int toIndex = _list.lastIndexOf(_factory.newEndTry());
//		
//		statement = new ArrayList<IStatement>(_list.subList(fromIndex + 1,
//				toIndex));
//		for (int i = toIndex; i >= fromIndex; i--) {
//			_list.remove(i);
//		}
//		
//		if (statement.size() > 0) {
//			add(_factory.newTryStatement(statement));
//		}
//		
//		return false;
//	}
	
	private void eval(IMethodBinding binding) {
		if (binding == null)
			return;

		add(_factory.newMethodStatement(binding));
	}

	private void eval(Statement stmt) {
		if (stmt == null)
			return;

		stmt.accept(this);
	}

	private void eval(Expression expression) {
		if (expression == null)
			return;
		
		switch (expression.getNodeType()) {
		case ASTNode.INFIX_EXPRESSION:
			break;
		case ASTNode.INSTANCEOF_EXPRESSION:
			break;
		case ASTNode.METHOD_INVOCATION:
			break;
		case ASTNode.PREFIX_EXPRESSION:
			break;
		case ASTNode.QUALIFIED_NAME:
			break;
		case ASTNode.SIMPLE_NAME:
			break;
		default:
			;
		}
		
		add(_factory.newExpressionStatement(expression));
	}

//	private void eval(CatchClause catchClause) {
//		if (catchClause == null)
//			return;
//
//		catchClause.accept(this);
//	}

	private void add(IStatement stmt) {
		_list.add(stmt);
	}
	
	private void addMethod(List<IStatement> stmt) {
		_methods.add(_factory.newMethodDeclaration(stmt));
	}

	public List<IMethodDeclaration> getMethods() {
		return _methods;
	}
}