/**
 * The MIT License
 * 
 * Copyright (c) 2009 kyozi0x7c00@gmail.com
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package jp.sourceforge.hhcp.statement;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.ConstructorInvocation;
import org.eclipse.jdt.core.dom.DoStatement;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.ForStatement;
import org.eclipse.jdt.core.dom.IMethodBinding;
import org.eclipse.jdt.core.dom.IfStatement;
import org.eclipse.jdt.core.dom.Initializer;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.MethodInvocation;
import org.eclipse.jdt.core.dom.Statement;
import org.eclipse.jdt.core.dom.SuperConstructorInvocation;
import org.eclipse.jdt.core.dom.SuperMethodInvocation;
import org.eclipse.jdt.core.dom.WhileStatement;

public class ASTVisitorList extends ASTVisitor {
	private StatementFactory _factory;
	private List<IStatement> _list;
	private List<IMethodDeclaration> _methods;

	public ASTVisitorList() {
		_factory = new StatementFactory();
		_list = Collections.synchronizedList(new ArrayList<IStatement>());
		_methods = Collections
				.synchronizedList(new ArrayList<IMethodDeclaration>());
	}

	@Override
	public boolean visit(Initializer node) {
		add(_factory.newConstructor());
		eval(node.getBody());
		add(_factory.newEndConstructor());

		List<IStatement> statement;
		int fromIndex = _list.lastIndexOf(_factory.newConstructor());
		int toIndex = _list.lastIndexOf(_factory.newEndConstructor());

		statement = Collections.synchronizedList(new ArrayList<IStatement>(
				_list.subList(fromIndex + 1, toIndex)));
		for (int i = toIndex; i >= fromIndex; i--) {
			_list.remove(i);
		}
		if (statement.size() > 0) {
			addMethod(statement);
		}

		return false;
	}

	@Override
	public boolean visit(MethodDeclaration node) {
		add(_factory.newMethod());
		eval(node.getBody());
		add(_factory.newEndMethod());

		List<IStatement> statement;
		int fromIndex = _list.lastIndexOf(_factory.newMethod());
		int toIndex = _list.lastIndexOf(_factory.newEndMethod());

		statement = Collections.synchronizedList(new ArrayList<IStatement>(
				_list.subList(fromIndex + 1, toIndex)));
		for (int i = toIndex; i >= fromIndex; i--) {
			_list.remove(i);
		}
		if (statement.size() > 0) {
			addMethod(statement);
		}

		return false;
	}

	@Override
	public boolean visit(ConstructorInvocation node) {
		eval(node.resolveConstructorBinding());

		return true;
	}

	@Override
	public boolean visit(MethodInvocation node) {
		eval(node.resolveMethodBinding());

		return true;
	}

	@Override
	public boolean visit(SuperConstructorInvocation node) {
		eval(node.resolveConstructorBinding());

		return true;
	}

	@Override
	public boolean visit(SuperMethodInvocation node) {
		eval(node.resolveMethodBinding());

		return true;
	}

	@Override
	public boolean visit(ForStatement node) {
		add(_factory.newLoop());
		for (Object expr : node.initializers()) {
			eval((Expression) expr);
		}
		eval(node.getExpression());
		eval(node.getBody());
		for (Object expr : node.updaters()) {
			eval((Expression) expr);
		}
		add(_factory.newEndLoop());

		return false;
	}

	@Override
	public boolean visit(WhileStatement node) {
		add(_factory.newLoop());
		eval(node.getExpression());
		eval(node.getBody());
		add(_factory.newEndLoop());

		return false;
	}

	@Override
	public boolean visit(DoStatement node) {
		add(_factory.newLoop());
		eval(node.getBody());
		eval(node.getExpression());
		add(_factory.newEndLoop());

		return false;
	}

	@Override
	public boolean visit(IfStatement node) {
		add(_factory.newIf());
		eval(node.getExpression());
		eval(node.getThenStatement());
		if (node.getElseStatement() != null) {
			add(_factory.newElse());
			eval(node.getElseStatement());
		}
		add(_factory.newEndIf());

		return false;
	}

	private void eval(IMethodBinding binding) {
		if (binding == null)
			return;

		add(_factory.newMethodStatement(binding));
	}

	private void eval(Statement stmt) {
		if (stmt == null)
			return;

		stmt.accept(this);
	}

	private void eval(Expression expression) {
		if (expression == null)
			return;

		expression.accept(this);
	}

	private void add(IStatement stmt) {
		_list.add(stmt);
	}

	private void addMethod(List<IStatement> stmt) {
		_methods.add(_factory.newMethodDeclaration(stmt));
	}

	public List<IMethodDeclaration> getMethods() {
		return _methods;
	}
}