﻿/*
  ==============================================================================

   This file is part of the async
   Copyright 2005-10 by Satoshi Fujiwara.

   async can be redistributed and/or modified under the terms of the
   GNU General Public License, as published by the Free Software Foundation;
   either version 2 of the License, or (at your option) any later version.

   async is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with async; if not, visit www.gnu.org/licenses or write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
   Boston, MA 02111-1307 USA

  ==============================================================================
*/
#include "StdAfx.h"
#include <commctrl.h>

#if _DEBUG
#define _CRTDBG_MAP_ALLOC
#include <crtdbg.h>
#define new new(_NORMAL_BLOCK, __FILE__, __LINE__)
#endif

#include "message_loop.h"
#include "sf_com.h"
#include "application.h"
#include "dout.h"
#include "async_reader.h"

#ifndef HINST_THISCOMPONENT
EXTERN_C IMAGE_DOS_HEADER __ImageBase;
#define HINST_THISCOMPONENT ((HINSTANCE)&__ImageBase)
#endif


using namespace std;

namespace sf {
#ifdef _DEBUG
  std::wstring application::app_id_(L"SF.async_debug");
#else
  std::wstring application::app_id_(L"SF.async");
#endif

  application::application() : 
    event_output_(::CreateEventEx(NULL, NULL, 0, EVENT_MODIFY_STATE | SYNCHRONIZE)),
    event_reader_(::CreateEventEx(NULL, NULL, 0, EVENT_MODIFY_STATE | SYNCHRONIZE)),
    not_enqueue_(false),output_counter_(0),repeat_mode_(false),exclusive_mode_(false)
  {
    instance_handle_ = HINST_THISCOMPONENT;
    status_.store(player_stop);
  }

  application::~application()
  {
      change_status(player_exit,boost::memory_order_release);

      if(output_thread_.joinable())
      {
        output_thread_.join();
      }

      if(reader_thread_.joinable())
      {
        reader_thread_.join();
      }
  };

  int application::execute(HINSTANCE hInstance,
    HINSTANCE hPrevInstance,
    LPTSTR    lpCmdLine,
    int       nCmdShow)
  {
#ifdef _DEBUG
    ::_CrtSetDbgFlag(_CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF);
#endif
    UNREFERENCED_PARAMETER(hPrevInstance);
    UNREFERENCED_PARAMETER(lpCmdLine);
    std::wcout.imbue(std::locale(""));

    // 2重起動の防止
    SECURITY_DESCRIPTOR sd;
    InitializeSecurityDescriptor(&sd,SECURITY_DESCRIPTOR_REVISION);
    SetSecurityDescriptorDacl(&sd, TRUE, 0, FALSE);	    
    SECURITY_ATTRIBUTES sec_attr;
    sec_attr.nLength = sizeof (sec_attr);
    sec_attr.lpSecurityDescriptor = &sd;
    sec_attr.bInheritHandle = TRUE; 
#ifdef _DEBUG 
    sf::handle_holder handle(::CreateMutex(&sec_attr, FALSE, _T("async_mutex_debug")));
#else
    sf::handle_holder handle(::CreateMutex(&sec_attr, FALSE, _T("async_mutex")));
#endif

    if(NULL == handle.get() || ::GetLastError() == ERROR_ALREADY_EXISTS)
    {
      return 0;
    }

    // プロセスの優先順位
    SetPriorityClass(GetCurrentProcess(), HIGH_PRIORITY_CLASS);

    // コモンコントロールの初期化 
    static const INITCOMMONCONTROLSEX common_ctrls =
    {
      sizeof(INITCOMMONCONTROLSEX),
      ICC_STANDARD_CLASSES | ICC_BAR_CLASSES
    };

    InitCommonControlsEx(&common_ctrls);

    // COMの初期化
    sf::com_initialize init();

    // アプリケーションIDの登録
    sf::throw_if_err<application::exception>()(SetCurrentProcessExplicitAppUserModelID(app_id_.c_str()));
    timeBeginPeriod(1);

    // ウィンドウの作成
    window_ = sf::create_toplevel_window(
      std::wstring(L"async"),std::wstring(L"asyncテスト"));

    // ファイルリーダースレッドの起動
    reader_thread_ = boost::thread(boost::bind(&application::reader_thread_main,this));
    // wasapi出力スレッドの起動
    output_thread_ = boost::thread(boost::bind(&application::output_thread_main,this));

 //   bool a = ringbuffer_.is_lock_free();
//    wdout << ( ? L"**** true ****":L"---- false ----") << std::endl;
    //sf::dialogbox(L"asyncテスト",L"asyncテスト");
    // メッセージループの実行
    // WPARAM ret = sf::run_message_loop()();

    WPARAM ret = sf::dialog_message_loop(reinterpret_cast<HWND>(window_->raw_handle()))();
    timeEndPeriod(1);
    return ret;
  }

  void application::set_play_position(uint64_t pos)
  {
    read_position_ = pos;
    status_backup_ = status_.load();
    change_status(player_seek);
  }

  void application::change_status(uint32_t v,boost::memory_order o)
  {
    status_.store(v,o);
    SetEvent(event_output_.get());
    SetEvent(event_reader_.get());
  };

  //void application::change_status_without_event(uint32_t v,boost::memory_order o)
  //{
  //  status_.store(v,o);
  //};

  void application::reader_thread_main()
  {
    // COMの初期化
    sf::com_initialize init();
    while(true)
    {
      // イベントを待つ
      WaitForSingleObject(event_reader_.get(),INFINITE);
      switch(status_.load(boost::memory_order_acquire))
      {
      case player_exit:
        goto loop_end;
        break;
      case player_play:
        {
          while(status_.load(boost::memory_order_acquire) == player_play)
          {
            if(reader_->more_data_available() || not_enqueue_)
            {
              if(!not_enqueue_){              
                uint32_t size = reader_->data_bytes_remaining() >  wasapi_->get_buffer_byte_size() ?  wasapi_->get_buffer_byte_size() : reader_->data_bytes_remaining();
                if(size == 0 && repeat_mode_)
                {
                  reader_->reset_data_position();
                  size = reader_->data_bytes_remaining() >  wasapi_->get_buffer_byte_size() ?  wasapi_->get_buffer_byte_size() : reader_->data_bytes_remaining();
                }
                reader_->read_data(read_buffer_[read_index_].get(),sizeof(uint8_t) * size);

                // ここに変換処理を入れる

                reader_->wait();
                if(size < wasapi_->get_buffer_byte_size())
                {
                  memset(read_buffer_[read_index_].get() + size,0,wasapi_->get_buffer_byte_size() - size);
                }
              }

              not_enqueue_ = false;
              while(!ringbuffer_.enqueue(read_buffer_[read_index_].get()))
              {
                if(status_.load() != player_play)
                {
                  if(status_.load(boost::memory_order_relaxed) == player_pause)
                  {
                    not_enqueue_ = true;
                  }
                  break;
                } else {
                  Sleep(1);
                }
              }

              ;
#ifdef _DEBUG
              wdout << boost::wformat(L"index:%d address:%x 差分:%x") 
                % read_index_ % read_buffer_[read_index_].get()
                % (read_buffer_[(read_index_ + 1) & (read_buffer_.size() - 1)].get() - read_buffer_[read_index_].get())
                << std::endl;
#endif
              if(!not_enqueue_){
                read_index_ = (++read_index_) & (read_buffer_.size() - 1);
              }

            } else {
              status_.store(player_end);
              break;
            }
          }
        }
        break;
      case player_pause:
#ifdef _DEBUG
        wdout << boost::wformat(L"**Pause** index:%d address:%x") % read_index_ % read_buffer_[read_index_].get() << std::endl;
#endif
        break;
      case player_prep_seek:

        break;
      case player_seek:
        reader_->seek(read_position_);
        ringbuffer_.reset();
        output_counter_ = read_position_;
        read_index_ = 0;
        change_status(status_backup_);
        break;
      case player_prep_stop:
        break;
      case player_stop:
        reader_->reset_data_position();
        ringbuffer_.reset();
        output_counter_ = 0;
        change_status(player_ready);
        break;
      }
      Sleep(1);
//      wdout << L"reader_threadは起動中" << std::endl;
    }
loop_end:
#ifdef _DEBUG
    wdout << L"##### reader_threadは終了！" << std::endl;
#endif
  ;};

  /// WASAPI出力スレッド
  void application::output_thread_main()
  {
    {
    // COMの初期化
    sf::com_initialize init();

    // MMCSSの初期化
    sf::av_mm_thread_characteristics avmm(wstring(L"Pro Audio"));
    avmm.set_priority(AVRT_PRIORITY::AVRT_PRIORITY_HIGH);

    while(true)
    { 
      //イベントを待つ
      WaitForSingleObject(event_output_.get(),INFINITE);
      switch(status_.load(boost::memory_order_acquire))
      {
      case player_exit:
        goto loop_end;
        break;
      case player_play:
        {BYTE* buffer;
//        static BYTE *buffer_backup = 0;
        uint32_t status;
//        uint32_t buffer_count = 0;
        while((status = status_.load(boost::memory_order_acquire)) == player_play)
        {
          if(ringbuffer_.dequeue(buffer))
          {
            wasapi_->play_buffer(buffer);
//            sf::wdout << boost::wformat(L"play_buffer:%d address:%x") % (buffer_count) % buffer << std::endl;
//            sf::wdout << boost::wformat(L"差分:%x") % (buffer - buffer_backup) << std::endl;
//            buffer_backup = buffer;
//            buffer_count = (++buffer_count) & 0x7f;
            output_counter_ += wasapi_->get_buffer_byte_size();
            if(output_counter_ > reader_->total_data_bytes() && repeat_mode_)
            {
              output_counter_ -= reader_->total_data_bytes();
            };
          } else {
            Sleep(0);
          }
        }

        if(status == player_end)
        {
          while(ringbuffer_.dequeue(buffer))
          {
            wasapi_->play_buffer(buffer);
            output_counter_ += wasapi_->get_buffer_byte_size();
          }
          stop();
        }
        }

        break;
      case player_pause:
        wasapi_->stop();
        break;
      case player_prep_seek:
        wasapi_->stop();
        change_status(player_seek);
        break;
      case player_seek:
        break;
      case player_prep_stop:
        wasapi_->stop();
        change_status(player_stop);
        break;
      case player_stop:
        break;
      }
      Sleep(0);
    }
loop_end:
#ifdef _DEBUG
    wdout << L"***** output_threadは終了!" << std::endl;
#endif
    ;}
;  };

  void application::setup(const std::wstring& file_path)
  {
    try {
      source_file_path_ = file_path;
//      reader_.reset(new wave_file_reader(file_path,false));
      reader_.reset(new async_reader(file_path,repeat_mode_));
      wasapi_setup();
      read_index_ = 0;
      //
      for(int i = 0,size = read_buffer_.size();i < size;++i)
      {
        read_buffer_[i].reset(reinterpret_cast<uint8_t*>(_aligned_malloc(sizeof(uint8_t) * wasapi_->get_buffer_byte_size(),16)));
      }
      status_.store(player_ready);
      window_->ready();
    } catch (sf::exception& e )
    {
      window_->message_box(wstring(L"エラーが発生しました。"),e.what_str());
    }
  };

  void application::pause()
  {
    if(status_.load() == player_pause)
    {
      change_status(player_play);
      window_->play();
    } else {
      change_status(player_pause);
      window_->pause();
    }
  }

  void application::play()
  {
    change_status(player_play);
    window_->play();
  }

  void application::stop()
  {
    change_status(player_prep_stop);
    window_->stop();
  }

  uint32_t application::get_status(boost::memory_order o)
  {
    return status_.load(o);
  }

  void application::exclusive_mode(bool v)
  {
    assert(get_status() == player_stop || get_status() == player_ready);
    exclusive_mode_ = v;
    if(wasapi_)
    {
      wasapi_setup();
    }
  };

  void application::wasapi_setup()
  {
    if(wasapi_)
    {
      wasapi_.reset();
    }
    if(exclusive_mode_){
      wasapi_.reset(new sf::wasapi_exclusive_timer(reader_->get_wave_format()));
    } else {
      wasapi_.reset(new sf::wasapi_shared_timer(reader_->get_wave_format()));
    }
    if(!wasapi_->is_enabled())
    {
      throw sf::win32_error_exception(*wasapi_->result());
    }
  }

}

