#pragma once

#include "StdAfx.h"

#include <comdef.h>
#include <comip.h>
#include <comutil.h>
#include <shlobj.h>
#include <oleauto.h>
#include <shellapi.h>

#include <string>
#include <list>
#include <map>
#include <iterator>

#include "SSAPI.h"
#include "file.h"
#include "safearray.h"
#include "exception.h"
#include "unknown.h"
#include "jscript.h"
#include "window.h"
#include "notify.h"
#include "infoz.h"

class EventsImpl : public IVSSEvents, private UnknownImpl, private MessageWindow::Listener {
private:
	friend struct Global;
	struct Global : public IDispatch {
		EventsImpl *that;

		HRESULT __stdcall QueryInterface(REFIID riid, void **ppv) throw () try {
			if (!ppv) throw _com_error(E_POINTER);
			*ppv = nullptr;
			if (::IsEqualIID(riid, IID_IUnknown)) {
				*ppv = static_cast<IUnknown *>(this);
			} else if (::IsEqualIID(riid, IID_IDispatch)) {
				*ppv = static_cast<IDispatch *>(this);
			} else {
				throw _com_error(E_NOINTERFACE);
			}
			this->AddRef(/* in vain */);
			return S_OK;
		} catch (_com_error &hr) {
			return hr.Error();
		} catch (...) {
			return E_UNEXPECTED;
		}

		ULONG __stdcall AddRef(void) throw() { return 1; }

		ULONG __stdcall Release(void) throw() { return 1; }

		HRESULT __stdcall GetTypeInfoCount(UINT *pctinfo) throw () try {
			if (!pctinfo) throw _com_error(E_POINTER);
			*pctinfo = 0;
			return S_OK;
		} catch (_com_error &hr) {
			return hr.Error();
		} catch (...) {
			return E_UNEXPECTED;
		}

		HRESULT __stdcall GetTypeInfo(UINT iTInfo, LCID lcid, ITypeInfo **ppTInfo) throw () try {
			throw _com_error(E_NOTIMPL);
		} catch (_com_error &hr) {
			return hr.Error();
		} catch (...) {
			return E_UNEXPECTED;
		}

		HRESULT __stdcall GetIDsOfNames(REFIID riid, LPOLESTR *rgszNames, UINT cNames, LCID lcid, DISPID *rgDispId) throw () try {
			if (!rgszNames) throw _com_error(E_INVALIDARG);
			if (!rgDispId) throw _com_error(E_POINTER);

			if (lstrcmpiW(*rgszNames, L"setTimeout") == 0) {
				*rgDispId = dispidSETTIMEOUT;
			} else if (lstrcmpiW(*rgszNames, L"clearTimeout") == 0) {
				*rgDispId = dispidCLEARTIMEOUT;
			} else if (lstrcmpiW(*rgszNames, L"alert") == 0) {
				*rgDispId = dispidALERT;
//			} else if (lstrcmpiW(*rgszNames, L"balloonAdd") == 0) {
//				*rgDispId = dispidBALLOONADD;
			} else {
				throw _com_error(DISP_E_UNKNOWNNAME);
			}
			return S_OK;
		} catch (_com_error &hr) {
			return hr.Error();
		} catch (...) {
			return E_UNEXPECTED;
		}

		HRESULT __stdcall Invoke(DISPID dispIdMember, REFIID /* riid */, LCID /* lcid */, WORD wFlags, DISPPARAMS *pDispParams, VARIANT *pVarResult, EXCEPINFO * /* pExcepInfo */, UINT * /* puArgErr */) throw () try {
			if ((dispIdMember == dispidSETTIMEOUT) && ((wFlags | DISPATCH_METHOD) != 0)) {
				if (pDispParams->cArgs != 2) throw _com_error(E_INVALIDARG);
				_variant_t m, e;
				_com_check(::VariantChangeType(m.GetAddress(), &(pDispParams->rgvarg[0]), 0, VT_I4));
				_com_check(::VariantChangeType(e.GetAddress(), &(pDispParams->rgvarg[1]), 0, VT_BSTR));
				int r = that->SetTimeout(e.bstrVal, m.intVal);
				if (pVarResult) {
					_com_check(::VariantClear(pVarResult));
					pVarResult->vt = VT_I4;
					pVarResult->intVal = r;
				}
				return S_OK;
			} else if ((dispIdMember == dispidCLEARTIMEOUT) && (wFlags == DISPATCH_METHOD)) {
				if (pDispParams->cArgs != 1) throw _com_error(E_INVALIDARG);
				_variant_t i;
				_com_check(::VariantChangeType(i.GetAddress(), &(pDispParams->rgvarg[0]), 0, VT_BSTR));
				that->ClearTimeout(i.intVal);
				return S_OK;
			} else if ((dispIdMember == dispidALERT) && (wFlags == DISPATCH_METHOD)) {
				if (pDispParams->cArgs != 1) throw _com_error(E_INVALIDARG);
				_variant_t b;
				_com_check(::VariantChangeType(b.GetAddress(), &(pDispParams->rgvarg[0]), 0, VT_BSTR));
				that->BalloonAdd(b.bstrVal);
				return S_OK;
//			} else if ((dispIdMember == dispidBALLOONADD) && (wFlags == DISPATCH_METHOD)) {
//				if (pDispParams->cArgs != 1) throw _com_error(E_INVALIDARG);
//				_variant_t b;
//				_com_check(::VariantChangeType(b.GetAddress(), &(pDispParams->rgvarg[0]), 0, VT_BSTR));
//				that->BalloonAdd(b.bstrVal);
//				return S_OK;
			}
			throw _com_error(E_NOTIMPL);
		} catch (_com_error &hr) {
			return hr.Error();
		} catch (...) {
			return E_UNEXPECTED;
		}
	};

private:
	_com_ptr_t<_com_IIID<IVSS, &IID_IVSS>> VSS;
	MessageWindow window;
	JScript jscript;
	Global global;
	NOTIFYICONDATAW notifyicondata;
	NotifyArea notifyarea;
	InfoList infolist;
	std::map<int, std::wstring> timeouts;

private:
	static BOOL __stdcall EnumResNameProc(HMODULE hModule, LPCTSTR lpszType, LPTSTR lpszName, LONG_PTR lParam) {
		HICON *result = reinterpret_cast<HICON *>(lParam);
		*result = ::LoadIcon(hModule, lpszName);
		return FALSE;
	}

	static HICON GetVSSIcon(void) {
		HICON result = 0;
		HINSTANCE h = ::GetModuleHandle(0);
		::EnumResourceNames(h, RT_GROUP_ICON, EnumResNameProc, reinterpret_cast<LONG_PTR>(&result));
		if (result == 0) throw _win32_error(::GetLastError());
		return result;
	}

private:
	enum {
		dispidSETTIMEOUT    = 0x1001L,
		dispidCLEARTIMEOUT  = 0x1002L,
		dispidALERT         = 0x1003L,
//		dispidBALLOONADD    = 0x1004L,
	};

	static const UINT WM_NOTIFYICONCALLBACK = WM_APP + 100;

	struct Constant {
		std::wstring JScriptFile;
		UINT TaskbarCreated;
		Constant(void) {
			wchar_t appdata[MAX_PATH];
			if (!::SHGetSpecialFolderPathW(0, appdata, CSIDL_APPDATA, TRUE)) throw _com_error(E_UNEXPECTED);
			JScriptFile = std::wstring(appdata) + L"\\vssjscript\\vssjscript.js";

			TaskbarCreated = ::RegisterWindowMessage(TEXT("TaskbarCreated"));
			if (TaskbarCreated == 0) throw _win32_error(::GetLastError());
		}
	};
	static const Constant CONSTANT;

private:
	void ShowError(const std::wstring &mssg) {
		try {
			BalloonAdd(mssg);
		} catch (...) {
			::MessageBoxW(0, mssg.c_str(), nullptr, MB_OK | MB_ICONERROR | MB_TASKMODAL);
		}
	}
	void ShowError(LPCWSTR mssg) {
		ShowError(std::wstring(mssg));
	}

private:
	struct Database {
		std::wstring SrcSafeIni;
		std::wstring UserName;
	};

	static Database GetDatabase(IVSS *vss) {
		_com_ptr_t<_com_IIID<IVSSDatabase, &IID_IVSSDatabase>> db;
		_com_check(vss->get_VSSDatabase(&db));
		_bstr_t srcsafeini;
		_com_check(db->get_SrcSafeIni(srcsafeini.GetAddress()));
		_bstr_t username;
		_com_check(db->get_Username(username.GetAddress()));
		Database result;
		result.SrcSafeIni = srcsafeini;
		result.UserName = username;
		return result;
	}

	struct Item {
		std::wstring Spec;
		long VersionNnmber;
	};

	static Item GetItem(IVSSItem *item) {
		_bstr_t spec;
		_com_check(item->get_Spec(spec.GetAddress()));
		long versionNnmber;
		_com_check(item->get_VersionNumber(&versionNnmber));
		Item result;
		result.Spec = spec;
		result.VersionNnmber = versionNnmber;
		return result;
	}

private:
	int SetTimeout(const std::wstring &code, int millisecond) {
		static int count = 0;
		int id = ++count;
		timeouts[id] = code;
		try {
			if (::SetTimer(window.GetHandle(), id, millisecond, nullptr) == 0) throw _win32_error(::GetLastError());
		} catch (...) {
			timeouts.erase(id);
			throw;
		}
		return id;
	}

	void ClearTimeout(int id) {
		if (timeouts.find(id) != timeouts.end()) {
			timeouts.erase(id);
			::KillTimer(window.GetHandle(), id);
		}
	}

	void BalloonAdd(const std::wstring &mssg) {
		const std::wstring szInfo = infolist.AddText(mssg);
		strcopy(szInfo , notifyicondata.szInfo, infolist.MaxLength);

		const size_t SZINFOTITLE = sizeof(notifyicondata.szInfoTitle) / sizeof(notifyicondata.szInfoTitle[0]);
		const Database db = GetDatabase(VSS);
		strcopy(db.SrcSafeIni, notifyicondata.szInfoTitle, SZINFOTITLE);

		notifyarea.SetIcon(&notifyicondata);
	}

private:
	bool InitScript(void) {
		try {
			File f(CONSTANT.JScriptFile);
			std::wstring code = f.Read();
			jscript.Reset(code, &global);
			return true;
		} catch (_com_error hr) {
			jscript.Reset(L"", &global);
			ShowError(std::wstring(L"can't load '") + CONSTANT.JScriptFile + std::wstring(L"'"));
			return false;
		}
	}

protected:
	LRESULT WindowProc(HWND h, UINT m, WPARAM w, LPARAM l) {
		try {
			if (m == CONSTANT.TaskbarCreated) {
				notifyarea.ClearIcon();
				notifyarea.SetIcon(&notifyicondata);
			} else if (m == WM_NOTIFYICONCALLBACK) {
				if (l == WM_LBUTTONDOWN) {
					notifyarea.SetIcon(&notifyicondata);
				} else if (l == WM_RBUTTONDOWN) {
					InitScript();
				}
			} else if (m == WM_TIMER) {
				::KillTimer(h, w);
				auto found = timeouts.find(w);
				if (found != timeouts.end()) {
					const std::wstring e = found->second;
					timeouts.erase(found);
					jscript.Eval(e);
				}
			}
		} catch (_com_error &hr) {
			ShowError(hr.ErrorMessage());
		} catch (...) {
			ShowError(L"Error @ WindowProc");
		}
		return ::DefWindowProc(h, m, w, l);
	}

protected:
	virtual ~EventsImpl(void) { }

public:
	EventsImpl(IVSS *pIVSS) : VSS(pIVSS), infolist(sizeof(notifyicondata.szInfo) / sizeof(notifyicondata.szInfo[0])) {
		global.that = this;
		window.SetListener(this);
		::ZeroMemory(&notifyicondata, sizeof(notifyicondata));
		notifyicondata.cbSize   = sizeof(notifyicondata);
		notifyicondata.hWnd     = window.GetHandle();
		notifyicondata.uID      = reinterpret_cast<UINT>(this);
		notifyicondata.uFlags   = NIF_ICON | NIF_TIP | NIF_MESSAGE | NIF_INFO;
		notifyicondata.uCallbackMessage  = WM_NOTIFYICONCALLBACK;
		notifyicondata.hIcon      = GetVSSIcon();
		notifyicondata.szTip[0x0] = L'V';
		notifyicondata.szTip[0x1] = L'S';
		notifyicondata.szTip[0x2] = L'S';
		notifyicondata.szTip[0x3] = L'J';
		notifyicondata.szTip[0x4] = L'S';
		notifyicondata.szTip[0x5] = L'c';
		notifyicondata.szTip[0x6] = L'r';
		notifyicondata.szTip[0x7] = L'i';
		notifyicondata.szTip[0x8] = L'p';
		notifyicondata.szTip[0x9] = L't';
		notifyicondata.szTip[0xa] = L'\0';
		notifyicondata.dwState        = 0;
		notifyicondata.dwStateMask    = 0;
		notifyicondata.szInfo[00]     = L'\0';
		notifyicondata.uTimeout       = 0;
		notifyicondata.szInfoTitle[0] = L'\0';
		notifyicondata.dwInfoFlags    = NIIF_INFO;
		notifyarea.SetIcon(&notifyicondata);
		InitScript();
	}

public:
	HRESULT __stdcall QueryInterface(REFIID riid, void **ppv) throw () try {
		if (!ppv) throw _com_error(E_POINTER);
		*ppv = nullptr;
		if (::IsEqualIID(riid, IID_IUnknown)) { 
			*ppv = static_cast<IUnknown *>(this);
		} else if (::IsEqualIID(riid, IID_IVSSEvents)) {
			*ppv = static_cast<IVSSEvents *>(this);
		} else {
			throw _com_error(E_NOINTERFACE);
		}
		this->AddRef();
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	ULONG __stdcall AddRef(void) throw() { return UnknownImpl::AddRef(); }

	ULONG __stdcall Release(void) throw() { return UnknownImpl::Release(); }

public:
	HRESULT __stdcall BeforeAdd(IVSSItem *pIPrj, BSTR Local, BSTR Comment, VARIANT_BOOL *pbContinue) throw () try {
		if (!pbContinue) throw _com_error(E_POINTER);
		*pbContinue = VARIANT_FALSE;
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIPrj);
		try { 
			jscript.BeforeAdd(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, Local, Comment);
			*pbContinue = VARIANT_TRUE;
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall AfterAdd(IVSSItem *pIItem, BSTR Local, BSTR Comment) throw () try {
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.AfterAdd(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, Local, Comment);
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall BeforeCheckout(IVSSItem *pIItem, BSTR Local, BSTR Comment, VARIANT_BOOL *pbContinue) throw () try {
		if (!pbContinue) throw _com_error(E_POINTER);
		*pbContinue = VARIANT_FALSE;
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.BeforeCheckout(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, Local, Comment);
			*pbContinue = VARIANT_TRUE;
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall AfterCheckout(IVSSItem *pIItem, BSTR Local, BSTR Comment) throw () try {
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.AfterCheckout(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, Local, Comment);
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall BeforeCheckin(IVSSItem *pIItem, BSTR Local, BSTR Comment, VARIANT_BOOL *pbContinue) throw () try {
		if (!pbContinue) throw _com_error(E_POINTER);
		*pbContinue = VARIANT_FALSE;
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.BeforeCheckin(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, Local, Comment);
			*pbContinue = VARIANT_TRUE;
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall AfterCheckin(IVSSItem *pIItem, BSTR Local, BSTR Comment) throw () try {
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.AfterCheckin(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, Local, Comment);
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall BeforeUndoCheckout(IVSSItem *pIItem, BSTR Local, VARIANT_BOOL *pbContinue) throw () try {
		if (!pbContinue) throw _com_error(E_POINTER);
		*pbContinue = VARIANT_FALSE;
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.BeforeUndoCheckout(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, Local);
			*pbContinue = VARIANT_TRUE;
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall AfterUndoCheckout(IVSSItem *pIItem, BSTR Local) throw () try {
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.AfterUndoCheckout(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, Local);
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall BeforeRename(IVSSItem *pIItem, BSTR NewName, VARIANT_BOOL *pbContinue) throw () try {
		if (!pbContinue) throw _com_error(E_POINTER);
		*pbContinue = VARIANT_FALSE;
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.BeforeRename(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, NewName);
			*pbContinue = VARIANT_TRUE;
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall AfterRename(IVSSItem *pIItem, BSTR OldName) throw () try {
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.AfterRename(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, OldName);
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall BeforeBranch(IVSSItem *pIItem, BSTR Comment, VARIANT_BOOL *pbContinue) throw () try {
		if (!pbContinue) throw _com_error(E_POINTER);
		*pbContinue = VARIANT_FALSE;
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.BeforeBranch(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, Comment);
			*pbContinue = VARIANT_TRUE;
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall AfterBranch(IVSSItem *pIItem, BSTR Comment) throw () try {
		Database db = GetDatabase(VSS);
		Item item = GetItem(pIItem);
		try {
			jscript.AfterBranch(db.SrcSafeIni, db.UserName, item.Spec, item.VersionNnmber, Comment);
		} catch (_com_error &hr) {
			ShowError(hr.Description());
		}
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall BeginCommand(long unused, VARIANT_BOOL *pbContinue) throw () try {
		if (!pbContinue) throw _com_error(E_POINTER);
		*pbContinue = VARIANT_TRUE;
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall EndCommand(long unused) throw () try {
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall BeforeEvent(long iEvent, IVSSItem *pIItem, BSTR Str, VARIANT var, VARIANT_BOOL *pbContinue) throw () try {
		if (!pbContinue) throw _com_error(E_POINTER);
		*pbContinue = VARIANT_TRUE;
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}

	HRESULT __stdcall AfterEvent(long iEvent, IVSSItem *pIItem, BSTR Str, VARIANT var) throw () try {
		return S_OK;
	} catch (_com_error &hr) {
		return hr.Error();
	} catch (...) {
		return E_UNEXPECTED;
	}
};
