/*
 * @file  http_request.cpp
 * @brief module of HTTP Request
 * @brief HTTP Request parser
 *
 * Copyright (C) 2009  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#include "http_request.h"

/*!
 * HTTP Request constructor.
 */
http_request::http_request()
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 20,
        "in/out_function : Constructor http_request::http_request(void)");
    }
    /*------ DEBUG LOG END ------*/
}

/*!
 * HTTP Request constructor.
 * Parse HTTP request header.
 *
 * @param[in]   header  full http request header string
 */
http_request::http_request(std::string header)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 21,
        "in/out_function : Constructor http_request::http_request(std::string header) : "
        "header(%s)", header.c_str());
    }
    /*------ DEBUG LOG END ------*/
    this->parse(header);
}

/*!
 * HTTP Request destructor.
 */
http_request::~http_request()
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 22,
        "in/out_function : Destructor http_request::~http_request(void)");
    }
    /*------ DEBUG LOG END ------*/
}

/*!
 * Get method function.
 *
 * @return    method
 */
std::string http_request::method() const
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 23,
        "in_function : std::string http_request::method(void)");
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 24,
        "out_function : std::string http_request::method(void) : "
        "return(%s)", this->_method.c_str());
    }
    /*------ DEBUG LOG END ------*/

    return this->_method;
}

/*!
 * Set method function.
 * Set new method and return old method.
 *
 * @param[in]   method  new method
 * @return  old method
 */
std::string http_request::method(std::string _method)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 25,
        "in_function : std::string http_request::method(std::string _method) : "
        "_method(%s)", _method.c_str());
    }
    /*------ DEBUG LOG END ------*/

    std::string ret = this->_method;
    this->_method = _method;
    this->modified = true;

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 26,
        "out_function : std::string http_request::method(std::string _method) : "
        "return(%s)", ret.c_str());
    }
    /*------ DEBUG LOG END ------*/

    return ret;
}

/*!
 * Get request URI function.
 *
 * @return    request URI
 */
std::string http_request::request_uri() const
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 27,
        "in_function : std::string http_request::request_uri(void)");
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 28,
        "out_function : std::string http_request::request_uri(void) : "
        "return(%s)", this->_request_uri.c_str());
    }
    /*------ DEBUG LOG END ------*/

    return this->_request_uri;
}

/*!
 * Set request URI function.
 * Set new request URI and return old request URI.
 *
 * @param[in]   _request_uri    new request URI
 * @return  old request URI
 */
std::string http_request::request_uri(std::string _request_uri)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 29,
        "in_function : std::string http_request::request_uri(std::string _request_uri) : "
        "_request_uri(%s)", _request_uri.c_str());
    }
    /*------ DEBUG LOG END ------*/

    std::string ret = this->_request_uri;
    this->_request_uri = _request_uri;
    this->modified = true;

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 30,
        "out_function : std::string http_request::_request_uri(std::string _request_uri) : "
        "return(%s)", ret.c_str());
    }
    /*------ DEBUG LOG END ------*/

    return ret;
}

/*!
 * Get HTTP version function.
 *
 * @return    HTTP version
 */
std::string http_request::http_version() const
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 31,
        "in_function : std::string http_request::http_version(void)");
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 32,
        "out_function : std::string http_request::http_version(void) : "
        "return(%s)", this->_http_version.c_str());
    }
    /*------ DEBUG LOG END ------*/

    return this->_http_version;
}

/*!
 * Set HTTP version function.
 * Set new HTTP version and return old HTTP version.
 *
 * @param[in]   _http_version   new HTTP version
 * @return  old HTTP version
 */
std::string http_request::http_version(std::string _http_version)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 33,
        "in_function : std::string http_request::http_version(std::string _http_version) : "
        "_http_version(%s)", _http_version.c_str());
    }
    /*------ DEBUG LOG END ------*/

    std::string ret = this->_http_version;
    this->_http_version = _http_version;
    this->modified = true;

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 34,
        "out_function : std::string http_request::http_version(std::string _http_version) : "
        "return(%s)", ret.c_str());
    }
    /*------ DEBUG LOG END ------*/

    return ret;
}

/*!
 * Get request line function.
 *
 * @return    request line
 */
std::string http_request::request_line() const
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 35,
        "in_function : std::string http_request::request_line(void)");
    }
    /*------ DEBUG LOG END ------*/

    std::string ret = this->method() + " " + this->request_uri() + " " + this->http_version() + "\r\n";

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 36,
        "out_function : std::string http_request::request_line(void) : "
        "return(%s)", ret.c_str());
    }
    /*------ DEBUG LOG END ------*/

    return ret;
}

/*!
 * Get full HTTP request function.
 *
 * @return    HTTP request
 */
std::string http_request::as_string()
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 37,
        "in_function : std::string http_request::as_string(void)");
    }
    /*------ DEBUG LOG END ------*/

    if (this->modified)
        this->rebuild();

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 38,
        "out_function : std::string http_request::as_string(void) : "
        "return(%s)", this->raw_message.c_str());
    }
    /*------ DEBUG LOG END ------*/

    return this->raw_message;
}

/*!
 * Parse HTTP request header function.
 *
 * @param[in]   request     full HTTP request header
 */
void http_request::parse(std::string request)
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 39,
        "in_function : void http_request::parse(std::string request) : "
        "request(%s)", request.c_str());
    }
    /*------ DEBUG LOG END ------*/

    // save raw request
    this->raw_message = request;

    // parse request
    HTTP_REQUEST_POSITION pos = REQUEST_METHOD;

    /*
     * RFC2616
     *  OCTET       : 8bit data
     *  CHAR        : US-ASCII(0-127)
     *  UPALPHA     : A-Z
     *  LOALPHA     : a-z
     *  ALPHA       : UPALPHA | LOALPHA
     *  DIGIT       : 0-9
     *  HEXDIG      : A-F | a-f | DIGIT
     *  SP          : SPace(32)
     *  HT          : Horizontal Tab(9)
     *  CR          : Carriage Return(13)
     *  LF          : Line Feed(10)
     *  CTL         : ConTLol char(0-31,127)
     *  LWS         : [CRLF] 1*(SP|HT)
     *  separators  : ()<>@,;:\"/[]?={} and SP, HT
     *  token       : 1*(CHAR not CTL, separators)
     */
    std::string::iterator ptr = request.begin();
    std::string::iterator end = request.end();
    std::string::iterator start = ptr;
    std::pair<std::string, std::string> field_pair;
    try {
        while (ptr != end) {
            switch(pos) {
            /*
             * REQUEST-LINE :
             *      METHOD SP REQUEST-URI SP HTTP-VERSION CRLF
             */
            /*
             * METHOD : token
             */
            case REQUEST_METHOD:
                if (*ptr == ' ') { // METHOD end with SP
                    this->_method.assign(start, ptr);
                    pos = REQUEST_METHOD_SP;
                } else if (!isalpha(*ptr) && !isdigit(*ptr)) { // XXX not enough
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 10,
                    "Parse error: Invalid request method.(%c)", *ptr);
                    throw -1;
                }
                break;
            /*
             * REQUEST-URI : * | absolute-URI | path-absolute | authority
             *
             * RFC3986
             *  absolute-URI    : scheme ":" hier-part [ "?" query ]
             *  path-absolute   : "/" [ segment-nz *( "/" segment ) ]
             *  authority       : [ userinfo "@" ] host [ ":" port ]
             *  scheme          : ALPHA *( ALPHA | DIGIT | "+" | "-" | "." )
             *  hier-part       : "//" authority path-abempty
             *                  / path-absolute
             *                  / path-rootless
             *                  / path-empty
             *  query           : *( pchar | "/" | "?" )
             *  path-abempty    : *( "/" segment )
             *  path-rootless   : segment-nz *( "/" segment )
             *  path-empty      : no char
             *  segment-nz      : 1*pchar
             *  segment         : *pchar
             *  userinfo        : *( unreserved | pct-encoded | sub-delims | ":" )
             *  host            : IP-literal | IPv4address | reg-name
             *  port            : *DIGIT
             *  unreserved      : ALPHA | DIGIT | "-" | "." | "_" | "~"
             *  pct-encoded     : "%" HEXDIG HEXDIG
             *  sub-delims      : !$&'()*+,;=
             *  pchar           : unreserved | pct-encoded | subdelims | ":" | "@"
             *  IP-literal      : "[" ( IPv6address | IPvFuture ) "]"
             *  IPvFuture       : "v" 1*HEXDIG "." 1*( unreserved | sub-delims | ":" )
             *  IPv6address     :                            6( h16 ":" ) ls32
             *                  /                       "::" 5( h16 ":" ) ls32
             *                  / [               h16 ] "::" 4( h16 ":" ) ls32
             *                  / [ *1( h16 ":" ) h16 ] "::" 3( h16 ":" ) ls32
             *                  / [ *2( h16 ":" ) h16 ] "::" 2( h16 ":" ) ls32
             *                  / [ *3( h16 ":" ) h16 ] "::"    h16 ":"   ls32
             *                  / [ *4( h16 ":" ) h16 ] "::"              ls32
             *                  / [ *5( h16 ":" ) h16 ] "::"              h16
             *                  / [ *6( h16 ":" ) h16 ] "::"
             *  h16             : 1*4HEXDIG
             *  ls32            : ( h16 ":" h16 ) | IPv4address
             *  IPv4address     : dec-octet "." dec-octet "." dec-octet "." dec-octet
             *  dec-octet       : 0-255
             *  reg-name        : *( unreserved | pct-encoded | sub-delims )
             */
            case REQUEST_METHOD_SP:
                // Request-URI start
                // XXX not enough?
                if (*ptr == '/' || isalpha(*ptr) || isdigit(*ptr) || *ptr == '-' ||
                    *ptr == '.' || *ptr == '_' || *ptr == '~' || *ptr == ':' || 
                    *ptr == '@' || *ptr == '!' || *ptr == '$' || *ptr == '&' ||
                    *ptr == '(' || *ptr == ')' || *ptr == '*' || *ptr == '+' ||
                    *ptr == ',' || *ptr == ';' || *ptr == '=' || *ptr == '%') {
                    pos = REQUEST_REQUEST_URI;
                    start = ptr;
                } else {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 11,
                    "Parse error: Invalid request-uri.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            case REQUEST_REQUEST_URI:
                if (*ptr == ' ') { // Request-URI end with SP
                    this->_request_uri.assign(start, ptr);
                    pos = REQUEST_REQUEST_URI_SP;
                } else if (!isalpha(*ptr) && !isdigit(*ptr) && *ptr != '/' && // XXX not enough?
                    *ptr != '.' && *ptr != '=' && *ptr != '%' && *ptr != '?' &&
                    *ptr != '&' && *ptr != '+' && *ptr != '~' && *ptr != ',' && 
                    *ptr != '@' && *ptr != '!' && *ptr != '$' && *ptr != '-' &&
                    *ptr != '(' && *ptr != ')' && *ptr != '*' && *ptr != '_' &&
                    *ptr != ':' && *ptr != ';') {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 12,
                    "Parse error: Invalid request-uri.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            /*
             * HTTP-VERSION     : "HTTP" "/" 1*DIGIT "." 1*DIGIT
             */
            case REQUEST_REQUEST_URI_SP:
                // HTTP-VERSION start
                if (*ptr == 'H') {
                    pos = REQUEST_HTTP_VERSION_H;
                    start = ptr;
                } else {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 13,
                    "Parse error: Invalid http-version.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            case REQUEST_HTTP_VERSION_H:
                // HTTP-VERSION next
                if (*ptr == 'T') {
                    pos = REQUEST_HTTP_VERSION_T1;
                } else {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 14,
                    "Parse error: Invalid http-version.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            case REQUEST_HTTP_VERSION_T1:
                // HTTP-VERSION next
                if (*ptr == 'T') {
                    pos = REQUEST_HTTP_VERSION_T2;
                } else {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 15,
                    "Parse error: Invalid http-version.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            case REQUEST_HTTP_VERSION_T2:
                // HTTP-VERSION next
                if (*ptr == 'P') {
                    pos = REQUEST_HTTP_VERSION_P;
                } else {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 16,
                    "Parse error: Invalid http-version.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            case REQUEST_HTTP_VERSION_P:
                // HTTP-VERSION next
                if (*ptr == '/') {
                    pos = REQUEST_HTTP_VERSION_SLASH;
                } else {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 17,
                    "Parse error: Invalid http-version.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            case REQUEST_HTTP_VERSION_SLASH:
                // HTTP-VERSION Mejor number
                if (isdigit(*ptr)) {
                    pos = REQUEST_HTTP_VERSION_MAJOR;
                } else {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 18,
                    "Parse error: Invalid http-version.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            case REQUEST_HTTP_VERSION_MAJOR:
                // HTTP-VERSION next dot
                if (*ptr == '.') {
                    pos = REQUEST_HTTP_VERSION_DOT;
                } else if (!isdigit(*ptr)) {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 19,
                    "Parse error: Invalid http-version.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            case REQUEST_HTTP_VERSION_DOT:
                // HTTP-VERSION Minor number
                if (isdigit(*ptr)) {
                    pos = REQUEST_HTTP_VERSION_MINOR;
                } else {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 20,
                    "Parse error: Invalid http-version.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            case REQUEST_HTTP_VERSION_MINOR:
                // HTTP-VERSION end with CR
                if (*ptr == '\r') {
                    pos = REQUEST_CR;
                    this->_http_version.assign(start, ptr);
                } else if (!isdigit(*ptr)) {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 21,
                    "Parse error: Invalid http-version.(%c)", *ptr);
                    throw -1;
                }
                break;
    
            case REQUEST_CR:
                // LF only
                if (*ptr == '\n') {
                    pos = REQUEST_LF;
                    http_message::parse(std::string(ptr + 1, end));
                } else {
                    LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 22,
                    "Parse error: No LF.(%c)", *ptr);
                    throw -1;
                }
                break;
            }
            if (pos == REQUEST_LF)
                break;
            ptr++;
        }
    }
    catch (...) {
        LOGGER_PUT_LOG_ERROR(LOG_CAT_PACKET_EDIT_HTTP, 23,
        "Exception occured by parsing HTTP request.");
    }

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 40,
        "out_function : void http_request::parse(std::string request)");
    }
    /*------ DEBUG LOG END ------*/
}

/*!
 * Rebuild HTTP request header function.
 */
void http_request::rebuild()
{
    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 41,
        "in_function : void http_request::rebuild()");
    }
    /*------ DEBUG LOG END ------*/

    this->raw_message = this->request_line();
    http_message::rebuild();

    /*-------- DEBUG LOG --------*/
    if (LOG_LV_DEBUG == logger_get_log_level(LOG_CAT_PACKET_EDIT_HTTP)) {
        LOGGER_PUT_LOG_DEBUG(LOG_CAT_PACKET_EDIT_HTTP, 42,
        "out_function : void http_request::rebuild()");
    }
    /*------ DEBUG LOG END ------*/
}
