/*
 * @file  http_message.h
 * @brief HTTP Request Message Header
 *
 * Copyright (C) 2009  NTT COMWARE Corporation.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 **********************************************************************/

#ifndef __HTTP_MESSAGE_H__
#define __HTTP_MESSAGE_H__

#include <boost/multi_index_container.hpp>
#include <boost/multi_index/member.hpp>
#include <boost/multi_index/hashed_index.hpp>
#include <boost/multi_index/sequenced_index.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/detail/case_conv.hpp>
#include <boost/tokenizer.hpp>
#include <string>
#include "http_message_enum.h"
#include "logger_wrapper.h"

using boost::multi_index_container;
using namespace boost::multi_index;

struct field_map; // tag
typedef std::pair<std::string, std::string> field; // header field
typedef multi_index_container<
  field,
  indexed_by<
    sequenced<>, // header field order
    hashed_non_unique< tag<field_map>, member<field, std::string, &field::first> >
  >
> header_container;
typedef header_container::index_iterator<field_map>::type field_map_iterator;
typedef std::pair<field_map_iterator, field_map_iterator> field_range;

//! HTTP Message Class (RFC2616)
class http_message
{
protected:
    /*
     * _header : keep header fields order and fast key lookup.
     *
     * RFC2616 Sec 4.2
     * The order in which header fields with the same field-name are
     * received is therefore significant to the interpretation of
     * the combined field value, and thus a proxy MUST NOT change
     * the order of these field values when a message is forwarded.
     */
    header_container    _header;
    std::string     _body;
    std::string     incomplete;
    std::string     raw_message;
    bool            modified;
    std::string convert_upper_camel_case(std::string) const;

public:
    http_message();
    http_message( std::string );
    ~http_message();

    field_range header( std::string ) const;
    void header( std::string, std::string );
    std::string body() const;
    std::string body( std::string );

    std::string as_string();
    void parse( std::string );
    void rebuild();
};

#endif //__HTTP_MESSAGE_H__
