﻿using System;
using System.Collections;
using System.Collections.Generic;

namespace SystemNeo.Collections.Generic
{
	/// <summary>
	/// 値を取り出す際の優先度を指定できるキューです。
	/// </summary>
	/// <typeparam name="TPriority"></typeparam>
	/// <typeparam name="TValue"></typeparam>
	public class PriorityQueue<TPriority, TValue> : AbstractEnumerable<TValue>
			where TPriority : IComparable<TPriority>
	{
		#region private fields
		private readonly SortedList<TPriority, Queue<TValue>> list;
		#endregion

		// public プロパティ //

		/// <summary>
		/// 
		/// </summary>
		public int Count
		{
			get {
				int result = 0;
				foreach (Queue<TValue> queue in this.list.Values) {
					result += queue.Count;
				}
				return result;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public bool IsEmpty
		{
			get {
				return this.GetFirstQueue() == null;
			}
		}

		// public コンストラクタ //

		/// <summary>
		/// 
		/// </summary>
		public PriorityQueue()
		{
			this.list = new SortedList<TPriority, Queue<TValue>>();
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="comparer"></param>
		public PriorityQueue(IComparer<TPriority> comparer)
		{
			this.list = new SortedList<TPriority, Queue<TValue>>(comparer);
		}

		// public メソッド //

		/// <summary>
		/// 
		/// </summary>
		public void Clear()
		{
			this.list.Clear();
		}

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public TValue Dequeue()
		{
			Queue<TValue> queue = this.GetFirstQueue();
			if (queue == null) {
				throw new InvalidOperationException();
			}
			return queue.Dequeue();
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="priority"></param>
		/// <param name="value"></param>
		public void Enqueue(TPriority priority, TValue value)
		{
			if (! this.list.ContainsKey(priority)) {
				this.list.Add(priority, new Queue<TValue>());
			}
			this.list[priority].Enqueue(value);
		}

		// protected メソッド //

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		protected override IEnumerator<TValue> GetEnumeratorInternal()
		{
			foreach (Queue<TValue> queue in this.list.Values) {
				foreach (TValue value in queue) {
					yield return value;
				}
			}
		}

		// private メソッド //

		/// <summary>
		/// 空でないキューのうち優先度が最も高いものを取得します。
		/// </summary>
		/// <returns></returns>
		private Queue<TValue> GetFirstQueue()
		{
			while (this.list.Count > 0) {
				// 優先度の順にキューを調べる
				foreach (var pair in this.list) {
					// 空でないキューが見つかったら、それを返す
					if (pair.Value.Count > 0) {
						return pair.Value;
					}
					// キューが空になっていたら削除して次を探す
					this.list.Remove(pair.Key);
					break;
				}
			}
			return null;
		}
	}
}
