/*
  This program is a conversion from original Swiss Ephemeris Software.
*/
/* Copyright (C) 1997, 1998 Astrodienst AG, Switzerland.  All rights reserved.
  
  This file is part of Swiss Ephemeris Free Edition.
  
  Swiss Ephemeris is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Swiss Ephemeris Public License
  ("SEPL" or the "License") for full details.
  
  Every copy of Swiss Ephemeris must include a copy of the License,
  normally in a plain ASCII text file named LICENSE.  The License grants you
  the right to copy, modify and redistribute Swiss Ephemeris, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notices and this notice be preserved on
  all copies.

  For uses of the Swiss Ephemeris which do not fall under the definitions
  laid down in the Public License, the Swiss Ephemeris Professional Edition
  must be purchased by the developer before he/she distributes any of his
  software or makes available any product or service built upon the use of
  the Swiss Ephemeris.

  Authors of the Swiss Ephemeris: Dieter Koch and Alois Treindl

  The authors of Swiss Ephemeris have no control or influence over any of
  the derived works, i.e. over software or services created by other
  programmers which use Swiss Ephemeris functions.

  The names of the authors or of the copyright holder (Astrodienst) must not
  be used for promoting any software, product or service which uses or contains
  the Swiss Ephemeris. This copyright notice is the ONLY place where the
  names of the authors can legally appear, except in cases where they have
  given special permission in writing.

  The trademarks 'Swiss Ephemeris' and 'Swiss Ephemeris inside' may be used
  for promoting such software, products or services.
*/
/*******************************************************
house and (simple) aspect calculation
*/

package jp.co.isic.SwissEphemeris;

//import java.lang.*;
//import java.io.*;

public class SweHouse {

	static final double VERY_SMALL = 1E-10;

	double cusp[];
	double ac;
	double mc;
	double vertex;
	double equasc;
	double coasc1;
	double coasc2;
	double polasc;

	public SweHouse() {
		cusp = new double[37];
		for (int i = 0; i < 37; i++) {
			cusp[i] = 0;
		}
		ac = 0;
		mc = 0;
		vertex = 0;
		equasc = 0;
		coasc1 = 0;
		coasc2 = 0;
		polasc = 0;
	}

	final double sind(double x) {
		return Math.sin(x * SwissEphemeris.DEGTORAD);
	}
	final double cosd(double x) {
		return Math.cos(x * SwissEphemeris.DEGTORAD);
	}

	final double tand(double x) {
		return Math.tan(x * SwissEphemeris.DEGTORAD);
	}

	final double asind(double x) {
		return (Math.asin(x) * SwissEphemeris.RADTODEG);
	}

	final double acosd(double x) {
		return (Math.acos(x) * SwissEphemeris.RADTODEG);
	}

	final double atand(double x) {
		return (Math.atan(x) * SwissEphemeris.RADTODEG);
	}
	final double atan2d(double y, double x) {
		return (Math.atan2(y, x) * SwissEphemeris.RADTODEG);
	}

	/* housasp.c 
	 * cusps are returned in double cusp[13],
	 *                           or cusp[37] with house system 'G'.
	 * cusp[1...12]	houses 1 - 12
	 * additional points are returned in ascmc[10].
	 * ascmc[0] = ascendant
	 * ascmc[1] = mc
	 * ascmc[2] = armc
	 * ascmc[3] = vertex
	 * ascmc[4] = equasc		* "equatorial ascendant" *
	 * ascmc[5] = coasc1		* "co-ascendant" (W. Koch) *
	 * ascmc[6] = coasc2		* "co-ascendant" (M. Munkasey) *
	 * ascmc[7] = polasc		* "polar ascendant" (M. Munkasey) *
	 */
	public int swe_houses(
		double tjd_ut,
		double geolat,
		double geolon,
		int hsys,
		double cusps[],
		double ascmc[]) {
		int i, retc = 0;
		double armc, eps, nutlo[];
		nutlo = new double[2];
		double tjde = tjd_ut + SwephLib.swe_deltat(tjd_ut);
		eps = SwephLib.swi_epsiln(tjde) * SwissEphemeris.RADTODEG;
		SwephLib.swi_nutation(tjde, nutlo);
		for (i = 0; i < 2; i++)
			nutlo[i] *= SwissEphemeris.RADTODEG;
		armc =
			SwephLib.swe_degnorm(
				SwephLib.swe_sidtime0(tjd_ut, eps + nutlo[1], nutlo[0]) * 15
					+ geolon);
		retc =
			swe_houses_armc(armc, geolat, eps + nutlo[1], hsys, cusps, ascmc);
		return retc;
	}

	/* housasp.c 
	 * cusps are returned in double cusp[13],
	 *                           or cusp[37] with house system 'G'.
	 * cusp[1...12]	houses 1 - 12
	 * additional points are returned in ascmc[10].
	 * ascmc[0] = ascendant
	 * ascmc[1] = mc
	 * ascmc[2] = armc
	 * ascmc[3] = vertex
	 * ascmc[4] = equasc		* "equatorial ascendant" *
	 * ascmc[5] = coasc1		* "co-ascendant" (W. Koch) *
	 * ascmc[6] = coasc2		* "co-ascendant" (M. Munkasey) *
	 * ascmc[7] = polasc		* "polar ascendant" (M. Munkasey) *
	 */
	public int swe_houses_ex(
		double tjd_ut,
		int iflag,
		double geolat,
		double geolon,
		int hsys,
		double cusp[],
		double ascmc[],
		SwissEphemeris swe) {
		int i, retc = 0;
		double armc, eps_mean, nutlo[];
		nutlo = new double[2];
		double tjde = tjd_ut + SwephLib.swe_deltat(tjd_ut);
		SwissEphemeris.sid_data sip = swe.swed.sidd;
		int ito;
		if (Character.toUpperCase((char) hsys) == 'G')
			ito = 36;
		else
			ito = 12;
		if ((iflag & SwissEphemeris.SEFLG_SIDEREAL) != 0
			&& !swe.swed.ayana_is_set)
			swe.swe_set_sid_mode(SwissEphemeris.SE_SIDM_FAGAN_BRADLEY, 0, 0);
		eps_mean = SwephLib.swi_epsiln(tjde) * SwissEphemeris.RADTODEG;
		SwephLib.swi_nutation(tjde, nutlo);
		for (i = 0; i < 2; i++)
			nutlo[i] *= SwissEphemeris.RADTODEG;
		/*houses_to_sidereal(tjde, geolat, hsys, eps, cusp, ascmc, iflag);*/
		armc =
			SwephLib.swe_degnorm(
				SwephLib.swe_sidtime0(tjd_ut, eps_mean + nutlo[1], nutlo[0])
					* 15
					+ geolon);
		if ((iflag & SwissEphemeris.SEFLG_SIDEREAL) != 0) {
			if ((sip.sid_mode & SwissEphemeris.SE_SIDBIT_ECL_T0) != 0)
				retc =
					sidereal_houses_ecl_t0(
						tjde,
						armc,
						eps_mean + nutlo[1],
						nutlo,
						geolat,
						hsys,
						cusp,
						ascmc,
						swe);
			else if ((sip.sid_mode & SwissEphemeris.SE_SIDBIT_SSY_PLANE) != 0)
				retc =
					sidereal_houses_ssypl(
						tjde,
						armc,
						eps_mean + nutlo[1],
						nutlo,
						geolat,
						hsys,
						cusp,
						ascmc,
						swe);
			else
				retc =
					sidereal_houses_trad(
						tjde,
						armc,
						eps_mean + nutlo[1],
						nutlo[0],
						geolat,
						hsys,
						cusp,
						ascmc,
						swe);
		} else {
			retc =
				swe_houses_armc(
					armc,
					geolat,
					eps_mean + nutlo[1],
					hsys,
					cusp,
					ascmc);
		}
		if ((iflag & SwissEphemeris.SEFLG_RADIANS) != 0) {
			for (i = 1; i <= ito; i++)
				cusp[i] *= SwissEphemeris.DEGTORAD;
			for (i = 0; i < SwissEphemeris.SE_NASCMC; i++)
				ascmc[i] *= SwissEphemeris.DEGTORAD;
		}
		return retc;
	}

	/*
	 * houses to sidereal
	 * ------------------
	 * there are two methods: 
	 * a) the traditional one
	 *    houses are computed tropically, then nutation and the ayanamsa
	 *    are subtracted.
	 * b) the projection on the ecliptic of t0
	 *    The house computation is then as follows:
	 *
	 * Be t the birth date and t0 the epoch at which ayanamsa = 0.
	 * 1. Compute the angle between the mean ecliptic at t0 and 
	 *    the true equator at t.
	 *    The intersection point of these two circles we call the 
	 *    "auxiliary vernal point", and the angle between them the 
	 *    "auxiliary obliquity".
	 * 2. Compute the distance of the auxiliary vernal point from the 
	 *    vernal point at t. (this is a section on the equator)
	 * 3. subtract this value from the armc of t = aux. armc.
	 * 4. Compute the axes and houses for this aux. armc and aux. obliquity.
	 * 5. Compute the distance between the auxiliary vernal point and the
	 *    vernal point at t0 (this is the ayanamsa at t, measured on the
	 *    ecliptic of t0)
	 * 6. subtract this distance from all house cusps.
	 * 7. subtract ayanamsa_t0 from all house cusps.
	 */
	int sidereal_houses_ecl_t0(
		double tjde,
		double armc,
		double eps,
		double nutlo[],
		double lat,
		int hsys,
		double cusp[],
		double ascmc[],
		SwissEphemeris swe) {
		int i, j, retc = SwissEphemeris.OK;
		double x[], xvpx[], x2[], epst0, xnorm[];
		x = new double[6];
		xvpx = new double[6];
		x2 = new double[6];
		xnorm = new double[6];
		double rxy, rxyz, c2, epsx, sgn, fac, dvpx, dvpxe;
		double armcx;
		SwissEphemeris.sid_data sip = swe.swed.sidd;
		int ito;
		if (Character.toUpperCase((char) hsys) == 'G')
			ito = 36;
		else
			ito = 12;
		/* epsilon at t0 */
		epst0 = SwephLib.swi_epsiln(sip.t0);
		/* cartesian coordinates of an imaginary moving body on the
		 * the mean ecliptic of t0; we take the vernal point: */
		x[0] = x[4] = 1;
		x[1] = x[2] = x[3] = x[5] = 0;
		/* to equator */
		SwephLib.swi_coortrf(x, 0, x, 0, -epst0);
		SwephLib.swi_coortrf(x, 3, x, 3, -epst0);
		/* to tjd_et */
		SwephLib.swi_precess(x, 0, sip.t0, SwissEphemeris.J_TO_J2000);
		SwephLib.swi_precess(x, 0, tjde, SwissEphemeris.J2000_TO_J);
		SwephLib.swi_precess(x, 3, sip.t0, SwissEphemeris.J_TO_J2000);
		SwephLib.swi_precess(x, 3, tjde, SwissEphemeris.J2000_TO_J);
		/* to true equator of tjd_et */
		SwephLib.swi_coortrf(
			x,
			0,
			x,
			0,
			(eps - nutlo[1]) * SwissEphemeris.DEGTORAD);
		SwephLib.swi_coortrf(
			x,
			3,
			x,
			3,
			(eps - nutlo[1]) * SwissEphemeris.DEGTORAD);
		SwephLib.swi_cartpol_sp(x, 0, x, 0);
		x[0] += nutlo[0] * SwissEphemeris.DEGTORAD;
		SwephLib.swi_polcart_sp(x, 0, x, 0);
		SwephLib.swi_coortrf(x, 0, x, 0, -eps * SwissEphemeris.DEGTORAD);
		SwephLib.swi_coortrf(x, 3, x, 3, -eps * SwissEphemeris.DEGTORAD);
		/* now, we have the moving point precessed to tjd_et.
		 * next, we compute the auxiliary epsilon: */
		SwephLib.swi_cross_prod(x, 0, x, 3, xnorm);
		rxy = xnorm[0] * xnorm[0] + xnorm[1] * xnorm[1];
		c2 = (rxy + xnorm[2] * xnorm[2]);
		rxyz = Math.sqrt(c2);
		rxy = Math.sqrt(rxy);
		epsx = Math.asin(rxy / rxyz) * SwissEphemeris.RADTODEG; /* 1a */
		/* auxiliary vernal point */
		if (Math.abs(x[5]) < 1e-15)
			x[5] = 1e-15;
		fac = x[2] / x[5];
		sgn = x[5] / Math.abs(x[5]);
		for (j = 0; j <= 2; j++)
			xvpx[j] = (x[j] - fac * x[j + 3]) * sgn; /* 1b */
		/* distance of the auxiliary vernal point from 
		 * the zero point at tjd_et (a section on the equator): */
		SwephLib.swi_cartpol(xvpx, 0, x2, 0);
		dvpx = x2[0] * SwissEphemeris.RADTODEG; /* 2 */
		/* auxiliary armc */
		armcx = SwephLib.swe_degnorm(armc - dvpx); /* 3 */
		/* compute axes and houses: */
		retc = swe_houses_armc(armcx, lat, epsx, hsys, cusp, ascmc); /* 4 */
		/* distance between auxiliary vernal point and
		 * vernal point of t0 (a section on the sidereal plane) */
		/* 5 */
		dvpxe =
			Math.acos(SwephLib.swi_dot_prod_unit(x, xvpx))
				* SwissEphemeris.RADTODEG;
		if (tjde < sip.t0)
			dvpxe = -dvpxe;
		for (i = 1; i <= ito; i++) /* 6, 7 */
			cusp[i] = SwephLib.swe_degnorm(cusp[i] - dvpxe - sip.ayan_t0);
		for (i = 0; i <= SwissEphemeris.SE_NASCMC; i++)
			ascmc[i] = SwephLib.swe_degnorm(ascmc[i] - dvpxe - sip.ayan_t0);
		return retc;
	}

	/*
	* Be t the birth date and t0 the epoch at which ayanamsa = 0.
	* 1. Compute the angle between the solar system rotation plane and 
	*    the true equator at t.
	*    The intersection point of these two circles we call the 
	*    "auxiliary vernal point", and the angle between them the 
	*    "auxiliary obliquity".
	* 2. Compute the distance of the auxiliary vernal point from the 
	*    zero point at t. (this is a section on the equator)
	* 3. subtract this value from the armc of t = aux. armc.
	* 4. Compute the axes and houses for this aux. armc and aux. obliquity.
	* 5. Compute the distance between the auxiliary vernal point at t
	*    and the zero point of the solar system plane J2000
	*    (a section measured on the solar system plane)
	* 6. subtract this distance from all house cusps.
	* 7. compute the ayanamsa of J2000 on the solar system plane, 
	*    referred to t0
	* 8. subtract ayanamsa_t0 from all house cusps.
	* 9. subtract ayanamsa_2000 from all house cusps.
	*/
	int sidereal_houses_ssypl(
		double tjde,
		double armc,
		double eps,
		double nutlo[],
		double lat,
		int hsys,
		double cusp[],
		double ascmc[],
		SwissEphemeris swe) {
		int i, j, retc = SwissEphemeris.OK;
		double x[], x0[], xvpx[], x2[], xnorm[];
		x = new double[6];
		x0 = new double[6];
		xvpx = new double[6];
		x2 = new double[6];
		xnorm = new double[6];
		double rxy, rxyz, c2, epsx, eps2000, sgn, fac, dvpx, dvpxe, x00;
		double armcx;
		SwissEphemeris.sid_data sip = swe.swed.sidd;
		int ito;
		if (Character.toUpperCase((char) hsys) == 'G')
			ito = 36;
		else
			ito = 12;
		/* epsilon at t0 */
		SwephLib.swi_epsiln(sip.t0);
		eps2000 = SwephLib.swi_epsiln(SwissEphemeris.J2000);
		/* cartesian coordinates of the zero point on the
		 * the solar system rotation plane */
		x[0] = x[4] = 1;
		x[1] = x[2] = x[3] = x[5] = 0;
		/* to ecliptic 2000 */
		SwephLib.swi_coortrf(x, 0, x, 0, -SwissEphemeris.SSY_PLANE_INCL);
		SwephLib.swi_coortrf(x, 3, x, 3, -SwissEphemeris.SSY_PLANE_INCL);
		SwephLib.swi_cartpol_sp(x, 0, x, 0);
		x[0] += SwissEphemeris.SSY_PLANE_NODE_E2000;
		SwephLib.swi_polcart_sp(x, 0, x, 0);
		/* to equator 2000 */
		SwephLib.swi_coortrf(x, 0, x, 0, -eps2000);
		SwephLib.swi_coortrf(x, 3, x, 3, -eps2000);
		/* to mean equator of t */
		SwephLib.swi_precess(x, 0, tjde, SwissEphemeris.J2000_TO_J);
		SwephLib.swi_precess(x, 3, tjde, SwissEphemeris.J2000_TO_J);
		/* to true equator of t */
		SwephLib.swi_coortrf(
			x,
			0,
			x,
			0,
			(eps - nutlo[1]) * SwissEphemeris.DEGTORAD);
		SwephLib.swi_coortrf(
			x,
			3,
			x,
			3,
			(eps - nutlo[1]) * SwissEphemeris.DEGTORAD);
		SwephLib.swi_cartpol_sp(x, 0, x, 0);
		x[0] += nutlo[0] * SwissEphemeris.DEGTORAD;
		SwephLib.swi_polcart_sp(x, 0, x, 0);
		SwephLib.swi_coortrf(x, 0, x, 0, -eps * SwissEphemeris.DEGTORAD);
		SwephLib.swi_coortrf(x, 3, x, 3, -eps * SwissEphemeris.DEGTORAD);
		/* now, we have the moving point precessed to tjd_et.
		 * next, we compute the auxiliary epsilon: */
		SwephLib.swi_cross_prod(x, 0, x, 3, xnorm);
		rxy = xnorm[0] * xnorm[0] + xnorm[1] * xnorm[1];
		c2 = (rxy + xnorm[2] * xnorm[2]);
		rxyz = Math.sqrt(c2);
		rxy = Math.sqrt(rxy);
		epsx = Math.asin(rxy / rxyz) * SwissEphemeris.RADTODEG; /* 1a */
		/* auxiliary vernal point */
		if (Math.abs(x[5]) < 1e-15)
			x[5] = 1e-15;
		fac = x[2] / x[5];
		sgn = x[5] / Math.abs(x[5]);
		for (j = 0; j <= 2; j++)
			xvpx[j] = (x[j] - fac * x[j + 3]) * sgn; /* 1b */
		/* distance of the auxiliary vernal point from 
		 * mean vernal point at tjd_et (a section on the equator): */
		SwephLib.swi_cartpol(xvpx, 0, x2, 0);
		dvpx = x2[0] * SwissEphemeris.RADTODEG; /* 2 */
		/* auxiliary armc */
		armcx = SwephLib.swe_degnorm(armc - dvpx); /* 3 */
		/* compute axes and houses: */
		retc = swe_houses_armc(armcx, lat, epsx, hsys, cusp, ascmc); /* 4 */
		/* distance between the auxiliary vernal point at t and
		 * the sidereal zero point of 2000 at t
		 * (a section on the sidereal plane).
		 */
		/* 5 */
		dvpxe =
			Math.acos(SwephLib.swi_dot_prod_unit(x, xvpx))
				* SwissEphemeris.RADTODEG;
		/* (always positive for dates after 5400 bc) */
		dvpxe -= SwissEphemeris.SSY_PLANE_NODE * SwissEphemeris.RADTODEG;
		/* ayanamsa between t0 and J2000, measured on solar system plane: */
		/* position of zero point of t0 */
		x0[0] = 1;
		x0[1] = x0[2] = 0;
		/* zero point of t0 in J2000 system */
		if (sip.t0 != SwissEphemeris.J2000)
			SwephLib.swi_precess(x0, 0, sip.t0, SwissEphemeris.J_TO_J2000);
		/* zero point to ecliptic 2000 */
		SwephLib.swi_coortrf(x0, 0, x0, 0, eps2000);
		/* to solar system plane */
		SwephLib.swi_cartpol(x0, 0, x0, 0);
		x0[0] -= SwissEphemeris.SSY_PLANE_NODE_E2000;
		SwephLib.swi_polcart(x0, 0, x0, 0);
		SwephLib.swi_coortrf(x0, 0, x0, 0, SwissEphemeris.SSY_PLANE_INCL);
		SwephLib.swi_cartpol(x0, 0, x0, 0);
		x0[0] += SwissEphemeris.SSY_PLANE_NODE;
		x00 = x0[0] * SwissEphemeris.RADTODEG; /* 7 */
		for (i = 1; i <= ito; i++) /* 6, 8, 9 */
			cusp[i] = SwephLib.swe_degnorm(cusp[i] - dvpxe - sip.ayan_t0 - x00);
		for (i = 0; i <= SwissEphemeris.SE_NASCMC; i++)
			ascmc[i] =
				SwephLib.swe_degnorm(ascmc[i] - dvpxe - sip.ayan_t0 - x00);
		return retc;
	}

	/* common simplified procedure */
	int sidereal_houses_trad(
		double tjde,
		double armc,
		double eps,
		double nutl,
		double lat,
		int hsys,
		double cusp[],
		double ascmc[],
		SwissEphemeris swe) {
		int i, retc = SwissEphemeris.OK;
		double ay;
		int ito;
		if (Character.toUpperCase((char) hsys) == 'G')
			ito = 36;
		else
			ito = 12;
		retc = swe_houses_armc(armc, lat, eps, hsys, cusp, ascmc);
		ay = swe.swe_get_ayanamsa(tjde);
		for (i = 1; i <= ito; i++)
			cusp[i] = SwephLib.swe_degnorm(cusp[i] - ay - nutl);
		for (i = 0; i < SwissEphemeris.SE_NASCMC; i++) {
			if (i == 2) /* armc */
				continue;
			ascmc[i] = SwephLib.swe_degnorm(ascmc[i] - ay - nutl);
		}
		return retc;
	}

	/* 
	 * this function is required for very special computations
	 * where no date is given for house calculation,
	 * e.g. for composite charts or progressive charts.
	 * cusps are returned in double cusp[13],
	 *                           or cusp[37] with house system 'G'.
	 * cusp[1...12]	houses 1 - 12
	 * additional points are returned in ascmc[10].
	 * ascmc[0] = ascendant
	 * ascmc[1] = mc
	 * ascmc[2] = armc
	 * ascmc[3] = vertex
	 * ascmc[4] = equasc		* "equatorial ascendant" *
	 * ascmc[5] = coasc1		* "co-ascendant" (W. Koch) *
	 * ascmc[6] = coasc2		* "co-ascendant" (M. Munkasey) *
	 * ascmc[7] = polasc		* "polar ascendant" (M. Munkasey) *
	 */
	public int swe_houses_armc(
		double armc,
		double geolat,
		double eps,
		int hsys,
		double cusps[],
		double ascmc[]) {
		int i, retc = 0;
		int ito;
		if (Character.toUpperCase((char) hsys) == 'G')
			ito = 36;
		else
			ito = 12;
		retc = CalcH(armc, geolat, eps, (char) hsys, 2);
		cusps[0] = 0;
		for (i = 1; i <= ito; i++) {
			cusps[i] = this.cusp[i];
		}
		ascmc[0] = this.ac; /* Asc */
		ascmc[1] = this.mc; /* Mid */
		ascmc[2] = armc;
		ascmc[3] = this.vertex;
		ascmc[4] = this.equasc;
		ascmc[5] = this.coasc1; /* "co-ascendant" (W. Koch) */
		ascmc[6] = this.coasc2; /* "co-ascendant" (M. Munkasey) */
		ascmc[7] = this.polasc; /* "polar ascendant" (M. Munkasey) */
		for (i = SwissEphemeris.SE_NASCMC; i < 10; i++)
			ascmc[i] = 0;
		return retc;
	}

	int CalcH(double th, double fi, double ekl, char hsy, int iteration_count)
	/* ********************************************************* 
	 *  Arguments: th = sidereal time (angle 0..360 degrees     
	 *             hsy = letter code for house system;         
	 *                   A  equal                         
	 *                   E  equal                         
	 *                   C  Campanus                        
	 *                   H  horizon / azimut
	 *                   K  Koch                             
	 *                   O  Porphyry                          
	 *                   P  Placidus                          
	 *                   R  Regiomontanus                  
	 *                   V  equal Vehlow                 
	 *                   X  axial rotation system/ Meridian houses 
	 *                   G  36 Gauquelin sectors
	 *             fi = geographic latitude             
	 *             ekl = obliquity of the ecliptic               
	 *             iteration_count = number of iterations in    
	 *             Placidus calculation; can be 1 or 2.        
	 * ********************************************************* 
	 *  Koch and Placidus don't work in the polar circle.        
	 *  We swap MC/IC so that MC is always before AC in the zodiac 
	 *  We than divide the quadrants into 3 equal parts.         
	 * *********************************************************
	 *  All angles are expressed in degrees.                     
	 *  Special trigonometric functions sind, cosd etc. are 
	 *  implemented for arguments in degrees.                 
	 ***********************************************************/ {
		double tane, tanfi, cosfi, tant, sina, cosa, th2;
		double a, c, f, fh1, fh2, xh1, xh2, rectasc, ad3, acmc, vemc;
		int i, ih, ih2, retc = SwissEphemeris.OK;
		double sine, cose;
		boolean isPorphyry = false;
		cose = cosd(ekl);
		sine = sind(ekl);
		tane = tand(ekl);
		/* north and south poles */
		if (Math.abs(Math.abs(fi) - 90) < VERY_SMALL) {
			if (fi < 0)
				fi = -90 + VERY_SMALL;
			else
				fi = 90 - VERY_SMALL;
		}
		tanfi = tand(fi);
		/* mc */
		if (Math.abs(th - 90) > VERY_SMALL
			&& Math.abs(th - 270) > VERY_SMALL) {
			tant = tand(th);
			this.mc = atand(tant / cose);
			if (th > 90 && th <= 270)
				this.mc = SwephLib.swe_degnorm(this.mc + 180);
		} else {
			if (Math.abs(th - 90) <= VERY_SMALL)
				this.mc = 90;
			else
				this.mc = 270;
		} /*  if */
		this.mc = SwephLib.swe_degnorm(this.mc);
		/* ascendant */
		this.ac = Asc1(th + 90, fi, sine, cose);
		this.cusp[1] = this.ac;
		this.cusp[10] = this.mc;
		if (hsy > 95)
			hsy = (char) (hsy - 32); /* translate into capital letter */
		switch (hsy) {
			case 'A' : /* equal houses */
			case 'E' :
				/*
				 * within polar circle we swap AC/DC if AC is on wrong side
				 */
				acmc = SwephLib.swe_difdeg2n(this.ac, this.mc);
				if (acmc < 0) {
					this.ac = SwephLib.swe_degnorm(this.ac + 180);
					this.cusp[1] = this.ac;
				}
				for (i = 2; i <= 12; i++)
					this.cusp[i] =
						SwephLib.swe_degnorm(this.cusp[1] + (i - 1) * 30);
				break;
			case 'C' : /* Campanus houses and Horizon or Azimut system */
			case 'H' :
				if (hsy == 'H') {
					if (fi > 0)
						fi = 90 - fi;
					else
						fi = -90 - fi;
					/* equator */
					if (Math.abs(Math.abs(fi) - 90) < VERY_SMALL) {
						if (fi < 0)
							fi = -90 + VERY_SMALL;
						else
							fi = 90 - VERY_SMALL;
					}
					th = SwephLib.swe_degnorm(th + 180);
				}
				fh1 = asind(sind(fi) / 2);
				fh2 = asind(Math.sqrt(3.0) / 2 * sind(fi));
				cosfi = cosd(fi);
				if (Math.abs(cosfi) == 0) { /* '==' should be save! */
					if (fi > 0)
						xh1 = xh2 = 90; /* cosfi = VERY_SMALL; */
					else
						xh1 = xh2 = 270; /* cosfi = -VERY_SMALL; */
				} else {
					xh1 = atand(Math.sqrt(3.0) / cosfi);
					xh2 = atand(1 / Math.sqrt(3.0) / cosfi);
				}
				this.cusp[11] = Asc1(th + 90 - xh1, fh1, sine, cose);
				this.cusp[12] = Asc1(th + 90 - xh2, fh2, sine, cose);
				if (hsy == 'H')
					this.cusp[1] = Asc1(th + 90, fi, sine, cose);
				this.cusp[2] = Asc1(th + 90 + xh2, fh2, sine, cose);
				this.cusp[3] = Asc1(th + 90 + xh1, fh1, sine, cose);
				/* within polar circle, when mc sinks below horizon and 
				 * ascendant changes to western hemisphere, all cusps
				 * must be added 180 degrees. 
				 * houses will be in clockwise direction */
				if (Math.abs(fi) >= 90 - ekl) { /* within polar circle */
					acmc = SwephLib.swe_difdeg2n(this.ac, this.mc);
					if (acmc < 0) {
						this.ac = SwephLib.swe_degnorm(this.ac + 180);
						this.mc = SwephLib.swe_degnorm(this.mc + 180);
						for (i = 1; i <= 12; i++)
							this.cusp[i] =
								SwephLib.swe_degnorm(this.cusp[i] + 180);
					}
				}
				if (hsy == 'H') {
					for (i = 1; i <= 3; i++)
						this.cusp[i] = SwephLib.swe_degnorm(this.cusp[i] + 180);
					for (i = 11; i <= 12; i++)
						this.cusp[i] = SwephLib.swe_degnorm(this.cusp[i] + 180);
					/* restore fi and th */
					if (fi > 0)
						fi = 90 - fi;
					else
						fi = -90 - fi;
					th = SwephLib.swe_degnorm(th + 180);
				}
				break;
			case 'K' : /* Koch houses */
				if (Math.abs(fi) >= 90 - ekl) { /* within polar circle */
					retc = SwissEphemeris.ERR;
					/*goto porphyry;*/
					isPorphyry = true;
					break;
				}
				sina = sind(this.mc) * sine / cosd(fi);
				/* always << 1, 
									   because fi < polar circle */
				cosa = Math.sqrt(1 - sina * sina); /* always >> 0 */
				c = atand(tanfi / cosa);
				ad3 = asind(sind(c) * sina) / 3.0;
				this.cusp[11] = Asc1(th + 30 - 2 * ad3, fi, sine, cose);
				this.cusp[12] = Asc1(th + 60 - ad3, fi, sine, cose);
				this.cusp[2] = Asc1(th + 120 + ad3, fi, sine, cose);
				this.cusp[3] = Asc1(th + 150 + 2 * ad3, fi, sine, cose);
				break;
			case 'O' : /* Porphyry houses */
				/*porphyry:*/
				isPorphyry = true;
				break;
			case 'R' : /* Regiomontanus houses */
				fh1 = atand(tanfi * 0.5);
				fh2 = atand(tanfi * cosd(30));
				this.cusp[11] = Asc1(30 + th, fh1, sine, cose);
				this.cusp[12] = Asc1(60 + th, fh2, sine, cose);
				this.cusp[2] = Asc1(120 + th, fh2, sine, cose);
				this.cusp[3] = Asc1(150 + th, fh1, sine, cose);
				/* within polar circle, when mc sinks below horizon and 
				 * ascendant changes to western hemisphere, all cusps
				 * must be added 180 degrees.
				 * houses will be in clockwise direction */
				if (Math.abs(fi) >= 90 - ekl) { /* within polar circle */
					acmc = SwephLib.swe_difdeg2n(this.ac, this.mc);
					if (acmc < 0) {
						this.ac = SwephLib.swe_degnorm(this.ac + 180);
						this.mc = SwephLib.swe_degnorm(this.mc + 180);
						for (i = 1; i <= 12; i++)
							this.cusp[i] =
								SwephLib.swe_degnorm(this.cusp[i] + 180);
					}
				}
				break;
			case 'T' : /* 'topocentric' houses */
				fh1 = atand(tanfi / 3.0);
				fh2 = atand(tanfi * 2.0 / 3.0);
				this.cusp[11] = Asc1(30 + th, fh1, sine, cose);
				this.cusp[12] = Asc1(60 + th, fh2, sine, cose);
				this.cusp[2] = Asc1(120 + th, fh2, sine, cose);
				this.cusp[3] = Asc1(150 + th, fh1, sine, cose);
				/* within polar circle, when mc sinks below horizon and 
				 * ascendant changes to western hemisphere, all cusps
				 * must be added 180 degrees.
				 * houses will be in clockwise direction */
				if (Math.abs(fi) >= 90 - ekl) { /* within polar circle */
					acmc = SwephLib.swe_difdeg2n(this.ac, this.mc);
					if (acmc < 0) {
						this.ac = SwephLib.swe_degnorm(this.ac + 180);
						this.mc = SwephLib.swe_degnorm(this.mc + 180);
						for (i = 1; i <= 12; i++)
							this.cusp[i] =
								SwephLib.swe_degnorm(this.cusp[i] + 180);
					}
				}
				break;
			case 'V' : /* equal houses after Vehlow */
				/*
				 * within polar circle we swap AC/DC if AC is on wrong side
				 */
				acmc = SwephLib.swe_difdeg2n(this.ac, this.mc);
				if (acmc < 0) {
					this.ac = SwephLib.swe_degnorm(this.ac + 180);
					this.cusp[1] = this.ac;
				}
				this.cusp[1] = SwephLib.swe_degnorm(this.ac - 15);
				for (i = 2; i <= 12; i++)
					this.cusp[i] =
						SwephLib.swe_degnorm(this.cusp[1] + (i - 1) * 30);
				break;
			case 'X' :
				{
					/* 
					 * Meridian or axial rotation system:
					 * ecliptic points whose rectascensions
					 * are armc + n * 30 
					 */
					int j;
					double aa = th;
					for (i = 1; i <= 12; i++) {
						j = i + 10;
						if (j > 12)
							j -= 12;
						aa = SwephLib.swe_degnorm(aa + 30);
						if (Math.abs(aa - 90) > VERY_SMALL
							&& Math.abs(aa - 270) > VERY_SMALL) {
							tant = tand(aa);
							this.cusp[j] = atand(tant / cose);
							if (aa > 90 && aa <= 270)
								this.cusp[j] =
									SwephLib.swe_degnorm(this.cusp[j] + 180);
						} else {
							if (Math.abs(aa - 90) <= VERY_SMALL)
								this.cusp[j] = 90;
							else
								this.cusp[j] = 270;
						} /*  if */
						this.cusp[j] = SwephLib.swe_degnorm(this.cusp[j]);
					}
				}
				break;
			case 'B' :
				{ /* Alcabitius */
					/* created by Alois 17-sep-2000, followed example in Matrix
					electrical library. The code reproduces the example!
					I think the Alcabitius code in Walter Pullen's Astrolog 5.40
					is wrong, because he remains in RA and forgets the transform to
					the ecliptic. */
					double dek, r, sna, sda, sn3, sd3;
					acmc = SwephLib.swe_difdeg2n(this.ac, this.mc);
					if (acmc < 0) {
						this.ac = SwephLib.swe_degnorm(this.ac + 180);
						this.cusp[1] = this.ac;
						acmc = SwephLib.swe_difdeg2n(this.ac, this.mc);
					}
					dek = asind(sind(this.ac) * sine);
					/* declination of Ascendant */
					/* must treat the case fi == 90 or -90 */
					r = -tanfi * tand(dek);
					/* must treat the case of abs(r) > 1; probably does not happen
					 * because dek becomes smaller when fi is large, as ac is close to
					 * zero Aries/Libra in that case.
					 */
					/* semidiurnal arc, measured on equator */
					sda = Math.acos(r) * SwissEphemeris.RADTODEG;
					sna = 180 - sda; /* complement, seminocturnal arc */
					sd3 = sda / 3;
					sn3 = sna / 3;
					rectasc = SwephLib.swe_degnorm(th + sd3); /* cusp 11 */
					this.cusp[11] = Asc1(rectasc, 0, sine, cose);
					rectasc = SwephLib.swe_degnorm(th + 2 * sd3); /* cusp 12 */
					this.cusp[12] = Asc1(rectasc, 0, sine, cose);
					rectasc = SwephLib.swe_degnorm(th + 180 - 2 * sn3);
					/* cusp 2 */
					this.cusp[2] = Asc1(rectasc, 0, sine, cose);
					rectasc = SwephLib.swe_degnorm(th + 180 - sn3);
					/* cusp 3 */
					this.cusp[3] = Asc1(rectasc, 0, sine, cose);
				}
				break;
			case 'G' : /* 36 Gauquelin sectors */
				for (i = 1; i <= 36; i++) {
					this.cusp[i] = 0;
				}
				if (Math.abs(fi) >= 90 - ekl) { /* within polar circle */
					retc = SwissEphemeris.ERR;
					/*goto porphyry;*/
					isPorphyry = true;
					break;
				}
				/*************** forth/second quarter ***************/
				/* note: Gauquelin sectors are counted in clockwise direction */
				a = asind(tand(fi) * tane);
				for (ih = 2; ih <= 9; ih++) {
					ih2 = 10 - ih;
					fh1 = atand(sind(a * ih2 / 9) / tane);
					rectasc = SwephLib.swe_degnorm((90 / 9) * ih2 + th);
					tant =
						tand(
							asind(sine * sind(Asc1(rectasc, fh1, sine, cose))));
					if (Math.abs(tant) < VERY_SMALL) {
						this.cusp[ih] = rectasc;
					} else {
						/* pole height */
						f = atand(sind(asind(tanfi * tant) * ih2 / 9) / tant);
						this.cusp[ih] = Asc1(rectasc, f, sine, cose);
						for (i = 1; i <= iteration_count; i++) {
							tant = tand(asind(sine * sind(this.cusp[ih])));
							if (Math.abs(tant) < VERY_SMALL) {
								this.cusp[ih] = rectasc;
								break;
							}
							/* pole height */
							f =
								atand(
									sind(asind(tanfi * tant) * ih2 / 9) / tant);
							this.cusp[ih] = Asc1(rectasc, f, sine, cose);
						}
					}
					this.cusp[ih + 18] =
						SwephLib.swe_degnorm(this.cusp[ih] + 180);
				}
				/*************** first/third quarter ***************/
				for (ih = 29; ih <= 36; ih++) {
					ih2 = ih - 28;
					fh1 = atand(sind(a * ih2 / 9) / tane);
					rectasc = SwephLib.swe_degnorm(180 - ih2 * 90 / 9 + th);
					tant =
						tand(
							asind(sine * sind(Asc1(rectasc, fh1, sine, cose))));
					if (Math.abs(tant) < VERY_SMALL) {
						this.cusp[ih] = rectasc;
					} else {
						f = atand(sind(asind(tanfi * tant) * ih2 / 9) / tant);
						/*  pole height */
						this.cusp[ih] = Asc1(rectasc, f, sine, cose);
						for (i = 1; i <= iteration_count; i++) {
							tant = tand(asind(sine * sind(this.cusp[ih])));
							if (Math.abs(tant) < VERY_SMALL) {
								this.cusp[ih] = rectasc;
								break;
							}
							f =
								atand(
									sind(asind(tanfi * tant) * ih2 / 9) / tant);
							/*  pole height */
							this.cusp[ih] = Asc1(rectasc, f, sine, cose);
						}
					}
					this.cusp[ih - 18] =
						SwephLib.swe_degnorm(this.cusp[ih] + 180);
				}
				this.cusp[1] = this.ac;
				this.cusp[10] = this.mc;
				this.cusp[19] = SwephLib.swe_degnorm(this.ac + 180);
				this.cusp[28] = SwephLib.swe_degnorm(this.mc + 180);
				break;
			default : /* Placidus houses */
				if (hsy != 'P')
					/*fprintf (stderr, "swe_houses: make Placidus, unknown key %c\n", hsy);*/
					System.err.println(
						"swe_houses: make Placidus, unknown key " + hsy + '\n'); //$NON-NLS-1$
				if (Math.abs(fi) >= 90 - ekl) { /* within polar circle */
					retc = SwissEphemeris.ERR;
					/*goto porphyry;*/
					isPorphyry = true;
					break;
				}
				a = asind(tand(fi) * tane);
				fh1 = atand(sind(a / 3) / tane);
				fh2 = atand(sind(a * 2 / 3) / tane);
				/* ************  house 11 ******************** */
				rectasc = SwephLib.swe_degnorm(30 + th);
				tant = tand(asind(sine * sind(Asc1(rectasc, fh1, sine, cose))));
				if (Math.abs(tant) < VERY_SMALL) {
					this.cusp[11] = rectasc;
				} else {
					/* pole height */
					f = atand(sind(asind(tanfi * tant) / 3) / tant);
					this.cusp[11] = Asc1(rectasc, f, sine, cose);
					for (i = 1; i <= iteration_count; i++) {
						tant = tand(asind(sine * sind(this.cusp[11])));
						if (Math.abs(tant) < VERY_SMALL) {
							this.cusp[11] = rectasc;
							break;
						}
						/* pole height */
						f = atand(sind(asind(tanfi * tant) / 3) / tant);
						this.cusp[11] = Asc1(rectasc, f, sine, cose);
					}
				}
				/* ************  house 12 ******************** */
				rectasc = SwephLib.swe_degnorm(60 + th);
				tant = tand(asind(sine * sind(Asc1(rectasc, fh2, sine, cose))));
				if (Math.abs(tant) < VERY_SMALL) {
					this.cusp[12] = rectasc;
				} else {
					f = atand(sind(asind(tanfi * tant) / 1.5) / tant);
					/*  pole height */
					this.cusp[12] = Asc1(rectasc, f, sine, cose);
					for (i = 1; i <= iteration_count; i++) {
						tant = tand(asind(sine * sind(this.cusp[12])));
						if (Math.abs(tant) < VERY_SMALL) {
							this.cusp[12] = rectasc;
							break;
						}
						f = atand(sind(asind(tanfi * tant) / 1.5) / tant);
						/*  pole height */
						this.cusp[12] = Asc1(rectasc, f, sine, cose);
					}
				}
				/* ************  house  2 ******************** */
				rectasc = SwephLib.swe_degnorm(120 + th);
				tant = tand(asind(sine * sind(Asc1(rectasc, fh2, sine, cose))));
				if (Math.abs(tant) < VERY_SMALL) {
					this.cusp[2] = rectasc;
				} else {
					f = atand(sind(asind(tanfi * tant) / 1.5) / tant);
					/*  pole height */
					this.cusp[2] = Asc1(rectasc, f, sine, cose);
					for (i = 1; i <= iteration_count; i++) {
						tant = tand(asind(sine * sind(this.cusp[2])));
						if (Math.abs(tant) < VERY_SMALL) {
							this.cusp[2] = rectasc;
							break;
						}
						f = atand(sind(asind(tanfi * tant) / 1.5) / tant);
						/*  pole height */
						this.cusp[2] = Asc1(rectasc, f, sine, cose);
					}
				}
				/* ************  house  3 ******************** */
				rectasc = SwephLib.swe_degnorm(150 + th);
				tant = tand(asind(sine * sind(Asc1(rectasc, fh1, sine, cose))));
				if (Math.abs(tant) < VERY_SMALL) {
					this.cusp[3] = rectasc;
				} else {
					f = atand(sind(asind(tanfi * tant) / 3) / tant);
					/*  pole height */
					this.cusp[3] = Asc1(rectasc, f, sine, cose);
					for (i = 1; i <= iteration_count; i++) {
						tant = tand(asind(sine * sind(this.cusp[3])));
						if (Math.abs(tant) < VERY_SMALL) {
							this.cusp[3] = rectasc;
							break;
						}
						f = atand(sind(asind(tanfi * tant) / 3) / tant);
						/*  pole height */
						this.cusp[3] = Asc1(rectasc, f, sine, cose);
					}
				}
				break;
		} /* end switch */
		if (isPorphyry) { /* porphyry */
			/*
			 * within polar circle we swap AC/DC if AC is on wrong side
			 */
			acmc = SwephLib.swe_difdeg2n(this.ac, this.mc);
			if (acmc < 0) {
				this.ac = SwephLib.swe_degnorm(this.ac + 180);
				this.cusp[1] = this.ac;
				acmc = SwephLib.swe_difdeg2n(this.ac, this.mc);
			}
			this.cusp[2] = SwephLib.swe_degnorm(this.ac + (180 - acmc) / 3);
			this.cusp[3] = SwephLib.swe_degnorm(this.ac + (180 - acmc) / 3 * 2);
			this.cusp[11] = SwephLib.swe_degnorm(this.mc + acmc / 3);
			this.cusp[12] = SwephLib.swe_degnorm(this.mc + acmc / 3 * 2);
		}
		if (hsy != 'G') {
			this.cusp[4] = SwephLib.swe_degnorm(this.cusp[10] + 180);
			this.cusp[5] = SwephLib.swe_degnorm(this.cusp[11] + 180);
			this.cusp[6] = SwephLib.swe_degnorm(this.cusp[12] + 180);
			this.cusp[7] = SwephLib.swe_degnorm(this.cusp[1] + 180);
			this.cusp[8] = SwephLib.swe_degnorm(this.cusp[2] + 180);
			this.cusp[9] = SwephLib.swe_degnorm(this.cusp[3] + 180);
		}
		/* vertex */
		if (fi >= 0)
			f = 90 - fi;
		else
			f = -90 - fi;
		this.vertex = Asc1(th - 90, f, sine, cose);
		/* with tropical latitudes, the vertex behaves strange, 
		 * in a similar way as the ascendant within the polar 
		 * circle. we keep it always on the western hemisphere.*/
		if (Math.abs(fi) <= ekl) {
			vemc = SwephLib.swe_difdeg2n(this.vertex, this.mc);
			if (vemc > 0)
				this.vertex = SwephLib.swe_degnorm(this.vertex + 180);
		}
		/* 
		 * some strange points:
		 */
		/* equasc (equatorial ascendant) */
		th2 = SwephLib.swe_degnorm(th + 90);
		if (Math.abs(th2 - 90) > VERY_SMALL
			&& Math.abs(th2 - 270) > VERY_SMALL) {
			tant = tand(th2);
			this.equasc = atand(tant / cose);
			if (th2 > 90 && th2 <= 270)
				this.equasc = SwephLib.swe_degnorm(this.equasc + 180);
		} else {
			if (Math.abs(th2 - 90) <= VERY_SMALL)
				this.equasc = 90;
			else
				this.equasc = 270;
		} /*  if */
		this.equasc = SwephLib.swe_degnorm(this.equasc);
		/* "co-ascendant" W. Koch */
		this.coasc1 = SwephLib.swe_degnorm(Asc1(th - 90, fi, sine, cose) + 180);
		/* "co-ascendant" M. Munkasey */
		if (fi >= 0)
			this.coasc2 = Asc1(th + 90, 90 - fi, sine, cose);
		else /* southern hemisphere */
			this.coasc2 = Asc1(th + 90, -90 - fi, sine, cose);
		/* "polar ascendant" M. Munkasey */
		this.polasc = Asc1(th - 90, fi, sine, cose);
		return retc;
	} /* procedure houses */

	/******************************/
	double Asc1(double x1, double f, double sine, double cose) {
		int n;
		double ass;
		x1 = SwephLib.swe_degnorm(x1);
		n = (int) ((x1 / 90) + 1);
		if (n == 1)
			ass = (Asc2(x1, f, sine, cose));
		else if (n == 2)
			ass = (180 - Asc2(180 - x1, -f, sine, cose));
		else if (n == 3)
			ass = (180 + Asc2(x1 - 180, -f, sine, cose));
		else
			ass = (360 - Asc2(360 - x1, f, sine, cose));
		ass = SwephLib.swe_degnorm(ass);
		if (Math.abs(ass - 90) < VERY_SMALL) /* rounding, e.g.: if */
			ass = 90; /* fi = 0 & st = 0, ac = 89.999... */
		if (Math.abs(ass - 180) < VERY_SMALL)
			ass = 180;
		if (Math.abs(ass - 270) < VERY_SMALL) /* rounding, e.g.: if */
			ass = 270; /* fi = 0 & st = 0, ac = 89.999... */
		if (Math.abs(ass - 360) < VERY_SMALL)
			ass = 0;
		return ass;
	} /* Asc1 */

	double Asc2(double x, double f, double sine, double cose) {
		double ass, sinx;
		ass = -tand(f) * sine + cose * cosd(x);
		if (Math.abs(ass) < VERY_SMALL)
			ass = 0;
		sinx = sind(x);
		if (Math.abs(sinx) < VERY_SMALL)
			sinx = 0;
		if (sinx == 0) {
			if (ass < 0)
				ass = -VERY_SMALL;
			else
				ass = VERY_SMALL;
		} else if (ass == 0) {
			if (sinx < 0)
				ass = -90;
			else
				ass = 90;
		} else {
			ass = atand(sinx / ass);
		}
		if (ass < 0)
			ass = 180 + ass;
		return (ass);
	} /* Asc2 */

	/* computes the house position of a planet or another point,
	 * in degrees: 0 - 30 = 1st house, 30 - 60 = 2nd house, etc.
	 * armc 	sidereal time in degrees
	 * geolat	geographic latitude
	 * eps		true ecliptic obliquity
	 * hsys		house system character
	 * xpin		array of 6 doubles: 
	 * 		only the first two of them are used: ecl. long., lat.
	 * serr		error message area
	 *
	 * house position is returned by function.
	 * 
	 * sidereal house positions:
	 *
	 * tropical and sidereal house positions of planets are always identical 
	 * if the traditional method of computing sidereal positions (subtracting
	 * the ayanamsha from tropical in order to get sidereal positions) is used.
	 *
	 * if the sidereal plane is not identical to the ecliptic of date, 
	 * sidereal and tropical house positions are identical with
	 * house systems that are independent on the ecliptic such as:
	 * - Campanus
	 * - Regiomontanus
	 * - Placidus
	 * - Azimuth/Horizon
	 * - Axial rotation system
	 * - "topocentric" system
	 * 
	 * in all these cases call swe_house_pos() with TROPICAL planetary positions.
	 *
	 * but there are different house positions for ecliptic-dependent systems
	 * such as:
	 * - equal
	 * - Porphyry
	 * - Koch
	 * 
	 * for these cases there is no function. 
	 */
	public double swe_house_pos(
		double armc,
		double geolat,
		double eps,
		int hsys,
		double xpin[],
		String serr) {
		final double MILLIARCSEC = (1.0 / 3600000.0);
		double xp[], xeq[], ra, de, mdd, mdn, sad, san;
		xp = new double[6];
		xeq = new double[6];
		double hpos, sinad, ad, a, admc, adp, samc, demc, asc, mc, acmc, tant;
		double fh, ra0, tanfi, fac;
		double sine = sind(eps);
		double cose = cosd(eps);
		boolean is_above_hor = false;
		serr = ""; //$NON-NLS-1$
		
		hsys = Character.toUpperCase((char) hsys);
		xeq[0] = xpin[0];
		xeq[1] = xpin[1];
		xeq[2] = 1;
		SwephLib.swe_cotrans(xpin, xeq, -eps);
		ra = xeq[0];
		de = xeq[1];
		mdd = SwephLib.swe_degnorm(ra - armc);
		mdn = SwephLib.swe_degnorm(mdd + 180);
		if (mdd >= 180)
			mdd -= 360;
		if (mdn >= 180)
			mdn -= 360;
		/* xp[0] will contain the house position, a value between 0 and 360 */
		switch (hsys) {
			case 'A' :
			case 'E' :
			case 'V' :
				asc = Asc1(SwephLib.swe_degnorm(armc + 90), geolat, sine, cose);
				demc = atand(sind(armc) * tand(eps));
				if (geolat >= 0 && 90 - geolat + demc < 0)
					asc = SwephLib.swe_degnorm(asc + 180);
				if (geolat < 0 && -90 - geolat + demc > 0)
					asc = SwephLib.swe_degnorm(asc + 180);
				xp[0] = SwephLib.swe_degnorm(xpin[0] - asc);
				if (hsys == 'V')
					xp[0] = SwephLib.swe_degnorm(xp[0] + 15);
				/* to make sure that a call with a house cusp position returns
				 * a value within the house, 0.001" is added */
				xp[0] = SwephLib.swe_degnorm(xp[0] + MILLIARCSEC);
				hpos = xp[0] / 30.0 + 1;
				break;
			case 'O' :
				asc = Asc1(SwephLib.swe_degnorm(armc + 90), geolat, sine, cose);
				demc = atand(sind(armc) * tand(eps));
				/* mc */
				if (Math.abs(armc - 90) > VERY_SMALL
					&& Math.abs(armc - 270) > VERY_SMALL) {
					tant = tand(armc);
					mc = SwephLib.swe_degnorm(atand(tant / cose));
					if (armc > 90 && armc <= 270)
						mc = SwephLib.swe_degnorm(mc + 180);
				} else {
					if (Math.abs(armc - 90) <= VERY_SMALL)
						mc = 90;
					else
						mc = 270;
				}
				/* while MC is always south, 
				 * Asc must always be in eastern hemisphere */
				if (geolat >= 0 && 90 - geolat + demc < 0) {
					asc = SwephLib.swe_degnorm(asc + 180);
				}
				if (geolat < 0 && -90 - geolat + demc > 0) {
					asc = SwephLib.swe_degnorm(asc + 180);
				}
				xp[0] = SwephLib.swe_degnorm(xpin[0] - asc);
				/* to make sure that a call with a house cusp position returns
				 * a value within the house, 0.001" is added */
				xp[0] = SwephLib.swe_degnorm(xp[0] + MILLIARCSEC);
				if (xp[0] < 180)
					hpos = 1;
				else {
					hpos = 7;
					xp[0] -= 180;
				}
				acmc = SwephLib.swe_difdeg2n(asc, mc);
				if (xp[0] < 180 - acmc)
					hpos += xp[0] * 3 / (180 - acmc);
				else
					hpos += 3 + (xp[0] - 180 + acmc) * 3 / acmc;
				break;
			case 'X' : /* Merdidian or axial rotation system */
				hpos = SwephLib.swe_degnorm(mdd - 90) / 30.0 + 1;
				break;
			case 'K' :
				demc = atand(sind(armc) * tand(eps));
				/* if body is within circumpolar region, error */
				if (90 - Math.abs(geolat) <= Math.abs(de)) {
					/*strcpy(serr, "no Koch house position, because planet is circumpolar.");*/
					serr =
						"no Koch house position, because planet is circumpolar."; //$NON-NLS-1$
					xp[0] = 0;
					hpos = 0; /* Error */
				} else if (90 - Math.abs(geolat) <= Math.abs(demc)) {
					/*strcpy(serr, "no Koch house position, because mc is circumpolar.");*/
					serr = "no Koch house position, because mc is circumpolar."; //$NON-NLS-1$
					xp[0] = 0;
					hpos = 0; /* Error */
				} else {
					admc = asind(tand(eps) * tand(geolat) * sind(armc));
					adp = asind(tand(geolat) * tand(de));
					samc = 90 + admc;
					if (mdd >= 0) { /* east */
						xp[0] =
							SwephLib.swe_degnorm(
								((mdd - adp + admc) / samc - 1) * 90);
					} else {
						xp[0] =
							SwephLib.swe_degnorm(
								((mdd + 180 + adp + admc) / samc + 1) * 90);
					}
					/* to make sure that a call with a house cusp position returns
					 * a value within the house, 0.001" is added */
					xp[0] = SwephLib.swe_degnorm(xp[0] + MILLIARCSEC);
					hpos = xp[0] / 30.0 + 1;
				}
				break;
			case 'C' :
				xeq[0] = SwephLib.swe_degnorm(mdd - 90);
				SwephLib.swe_cotrans(xeq, xp, -geolat);
				/* to make sure that a call with a house cusp position returns
				 * a value within the house, 0.001" is added */
				xp[0] = SwephLib.swe_degnorm(xp[0] + MILLIARCSEC);
				hpos = xp[0] / 30.0 + 1;
				break;
			case 'H' :
				xeq[0] = SwephLib.swe_degnorm(mdd - 90);
				SwephLib.swe_cotrans(xeq, xp, 90 - geolat);
				/* to make sure that a call with a house cusp position returns
				 * a value within the house, 0.001" is added */
				xp[0] = SwephLib.swe_degnorm(xp[0] + MILLIARCSEC);
				hpos = xp[0] / 30.0 + 1;
				break;
			case 'R' :
				if (Math.abs(mdd) < VERY_SMALL)
					xp[0] = 270;
				else if (180 - Math.abs(mdd) < VERY_SMALL)
					xp[0] = 90;
				else {
					if (90 - Math.abs(geolat) < VERY_SMALL) {
						if (geolat > 0)
							geolat = 90 - VERY_SMALL;
						else
							geolat = -90 + VERY_SMALL;
					}
					if (90 - Math.abs(de) < VERY_SMALL) {
						if (de > 0)
							de = 90 - VERY_SMALL;
						else
							de = -90 + VERY_SMALL;
					}
					a = tand(geolat) * tand(de) + cosd(mdd);
					xp[0] = SwephLib.swe_degnorm(atand(-a / sind(mdd)));
					if (mdd < 0)
						xp[0] += 180;
					xp[0] = SwephLib.swe_degnorm(xp[0]);
					/* to make sure that a call with a house cusp position returns
					 * a value within the house, 0.001" is added */
					xp[0] = SwephLib.swe_degnorm(xp[0] + MILLIARCSEC);
				}
				hpos = xp[0] / 30.0 + 1;
				break;
			case 'T' :
				mdd = SwephLib.swe_degnorm(mdd);
				if (de > 90 - VERY_SMALL)
					de = 90 - VERY_SMALL;
				if (de < -90 + VERY_SMALL)
					de = -90 + VERY_SMALL;
				sinad = tand(de) * tand(geolat);
				ad = asind(sinad);
				a = sinad + cosd(mdd);
				if (a >= 0)
					is_above_hor = true;
				/* mirror everything below the horizon to the opposite point
				 * above the horizon */
				if (!is_above_hor) {
					ra = SwephLib.swe_degnorm(ra + 180);
					de = -de;
					mdd = SwephLib.swe_degnorm(mdd + 180);
				}
				/* mirror everything on western hemisphere to eastern hemisphere */
				if (mdd > 180) {
					ra = SwephLib.swe_degnorm(armc - mdd);
				}
				/* binary search for "topocentric" position line of body */
				tanfi = tand(geolat);
				fh = geolat;
				ra0 = SwephLib.swe_degnorm(armc + 90);
				xp[1] = 1;
				xeq[1] = de;
				fac = 2;
				while (Math.abs(xp[1]) > 0.000001) {
					if (xp[1] > 0) {
						fh = atand(tand(fh) - tanfi / fac);
						ra0 -= 90 / fac;
					} else {
						fh = atand(tand(fh) + tanfi / fac);
						ra0 += 90 / fac;
					}
					xeq[0] = SwephLib.swe_degnorm(ra - ra0);
					SwephLib.swe_cotrans(xeq, xp, 90 - fh);
					fac *= 2;
				}
				hpos = SwephLib.swe_degnorm(ra0 - armc);
				/* mirror back to west */
				if (mdd > 180)
					hpos = SwephLib.swe_degnorm(-hpos);
				/* mirror back to below horizon */
				if (!is_above_hor)
					hpos = SwephLib.swe_degnorm(hpos + 180);
				hpos = SwephLib.swe_degnorm(hpos - 90) / 30 + 1;
				break;
			case 'P' :
			case 'G' :
			default :
				/* circumpolar region */
				if (90 - Math.abs(de) <= Math.abs(geolat)) {
					if (de * geolat < 0)
						xp[0] = SwephLib.swe_degnorm(90 + mdn / 2);
					else
						xp[0] = SwephLib.swe_degnorm(270 + mdd / 2);
					/*strcpy(serr, "Otto Ludwig procedure within circumpolar regions.");*/
					serr = "Otto Ludwig procedure within circumpolar regions."; //$NON-NLS-1$
				} else {
					sinad = tand(de) * tand(geolat);
					ad = asind(sinad);
					a = sinad + cosd(mdd);
					if (a >= 0)
						is_above_hor = true;
					sad = 90 + ad;
					san = 90 - ad;
					if (is_above_hor)
						xp[0] = (mdd / sad + 3) * 90;
					else
						xp[0] = (mdn / san + 1) * 90;
					/* to make sure that a call with a house cusp position returns
					 * a value within the house, 0.001" is added */
					xp[0] = SwephLib.swe_degnorm(xp[0] + MILLIARCSEC);
				}
				if (hsys == 'G') {
					xp[0] = 360 - xp[0];
					/* Gauquelin sectors are in clockwise direction */
					hpos = xp[0] / 10.0 + 1;
				} else {
					hpos = xp[0] / 30.0 + 1;
				}
				break;
		}
		return hpos;
	}
	/* get house & armc name */
	public String swe_get_house_name(int ipl, String spname) {
		switch (ipl) {
			case 1 :
				spname = "house 1"; //$NON-NLS-1$
				break;
			case 2 :
				spname = "house 2"; //$NON-NLS-1$
				break;
			case 3 :
				spname = "house 3"; //$NON-NLS-1$
				break;
			case 4 :
				spname = "house 4"; //$NON-NLS-1$
				break;
			case 5 :
				spname = "house 5"; //$NON-NLS-1$
				break;
			case 6 :
				spname = "house 6"; //$NON-NLS-1$
				break;
			case 7 :
				spname = "house 7"; //$NON-NLS-1$
				break;
			case 8 :
				spname = "house 8"; //$NON-NLS-1$
				break;
			case 9 :
				spname = "house 9"; //$NON-NLS-1$
				break;
			case 10 :
				spname = "house 10"; //$NON-NLS-1$
				break;
			case 11 :
				spname = "house 11"; //$NON-NLS-1$
				break;
			case 12 :
				spname = "house 12"; //$NON-NLS-1$
				break;
			case 13 :
				spname = "Ascendant"; //$NON-NLS-1$
				break;
			case 14 :
				spname = "MC"; //$NON-NLS-1$
				break;
			case 15 :
				spname = "ARMC"; //$NON-NLS-1$
				break;
			case 16 :
				spname = "Vertex"; //$NON-NLS-1$
				break;
			default :
				spname = ""; //$NON-NLS-1$
				break;
		}
		return spname;
	}

}
