/* $Id: BookmarkView.java 302 2011-08-18 21:46:39Z minao $ */
package smart_gs.drawing_tool.view;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.Point2D.Double;

import javax.swing.ImageIcon;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import smart_gs.GSConstants;
import smart_gs.debugprint.Debugprint;
import smart_gs.logical.URICreator;


public class BookmarkView extends View {
	public static final double HUGE = 1.5;
	public static final double LARGE = 1.25;
	public static final double NORMAL = 1;
	public static final double SMALL = 0.75;
	public static final double TINY = 0.5;
	
	public static double DEFAULT_WIDTH = 45;
	public static double DEFAULT_HEIGHT = 42;
	
	private Rectangle2D rectangle;
	private String title;
	private double size;
	private Point origin;
	private double imageScale;

	
	public BookmarkView(Point origin, String title, double size) {
		this.origin = origin;
		this.title = title;
		this.changeSize(size);
	}

	@Override
	public boolean contains(Point2D point) {
		double w = (this.rectangle.getWidth()/2) / imageScale;
		double h = (this.rectangle.getHeight()) / imageScale;
		double x = this.origin.getX() - w/2;
		double y = this.origin.getY() - h;
		Rectangle2D rect = new Rectangle.Double(x,y,w,h);
		return rect.contains(point);
	}

	@Override
	public Element createXMLElement(Document document) {
		Element element = document.createElement("view");
		element.setAttribute("type", URICreator.BOOKMARK);
		element.setAttribute("size",((int)this.size*100)+"");
		element.setAttribute("title",this.title);
		element.setAttribute("originX",(int)this.origin.getX()+"");
		element.setAttribute("originY",(int)this.origin.getY()+"");
		return element;
	}
	public static View restore(Element elem) {
		double size = new Integer(elem.getAttribute("size")) / 100.0;
		String title = elem.getAttribute("title");
		int originX = new Integer(elem.getAttribute("originX"));
		int originY = new Integer(elem.getAttribute("originY"));
		BookmarkView view = new BookmarkView(new Point(originX, originY), title, size);
		view.size = size;
		return view;
	}

	@Override
	public void drawLinkedShape(Graphics2D g) {
		drawShape(g);
	}

	@Override
	protected void drawShape(Graphics2D g) {
		String imageFileName;
		if (this.isSelected) {
			imageFileName = "bookmark_selected.png";
		} else {
			imageFileName = "bookmark.png";
		}
		ImageIcon imageIcon = new ImageIcon(
				GSConstants.ICON_PATH + imageFileName);
		Image image = imageIcon.getImage();
		int x = (int)this.rectangle.getX();
		int y = (int)this.rectangle.getY();
		int width = (int)this.rectangle.getWidth();
		int height = (int)this.rectangle.getHeight();
		g.drawImage(image, x, y, width, height, null);
		if (this.isProtected) {
			int xx = (int)this.rectangle.getX();
			int yy = (int)this.rectangle.getY();
			int ww = (int)(this.rectangle.getWidth() / 2);
			int hh = (int)(this.rectangle.getHeight());
			Rectangle rect = new Rectangle(xx, yy, ww, hh);
			g.setStroke(new BasicStroke(1));
			g.setColor(Color.BLACK);
			g.draw(rect);
		}
	}

	@Override
	public BookmarkView enlargedView(double ratio, double gapX, double gapY) {
		this.imageScale = ratio;
		int x = (int)(this.origin.getX() * ratio + gapX);
		int y = (int)(this.origin.getY() * ratio + gapY);
		Point newOrigin = new Point(x,y);
		BookmarkView view = new BookmarkView(newOrigin, this.title, this.size);
		view.setIsSelected(this.isSelected);
		view.isProtected = this.isProtected;
		return view;
	}

	@Override
	public Point getCenterPoint() {
		return new Point((int)rectangle.getCenterX(), (int)rectangle.getCenterY());
	}

	@Override
	public Rectangle2D getRectangle2D() {
		return this.rectangle;
	}

	@Override
	public int getType() {
		return View.BOOKMARK;
	}

	@Override
	public String getTypeString() {
		return URICreator.BOOKMARK;
	}

	@Override
	public void moveBy(Point2D d) {
		int x = (int)(this.origin.getX()+d.getX());
		int y = (int)(this.origin.getY()+d.getY());
		this.origin = new Point(x,y);
		this.changeSize(this.size);
	}

	public String getTitle() {
		return this.title;
	}

	@Override
	public String getTypeStringForDisplay() {
		return "Bookmark";
	}
	public void changeSize(double size) {
		this.size = size;
		double originX = this.origin.getX();
		double originY = this.origin.getY();
		double w = DEFAULT_WIDTH * size;
		double h = DEFAULT_HEIGHT * size;
		double x = originX - (w/4);
		double y = originY - h;
		this.rectangle = new Rectangle2D.Double(x, y, w, h);
	}

	public double getSize() {
		return this.size;
	}

	public static boolean properSize(double bookmarkSize) {
		return bookmarkSize == HUGE 
			|| bookmarkSize == LARGE
			|| bookmarkSize == NORMAL
			|| bookmarkSize == SMALL
			|| bookmarkSize == TINY;
	}


}
