package smart_gs.network;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.swing.DefaultListModel;
import javax.swing.JFileChooser;
import javax.swing.JList;
import javax.swing.JOptionPane;

import smart_gs.logical.Preference;
import smart_gs.network.ProjectActionListener.ProjectAction;
import smart_gs.swingui.WorkspaceWindow;

import jp.ac.nii.hcp.client.core.FileAttachmentManager;
import jp.ac.nii.hcp.client.core.HCPController;
import jp.ac.nii.hcp.client.core.ProjectEditor;
import jp.ac.nii.hcp.client.net.AuthenticationCache;
import jp.ac.nii.hcp.client.net.UnauthorizedException;
import jp.ac.nii.hcp.client.shared.ClientConfiguration;
import jp.ac.nii.hcp.client.shared.ClientStatus;
import jp.ac.nii.hcp.client.shared.HCPClientException;
import jp.ac.nii.hcp.shared.model.HCPImage;
import jp.ac.nii.hcp.shared.model.HCPProject;
import jp.ac.nii.hcp.shared.model.HCPProjectContent;
import jp.ac.nii.hcp.shared.model.HCPProjectHistory;
import jp.ac.nii.hcp.shared.model.HCPProjectList;
import jp.ac.nii.hcp.shared.model.HCPUserGroup;
import jp.ac.nii.hcp.shared.model.ProjectDifference;

public class ProjectHandler {

	private String action;
	private String repositoryAPIUrl;
	private String authUrl;
	private String imageServerUrl;
	private String workspaceDirectory;
	private HCPProject projectInWorkspace;
	private HCPProject projectInRepository;
	private ClientConfiguration config;
	private HCPController controller;
	private AuthenticationCache authenticationCache;
	private HCPUserGroup group;
	private String newProjectName;
	private String commitComment;
	private int revision = -1;
	private int headRevision;
	private static ProjectHandler singleton = new ProjectHandler(); 

	public ProjectHandler() {
		this.repositoryAPIUrl = Preference.getInstance().getRepositoryAPIUri();
		this.authUrl = Preference.getInstance().getAuthUri();
		this.imageServerUrl = Preference.getInstance().getImageServerUri();
		this.workspaceDirectory = Preference.getInstance().getWorkspaceDirectory();
		this.config = this.makeConfiguration();
		this.controller = this.makeHcpController();
	}
	
	public static ProjectHandler getInstance() {
		return singleton;
	}

	public List<HCPProject> getProjectsFromWorkspace() {
		HCPController hcp = HCPController.get(this.workspaceDirectory, this.config);
		return hcp.listProjects();
	}

	public List<HCPProject> getProjectsFromRepository() {
		HCPProjectList list;
		try {
			list = this.controller.searchProjects();
			return list.projects;
		} catch (IOException e) {
			e.printStackTrace();
			new ShowResultDialog("", e);
		} catch (UnauthorizedException e) {
			e.printStackTrace();
			new ShowResultDialog("", e);
		}
		return null;
	}

	public String getRepositoryAPIUrl() {
		return repositoryAPIUrl;
	}

	public void setRepositoryAPIUrl(String string) {
		this.repositoryAPIUrl = string;
	}

	public String getAuthUrl() {
		return authUrl;
	}

	public void setAuthUrl(String authUrl) {
		this.authUrl = authUrl;
	}

	public String getImageServerUrl() {
		return imageServerUrl;
	}

	public void setImageServerUrl(String imageServerUrl) {
		this.imageServerUrl = imageServerUrl;
	}

	public String getWorkspaceDirectory() {
		return workspaceDirectory;
	}

	public void setWorkspaceDirectory(String workspaceDirectory) {
		this.workspaceDirectory = workspaceDirectory;
	}

	public void create() {
		if (this.newProjectName == null) {
			new NewProjectDialogFrame().setVisible(true);
		} else {
			try {
				HCPProject project = this.controller.createProject(this.newProjectName);
				this.newProjectName = null;
				WorkspaceWindow.getInstance().openProject(project,this.workspaceDirectory);
				Preference.getInstance().setWorkspaceDirectory(this.workspaceDirectory);
				ProjectExplorer.getInstance().update();
			} catch (HCPClientException e) {
				e.printStackTrace();
				new ShowResultDialog("Create", e).setVisible(true);
				this.newProjectName = null;
			}
		}
	}
	
	private ClientConfiguration makeConfiguration() {
		ClientConfiguration config = new ClientConfiguration();
		config.repositoryAPIURL(this.repositoryAPIUrl);
		config.authURL(this.authUrl);
		config.imageServerURL(this.imageServerUrl);
		return config;
	}

	public void authenticate() {
		if (!this.authenticated()) {
			new AuthFrame("User Authentication", this).setVisible(true);
		} else {
			this.showAlreadyAuthenticatedMessage();
		}
	}

	private ClientConfiguration config() {
		ClientConfiguration config =  new ClientConfiguration();
        config.repositoryAPIURL(this.repositoryAPIUrl);
        config.authURL(this.authUrl);
        config.imageServerURL(this.imageServerUrl);
        return config;
	}
	private HCPController makeHcpController() {
		return HCPController.get(this.workspaceDirectory, this.config());
	}
	

	public List<HCPUserGroup> getGroups(AuthenticationCache auth) throws IOException, UnauthorizedException {
		return this.controller.getGroups(auth);
	}

	public AuthenticationCache makeAuthenticationCache(String id, String pass) throws IOException {
        return this.controller.login(id, pass);
	}

	public void setAuthenticationCache(AuthenticationCache auth) {
		this.authenticationCache = 	auth;
	}

	public AuthenticationCache getAuthenticationCache() {
		return this.authenticationCache;
	}

	public void selectGroup() {
		new SelectGroupFrame("Select Group", this).setVisible(true);			
	}

	public void setGroup(HCPUserGroup group) {
		this.group = group;
	}
	
	private boolean validRevsion(int revision) {
		return 0 <= revision && revision <= this.getHeadRevision();
	}

	public void setNewProjectName(String name) {
		this.newProjectName = name;
	}

	public HCPController getController() {
		return this.controller;
		
	}

	public String getAction() {
		return this.action;
	}
	
	public void setAction(String action) {
		this.action = action;	
	}

	public void setRevision(int revision) {
		this.revision = revision;
	}

	public int getRevision() {
		return revision;
	}

	public void setHeadRevision(int headRevision) {
		this.headRevision = headRevision;
	}

	public int getHeadRevision() {
		return headRevision;
	}
	
	public void makeList(JList<String> list, List<HCPProject> projects, ProjectAction action) throws HCPClientException {
		HCPController controller = this.getController();
		DefaultListModel<String> model = new DefaultListModel<String>();
		if (action == ProjectAction.NONE || action == ProjectAction.CHECKOUT || action == ProjectAction.EXPORT) {
			for (int i = 0; i < projects.size(); i++) {
				model.addElement(projects.get(i).name);
			}
		} else {
			for (int i = 0; i < projects.size(); i++) {
				String name = projects.get(i).name;
				ProjectDifference diff = controller.status(name);
				int revision = diff.revision;
				this.setHeadRevision(diff.revision);
				this.setRevision(diff.revision);
				model.addElement(name + " (head revision: " + revision + ")");
			}
		}
		list.setModel(model);		
	}

	public boolean authenticated() {
		return this.authenticationCache != null;
	}

	public void open(HCPProject project) {
		WorkspaceWindow window = WorkspaceWindow.getInstance();
		window.openProject(project, this.getWorkspaceDirectory());
		window.setFileOpened(true);
		window.autosave(window.getCurrentFile());
		window.unlock();
		window.setLastsave(0);
	}

	public void imprt() {
		HCPProject project = this.projectInWorkspace;
		if (!this.authenticated()) {
			this.authenticate();
		} else if (this.group == null) {
			this.selectGroup();
		} else if (project == null) {
			this.showNoProjectSelectedMessage(Constants.WORKSPACE);
		} else {
			try {
				this.controller.importProject(project.name, this.group.id);
				ProjectExplorer.getInstance().update();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				new ShowResultDialog("Import", e).setVisible(true);
			} catch (UnauthorizedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				new ShowResultDialog("Import", e).setVisible(true);
			} catch (HCPClientException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				new ShowResultDialog("Import", e).setVisible(true);
			}			
		}
	}

	public void commit() {
		if (!this.authenticated()) {
			this.authenticate();
		} else if (this.projectInWorkspace == null) {
			this.showNoProjectSelectedMessage(Constants.WORKSPACE);
		} else {
			CommitDialog.getInstance().setVisible(true);
			String name = this.projectInWorkspace.name;
			String comment = this.commitComment;
			HCPController controller = this.controller;
			ProjectDifference diff;
			try {
				diff = controller.status(this.projectInWorkspace.name);
			} catch (HCPClientException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				new ShowResultDialog("Commit", e).setVisible(true);
			}
			ProjectDifference commitDiff = null;
			try {
				try {
					commitDiff = controller.commitProject(name, comment, false);
					ProjectExplorer.getInstance().update();
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
					new ShowResultDialog("Commit", e).setVisible(true);
				} catch (UnauthorizedException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
					new ShowResultDialog("Commit", e).setVisible(true);
				}
			}catch(HCPClientException e) {
				if(e.getStatus() == ClientStatus.COMMIT_CONFLICTS) {
					//Փ˔AcommitDiffɏՓ˂Reci[ĂB
					System.out.println(commitDiff);
				} else {
					try {
						throw e;
					} catch (HCPClientException e1) {
						// TODO Auto-generated catch block
						e1.printStackTrace();
						new ShowResultDialog("Commit", e).setVisible(true);
					}
				}
			}
		}
	}
	
	public void checkout() {
		if (!this.authenticated()) {
			this.authenticate();
		} else if (this.projectInRepository == null) {
			this.showNoProjectSelectedMessage(Constants.REPOSITORY);
		} else {
			try {
				
				this.controller.checkoutProject(this.projectInRepository.name);
				ProjectExplorer.getInstance().update();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				new ShowResultDialog("Checkout", e).setVisible(true);
			} catch (UnauthorizedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				new ShowResultDialog("Checkout", e).setVisible(true);
			} catch (HCPClientException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				new ShowResultDialog("Checkout", e).setVisible(true);
			}
		}
	}
	
	public void update() {
		if (!this.authenticated()) {
			this.authenticate();
		} else if (this.projectInWorkspace == null) {
			this.showNoProjectSelectedMessage(Constants.WORKSPACE);
		} else {
			try {
				List<HCPProjectContent> notUpdated = new ArrayList<HCPProjectContent>();
//				List<HCPProjectContent> updated = new ArrayList<HCPProjectContent>();
				this.controller.updateProject(this.projectInWorkspace.name, false,
						notUpdated);
				new ShowResultDialog("Update", notUpdated).setVisible(true);
				ProjectExplorer.getInstance().update();

			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				new ShowResultDialog("Update", e).setVisible(true);
			} catch (UnauthorizedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				new ShowResultDialog("Update", e).setVisible(true);
			} catch (HCPClientException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
				new ShowResultDialog("Update", e).setVisible(true);
			}
		}
	}

	public void export() {
		if (!this.authenticated()) {
			this.authenticate();
		} else if (this.projectInRepository == null) {
			this.showNoProjectSelectedMessage(Constants.REPOSITORY);
		} else {
			new ExportOptionDialog(this.projectInRepository).setVisible(true);
		}
		
	}

	public void compare() {
		if (this.projectInWorkspace == null) {
			this.showNoProjectSelectedMessage(Constants.WORKSPACE);
		} else {
			//TODO
		}

	}
	
	public void remove() {
		if (this.projectInWorkspace == null) {
			this.showNoProjectSelectedMessage(Constants.WORKSPACE);
		} else {
			int option = this.showRemovalConfirmDialog();
			if (option == JOptionPane.OK_OPTION) {
				this.controller.removeProject(this.projectInWorkspace.name);
				ProjectExplorer.getInstance().update();
			}
		}
		
	}


	
	private int showRemovalConfirmDialog() {
		return JOptionPane.showConfirmDialog(ProjectExplorer.getInstance(),
				"Do you want to remove \"" + this.projectInWorkspace.name + "\"?",
				"Remove Project",
				JOptionPane.OK_CANCEL_OPTION);
	}

	private void showNoProjectSelectedMessage(String location) {
		JOptionPane.showMessageDialog(null,
				"Select a project from " + location + ".", "Message",
				JOptionPane.INFORMATION_MESSAGE);		
	}
	private void showAlreadyAuthenticatedMessage() {
		JOptionPane.showMessageDialog(null,
				"Already authenticated.", "Message",
				JOptionPane.INFORMATION_MESSAGE);				
	}

	public void setProjectInWorkspace(HCPProject projectInWorkspace) {
		this.projectInWorkspace = projectInWorkspace;
	}

	public HCPProject getProjectInWorkspace() {
		return projectInWorkspace;
	}

	public void setProjectInRepository(HCPProject projectInRepository) {
		this.projectInRepository = projectInRepository;
	}

	public HCPProject getProjectInRepository() {
		return projectInRepository;
	}

	public List<HCPProjectHistory> fetchHistories(String name, int i, int j) throws IOException, UnauthorizedException {
		return this.controller.fetchHistories(name, i, j);
	}

	
	public void perform() {
		if (action == Constants.OPEN) {
			this.open(this.projectInWorkspace);
		} else if (action == Constants.IMPORT) {
			this.imprt();
		} else if (action == Constants.CHECKOUT) {
			this.checkout();
		} else if (action == Constants.COMMIT) {
			this.commit();
		} else if (action == Constants.EXPORT) {
			this.export();
		} else if (action == Constants.UPDATE) {
			this.update();
		} else if (action == Constants.CREATE) {
			this.create();
		} else if (action == Constants.AUTHENTICATE) {
			this.authenticate();
		} else if (action == Constants.REMOVE) {
			this.remove();
		} else if (action == Constants.COMPARE) {
			this.compare();
		} else if (action == Constants.ADD_ATTACHMENT) {
			this.addAttachment();
		}
	}

	public List<HCPImage> getImagesFromWorkspace() {
		// TODO Auto-generated method stub
		return null;
	}

	public void addImages() {
		String workspaceDir = Preference.getInstance().getWorkspaceDirectory();
		String imageFolderPath = workspaceDir + "/" + "image";
		JFileChooser fileChooser = new JFileChooser(imageFolderPath);
		fileChooser.setDialogTitle("Select an image");
		int option = fileChooser.showDialog(WorkspaceWindow.getInstance(), "Select");
		if (option != JFileChooser.APPROVE_OPTION) return;
		File file = fileChooser.getSelectedFile();
		int beginIndex = imageFolderPath.length();
		String imagePath = file.getAbsolutePath().substring(beginIndex);
		new AddImageToProjectDialog(WorkspaceWindow.getInstance(), Constants.ADD_IMAGE, file.getName(), imagePath).setVisible(true);
	}
	public void addAttachment() {
		if (!ProjectHandler.getInstance().authenticated()) {
			ProjectHandler.getInstance().authenticate();
		} else {
			JFileChooser fileChooser = new JFileChooser(".");
			fileChooser.setDialogTitle("Select an attachment file");
			int option = fileChooser.showDialog(WorkspaceWindow.getInstance(), "Select");
			if (option != JFileChooser.APPROVE_OPTION) return;
			File file = fileChooser.getSelectedFile();
			new AddAttachmentToProjectDialog(WorkspaceWindow.getInstance(), "Add an attachment file", file).setVisible(true);
		}
	}

	private void showNoProjectSelectedMessage() {
		JOptionPane.showMessageDialog(ProjectExplorer.getInstance(), "No project is selected.", "Error", JOptionPane.ERROR_MESSAGE);
	}

	public ProjectEditor getEditor(HCPProject project) throws HCPClientException {
		return this.controller.openProject(project.name);
	}

	public FileAttachmentManager getAttachmentManager(String name) throws HCPClientException {
		return this.controller.getAttachmentManager(name);
	}

	

}