#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
//#include <malloc.h>

#include "posix_key.hpp"

posix_key::posix_key ( void )
{
    kbport = -1;
    settings = get_termsettings(ctermid(NULL));
}

posix_key::~posix_key ( void )
{
    set_termsettings(ctermid(NULL),settings);
}

bool posix_key::kbhit(void)
{
    int rtnval = 0;

    fd_set rset;
    static struct timeval tvptr;
    struct termios tty;

    int status = 0;

    if (kbport == -1)
    {

            /* Open the controlling terminal for read-only access */

        kbport = open(ctermid(NULL),O_RDONLY);

        if (kbport > -1)
        {
                /* Get the terminal attributes */

            if (tcgetattr(kbport,&tty) < 0)
            {
                fprintf(stderr,"\nERROR:  unable to get tty attributes for %s\n",ctermid(NULL));
                exit(-1);
            }
            else
            {
                tty.c_iflag = 0x0000; /* We don't need any input flags */
                tty.c_oflag = ONLCR | OPOST;
                tty.c_cflag = CREAD | CSIZE;
                tty.c_lflag = IEXTEN; /* Note - noncanonical mode, since no ICANON */

                tty.c_cc[VERASE] = 0x7F;   /* Since we're in noncanonical mode, we ignore
                                              backspace, and its value *should* be 127;
                                              to enable me to handle it, I make *sure* it's
                                              127, and then have a special case to map 127
                                              to 8 (^H) inside getkey()               */

                if (tcsetattr(kbport,TCSANOW,&tty) < 0)
                {
                    fprintf(stderr,"\nERROR:  unable to set tty attributes for %s\n",ctermid(NULL));
                }
            }

                /* Set up our "return immediately" structure for select */

            tvptr.tv_sec = 0;
            tvptr.tv_usec = 0;

        }
    }

        /* Initialize the read set to zero */

    FD_ZERO(&rset);

        /* And now turn on the read set */

    FD_SET(kbport,&rset);

    status = select(kbport + 1,&rset,NULL,NULL,&tvptr);

    if (status == -1) /* Error */
    {
        fprintf(stderr,"\nERROR:  kbhit():  select() returned -1\n");
        exit(-1);
    }
    else if (status > 0)
    {
        if (FD_ISSET(kbport,&rset))
        {
            rtnval = 1;
        }
    }

    return(rtnval);

}

bool posix_key::getkey(char *dst)
{
    char *extended_key = NULL;
    int extended_key_length = 0, numread = 0;
    char ch = 0;
    char rtnval = 0;

    if (kbport != -1)
    {
        read(kbport,&ch,1);

        if (ch == 27 && kbhit())
        {
            read(kbport,&ch,1);

            if (ch == 91 && kbhit())
            {
                do
                {
                    numread = read(kbport,&ch,1);

                    if ((numread == 1) && (ch != 126))
                    {
                        extended_key_length++;

                        extended_key = (char *) realloc(extended_key,extended_key_length + 1);

                        extended_key[extended_key_length - 1] = ch;

                            /* Set the last element to zero */

                        extended_key[extended_key_length] = 0;
                    }
                }
                while ((numread == 1) && (ch != 126));

                ch = (char) atoi(extended_key);
                free(extended_key);

                *dst = ch;
                rtnval = 1; /* Extended key found */

            }
            else
            {
                *dst = ch;
                rtnval = 0;
            }
        }
        else if (ch == 127) /* Backspace */
        {
            *dst = 8;         /* Return ^H */
            rtnval = 0;
        }
        else
        {
            *dst = ch;
            rtnval = 0; /* Not a funckey */
        }
    }
    return (rtnval == 1);
}

struct termios *posix_key::get_termsettings(char *devname)
{
    int port = 0;
    struct termios *rtnval = NULL;

    if (devname != (char *) NULL)
    {
        port = open(devname,O_RDONLY);

        if (port > -1)
        {

            rtnval = (struct termios *) malloc(sizeof(struct termios));

            if (rtnval != (struct termios *) NULL)
            {
                if (tcgetattr(port,rtnval) == -1)
                {
                    free(rtnval);
                    rtnval = (struct termios *) NULL;
                }
            }

            close(port);
        }
    }

    return(rtnval);
}

int posix_key::set_termsettings(char *devname, struct termios *termios)
{
    int port = 0;
    int rtnval = 0;

    if (devname != (char *) NULL)
    {
        port = open(devname,O_RDWR);

        if (port > -1)
        {

            if (tcsetattr(port,TCSANOW,termios) != -1)
            {
                rtnval = 1;
            }

            close(port);
        }
    }

    return(rtnval);
}

