#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <posix/posix_serial.hpp>

extern "C"
{
#include <xmodem/xmodem.h>
}


static FILE *fp = NULL;
static posix_serial *sp;

extern "C" void posix_putchar( int c )
{
    sp->write( c & 0xff );
}

extern "C" int posix_getchar( int timeout )
{
    if (sp->dataready(timeout))
    {
        char c;
        if ( sp->read(&c) )
        {
            return c;
        }
    }
    return -1;
}

char buf[XMODEM_BLOCK_SIZE];
extern "C" xmodem_buffer_t get_data( int block_no )
{
    xmodem_buffer_t data;
    if ( fp )
    {
        data.size = fread( buf, sizeof( char), XMODEM_BLOCK_SIZE, fp );
        data.buffer = (unsigned char *)buf;
    }
    else
    {
        data.size = 0;
    }
    return data;
}

static void usage( char *name )
{
    fprintf ( stderr, "%s:  %s -[XYZRS] [filename]\n", name, name );
    fprintf ( stderr, "\t-X: xmodem\n" );
    fprintf ( stderr, "\t-Y: ymodem\n" );
    fprintf ( stderr, "\t-Z: zmodem\n" );
    fprintf ( stderr, "\t-S:  send\n" );
    fprintf ( stderr, "\t-R:  recv\n" );
    
    exit(-1);
}

static void opt_error_modem_mode( char *name )
{
    fprintf ( stderr, "%s:  You must set one of X, Y, Z\n", name );
    exit(-1);
}

static void opt_error_send_mode( char *name )
{
    fprintf ( stderr, "%s:  You must set one of S, R\n", name );
    exit(-1);
}

typedef enum modem_mode_t
{
    NO_MODEM_MODE,
    XMODEM_MODE,
    YMODEM_MODE,
    ZMODEM_MODE
};

typedef enum send_mode_t
{
    NO_SEND_MODE,
    SEND_MODE,
    RECV_MODE
};

int main(int argc, char **argv)
{
    int opt;
    modem_mode_t modem_mode = NO_MODEM_MODE;
    send_mode_t  send_mode  = NO_SEND_MODE;

    while((opt = getopt(argc, argv, "XxYyZzSsRr")) != -1)
    {
        
        switch(opt)
        {
            case 'x':
            case 'X':
                if ( modem_mode != NO_MODEM_MODE ) opt_error_modem_mode( argv[0] );
                modem_mode = XMODEM_MODE;
                break;
            case 'y':
            case 'Y':
                if ( modem_mode != NO_MODEM_MODE ) opt_error_modem_mode( argv[0] );
                modem_mode = YMODEM_MODE;
                break;
            case 'z':
            case 'Z':
                if ( modem_mode != NO_MODEM_MODE ) opt_error_modem_mode( argv[0] );
                modem_mode = ZMODEM_MODE;
                break;
            case 's':
            case 'S':
                if ( send_mode != NO_SEND_MODE ) opt_error_send_mode( argv[0] );
                send_mode = SEND_MODE;
                break;
            case 'r':
            case 'R':
                if ( send_mode != NO_SEND_MODE ) opt_error_send_mode( argv[0] );
                send_mode = RECV_MODE;
                break;
        }
    }

    switch( send_mode )
    {
        case SEND_MODE:
            switch( modem_mode )
            {
                case XMODEM_MODE:
                    if (optind < argc)
                    {
                        sp = new posix_serial( argv[optind] );
                        fp = fopen( argv[2], "rb" );
                        if ( sp->open( 115200 ) )
                        {
                            xmodem_send( 10, get_data, posix_putchar, posix_getchar );
                        }
                        fclose( fp );
                    }
                    else
                    {
                        fprintf ( stderr, "%s: No send filename.\n", argv[0] );
                    }
                    break;
                case YMODEM_MODE:
                    fprintf ( stderr, "%s: does not support YMODEM Send\n", argv[0] );
                    break;
                case ZMODEM_MODE:
                    fprintf ( stderr, "%s: does not support ZMODEM Send\n", argv[0] );
                    break;
                case NO_MODEM_MODE:
                    usage( argv[0] );
            }
            break;
        case RECV_MODE:
            switch( modem_mode )
            {
                case XMODEM_MODE:
                    fprintf ( stderr, "%s: does not support XMODEM Recv\n", argv[0] );
                    break;
                case YMODEM_MODE:
                    fprintf ( stderr, "%s: does not support YMODEM Recv\n", argv[0] );
                    break;
                case ZMODEM_MODE:
                    fprintf ( stderr, "%s: does not support ZMODEM Recv\n", argv[0] );
                    break;
                case NO_MODEM_MODE:
                    usage( argv[0] );
            }
            break;
        case NO_SEND_MODE:
            usage( argv[0] );
    }
}
