#include <string.h>

#include <internal_api.h>
#include <plathome.h>
#include <queue.h>

#include <shell/shell_printf.h>
#include <shell/getline.h>
int shell_execline( char *line );


static bios_shell_mode_t current_mode = UART_DIRECT;
static char shell_prompt[32];

typedef struct
{
    int read_timeout;
    int write_timeout;
    int send_interrupt_enabled : 1;
    int recv_interrupt_enabled : 1;
    bios_uart_callback_t recv_handler;
    bios_uart_callback_t send_handler;
} uart_port_config_t;

static uart_port_config_t port_config[UART_NUM_OF_PORTS];

QUEUE_INIT(UART_RX, UART_QUEUE_SIZE, UART_NUM_OF_PORTS);
QUEUE_INIT(UART_TX, UART_QUEUE_SIZE, UART_NUM_OF_PORTS);

unsigned char uart_buffer[ 1 << UART_QUEUE_SIZE ];

int _uart_init( int port, bios_uart_param_t *params )
{
    _uart_send_interrupt_disable( port );
    _uart_recv_interrupt_disable( port );
    port_config[port].read_timeout  = -1;
    port_config[port].write_timeout = -1;
    QUEUE_CLEAR(UART_RX, port);
    QUEUE_CLEAR(UART_TX, port);
    return _plathome_uart_init( port, params );
}

bios_shell_mode_t _shell_mode_get( void )
{
    return current_mode;
}

void _shell_mode_set( bios_shell_mode_t mode, char *prompt )
{
    if ( mode == UART_SHELL_SINGLE )
    {
        int recent_status;
        bios_shell_mode_t recent_mode;
        _uart_init( SHELL_UART_PORT, 0 );
        recent_mode = current_mode;
        recent_status = _interrupt_disable();
        current_mode = mode;
        shell(prompt);
        _interrupt_restore( recent_status );

        mode = recent_mode;
    }

    if ( mode == UART_SHELL_MULTI || mode == UART_SHELL_GDB )
    {
        _uart_init( SHELL_UART_PORT, 0 );
        _uart_interrupt_enable( SHELL_UART_PORT );
        _uart_recv_interrupt_enable( SHELL_UART_PORT );
        _uart_send_interrupt_enable( SHELL_UART_PORT );
        if ( mode == UART_SHELL_MULTI && current_mode != UART_SHELL_MULTI )
        {
            strncpy( shell_prompt, prompt, 32 );
            shell_prompt[31] = 0;
            shell_printf ( "%s", shell_prompt ); // vvg\
        }
        current_mode = mode;
    }
}
    

int _uart_recv_interrupt_enable( int port )
{
    QUEUE_CLEAR(UART_RX, port);
    _plathome_uart_recv_interrupt_enable( port );
    return BIOS_NOERR;
}

int _uart_recv_interrupt_disable( int port )
{
    _plathome_uart_recv_interrupt_disable( port );
    QUEUE_CLEAR(UART_RX, port);
    return BIOS_NOERR;
}

int _uart_send_interrupt_enable( int port )
{
    QUEUE_CLEAR(UART_TX, port);
    _plathome_uart_send_interrupt_enable( port );
    return BIOS_NOERR;
}

int _uart_send_interrupt_disable( int port )
{
    _plathome_uart_send_interrupt_disable( port );
    QUEUE_CLEAR(UART_TX, port);
    return BIOS_NOERR;
}

int _uart_set_timeout( int port, int read_timeout, int write_timeout )
{
    port_config[port].read_timeout = read_timeout;
    port_config[port].write_timeout = write_timeout;
    return BIOS_NOERR;
}

void _uart_recv_callback( int port )
{
    int i;
    if ( current_mode == UART_SHELL_MULTI )
    {
        char *linebuf;
        int c;
        c = _uart_getc(port);
        if ( (linebuf = getline( c )) != 0 )
        {
            shell_execline( linebuf );
            shell_printf ( "%s", shell_prompt ); // vvg\
        }
    }
    else
    {
        while( _uart_rx_ready( port ) )
        {
            int c;
            c = _uart_getc(port);
            if  ( c == 0x03 )
            {
                _handle_exception();
            }
            
            QUEUE_IN(UART_RX, port, (unsigned char)c);
            if ( QUEUE_ISFULL(UART_RX, port) )
            {
                if ( port_config[port].recv_handler )
                {
                    bios_uart_buffer_t buf;
                    buf.length = QUEUE_STATUS(UART_RX, port);
                    for( i = 0; i < buf.length; i ++ )
                    {
                        QUEUE_OUT(UART_RX, port, uart_buffer[i] );
                    }
                    buf.buffer = uart_buffer;
                    ((bios_uart_callback_t)(port_config[port].recv_handler))(&buf);
                }
                else
                {
                    QUEUE_CLEAR(UART_RX, port);
                }
            }
        }
        if ( !QUEUE_ISEMPTY(UART_RX, port) )
        {
            if( port_config[port].recv_handler )
            {
                bios_uart_buffer_t buf;
                buf.length = QUEUE_STATUS(UART_RX, port);
                for( i = 0; i < buf.length; i ++ )
                {
                    QUEUE_OUT(UART_RX, port, uart_buffer[i] );
                }
                buf.buffer = uart_buffer;
                ((bios_uart_callback_t)(port_config[port].recv_handler))(&buf);
            }
        }
    }
}

void _uart_send_callback( int port )
{
    int i;
    while( _uart_tx_ready( port ) )
    {
        unsigned char data;
        if( QUEUE_ISEMPTY(UART_TX, port) )
        {
            if ( port_config[port].send_handler )
            {
                bios_uart_buffer_t buf;
                buf.buffer = uart_buffer;
                buf.length = ((bios_uart_callback_t)(port_config[port].send_handler))(&buf);
                buf.length = QUEUE_STATUS(UART_RX, port);
                for( i = 0; i < buf.length; i ++ )
                {
                    QUEUE_IN(UART_TX, port, uart_buffer[i] );
                }
            }
        }
        if ( !QUEUE_ISEMPTY( UART_TX, port ) )
        {
            QUEUE_OUT(UART_TX, port, data);
            _uart_putc(port, data);
        }
        else
        {
            break;
        }
    }
}

int _uart_recv_interrupt_set_handler( int port, bios_uart_callback_t handler )
{
    if ( port < 0 || port >= UART_NUM_OF_PORTS ) return BIOS_UART_INVALID_PORT;
    port_config[port].recv_handler = handler;
    return BIOS_NOERR;
}

int _uart_send_interrupt_set_handler( int port, bios_uart_callback_t handler )
{
    if ( port < 0 || port >= UART_NUM_OF_PORTS ) return BIOS_UART_INVALID_PORT;
    port_config[port].send_handler = handler;
    return BIOS_NOERR;
}

typedef int (*uart_check_func_t)( int port );
static int _uart_wait_timeout( uart_check_func_t check_func, int port )
{
    int i, j;
    if ( port_config[port].read_timeout == -1 )
    {
        while ( !check_func( port ) ) {}
        return 0;
    }
    else
    {
        for( j = 0; j < port_config[port].read_timeout; j++ )
        {
            for( i = 0; i < UART_CHECK_COUNT_1MS; i++ )
            {
                if ( check_func( port ) )
                {
                    return 0;
                }
            }
        }
    }
    return 1;
}

int _uart_read_noi( int port, char *str, int length )
{
    int count = 0;
    while( (length-- > 0) && (_uart_wait_timeout( _uart_rx_ready, port ) == 0) )
    {
        *(str+count) =  (char)_uart_getc( port );
        count ++;
    }
    return count;
}

int _uart_write_noi( int port, char *str, int length )
{
    int count = 0;
    while( (length-- > 0) && (_uart_wait_timeout( _uart_tx_ready, port ) == 0) )
    {
        _uart_putc( port, *(str + count) );
        count ++;
    }
    return count;
}

int _uart_read_i( int port, char *str, int length )
{
    int count = 0;
    while( (length-- > 0) && (!QUEUE_ISEMPTY(UART_RX, port)) )
    {
        QUEUE_OUT( UART_RX, port, *(str + count) );
        count ++;
    }
    return count;
}

int _uart_write_i( int port, char *str, int length )
{
    int count = 0;
    while( (length-- > 0) && (!QUEUE_ISFULL(UART_TX, port)) )
    {
        QUEUE_IN( UART_TX, port, *(str + count) );
        count ++;
    }
    while ( _uart_tx_ready( port ) && (!QUEUE_ISEMPTY(UART_TX, port)) )
    {
        unsigned char ch;
        QUEUE_OUT( UART_TX, port, ch );
        _uart_putc(port, ch);
    }

    return count;
}

