#include <at91/reg_r40807.h>
#include <bios_api.h>
#include <drivers.h>
#include <plathome.h>
#include <queue.h>

static StructUSART * const ports[2] = 
{
    USART0_BASE,
    USART1_BASE
};

QUEUE_EXT(UART_RX, UART_QUEUE_SIZE, UART_NUM_OF_PORTS);
QUEUE_EXT(UART_TX, UART_QUEUE_SIZE, UART_NUM_OF_PORTS);

extern void _uart_recv_callback( int port );
extern void _uart_send_callback( int port );

static void uart0_handler( void )
{
    if ( (USART0_BASE->US_CSR & USART0_BASE->US_IMR) & US_RXRDY )
    {
        _uart_recv_callback(0);
    }

    if ( (USART0_BASE->US_CSR & USART0_BASE->US_IMR) & US_TXRDY )
    {
        _uart_send_callback(0);
    }
}

static void uart1_handler( void )
{
    if ( (USART1_BASE->US_CSR & USART1_BASE->US_IMR) & US_RXRDY )
    {
        _uart_recv_callback(1);
    }

    if ( (USART1_BASE->US_CSR & USART1_BASE->US_IMR) & US_TXRDY )
    {
        _uart_send_callback(1);
    }
}

int _aki_arm7_uart_init( int port, bios_uart_param_t *params )
{
    unsigned int control;
    switch ( port )
    {
        case 0:
            PIO_PDR = PIOTXD0 + PIORXD0;
            break;
        case 1:
            PIO_PDR = PIOTXD1 + PIORXD1;
            break;
    }

    ports[port]->US_CR = US_RXDIS + US_TXDIS + US_RSTRX + US_RSTTX;
    
    if ( params == 0 )
    {
        control = US_ASYNC_MODE; //Default Mode
    }
    else
    {
        control = US_CHMODE_NORMAL;
        switch( params->length )
        {
            case CHAR_5BIT: control |= US_CHRL_5; break;
            case CHAR_6BIT: control |= US_CHRL_6; break;
            case CHAR_7BIT: control |= US_CHRL_7; break;
            case CHAR_8BIT: control |= US_CHRL_8; break;
        }
        
        switch( params->parity )
        {
            case PARITY_EVEN: control |= US_PAR_EVEN; break;
            case PARITY_ODD:  control |= US_PAR_ODD;  break;
            case PARITY_NONE: control |= US_PAR_NO;   break;
        }
        
        switch( params->stopbit )
        {
            case STOPBIT_1:  control |= US_NBSTOP_1;   break;
            case STOPBIT_15: control |= US_NBSTOP_1_5; break;
            case STOPBIT_2:  control |= US_NBSTOP_2;   break;
        }
        control |= US_CLKS_MCK;
    }
    ports[port]->US_MR = control;

	ports[port]->US_RCR = 0;
	ports[port]->US_TCR = 0;

    if ( params == 0 )
    {
        ports[port]->US_BRGR= US_ASYNC_BAUD_RATE( 33333333, 115200 );
    }
    else
    {
        switch( params->speed )
        {
            case B9600:  ports[port]->US_BRGR= US_ASYNC_BAUD_RATE( 33333333, 9600 );   break;
            case B38400: ports[port]->US_BRGR= US_ASYNC_BAUD_RATE( 33333333, 38400 );  break;
            case B57600: ports[port]->US_BRGR= US_ASYNC_BAUD_RATE( 33333333, 57600 );  break;
            case B115200:ports[port]->US_BRGR= US_ASYNC_BAUD_RATE( 33333333, 115200 ); break;
        }
    }
	ports[port]->US_CR  = US_RXEN + US_TXEN;
    
    return BIOS_NOERR;
}

int _aki_arm7_uart_rx_ready( int port )
{
	return (ports[port]->US_CSR & US_RXRDY);
}

int _aki_arm7_uart_tx_ready( int port )
{
    return (ports[port]->US_CSR & (US_TXRDY | US_TXEMPTY));
}

int _aki_arm7_uart_getc( int port )
{
	if ( (ports[port]->US_CSR & US_RXRDY) != 0 )
    {
        return ports[port]->US_RHR;
    }
    return BIOS_UART_RCVCHAR_ERROR;
}

int _aki_arm7_uart_putc( int port, unsigned char c )
{
    if ( (ports[port]->US_CSR & (US_TXRDY | US_TXEMPTY)) )
    {
        ports[port]->US_THR = c;
        return BIOS_NOERR;
    }
    return BIOS_UART_SENDCHAR_ERROR;
}

int _aki_arm7_uart_interrupt_enable( int port )
{
    if ( port != 0 && port != 1 ) return BIOS_UART_INVALID_PORT;
    if ( port == 0 )
    {
        _aki_arm7_interrupt_set_handler( port+2, (bios_interrupt_handler_t)uart0_handler );
    }
    if ( port == 1 )
    {
        _aki_arm7_interrupt_set_handler( port+2, (bios_interrupt_handler_t)uart1_handler );
    }
    QUEUE_CLEAR( UART_RX, port );
    QUEUE_CLEAR( UART_TX, port );
    _aki_arm7_interrupt_enable_num( port+2 );
    return BIOS_NOERR;
}

int _aki_arm7_uart_interrupt_disable( int port )
{
    if ( port != 0 && port != 1 ) return BIOS_UART_INVALID_PORT;
    _aki_arm7_interrupt_disable_num( port+2 );
    QUEUE_CLEAR( UART_RX, port );
    QUEUE_CLEAR( UART_TX, port );
    return BIOS_NOERR;
}

int _aki_arm7_uart_send_interrupt_enabled( int port )
{
    return (ports[port]->US_IMR & US_TXRDY);
}

int _aki_arm7_uart_send_interrupt_enable( int port )
{
    ports[port]->US_IER = US_TXRDY;
    return BIOS_NOERR;
}

int _aki_arm7_uart_send_interrupt_disable( int port )
{
    ports[port]->US_IDR = US_TXRDY;
    return BIOS_NOERR;
}

int _aki_arm7_uart_recv_interrupt_enabled( int port )
{
    return (ports[port]->US_IMR & US_RXRDY);
}

int _aki_arm7_uart_recv_interrupt_enable( int port )
{
    ports[port]->US_IER = US_RXRDY;
    return BIOS_NOERR;
}

int _aki_arm7_uart_recv_interrupt_disable( int port )
{
    ports[port]->US_IDR = US_RXRDY;
    return BIOS_NOERR;
}
