require 'timeout'
require 'socket'
require 'net/ldap'

module LDAP
  def self.setup_connection(options)
    h = {
      :host => options[:ldap_host],
      :port => options[:ldap_port],
      :base => options[:ldap_base_dn]
    }

    if (user = options[:ldap_bind_dn]) && (pass = options[:ldap_bind_password])
      h[:auth] = {:method => :simple, :username => user, :password => pass}
    end

    @attr_name = {
      :user => options[:ldap_user_attribute],
      :first_name => options[:ldap_first_name_attribute],
      :last_name => options[:ldap_last_name_attribute],
      :mail => options[:ldap_mail_attribute]
    }
    @connection = Net::LDAP.new(h)
  end

  def self.check_connection
    return false unless @connection
    timeout(0.5) { TCPSocket.new(@connection.host, @connection.port).close }
    @connection.bind
  rescue Timeout::Error
    false
  end

  def self.check_bind(username, password)
    return false unless check_connection

    filter = Net::LDAP::Filter.eq(@attr_name[:user], username)
    !!@connection.bind_as(:filter => filter, :password => password)
  end

  def self.search_attributes(username)
    return {} unless @connection

    filter = Net::LDAP::Filter.eq(@attr_name[:user], username)
    attrs = [:mail, :first_name, :last_name].map {|key|
      name = @attr_name[key]
      name.nil? || name.empty? ? nil : name
    }.compact
    h = {}
    @connection.search(:filter => filter, :attributes => attrs) {|entry|
      attrs.each {|attr| h[attr] = entry[attr].first}
    }
    h
  end
end
