require 'find'
require 'service'

module BackupRestore
  INSTALL_DIR = File.expand_path(ENV["PACKAGE_HOME"])

  def targets
    unless @targets
      @targets = %w[jenkins opends redmine subversion]
      @targets.delete("opends") unless Service.config["opends"]
    end
    @targets
  end

  def sync(src, dest, excludes=nil)
    [src, dest].each {|d| 
      raise "#{d} is not directory." unless File.directory?(d)
    }

    src = File.expand_path(src)
    dest = File.expand_path(dest)
    updated = {}

    Dir.chdir(dest) {
      Find.find(".") {|old_file|
        next if old_file == "."
        Find.prune if excludes && excludes.any? {|ex| old_file.include?(ex)}

        new_file = File.join(src, old_file)
        unless File.exists?(new_file)
          puts "Remove " + File.expand_path(old_file)
          FileUtils.rm_rf(old_file)
          next
        end

        next if File.directory?(old_file)

        old_st = File.stat(old_file)
        new_st = File.stat(new_file)
        if old_st.mtime != new_st.mtime || old_st.size != new_st.size
          updated[old_file] = true
          File.delete(old_file)
        end
      }
    }

    Dir.chdir(src) {
      Find.find(".") {|from|
        next if from == "."
        Find.prune if excludes && excludes.any? {|ex| from.include?(ex)}

        to = File.join(dest, from[1..-1])
        next if File.exists?(to)

        puts (updated[from] ? "Update " : "Add ") + to
        if File.directory?(from)
          Dir.mkdir(to)
        else
          st = File.stat(from)
          FileUtils.cp(from, to)
          File.utime(st.atime, st.mtime, to)
        end
      }
    }
  end

  def system_or_raise(command)
    raise "\"#{command}\" failed" unless system command
  end

  def exec(method, args)
    method = method.to_sym
    target, dir = args
    unless target == "all" || targets.include?(target)
      raise "invalid target #{target}"
    end
    dir = dir ? File.expand_path(dir) : File.join(INSTALL_DIR, "backup")

    service_script = File.join(INSTALL_DIR, "script/service.bat")
    system(service_script, "stop") if method == :restore

    Dir.chdir(INSTALL_DIR) {
      (target == "all" ? targets : [target]).each {|target|
        puts "--- #{target} ---"

        backup_dir = File.join(dir, target)
        case method
        when :backup
          FileUtils.mkdir_p(backup_dir)
        when :restore
          unless File.directory?(backup_dir)
            puts "Backup files not found"
            next
          end
        else
          raise "Invalid method #{method}"
        end

        
        case target
          when "jenkins"
            jenkins_dir = "jenkins/home"
            excludes = ["/workspace", "./plugins", "./updates", "./war"]
            if method == :backup
              sync(jenkins_dir, backup_dir, excludes)
            else
              sync(backup_dir, jenkins_dir, excludes)
            end

          when "opends"
            Dir.chdir("opends/bat") {
              if method == :backup
                FileUtils.rm_rf(backup_dir)
                system_or_raise(%[backup --compress -a -d "#{backup_dir}"])
              else
                Dir.glob(backup_dir + "/*/").each {|d|
                  system_or_raise(%[restore -d "#{d}"])
                }
              end
            }

          when "redmine"
            backup_db = backup_dir + "/redmine.db"
            files_dir = backup_dir + "/files"

            if method == :backup
              puts "Backup database"
              system_or_raise(%[sqlite3 redmine/db/redmine.db ".backup '#{backup_db}'"])
              Dir.mkdir(files_dir) unless File.exists?(files_dir)
              sync("redmine/files", files_dir)
            else
              if File.file?(backup_db)
                puts "Restore database"
                system_or_raise(%[sqlite3 redmine/db/redmine.db ".restore '#{backup_db}'"])
              end

              sync(files_dir, "redmine/files") if File.directory?(files_dir)
            end

          when "subversion"
            if method == :backup
              Dir.chdir("subversion/repos") {
                Dir.glob("*/").each {|svndir|
                  puts "Backup repository #{svndir[0..-2]}"
                  to = File.join(backup_dir, svndir)
                  FileUtils.rm_rf(to)
                  system_or_raise(%[svnadmin hotcopy #{svndir} "#{to}"])
                }
              }
            else
              svndirs = Dir.chdir(backup_dir) {Dir.glob("*/")}
              svndirs.each {|svndir|
                puts "Restore repository #{svndir[0..-2]}"
                from = File.join(backup_dir, svndir)
                to = "subversion/repos/" + svndir
                FileUtils.rm_rf(to)
                system_or_raise(%[svnadmin hotcopy #{from} #{to}])
              }
            end
        end
      }
    }

    system(service_script, "start") if method == :restore
  end

  def backup(args)
    exec :backup, args
  end

  def restore(args)
    exec :restore, args
  end

  def restore_usage
    warn <<-EOT
Usage: restore <target> [<source>]

By default, restore from RedmineLE/backup directory.

Targets:
  all
  #{targets.join("\n  ")}
    EOT
  end
end
