require 'subversion_hook/base'
require 'iconv'

module SubversionHook
  class PreCommit < Base
    attr_reader :errors

    def initialize(repos, transaction)
      super repos
      self.transaction = transaction

      @errors = {}
      (load_config["pre_commit"] || {}).each {|method, options|
        self.send(method, options)
      }

      unless @errors.size == 0
        warn_error_messages
        exit 1
      end
    end

    def warn_error_messages
      if a = @errors[:commit_message]
        warn "The following keywords are not found in commit message:"
        warn a.join("\n")
      end

      if h = @errors[:encoding]
        h.each {|encoding, messages|
          warn "The following files are not #{encoding}:"
          warn messages.join("\n")
        }
      end

      if h = @errors[:whitespace]
        h.each {|type, files|
          warn (type != "both") ?
            "The following files are not indented with #{type} only:" :
            "The following files are contains a mixture of tabs and spaces:"
          warn files.join("\n")
        }
      end

      if h = @errors[:linefeed]
        h.each {|type, files|
          warn (type != "both") ?
            "The following files are not ended with #{type.upcase} only:" :
            "The following files contain a mixture of line endings:"
          warn files.join("\n")
        }
      end
    end

    def check_commit_message(options)
      path = options["path"]
      keywords = options["keywords"]
      return unless changed_files(:in => path).size > 0

      keywords.each {|keyword|
        begin
          case keyword
          when String
            if m = keyword.match(%r{^/(.*)/([imx]*)$})
              opt = {
                "i" => Regexp::IGNORECASE,
                "m" => Regexp::MULTILINE,
                "x" => Regexp::EXTENDED
              }.inject(0) {|n, h|
                m[2].include?(h[0]) ? n + h[1] : n
              }

              r = Regexp.new(m[1], opt > 0 ? opt : nil, "UTF8")
              raise unless r === commit_message
            else
              raise unless commit_message.include?(keyword)
            end
          when Array
            raise unless keyword.any? {|word| commit_message.include?(word)}
          end
        rescue
          (@errors[:commit_message] ||= []).push(
            keyword.is_a?(Array) ? keyword.join(" or ") : keyword
          )
        end
      }
    end

    def check_text_files(options)
      h = {
        "space" => " ", "tab" => "\t",
        "cr" => "\r", "lf" => "\n", "crlf" => "\r\n"
      }

      options["targets"].each {|target|
        path = target["path"]
        filetype = target["filetype"]
        encoding = target["encoding"]
        whitespace = target["whitespace"]
        linefeed = target["linefeed"]

        changed_files(:type => :text, :in => path, :extension => filetype).each {|file|
          if encoding
            begin
              Iconv.iconv(encoding, encoding, cat(file))
            rescue Iconv::IllegalSequence
              e = @errors[:encoding] ||= {}
              (e[encoding] ||= []).push(file)
            end
          end

          next unless whitespace || linefeed
          content = cat(file)

          if whitespace
            sp = h[whitespace]
            content.each{|line|
              next unless s = line.match(/^[ \t]+/)
              s = s.to_s
              sp ||= s[0,1]

              if s.count(sp) != s.size
                warn s.inspect
                e = @errors[:whitespace] ||= {}
                (e[whitespace] ||= []).push(file)
                break
              end
            }
          end

          if linefeed
            lf = h[linefeed]
            content.scan(/\r\n|\r|\n/) {|s|
              lf ||= s
              if s != lf
                e = @errors[:linefeed] ||= {}
                (e[linefeed] ||= []).push(file)
                break
              end
            }
          end
        }
      }
    end
  end
end
