﻿// Copyright(C) 2008, 2010, 2011, 2013 panacorn <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: MujinzouUpdator.cs 492 2013-09-14 15:24:10Z panacoran $

using System;
using Protra.Lib.Data;

namespace Protra.Lib.Update
{
    /// <summary>
    /// 無尽蔵を利用して株価データの更新を行うクラス。
    /// </summary>
    public class MujinzouUpdator : PriceDataUpdator
    {
        /// <summary>
        /// データが存在する最初の日付を取得する。
        /// </summary>
        public override DateTime DataSince
        {
            get { return new DateTime(1996, 1, 4); }
        }

        /// <summary>
        /// 新しいデータが置かれる時刻に達しているか。
        /// </summary>
        /// <param name="time">時刻</param>
        /// <returns></returns>
        protected override bool IsDataAvailable(DateTime time)
        {
            return time.Hour >= 17;
        }

        /// <summary>
        /// データのURLを取得する。
        /// </summary>
        /// <returns>URL</returns>
        protected override string DownloadUrl
        {
            get { return "http://souba-data.com/k_data/" + Date.ToString("yyyy/yy_MM/TyyMMdd") + ".lzh"; }
        }

        /// <summary>
        /// 文字列を解析して価格データを返す。
        /// </summary>
        /// <param name="line">文字列</param>
        /// <returns>価格データ</returns>
        protected override Price ParseLine(string line)
        {
            var tokens = line.Split(new[] {'\t', ','});
            if (tokens.Length < 9)
                return null;
            var r = new Price();
            try
            {
                r.Date = DateTime.Parse(tokens[0]);
                r.Code = tokens[1];
                if (String.Compare(r.Code, "1002", StringComparison.Ordinal) > 0 &&
                    String.Compare(r.Code, "1300", StringComparison.Ordinal) < 0)
                    return null; // 各種指数を無視する。
                // 指数の値に小数が含まれているのでdouble.Parseを利用する。
                r.Open = (int)double.Parse(tokens[4]);
                r.High = (int)double.Parse(tokens[5]);
                r.Low = (int)double.Parse(tokens[6]);
                r.Close = (int)double.Parse(tokens[7]);
                r.Volume = double.Parse(tokens[8]) / 1000;
                switch (tokens[2])
                {
                    case "11":
                        r.Market = "T1";
                        break;
                    case "12":
                        r.Market = "T2";
                        break;
                    case "13":
                        r.Market = "M";
                        break;
                    case "14":
                        r.Market = "T1";
                        break;
                    case "20": //この大証は１部、２部混在しているので１部にする
                    case "21":
                        r.Market = "O1";
                        break;
                    case "22":
                        r.Market = "O2";
                        break;
                    case "23":
                    case "24":
                        r.Market = "H";
                        break;
                    case "32":
                        if (tokens[9] == "東証２部") // データのミスに対応する。
                            r.Market = "T2";
                        else
                            return null;
                        break;
                    case "91":
                        r.Market = "J";
                        break;
                    default:
                        return null;
                }
            }
            catch (FormatException)
            {
                // フォーマットエラーは全部無視する。
                return null;
            }
            return r;
        }
    }
}