﻿// Copyright (C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008, 2013 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: SuffixedExpressionNode.cs 472 2013-06-26 15:26:49Z panacoran $

using System;
using System.Collections.Generic;

namespace Protra.Lib.Lang
{
    /// <summary>
    /// 後置式のノードを表すクラス。
    /// </summary>
    public class SuffixedExpressionNode : ExpressionNode
    {
        /// <summary>
        /// 配列を指す式
        /// </summary>
        private ExpressionNode _arrayNode;

        /// <summary>
        /// 配列に対するインデックスとなるノードのリスト
        /// </summary>
        private List<ExpressionNode> _nodeList;

        /// <summary>
        /// 構文を解析する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.ParseException">
        /// 構文解析中にエラーが発生したときにthrowされる。
        /// </exception>
        /// <return>
        /// 解析した結果のノード。
        /// </return>
        public override ExpressionNode Parse(Scanner scanner)
        {
            Token = scanner.Token;
            var node = new PrimaryNode().Parse(scanner);
            var t = scanner.Token;
            if (t.Value != "[")
                return node;
            _arrayNode = node;
            _nodeList = new List<ExpressionNode>();
            do
            {
                scanner.Scan();
                _nodeList.Add(new ExpressionNode().Parse(scanner));
                t = scanner.Token;
                if (t.Value != "]")
                    throw new ParseException("']' expected.", Token);
                scanner.Scan();
                t = scanner.Token;
            } while (t.Value == "[");
            return this;
        }

        /// <summary>
        /// 式を評価する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// 評価中にエラーが発生した場合にthrowされる。
        /// </exception>
        /// <param name="resource">リソース</param>
        /// <param name="at">int型@作用素の値</param>
        /// <param name="ats">string型@作用素の値</param>
        /// <returns>値</returns>
        public override Value Evaluate(Resource resource, int at, string ats)
        {
            try
            {
                var val = _arrayNode.Evaluate(resource, at, ats);
                for (var i = 0; i < _nodeList.Count; i++)
                {
                    var index = _nodeList[i].Evaluate(resource, at, ats);
                    val = val[index];
                }
                return val;
            }
            catch (RuntimeException e)
            {
                throw new RuntimeException(e.Message, e.Token ?? Token, e);
            }
            catch (NullReferenceException e)
            {
                throw new RuntimeException(e.Message, Token);
            }
        }

        /// <summary>
        /// 代入を実行する。
        /// </summary>
        /// <param name="resource">リソース</param>
        /// <param name="at">int型@作用素の値</param>
        /// <param name="ats">string型@作用素の値</param>
        /// <param name="val">代入する値</param>
        /// <returns>実行結果</returns>
        public Value Assign(Resource resource, int at, string ats, Value val)
        {
            try
            {
                var arr = _arrayNode.Evaluate(resource, at, ats);
                var index = _nodeList[0].Evaluate(resource, at, ats);
                for (var i = 1; i < _nodeList.Count; i++)
                {
                    arr = arr[index];
                    index = _nodeList[i].Evaluate(resource, at, ats);
                }
                return arr[index] = val;
            }
            catch (RuntimeException e)
            {
                if (e.Token == null)
                    e.Token = Token;
                throw;
            }
            catch (NullReferenceException e)
            {
                throw new RuntimeException(e.Message, Token);
            }
        }
    }
}