﻿// Copyright (C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008, 2010, 2013 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: Interpreter.cs 455 2013-06-08 11:52:33Z panacoran $

using System.Collections.Generic;
using Protra.Lib.Lang.Builtins;

namespace Protra.Lib.Lang
{
    /// <summary>
    /// プログラムの実行処理を担うクラス。
    /// </summary>
    public class Interpreter
    {
        /// <summary>
        /// プログラムのノード
        /// </summary>
        private readonly Node _node = new ProgramNode();

        /// <summary>
        /// リソース
        /// </summary>
        private readonly Resource _resource;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.ParseException">
        /// プログラムの構文解析中にエラーが発生した場合にthrowされる。
        /// </exception>
        /// <param name="file">プログラムファイルのパス</param>
        public Interpreter(string file)
        {
            Node.Scanner = new Scanner(file);
            // 関数の外のローカル変数の記号表を用意する。
            Node.LvtStack.Push(new Dictionary<string, int>());
            try
            {
                _node = new ProgramNode();
                _node.Parse();
            }
            catch (ParseException exc)
            {
                var t = exc.Token;
                throw new ParseException(
                    "parse error in " + t.Filename + ":" + t.LineNo + ": " + exc.Message, null);
            }
            finally
            {
                Node.Scanner.Close();
            }
            _resource = new Resource();
        }

        /// <summary>
        /// プログラムを実行する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされる。
        /// </exception>
        /// <returns>プログラムの戻り値。</returns>
        public Value Execute()
        {
            _resource.Stack.Clear();
            _resource.FunctionTable.Clear();
            try
            {
                return _node.Execute(_resource, 0, null);
            }
            catch (RuntimeException e)
            {
                var token = e.Token;
                throw new RuntimeException("runtime error in " + token.Filename + ":" + token.LineNo + ": " + e.Message, null, e);
            }
        }

        /// <summary>
        /// グローバル変数のテーブルを取得または設定する。
        /// </summary>
        public Dictionary<string, Value> GlobalVariableTable
        {
            get { return _resource.GlobalVariableTable; }
            set { _resource.GlobalVariableTable = value; }
        }

        /// <summary>
        /// 組み込み関数を処理するクラスのインスタンスを取得または設定する。
        /// </summary>
        public BasicBuiltins Builtins
        {
            get { return _resource.Builtins; }
            set { _resource.Builtins = value; }
        }
    }
}