﻿// Copyright (C) 2003, 2005 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2008 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: Value.cs 444 2013-03-11 04:41:16Z panacoran $

using System;

namespace Protra.Lib.Lang
{
    /// <summary>
    /// プログラムで扱う値を統一的に表現するためのクラスです。
    /// </summary>
    [Serializable]
    public class Value : IComparable
    {
        /// <summary>
        /// 値の型を表す列挙型です。
        /// </summary>
        public enum Type : byte
        {
            /// <summary>
            /// int
            /// </summary>
            Int,
            /// <summary>
            /// float
            /// </summary>
            Float,
            /// <summary>
            /// string
            /// </summary>
            String,
            /// <summary>
            /// array
            /// </summary>
            Array
        }

        /// <summary>
        /// 値の実体
        /// </summary>
        private readonly object val;
        /// <summary>
        /// 値の型
        /// </summary>
        private readonly Type type;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="val">整数値</param>
        public Value(int val)
        {
            this.val = val;
            type = Type.Int;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="val">実数値</param>
        public Value(double val)
        {
            this.val = val;
            type = Type.Float;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="val">論理値</param>
        public Value(bool val)
        {
            this.val = val ? 1 : 0;
            type = Type.Int;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="val">文字列</param>
        public Value(string val)
        {
            this.val = val;
            type = Type.String;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="val">配列</param>
        public Value(Value[] val)
        {
            this.val = val;
            type = Type.Array;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="val">値</param>
        public Value(Value val)
        {
            this.val = val.val;
            type = val.type;
        }

        /// <summary>
        /// 内部的値の実体を取得します。
        /// </summary>
        public object InnerValue
        {
            get { return val; }
        }

        /// <summary>
        /// 型を示す値を取得します。
        /// </summary>
        public Type ValueType
        {
            get { return type; }
        }

        /// <summary>
        /// 値がtrueと評価されるかどうかを示す論理値を取得します。
        /// 0はfalse、それ以外はtrueになります。
        /// </summary>
        public bool IsTrue
        {
            get
            {
                return !IsFalse;
            }
        }

        /// <summary>
        /// 計算機イプシロン
        /// </summary>
        private const double epsilon = 1.11022302462516E-16;

        /// <summary>
        /// 値がfalseと評価されるかどうかを示す論理値を取得します。
        /// 0はfalse、それ以外はtrueになります。
        /// </summary>
        public bool IsFalse
        {
            get
            {
                return type == Type.Int && (int)val == 0 ||
                       type == Type.Float && Math.Abs((double)val - 0.0) <= epsilon;

            }
        }

        /// <summary>
        /// 配列の要素を取得または設定します。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        public Value this[Value index]
        {
            get
            {
                if (index == null)
                    throw new RuntimeException("assigned null for index", null);
                if (type != Type.Array)
                    throw new RuntimeException("assigned index for non-array", null);
                if (index.type != Type.Int)
                    throw new RuntimeException("assigned non-int value for array index", null);

                try
                {
                    return ((Value[])val)[(int)index.val];
                }
                catch (IndexOutOfRangeException)
                {
                    throw new RuntimeException("array index out of range", null);
                }
            }

            set
            {
                if (index == null)
                    throw new RuntimeException("assigned null for index", null);
                if (type != Type.Array)
                    throw new RuntimeException("assigned index for non-array", null);
                if (index.type != Type.Int)
                    throw new RuntimeException("assigned non-int value for array index", null);

                try
                {
                    ((Value[])val)[(int)index.val] = value;
                }
                catch (IndexOutOfRangeException)
                {
                    throw new RuntimeException("array index out of range", null);
                }
            }
        }

        /// <summary>
        /// 指定された型にキャストします。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="target">型</param>
        /// <returns>キャスト結果です。</returns>
        public Value Cast(Type target)
        {
            if (type == Type.Array)
                throw new RuntimeException("array casted", null);
            if (target == Type.String)
                return new Value(val.ToString());

            switch (type)
            {
                case Type.Int:
                    switch (target)
                    {
                        case Type.Int:
                            return new Value((int)val);
                        case Type.Float:
                            return new Value((double)(int)val);
                    }
                    break;
                case Type.Float:
                    switch (target)
                    {
                        case Type.Int:
                            return new Value((int)(double)val);
                        case Type.Float:
                            return new Value((double)val);
                    }
                    break;
                case Type.String:
                    switch (target)
                    {
                        case Type.Int:
                            try
                            {
                                return new Value(int.Parse((string)val));
                            }
                            catch (FormatException)
                            {
                                return new Value(0);
                            }
                        case Type.Float:
                            try
                            {
                                return new Value(double.Parse((string)val));
                            }
                            catch (FormatException)
                            {
                                return new Value(0.0);
                            }
                    }
                    break;
            }

            throw new RuntimeException("unexpected error", null);
        }

        /// <summary>
        /// 指定したオブジェクトが現在のオブジェクトと等しいかどうかを判断する。
        /// </summary>
        /// <param name="o">比較するオブジェクト。</param>
        /// <returns>等しい場合はtrue。それ以外の場合はfalse。</returns>
        public override bool Equals(object o)
        {
            var v = o as Value;
            if ((object)v == null)
                return false;
            if (type == Type.String && v.type == Type.String)
                return (string)val == (string)v.val;
            if (type == Type.Array && v.type == Type.Array)
                return val == v.val;
            if (type == Type.Array || type == Type.String ||
                v.type == Type.Array || v.type == Type.String)
                return false;
            if (v.type == Type.Int)
            {
                if (type == Type.Int)
                    return (int)val == (int)v.val;
                return Math.Abs((double)val - (int)v.val) <= epsilon;
            }
            if (type == Type.Int)
                return Math.Abs((double)v.val - (int)val) <= epsilon;
            return Math.Abs((double)v.val - (double)val) <= epsilon;
        }

        /// <summary>
        /// ハッシュ値を計算します。
        /// </summary>
        /// <returns>ハッシュ値</returns>
        public override int GetHashCode()
        {
            return val.GetHashCode();
        }

        /// <summary>
        /// 現在のオブジェクトを指定したオブジェクトと比較する。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされる。
        /// </exception>
        /// <param name="o">比較するオブジェクト。</param>
        /// <returns>比較結果を表す整数。</returns>
        public int CompareTo(Object o)
        {
            var v = (Value)o;
            if (type == Type.Array || v.type == Type.Array)
                throw new RuntimeException("array compared", null);
            switch (type)
            {
                case Type.Int:
                    switch (v.type)
                    {
                        case Type.Int:
                            return (int)val - (int)v.val;
                        case Type.Float:
                            return Math.Sign((int)val - (double)v.val);
                        case Type.String:
                            throw new RuntimeException("int compared with string", null);
                    }
                    break;
                case Type.Float:
                    switch (v.type)
                    {
                        case Type.Int:
                            return Math.Sign((double)val - (int)v.val);
                        case Type.Float:
                            return Math.Sign((double)val - (double)v.val);
                        case Type.String:
                            throw new RuntimeException("float compared with string", null);
                    }
                    break;
                case Type.String:
                    switch (v.type)
                    {
                        case Type.Int:
                            throw new RuntimeException("string compared with int", null);
                        case Type.Float:
                            throw new RuntimeException("string compared with float", null);
                        case Type.String:
                            return ((string)val).CompareTo(v.val);
                    }
                    break;
            }
            throw new RuntimeException("unexpected error", null);
        }

        /// <summary>
        /// 等価比較を行います。
        /// </summary>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>比較結果です。</returns>
        public static bool operator ==(Value v1, Value v2)
        {
            if ((object)v1 != null)
                return v1.Equals(v2);
            return (object)v2 == null;
        }

        /// <summary>
        /// 等価比較を行います。
        /// </summary>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>比較結果です。</returns>
        public static bool operator !=(Value v1, Value v2)
        {
            return !(v1 == v2);
        }

        /// <summary>
        /// 大小比較を行います。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>比較結果です。</returns>
        public static bool operator <(Value v1, Value v2)
        {
            if (v1 == null || v2 == null)
                throw new RuntimeException("null compared", null);
            return v1.CompareTo(v2) < 0;
        }

        /// <summary>
        /// 大小比較を行います。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>比較結果です。</returns>
        public static bool operator >(Value v1, Value v2)
        {
            if (v1 == null || v2 == null)
                throw new RuntimeException("null compared", null);
            return v1.CompareTo(v2) > 0;
        }

        /// <summary>
        /// 大小比較を行います。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>比較結果です。</returns>
        public static bool operator <=(Value v1, Value v2)
        {
            if (v1 == null || v2 == null)
                throw new RuntimeException("null compared", null);
            return v1.CompareTo(v2) <= 0;
        }

        /// <summary>
        /// 大小比較を行います。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>比較結果です。</returns>
        public static bool operator >=(Value v1, Value v2)
        {
            if (v1 == null || v2 == null)
                throw new RuntimeException("null compared", null);
            return v1.CompareTo(v2) >= 0;
        }

        /// <summary>
        /// 和を計算します。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>演算結果です。</returns>
        public static Value operator +(Value v1, Value v2)
        {
            if (v1 == null || v2 == null)
                throw new RuntimeException("binary operator '+' isn't defined for null", null);
            if (v1.type == Type.Array || v2.type == Type.Array)
                throw new RuntimeException("binary operator '+' isn't defined for array", null);
            switch (v1.type)
            {
                case Type.Int:
                    switch (v2.type)
                    {
                        case Type.Int:
                            return new Value((int)v1.val + (int)v2.val);
                        case Type.Float:
                            return new Value((int)v1.val + (double)v2.val);
                        case Type.String:
                            return new Value(v1.val + v2.val.ToString());
                    }
                    break;
                case Type.Float:
                    switch (v2.type)
                    {
                        case Type.Int:
                            return new Value((double)v1.val + (int)v2.val);
                        case Type.Float:
                            return new Value((double)v1.val + (double)v2.val);
                        case Type.String:
                            return new Value(v1.val + v2.val.ToString());
                    }
                    break;
                case Type.String:
                    return new Value(v1.val + v2.val.ToString());
            }

            throw new RuntimeException("unexpected error", null);
        }

        /// <summary>
        /// 差を計算します。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>演算結果です。</returns>
        public static Value operator -(Value v1, Value v2)
        {
            if (v1 == null || v2 == null)
                throw new RuntimeException("binary operator '-' isn't defined for null", null);
            if (v1.type == Type.String || v2.type == Type.String)
                throw new RuntimeException("binary operator '-' isn't defined for string", null);
            if (v1.type == Type.Array || v2.type == Type.Array)
                throw new RuntimeException("binary operator '-' isn't defined for array", null);
            switch (v1.type)
            {
                case Type.Int:
                    switch (v2.type)
                    {
                        case Type.Int:
                            return new Value((int)v1.val - (int)v2.val);
                        case Type.Float:
                            return new Value((int)v1.val - (double)v2.val);
                    }
                    break;
                case Type.Float:
                    switch (v2.type)
                    {
                        case Type.Int:
                            return new Value((double)v1.val - (int)v2.val);
                        case Type.Float:
                            return new Value((double)v1.val - (double)v2.val);
                    }
                    break;
            }

            throw new RuntimeException("unexpected error", null);
        }

        /// <summary>
        /// 積を計算します。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>演算結果です。</returns>
        public static Value operator *(Value v1, Value v2)
        {
            if (v1 == null || v2 == null)
                throw new RuntimeException("binary operator '*' isn't defined for null", null);
            if (v1.type == Type.String || v2.type == Type.String)
                throw new RuntimeException("binary operator '*' isn't defined for string", null);
            if (v1.type == Type.Array || v2.type == Type.Array)
                throw new RuntimeException("binary operator '*' isn't defined for array", null);
            switch (v1.type)
            {
                case Type.Int:
                    switch (v2.type)
                    {
                        case Type.Int:
                            return new Value((int)v1.val * (int)v2.val);
                        case Type.Float:
                            return new Value((int)v1.val * (double)v2.val);
                    }
                    break;
                case Type.Float:
                    switch (v2.type)
                    {
                        case Type.Int:
                            return new Value((double)v1.val * (int)v2.val);
                        case Type.Float:
                            return new Value((double)v1.val * (double)v2.val);
                    }
                    break;
            }

            throw new RuntimeException("unexpected error", null);
        }

        /// <summary>
        /// 商を計算します。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>演算結果です。</returns>
        public static Value operator /(Value v1, Value v2)
        {
            if (v1 == null || v2 == null)
                throw new RuntimeException("binary operator '/' isn't defined for null", null);
            if (v1.type == Type.String || v2.type == Type.String)
                throw new RuntimeException("binary operator '/' isn't defined for string", null);
            if (v1.type == Type.Array || v2.type == Type.Array)
                throw new RuntimeException("binary operator '/' isn't defined for array", null);
            switch (v1.type)
            {
                case Type.Int:
                    switch (v2.type)
                    {
                        case Type.Int:
                            if ((int)v2.val == 0)
                                throw new RuntimeException("divided by 0", null);
                            return new Value((int)v1.val / (int)v2.val);
                        case Type.Float:
                            if (Math.Abs((double)v2.val - 0.0) <= epsilon)
                                throw new RuntimeException("divided by 0", null);
                            return new Value((int)v1.val / (double)v2.val);
                    }
                    break;
                case Type.Float:
                    switch (v2.type)
                    {
                        case Type.Int:
                            if ((int)v2.val == 0)
                                throw new RuntimeException("divided by 0", null);
                            return new Value((double)v1.val / (int)v2.val);
                        case Type.Float:
                            if (Math.Abs((double)v2.val - 0.0) <= epsilon)
                                throw new RuntimeException("divided by 0", null);
                            return new Value((double)v1.val / (double)v2.val);
                    }
                    break;
            }

            throw new RuntimeException("unexpected error", null);
        }

        /// <summary>
        /// 余りを計算します。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v1">オペランド1</param>
        /// <param name="v2">オペランド2</param>
        /// <returns>演算結果です。</returns>
        public static Value operator %(Value v1, Value v2)
        {
            if (v1 == null || v2 == null)
                throw new RuntimeException("binary operator '%' isn't defined for null", null);
            if (v1.type == Type.Float || v2.type == Type.Float)
                throw new RuntimeException("binary operator '%' isn't defined for float", null);
            if (v1.type == Type.String || v2.type == Type.String)
                throw new RuntimeException("binary operator '%' isn't defined for string", null);
            if (v1.type == Type.Array || v2.type == Type.Array)
                throw new RuntimeException("binary operator '%' isn't defined for array", null);
            switch (v1.type)
            {
                case Type.Int:
                    switch (v2.type)
                    {
                        case Type.Int:
                            if ((int)v2.val == 0)
                                throw new RuntimeException("divided by 0", null);
                            return new Value((int)v1.val % (int)v2.val);
                    }
                    break;
            }

            throw new RuntimeException("unexpected error", null);
        }

        /// <summary>
        /// 否定を計算します。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v">オペランド</param>
        /// <returns>演算結果です。</returns>
        public static Value operator !(Value v)
        {
            return new Value(v == null || v.IsFalse);
        }

        /// <summary>
        /// プラス符号を計算します。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v">オペランド</param>
        /// <returns>演算結果です。</returns>
        public static Value operator +(Value v)
        {
            switch (v.type)
            {
                case Type.Int:
                case Type.Float:
                    return new Value(v);
                case Type.String:
                    throw new RuntimeException("unary operator '+' isn't defined for string", null);
                case Type.Array:
                    throw new RuntimeException("unary operator '+' isn't defined for array", null);
            }

            throw new RuntimeException("unexpected error", null);
        }

        /// <summary>
        /// マイナス符号を計算します。
        /// </summary>
        /// <exception cref="Protra.Lib.Lang.RuntimeException">
        /// プログラム実行中にエラーが発生した場合にthrowされます。
        /// </exception>
        /// <param name="v">オペランド</param>
        /// <returns>演算結果です。</returns>
        public static Value operator -(Value v)
        {
            switch (v.type)
            {
                case Type.Int:
                    return new Value(-(int)v.val);
                case Type.Float:
                    return new Value(-(double)v.val);
                case Type.String:
                    throw new RuntimeException("unary operator '-' isn't defined for string", null);
                case Type.Array:
                    throw new RuntimeException("unary operator '-' isn't defined for array", null);
            }

            throw new RuntimeException("unexpected error", null);
        }
    }
}
