﻿// Copyright (C) 2003 Daisuke Arai <darai@users.sourceforge.jp>
// Copyright (C) 2004, 2005 panacoran <panacoran@users.sourceforge.jp>
// 
// This program is part of Protra.
//
// Protra is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, see <http://www.gnu.org/licenses/>.
// 
// $Id: System.cs 306 2010-03-19 13:21:22Z panacoran $

using System;
using System.Collections;

namespace Protra.Lib.Db
{
	/// <summary>
	/// Systemテーブルのレコードを表す。
	/// </summary>
	public class TradingSystem: IRecord
	{
		private int id;
		private string file;
		private int mode;

		/// <summary>
		/// 既定のコンストラクタ
		/// </summary>
		public TradingSystem() {}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		private TradingSystem(int id, string file, int mode)
		{
			this.id = id;
			this.file = file;
			this.mode = mode;
		}

		/// <summary>
		/// システムIDを取得する。
		/// </summary>
		public int Id
		{
			get { return id; }
		}

		/// <summary>
		/// ファイル名を取得する。
		/// </summary>
		public string File
		{
			get { return file; }
		}

		/// <summary>
		/// モードを取得する。日足モードなら0、週足モードなら1。
		/// </summary>
		public int Mode
		{
			get { return mode; }
		}

		/// <summary>
		/// データベースのレコードをオブジェクトに変換する。
		/// </summary>
		/// <param name="values">ネイティブ形式の値の配列</param>
		/// <returns>変換後のTradingSystemオブジェクトを返す。</returns>
		public Object ToObject(Object[] values)
		{
			return new TradingSystem((int)values[0], // @Id
									 (string)values[1], // @File
									 (int)values[2]); // @Mode
		}

		/// <summary>
		/// インスタンスを文字列表現に変換する。
		/// </summary>
		/// <returns>文字列表現</returns>
		public override string ToString()
		{
			return string.Format("TradingSystem[Id={0}, File={1}, Mode={2}]",
								 id, file, mode);
		}
	}

	/// <summary>
	/// Systemテーブルを操作する。
	/// </summary>
	public class SystemTable
	{
		private static Connection conn;
		private static Connection Conn
		{
			get 
			{
				if (conn != null)
					return conn;
				conn = new Connection("protra");
				return conn;
			}
		}

		/// <summary>
		/// 指定されたシステムIDのレコードを取得する。
		/// </summary>
		/// <param name="id">システムID</param>
		/// <returns>TradingSystemオブジェクトを返す。レコードが存在しない場合にはnullを返す。</returns>
		public static TradingSystem GetRecord(int id)
		{
			string sql =
				string.Format("SELECT * FROM System WHERE @Id={0}", id);
			ArrayList result = Conn.Query(sql, new TradingSystem());
			return result.Count == 0 ? null : (TradingSystem)result[0];
		}

		/// <summary>
		/// 指定されたファイル名とモードのレコードを取得する。
		/// </summary>
		/// <param name="file">ファイル名</param>
		/// <param name="mode">モード</param>
		/// <returns>TradingSystemオブジェクトを返す。レコードが存在しない場合にはnullを返す。</returns>
		public static TradingSystem GetRecord(string file, int mode)
		{
			string sql = string.Format(
				"SELECT * FROM System WHERE @File='{0}' AND @Mode={1}",
				file, mode);
			ArrayList result = Conn.Query(sql, new TradingSystem());
			return result.Count == 0 ? null : (TradingSystem)result[0];
		}

		/// <summary>
		/// すべてのレコードを取得する。
		/// </summary>
		/// <returns>TradingSystemオブジェクトの配列を返す。</returns>
		public static TradingSystem[] GetAllRecords()
		{
			string sql = "SELECT * FROM System ORDER BY @Id";
			ArrayList result = Conn.Query(sql, new TradingSystem());
			return (TradingSystem[])result.ToArray(typeof(TradingSystem));
		}

		/// <summary>
		/// レコードを追加する。
		/// </summary>
		/// <param name="file">ファイル名</param>
		/// <param name="mode">モード</param>
		/// <returns>追加されたレコードを返す</returns>
		public static TradingSystem Add(string file, int mode)
		{
			string sql = string.Format(
				"INSERT INTO System(@File, @Mode) VALUES('{0}', {1})",
				file, mode);
			Conn.Execute(sql);
			TradingSystem[] all = GetAllRecords();
			return all[all.Length - 1];
		}

		/// <summary>
		/// 指定されたシステムIDのデータを削除する。
		/// </summary>
		/// <param name="id">システムID</param>
		public static void Delete(int id)
		{
			string sql = string.Format("DELETE FROM Price WHERE @Id={0}", id);
			Conn.Execute(sql);
		}

		/// <summary>
		/// 全データを削除する。
		/// </summary>
		public static void Delete()
		{
			string sql = "DELETE FROM System";
			Conn.Execute(sql);
		}
	}
}
