/***********************************************************************//**
	@file
	$Revision$
	$Author$
	$Date::                           $
***************************************************************************/
#include <sstream>
#include <assert.h>
#include "common/Command.h"
#include "Client.h"

namespace openmj {
namespace client {

const Version Client::UMP_VERSION(0, 1);
/***********************************************************************//**
	コンストラクタ.
***************************************************************************/
Client::Client()
    : lastSutehai(0), 
      tsumibou_(0), 
      kyotaku_(0)
{}
/***********************************************************************//**
	デストラクタ.
***************************************************************************/
Client::~Client() {}
/***********************************************************************//**
	サーバに接続する
	@param	host	ホスト名
	@param	port	ポート
        @return		接続に成功したら真
***************************************************************************/
bool Client::connect(const char* host, int port) {
    return socket_.connect(host, port);
}
/***********************************************************************//**
	@return	サーバに接続しているとき真
***************************************************************************/
bool Client::isConnect() const {
    return socket_.isOpen();
}
/***********************************************************************//**
	親を返す
	@return	親(荘家=0)
***************************************************************************/
int Client::getOya() const {
    return oya_;
}
/***********************************************************************//**
	積み棒の数(本場)を返す
	@return	本場
***************************************************************************/
int Client::getTsumibou() const {
    return tsumibou_;
}
/***********************************************************************//**
	供託の点数を返す
	@return	供託の点数
***************************************************************************/
int Client::getKyotaku() const {
    return kyotaku_;
}
/***********************************************************************//**
	場風を返す
	@return	場風を示す牌インスタンスのポインタ
***************************************************************************/
const mahjong::Hai* Client::getBakaze() const {
    return bakaze_;
}
/***********************************************************************//**
	最後の捨て牌を返す
	@return	最後の捨て牌を示すポインタ
***************************************************************************/
const mahjong::Sutehai* Client::getLastSutehai() const {
    return lastSutehai;
}
/***********************************************************************//**
	プレイヤー.
***************************************************************************/
Player* Client::getPlayer(const std::string& seki) {
    int index = seki[0] - 'A';
    if(index >= 0 && index < playerList.size()) {
        return &playerList.at(index);
    }
    return 0;
}
Player* Client::getPlayer() {
    return getPlayer(std::string(1, 'A' + seki));
}
std::vector<const Player*> Client::getPlayers() {
    std::vector<const Player*> players;
    int playerNum = playerList.size();
    for(int i = 0; i < playerNum; i++) {
        players.push_back(&playerList[(seki + i) % playerNum]);
    }
    return players;
}
/***********************************************************************//**
	実行する
***************************************************************************/
void Client::exec() {
    Command command;
    while(command.getCommand() != Command::CMD_GAMEEND) {
        if(!socket_.recvCommand(command)) {
            break;
        }
        std::string buff = "> ";
        buff.append(command.toString()).append("\n");
        log(buff.c_str());
        recv(command);
    }
    socket_.close();
}
/***********************************************************************//**
	コマンドをサーバに送る
	@param	command	送信するコマンド
***************************************************************************/
void Client::send(const Command& command) {
    std::string buff = command.toString();
    buff.append(Socket::EOL);
    log("< ");
    log(buff.c_str());
    socket_.send(buff.c_str());
}
/***********************************************************************//**
	コマンド受信
	@param	command	受信したコマンド
***************************************************************************/
void Client::recv(const Command& command) {
    switch(command.getCommand()) {
    case Command::CMD_GAMESTART:
        execGameStart(command);
        break;
    case Command::CMD_KYOKUSTART:
        execKyokuStart(command);
        break;
    case Command::CMD_POINT:
        execPoint(command);
        break;
    case Command::CMD_HAIPAI:
        execHaipai(command);
        break;
    case Command::CMD_TSUMO:
        execTsumo(command);
        break;
    case Command::CMD_OPEN:
        execOpen(command);
        break;
    case Command::CMD_SUTEHAI:
        execSutehai(command);
        break;
    case Command::CMD_SAY:
        execSay(command);
        break;
    case Command::CMD_AGARI:
        execAgari(command);
        break;
    case Command::CMD_RYUKYOKU:
        execRyukyoku(command);
        break;
    default:
        break;
    }
    recvCommand(command);
}
/***********************************************************************//**
	
***************************************************************************/
void Client::recvCommand(const Command& command) {
    replyCommand(command);
}
/***********************************************************************//**
	
***************************************************************************/
void Client::replyCommand(const Command& command) {
    switch(command.getCommand()) {
    case Command::CMD_HELLO:
        replyHello(command);
        break;
    case Command::CMD_READY_Q:
        replyReady(command);
        break;
    case Command::CMD_TENPAI_Q:
        replyTenpai(command);
        break;
    case Command::CMD_NAKU_Q:
        replyNaku(command);
        break;
    case Command::CMD_SUTEHAI_Q:
        replySutehai(command);
        break;
    default:
        break;
    }
}
/***********************************************************************//**
	helloに答える
	@param	command	コマンド
***************************************************************************/
void Client::replyHello(const Command& command) {
    Command response(command.getSerial(), Command::CMD_HELLO);
    std::string option;
    option = "ump=";
    option.append(UMP_VERSION.toString());
    response.append(option.c_str());
    send(response);
}
/***********************************************************************//**
	ready?に答える
	@param	command	コマンド
***************************************************************************/
void Client::replyReady(const Command& command) {
    send(Command(command.getSerial(), Command::CMD_OK));
}
/***********************************************************************//**
	tenpai?に答える
	@param	command	コマンド
***************************************************************************/
void Client::replyTenpai(const Command& command) {
    send(Command(command.getSerial(), Command::CMD_YES));
}
/***********************************************************************//**
	naku?に答える
	@param	command	コマンド
***************************************************************************/
void Client::replyNaku(const Command& command) {
    send(Command(command.getSerial(), Command::CMD_NO));
}
/***********************************************************************//**
	sutehai?に答える
	@param	command	コマンド
***************************************************************************/
void Client::replySutehai(const Command& command) {
}
/***********************************************************************//**
	ゲーム(半荘)開始.
***************************************************************************/
void Client::execGameStart(const Command& command) {
    playerList.clear();
    seki = command.getArg(0)[0] - 'A';
    for(int i = 1; i < command.getArgNum(); i++) {
        playerList.push_back(Player(command.getArg(i).c_str()));
    }
}
/***********************************************************************//**
	局開始.
***************************************************************************/
void Client::execKyokuStart(const Command& command) {
    restYama = 0;
    lastSutehai = 0;
    bakaze_ = mahjong::Hai::Get(command.getArg(0).c_str());
    oya_ = command.getArg(1)[0] - 'A';
    tsumibou_ = atoi(command.getArg(2).c_str());
    kyotaku_ = atoi(command.getArg(3).c_str());
}
/***********************************************************************//**
	ポイント.
***************************************************************************/
void Client::execPoint(const Command& command) {
    Player* player = getPlayer(command.getArg(0));
    const char* point = command.getArg(1).c_str();
    switch(point[0]) {
    case '=':
        player->setPoint(atoi(point + 1));
        break;
    case '+':
    case '-':
        player->addPoint(atoi(point));
        break;
    default:
        break;
    }
}
/***********************************************************************//**
	配牌.
***************************************************************************/
void Client::execHaipai(const Command& command) {
    Player* player = getPlayer(command.getArg(0));
    player->reset();
    if(command.getArgNum() > 1) {
        player->setHaipai(mahjong::HaiArray(command.getArg(1).c_str()));
    }
}
/***********************************************************************//**
	自摸.
***************************************************************************/
void Client::execTsumo(const Command& command) {
    Player* player = getPlayer(command.getArg(0));
    restYama = atoi(command.getArg(1).c_str());
    player->tsumo((command.getArgNum() > 2)
                  ? mahjong::Hai::Get(command.getArg(2)) : 0);
    lastSutehai = 0;
}
/***********************************************************************//**
	捨て牌.
***************************************************************************/
void Client::execSutehai(const Command& command) {
    Player* player = getPlayer(command.getArg(0));
    lastSutehai = &player->sutehai(mahjong::Hai::Get(command.getArg(1)), 
                                   (command.getArg(2) == "tsumogiri"));
}
/***********************************************************************//**
	手牌を晒す.
***************************************************************************/
void Client::execOpen(const Command& command) {
    Player* player = getPlayer(command.getArg(0));
    player->open(command.getArg(1).c_str(), 
                 (command.getArgNum() > 2)
                 ? mahjong::Hai::Get(command.getArg(2)) : 0);
}
/***********************************************************************//**
	発声
***************************************************************************/
void Client::execSay(const Command& command) {
    Player* player = getPlayer(command.getArg(0));
    if(command.getArg(1) == "ron" ||
       command.getArg(1) == "kan" ||
       command.getArg(1) == "pon" ||
       command.getArg(1) == "chi") {
        assert(lastSutehai);
        lastSutehai->setNaki();
    }
    else if(command.getArg(1) == "richi") {
        player->setRichi();
    }
    else if(command.getArg(1) == "tenpai") {
        player->setOpen();
    }
    else if(command.getArg(1) == "noten") {
        player->setClose();
    }
}
/***********************************************************************//**
	和了
***************************************************************************/
void Client::execAgari(const Command& command) {
    Player* player = getPlayer(command.getArg(0));
    player->setOpen();
}
/***********************************************************************//**
	流局
***************************************************************************/
void Client::execRyukyoku(const Command& command) {
    lastSutehai = 0;
}
/***********************************************************************//**
	デバッグ表示.
***************************************************************************/
void Client::dump() {
    char seki = 'A';
    for(PlayerList::iterator iter = playerList.begin();
        iter < playerList.end();
        iter++, seki++) {
        fprintf(stderr, "%c %s %d\n", seki, (*iter).getName().c_str(), 
                (*iter).getPoint());
        fprintf(stderr, "%s\n", (*iter).toString().c_str());
    }
}
/***********************************************************************//**
	$Id$
***************************************************************************/
}	/* namespace client */
}	/* namespace openmj */
