/***********************************************************************//**
	@file
	$Revision$
	$Author$
	$Date::                           $
***************************************************************************/
#include <algorithm>
#include "libmahjong.h"
#include "Agari.h"

namespace mahjong {
/***********************************************************************//**
	コンストラクタ.
***************************************************************************/
Hand::Hand() {
    reset();
}
/***********************************************************************//**
	初期化.
***************************************************************************/
void Hand::reset() {
    menzen_.clear();
    kawa_.clear();
    flag_.reset();
    tsumoHai_ = 0;
    fusehai_ = 13;
}
/***********************************************************************//**
	配牌をセットする.
	@param	haipai	配牌を表す牌の配列
***************************************************************************/
void Hand::setHaipai(const HaiArray& haipai) {
    reset();
    menzen_ = haipai;
    fusehai_ = 0;
    openMentsu_.clear();
}
/***********************************************************************//**
	晒す.
	@param	str	晒した牌を表す文字列
	@param	sutehai	鳴いた牌(鳴きじゃないときは0)
***************************************************************************/
void Hand::open(const char* str, const Hai* sutehai) {
    HaiArray haiArray;
    char c;
    do {
        c = *str;
        switch(c) {
        case '(':
        case '<':
        case '\0':
            if(!haiArray.isEmpty()) {
                if(fusehai_ > 0) {
                    fusehai_ -= haiArray.getSize();
                }
                menzen_ = haiArray;
                haiArray.clear();
                openMentsu_.clear();
            }
            str++;
            break;
        case ')':
        case '>':
            appendMentsu(Mentsu(haiArray, (c == ')' ? true : false)));
            if(sutehai) {
                haiArray.removeEqual(sutehai);
                sutehai = 0;
            }
            drop(haiArray);
            haiArray.clear();
            str++;
            break;
        default:
            haiArray.append(Hai::Get(str));
            str += 2;
            break;
        }
    } while(c != '\0');
}
/***********************************************************************//**
	面前の牌の配列を返す
	@return	面前の牌の配列
***************************************************************************/
const HaiArray& Hand::getMenzen() const {
    return menzen_;
}
/***********************************************************************//**
	捨て牌の河を返す
	@return	捨て牌の河
***************************************************************************/
const Kawa& Hand::getKawa() const {
    return kawa_;
}
/***********************************************************************//**
	伏せ牌の数を返す
	@return	伏せ牌の数
***************************************************************************/
int Hand::getFusehaiNum() const {
    return fusehai_;
}
/***********************************************************************//**
	牌を数える
	@param	hai	数える牌
	@return	牌の数
***************************************************************************/
int Hand::countHai(const Hai* hai) const {
    int c = getMenzen().count(hai);
    for(int i = 0; i < getOpenMentsuNum(); i++) {
        c += getOpenMentsu(i).countHai(hai);
    }
    return c;
}
/***********************************************************************//**
	面前か調べる
	@return	面前のとき真
***************************************************************************/
bool Hand::isMenzen() const {
    return !flag_.test(FLAG_NOT_MENZEN);
}
/***********************************************************************//**
	ツモ番か調べる
	@return	ツモ番のとき真
***************************************************************************/
bool Hand::isTurn() const {
    return (tsumoHai_ != 0);
}
/***********************************************************************//**
	牌を自摸る.
	@param	hai	ツモった牌
***************************************************************************/
void Hand::tsumo(const Hai* hai) {
    tsumoHai_ = hai;
    menzen_.sort();
    append(hai);
}
/***********************************************************************//**
	牌を面前に追加する
	@param	牌
***************************************************************************/
void Hand::append(const Hai* hai) {
    if(hai) {
        menzen_.append(hai);
    }
    else {
        fusehai_++;
    }
}
/***********************************************************************//**
	牌を河に捨てる
	@param	hai		捨て牌
	@param	isTsumogiri	ツモ切りのとき真
***************************************************************************/
Sutehai& Hand::sutehai(const Hai* hai, bool isTsumogiri) {
    if(isRichi()) {
        hai = tsumoHai_;
        isTsumogiri = true;
    }
    drop(hai);
    menzen_.sort();
    Sutehai& shai = kawa_.append(hai);
    if(isTsumogiri) {
        shai.setTsumogiri();
    }
    tsumoHai_ = 0;
    return shai;
}
/***********************************************************************//**
	牌を捨てる.
	@param	hai	捨て牌
***************************************************************************/
void Hand::drop(const Hai* hai) {
    if(fusehai_ == 0) {
        menzen_.removeEqual(hai);
    }
    else {
        fusehai_--;
    }
}
/***********************************************************************//**
	複数の牌を捨てる.
	@param	hais	捨てる牌の配列.
***************************************************************************/
void Hand::drop(const HaiArray& hais) {
    for(int i = 0; i < hais.getSize(); i++) {
        drop(hais[i]);
    }
}
/***********************************************************************//**
	牌を鳴く.
	@param	hai	鳴いた牌.
	@param	openHais	手から晒した牌.
***************************************************************************/
void Hand::naki(const Hai* hai, const HaiArray& openHais) {
    tsumoHai_ = 0;
    if(openHais.getSize() == 3) {
        flag_.set(FLAG_RINSHAN);
    }
    for(int i = 0; i < openHais.getSize(); i++) {
        drop(openHais[i]);
    }
    HaiArray hais = openHais;
    hais.append(hai);
    appendMentsu(Mentsu(hais, false));
}
/***********************************************************************//**
	晒した面子を追加する.
	@param	mentsu	晒した面子.
***************************************************************************/
void Hand::appendMentsu(const Mentsu& mentsu) {
    if(!mentsu.isMenzen()) {
        flag_.set(FLAG_NOT_MENZEN);
    }
    openMentsu_.push_back(mentsu);
}
/***********************************************************************//**
	和了形か調べる
	@return	和了形のとき真
***************************************************************************/
bool Hand::isAgari() const {
    return getAgari();
}
/***********************************************************************//**
	テンパイ形か調べる
	@return	テンパイのとき真
***************************************************************************/
bool Hand::isTenpai() const {
    Hand hand(*this);
    const Hai* hai = Hai::Each(0);
    while(hai) {
        if(countHai(hai) < 4) {
            hand.append(hai);
            if(hand.isAgari()) {
                return true;
            }
            hand.drop(hai);
            hai = Hai::Each(hai);
        }
    }
    return false;
}
/***********************************************************************//**
	リーチがかけられるか調べる
	@return	リーチがかけられる捨て牌の配列
***************************************************************************/
HaiArray Hand::canRichi() const {
    HaiArray hais;
    if(!isMenzen() || isRichi()) {
        return hais;
    }
    Hand hand(*this);
    HaiArray unique = getMenzen().getUnique();
    for(int i = 0; i < unique.getSize(); i++) {
        const Hai* hai = unique[i];
        hand.drop(hai);
        if(hand.isTenpai()) {
            hais.append(hai);
        }
        hand.append(hai);
    }
    return hais;
}
/***********************************************************************//**
	和了をセットする
        @param	hai	和了牌
	@param	isRon	ロン和了のとき真
	@return		和了っているとき真
***************************************************************************/
bool Hand::getAgari(const Hai* hai, bool isRon, Agari* result) const {
    if(result) {
        result->clear();
    }
    if(isKokushi()) {
        if(result) {
            result->appendYaku(Agari::YAKUMAN_KOKUSHI);
        }
        return true;
    }
    Agari::Vector agariVector;
    //七対子チェック
    checkChitoi(agariVector);
    //面子分解
    agariVector.push_back(Agari());
    for(Mentsu::Vector::const_iterator mentsu = openMentsu_.begin();
        mentsu != openMentsu_.end();
        mentsu++) {
        agariVector.back().pushMentsu(*mentsu);
    }
    HaiArray menzen = getMenzen();
    menzen.sort();
    HaiArray uniq = menzen.getUnique();
    for(int i = 0; i < uniq.getSize(); i++) {
        const Hai* hai = uniq[i];
        if(getMenzen().count(hai) >= 2) {
            HaiArray hais = menzen;
            HaiArray atama;
            atama.append(hais.removeSame(hai));
            atama.append(hais.removeSame(hai));
            agariVector.back().pushMentsu(Mentsu(atama));
            divideMentsu(hais, agariVector);
            agariVector.back().popMentsu();
        }
    }
    agariVector.pop_back();
    if(agariVector.empty()) {
        return false;
    }
    if(!hai || !result) {
        return true;
    }
    for(Agari::Vector::iterator iter = agariVector.begin();
        iter != agariVector.end();
        iter++) {
        Agari agari = iter->setAgariHai(hai, isRon);
        if(*result < agari) {
            *result = agari;
        }
    }
    if(isRichi()) {
        result->appendYaku(Agari::YAKU_RICHI);
    }
    return true;
}
/***********************************************************************//**
	面子に分解する.
***************************************************************************/
void Hand::divideMentsu(const HaiArray& haiArray, 
                        Agari::Vector& agari) const {
    if(haiArray.isEmpty()) {
        agari.push_back(agari.back());
    }
    else {
        const Hai* hai = haiArray[0];
        if(haiArray.count(hai) >= 3) {
            HaiArray hais = haiArray;
            HaiArray mentsuHais;
            for(int i = 0; i < 3; i++) {
                mentsuHais.append(hais.removeSame(hai));
            }
            agari.back().pushMentsu(Mentsu(mentsuHais));
            divideMentsu(hais, agari);
            agari.back().popMentsu();
        }
        if(!hai->isZihai() && hai->number <= 7) {
            if(haiArray.isInclude(hai->succ(1)) &&
               haiArray.isInclude(hai->succ(2))) {
                HaiArray hais = haiArray;
                HaiArray mentsuHais;
                mentsuHais.append(hais.removeSame(hai));
                mentsuHais.append(hais.removeSame(hai->succ(1)));
                mentsuHais.append(hais.removeSame(hai->succ(2)));
                agari.back().pushMentsu(Mentsu(mentsuHais));
                divideMentsu(hais, agari);
                agari.back().popMentsu();
            }
        }
    }
}
/***********************************************************************//**
	国士無双か調べる
***************************************************************************/
bool Hand::isKokushi() const {
    if(!isMenzen()) {
        return false;
    }
    HaiArray uniq = getMenzen().getUnique();
    if(uniq.getSize() != 13) {
        return false;
    }
    for(int i = 0; i < uniq.getSize(); i++) {
        if(!uniq[i]->isYaochu()) {
            return false;
        }
    }
    return true;
}
/***********************************************************************//**
	七対子か調べる
	@param	agariVector	和了を入れるベクタ
***************************************************************************/
void Hand::checkChitoi(Agari::Vector& agariVector) const {
    if(!isMenzen()) {
        return;
    }
    Agari agari;
    HaiArray menzen = getMenzen();
    menzen.sort();
    HaiArray uniq = menzen.getUnique();
    for(int i = 0; i < uniq.getSize(); i++) {
        if(menzen_.count(uniq[i]) != 2) {
            return;
        }
        HaiArray mentsu;
        mentsu.append(menzen.removeSame(uniq[i]));
        mentsu.append(menzen.removeSame(uniq[i]));
        agari.pushMentsu(Mentsu(mentsu));
    }
    agariVector.push_back(agari);
}
/***********************************************************************//**
	文字列を返す
	@return	手牌を表す文字列
***************************************************************************/
std::string Hand::toString() const {
    std::string str("");
    for(int i = 0; i < fusehai_; i++) {
        str.append("[]");
    }
    str.append(menzen_.toString());
    for(size_t i = 0; i < openMentsu_.size(); i++) {
        str.append(openMentsu_.at(i).toString());
    }
    //str.append("\n");
    //str.append(kawa_.toString());
    return str;
}
/***********************************************************************//**
	ポンで晒せる牌の配列を返す
	@param	hai	ポンする牌
	@return		晒せる牌の配列
***************************************************************************/
HaiArray Hand::getPon(const Hai* hai) {
    HaiArray hais;
    for(int i = 0; i < menzen_.getSize(); i++) {
        if(menzen_[i]->isSame(hai)) {
            hais.append(menzen_[i]);
        }
    }
    return hais;
}
/***********************************************************************//**
	チーで晒せる牌の配列を返す
	@param	hai	チーする牌
	@return		晒せる牌の配列
***************************************************************************/
HaiArray Hand::getChi(const Hai* hai) {
    HaiArray mask;
    if(hai->number > 1 && menzen_.isInclude(hai->succ(-1))) {
        if(hai->number > 2 && menzen_.isInclude(hai->succ(-2))) {
            mask.append(hai->succ(-1));
            mask.append(hai->succ(-2));
        }
        if(hai->number < 9 && menzen_.isInclude(hai->succ(1))) {
            mask.append(hai->succ(-1));
            mask.append(hai->succ(1));
        }
    }
    if(hai->number < 8 && menzen_.isInclude(hai->succ(1)) &&
       menzen_.isInclude(hai->succ(2))) {
        mask.append(hai->succ(1));
        mask.append(hai->succ(2));
    }
    HaiArray hais;
    for(int i = 0; i < menzen_.getSize(); i++) {
        if(!hais.isIncludeEqual(menzen_[i]) &&
           mask.isInclude(menzen_[i])) {
            hais.append(menzen_[i]);
        }
    }
    return hais;
}
/***********************************************************************//**
	晒した面子の数を返す.
	@return	晒した面子の数
***************************************************************************/
int Hand::getOpenMentsuNum() const {
    return openMentsu_.size();
}
/***********************************************************************//**
	晒した面子を返す
	@param	index	インデックス
	@return		面子
***************************************************************************/
const Mentsu& Hand::getOpenMentsu(int index) const {
    return openMentsu_.at(index);
}
/***********************************************************************//**
	リーチをかける
***************************************************************************/
void  Hand::setRichi() {
    kawa_.getLast().setRichi();
    flag_.set(FLAG_RICHI);
}
/***********************************************************************//**
	リーチがかかってるか調べる
	@return	リーチがかかってるとき真
***************************************************************************/
bool Hand::isRichi() const {
    return flag_.test(FLAG_RICHI);
}
/***********************************************************************//**
	手牌を開く(和了、テンパイ宣言)
***************************************************************************/
void Hand::setOpen() {
    flag_.set(FLAG_OPEN);
}
/***********************************************************************//**
	手牌が開いているか調べる
	@return	手牌が開いているとき真
***************************************************************************/
bool Hand::isOpen() const {
    return flag_.test(FLAG_OPEN);
}
/***********************************************************************//**
	手牌を閉じる(ノーテン宣言)
***************************************************************************/
void Hand::setClose() {
    flag_.set(FLAG_CLOSE);
}
/***********************************************************************//**
	手牌が閉じているか調べる
	@return	手牌が閉じているとき真
***************************************************************************/
bool Hand::isClose() const {
    return flag_.test(FLAG_CLOSE);
}
/***********************************************************************//**
	$Id$
***************************************************************************/
}	/* namespace mahjong */
