/***********************************************************************//**
	@file
	$Revision$
	$Author$
	$Date::                           $
***************************************************************************/
#include <assert.h>
#include <algorithm>
#include "HaiArray.h"

namespace mahjong {
/***********************************************************************//**
	文字列からインスタンスを生成する.
	@param	str	文字列
***************************************************************************/
HaiArray::HaiArray(const char* str) {
    const int len = strlen(str);
    for(int i = 0; i < len; i += 2) {
        append(Hai::Get(&str[i]));
    }
}
/***********************************************************************//**
	インデックスの位置の牌を返す.
	@param	index	インデックス
	@return		インデックス位置の牌
***************************************************************************/
const Hai* HaiArray::operator[](int index) const {
    return array_.at(index);
}
/***********************************************************************//**
	牌を追加する.
	@param	hai	追加する牌
	@return		追加したあとの自分自身
***************************************************************************/
HaiArray& HaiArray::append(const Hai* hai) {
    assert(hai);
    array_.push_back(hai);
    return *this;
}
/***********************************************************************//**
	配列を連結する.
	@param	other	連結する配列.
	@return		追加したあとの自分自身
***************************************************************************/
HaiArray& HaiArray::append(const HaiArray& other) {
    array_.insert(array_.end(), other.array_.begin(), other.array_.end());
    return *this;
}
/***********************************************************************//**
	牌の数を数える.
	@param	hai	数える牌
	@return		牌の数
***************************************************************************/
int HaiArray::count(const Hai* hai) const {
    int num = 0;
    for(HaiVector::iterator iter = array_.begin();
        iter != array_.end();
        iter++) {
        if((*iter)->isSame(hai)) {
            num++;
        }
    }
    return num;
}
/***********************************************************************//**
	完全に同じ牌を数える.
	@param	hai	数える牌.
	@return		完全に同じ牌の数
***************************************************************************/
int HaiArray::countEqual(const Hai* hai) const {
    int num = 0;
    for(HaiVector::iterator iter = array_.begin();
        iter != array_.end();
        iter++) {
        if((*iter)->isEqual(hai)) {
            num++;
        }
    }
    return num;
}
/***********************************************************************//**
	完全にユニークな牌を数える.
	@return	完全にユニークな牌の数.
***************************************************************************/
int HaiArray::countUniqueEqual() const {
    HaiArray uniq = *this;
    uniq.uniqueEqual();
    return uniq.getSize();
}
/***********************************************************************//**
	牌が含まれているか調べる.
	@param	hai	調べる牌
	@return		含まれているとき真
***************************************************************************/
bool HaiArray::isInclude(const Hai* hai) const {
    return (count(hai) > 0);
}
/***********************************************************************//**
	完全に同じ牌が含まれているか調べる
	@param	hai	調べる牌
	@return		完全に同じ牌が含まれているとき真
***************************************************************************/
bool HaiArray::isIncludeEqual(const Hai* hai) const {
    return (countEqual(hai) > 0);
}
/***********************************************************************//**
	配列のサイズを返す.
	@return	配列のサイズ
***************************************************************************/
int HaiArray::getSize() const {
    return array_.size();
}
/***********************************************************************//**
	配列が空か調べる.
	@return		空のとき真
***************************************************************************/
bool HaiArray::isEmpty() const {
    return (getSize() == 0);
}
/***********************************************************************//**
	配列をクリアする.
***************************************************************************/
void HaiArray::clear() {
    array_.clear();
}
/***********************************************************************//**
	理牌する.
***************************************************************************/
void HaiArray::sort() {
    std::sort(array_.begin(), array_.end(), Hai::CompareFunc());
}
/***********************************************************************//**
	洗牌する.
***************************************************************************/
void HaiArray::shuffle() {
    std::random_shuffle(array_.begin(), array_.end());
}
/***********************************************************************//**
	文字列に変換する.
	@return	変換した文字列
***************************************************************************/
std::string HaiArray::toString() const {
    std::string str = "";
    for(std::vector<const Hai*>::iterator iter = array_.begin();
        iter != array_.end();
        iter++) {
        str.append((*iter)->toString());
    }
    return str;
}
/***********************************************************************//**
	牌を削除する.
	@param	hai	削除する牌
	@param	num	削除する最大数
	@return		削除した数
***************************************************************************/
int HaiArray::remove(const Hai* hai, int num) {
    assert(num > 0);
    int removeNum = 0;
    HaiVector::iterator iter = array_.begin();
    while(iter != array_.end()) {
        HaiVector::iterator next = iter + 1;
        if((*iter)->isEqual(hai)) {
            array_.erase(iter);
            removeNum++;
            if((--num) == 0) {
                break;
            }
        }
        iter = next;
    }
    return removeNum;
}
/***********************************************************************//**
	同じ牌を削除する
	@param	hai	削除する牌
	@return		削除した牌(削除できなかったときは0)
***************************************************************************/
const Hai* HaiArray::removeSame(const Hai* hai) {
    for(HaiVector::iterator iter = array_.begin();
        iter != array_.end();
        iter++) {
        if((*iter)->isSame(hai)) {
            hai = *iter;
            array_.erase(iter);
            return hai;
        }
    }
    return 0;
}
/***********************************************************************//**
	完全に同じ牌を削除する
	@param	hai	削除する牌
	@return		削除した牌(削除できなかったときは0)
***************************************************************************/
const Hai* HaiArray::removeEqual(const Hai* hai) {
    for(HaiVector::iterator iter = array_.begin();
        iter != array_.end();
        iter++) {
        if((*iter)->isEqual(hai)) {
            hai = *iter;
            array_.erase(iter);
            return hai;
        }
    }
    return 0;
}
/***********************************************************************//**
	重複を取り除く.
	@return	重複を取り除いた自分自身
***************************************************************************/
HaiArray& HaiArray::unique() {
    sort();
    const Hai* hai = 0;
    HaiVector::iterator iter = array_.begin();
    while(iter != array_.end()) {
        if(hai && (*iter)->isSame(hai)) {
            array_.erase(iter);
        }
        else {
            hai = *iter++;
        }
    }
    return *this;
}
/***********************************************************************//**
	重複を取り除いた配列を返す.
	@return	ユニークな配列
***************************************************************************/
HaiArray HaiArray::getUnique() const {
    HaiArray uniq = *this;
    return uniq.unique();
}
/***********************************************************************//**
	完全な重複を取り除く.
	@return	完全な重複を取り除いた自分自身
***************************************************************************/
HaiArray& HaiArray::uniqueEqual() {
    sort();
    const Hai* hai = 0;
    HaiVector::iterator iter = array_.begin();
    while(iter != array_.end()) {
        if(hai && (*iter)->isEqual(hai)) {
            array_.erase(iter);
        }
        else {
            hai = *iter++;
        }
    }
    return *this;
}
/***********************************************************************//**
	$Id$
***************************************************************************/
}	/* namespace mahjong */
