/*
 * OpenI2CRADIO
 * I2C EEPROM Handler
 * Copyright (C) 2013-08-25 K.Ohta <whatisthis.sowhat at gmail.com>
 * License: GPL2+LE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2,
 *  or (at your option) any later version.
 *  This library / program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; see the file COPYING. If not, write to the
 *  Free Software Foundation, 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301, USA.
 *
 *  As a special exception, if you link this(includeed from sdcc) library
 *  with other files, some of which are compiled with SDCC,
 *  to produce an executable, this library does not by itself cause
 *  the resulting executable to be covered by the GNU General Public License.
 *  This exception does not however invalidate any other reasons why
 *  the executable file might be covered by the GNU General Public License.
 */

#include "i2c_eeprom.h"


#ifndef __SDCC
void i2c_writechar(unsigned char c)
{
    WriteI2C(c);
    while(SSPCON2bits.ACKSTAT);
}
#endif


static unsigned char i2c_eeprom_write_1block(unsigned char i2caddr, unsigned int addr, unsigned char *data, unsigned int size)
{
    unsigned char i;
    unsigned char *p = data;
    unsigned char page_hi = addr >> 8;
    unsigned char page_lo = addr & 0xff;

#ifdef __SDCC
    i2c_open(I2C_MASTER, I2C_SLEW_ON, 5);
    I2C_START();
    i2c_writechar(i2caddr & 0xfe);
    i2c_writechar(page_hi);
    i2c_writechar(page_lo);
    for(i = 0; i < 64; i++){
        i2c_writechar(*p++);
    }
    I2C_STOP();
    i2c_close();
#else
    if((size == 0) || (size > I2C_ROM_PAGE_SIZE)) return 0x00;
    OpenI2C(MASTER, SLEW_ON);
    SSPADD = 0x5;
    IdleI2C();
    StartI2C();
    while(SSPCON2bits.SEN);

    i2c_writechar(i2caddr & 0xfe);
    i2c_writechar(page_hi);
    i2c_writechar(page_lo);

    if(!SSPCON2bits.ACKSTAT){
        for(i = 0; i < size ; i++){
            if(SSPCON1bits.SSPM3) {
                i2c_writechar(*p);
                IdleI2C();
                p++;
            }
        }
    }
    StopI2C();
    while (SSPCON2bits.PEN);
    idle_time_ms(2);
    do {
        StartI2C();
        WriteI2C(i2caddr & 0xfe);
        idle_time_ms(1);
    } while(SSPCON2bits.ACKSTAT);
    StopI2C();
    while (SSPCON2bits.PEN);

    CloseI2C();
#endif  //    i2c_idle();
    return 0xff;
}



// I2C_IO
unsigned char i2c_eeprom_bytewrite(unsigned char i2caddr, unsigned int addr, unsigned char data)
{
    unsigned char page_hi = addr >> 8;
    unsigned char page_lo = addr & 0xff; // 64byte?
    
#ifdef __SDCC
    i2c_open(I2C_MASTER, I2C_SLEW_ON, 5);
    I2C_START();
    i2c_writechar(i2caddr & 0xfe);
    i2c_writechar(page_hi);
    i2c_writechar(page_lo);
    i2c_writechar(data);
    I2C_STOP();
    i2c_close();
#else
    OpenI2C(MASTER, SLEW_ON);
    SSPADD = 5; // 100KHz
    IdleI2C();
    StartI2C();
    while(SSPCON2bits.SEN);

    i2c_writechar(i2caddr & 0xfe);
    i2c_writechar(page_hi);
    i2c_writechar(page_lo);
    i2c_writechar(data);

    StopI2C();

    while(SSPCON2bits.PEN);
    do {
        StartI2C();
        WriteI2C(i2caddr & 0xfe);
        idle_time_ms(1);
    } while(SSPCON2bits.ACKSTAT);
    StopI2C();
    while (SSPCON2bits.PEN);
    CloseI2C();
#endif  //    i2c_idle();
    return 0xff;
}


unsigned char i2c_eeprom_burstwrite(unsigned char i2caddr, unsigned int addr, unsigned char *data, unsigned int bytes)
{
    unsigned int b = bytes;
    unsigned int bb;
    unsigned char sts;

    if((addr % I2C_ROM_PAGE_SIZE) != 0) {
        bb = I2C_ROM_PAGE_SIZE - (addr % I2C_ROM_PAGE_SIZE);
        if(b <= bb) bb = b;
        sts = i2c_eeprom_write_1block(i2caddr, addr, data, bb);
        if(sts == 0) return 0;
        b -= bb;
        addr += bb;
        data += bb;
    }

    while(b >= I2C_ROM_PAGE_SIZE){
        sts = i2c_eeprom_write_1block(i2caddr,  addr, data, I2C_ROM_PAGE_SIZE);
//        idle_time_ms(15);
        if(sts == 0) return 0;
        addr += I2C_ROM_PAGE_SIZE;
        data += I2C_ROM_PAGE_SIZE;
        b -= I2C_ROM_PAGE_SIZE;
    }

    if(b != 0){
        sts = i2c_eeprom_write_1block(i2caddr,  addr, data, b);
    }
    return sts;
}


static unsigned char i2c_eeprom_read_1block(unsigned char i2caddr, unsigned int addr, unsigned char *data, unsigned int size)
{
    unsigned char i;
    unsigned char page_hi = addr >> 8;
    unsigned char page_lo = addr & 0xff;

  #ifdef __SDCC
    i2c_open(I2C_MASTER, I2C_SLEW_ON, 5);
    I2C_START();
    i2c_writechar(addr & 0xfe);
    i2c_writechar(page_hi);
    i2c_writechar(page_lo);

//   delay100tcy(2);
    I2C_START();
    i2c_writechar(addr | 0x01);
    for(i = 0; i < I2C_ROM_PAGE_SIZE - 1 ; i++){
        *data = i2c_readchar();
        data++;
        I2C_ACK();
    }
    *data = i2c_readchar();
    I2C_NACK();
    I2C_STOP();
    i2c_close();
#else
    if((size == 0) || (size > I2C_ROM_PAGE_SIZE)) return 0x00;
    OpenI2C(MASTER, SLEW_ON);
    SSPADD = 0x5;
    StartI2C();
    while(SSPCON2bits.SEN);
    WriteI2C(i2caddr & 0xfe);
  //  delay1ktcy(8);
    while(SSPCON2bits.SEN);
    WriteI2C(page_hi);
    while(SSPCON2bits.SEN);
    WriteI2C(page_lo);
  //  delay1ktcy(8);
    StopI2C();

    IdleI2C();
    StartI2C();
    WriteI2C(i2caddr | 1);

    for(i = 0; i < size - 1 ;i++){
       if (!SSPCON2bits.ACKSTAT){
        SSPCON2bits.RCEN = 1;
        while(SSPCON2bits.RCEN);
         AckI2C();
//          while (!SSPCON2bits.ACKEN);
        *data = SSPBUF;
         data++;
        }
    }

    if (!SSPCON2bits.ACKSTAT){
      SSPCON2bits.RCEN = 1;
      while(SSPCON2bits.RCEN);
      NotAckI2C();
      while (SSPCON2bits.ACKEN);
      StopI2C();
      while (SSPCON2bits.PEN);
    }
    *data = SSPBUF;

    CloseI2C();
#endif
    //    CLOSEASMASTER();
    return 0xff;
}


// I2C_IO
unsigned char i2c_eeprom_byteread(unsigned char i2caddr, unsigned int addr)
{
    unsigned char page_hi = addr >> 8;
    unsigned char page_lo = addr & 0xff; // 64byte?
    unsigned char c;

  #ifdef __SDCC
    i2c_open(I2C_MASTER, I2C_SLEW_ON, 5);
    I2C_START();
    i2c_writechar(i2caddr);
    i2c_writechar(page_hi);
    i2c_writechar(page_lo);
    I2C_STOP();
    i2c_idle();
//   delay100tcy(2);
    I2C_START();
    i2c_writechar(i2caddr | 1);
    c = i2c_readchar();
    I2C_ACK();
    I2C_STOP();
    i2c_close();
#else
    OpenI2C(MASTER, SLEW_ON);
    SSPADD = 0x5;
    StartI2C();
    while(SSPCON2bits.SEN);
    WriteI2C(i2caddr);
    while(SSPCON2bits.SEN);
    WriteI2C(page_hi);
    while(SSPCON2bits.SEN);
    WriteI2C(page_lo);
    StopI2C();

    IdleI2C();
    StartI2C();
    WriteI2C(i2caddr | 1);

    if (!SSPCON2bits.ACKSTAT){
      SSPCON2bits.RCEN = 1;
      while(SSPCON2bits.RCEN);
      NotAckI2C();
      while (SSPCON2bits.ACKEN);
      StopI2C();
      while (SSPCON2bits.PEN);
    }
    c = SSPBUF;
    CloseI2C();
#endif
    //    CLOSEASMASTER();
    return c;
}


unsigned char i2c_eeprom_burstread(unsigned char i2caddr, unsigned int addr, unsigned char *data, unsigned int bytes)
{
    unsigned char sts;
    unsigned int b = bytes;
    unsigned int bb;

    if((addr % I2C_ROM_PAGE_SIZE) != 0) {
        bb = I2C_ROM_PAGE_SIZE - (addr % I2C_ROM_PAGE_SIZE);
        if(b <= bb) bb = b;
        sts = i2c_eeprom_read_1block(i2caddr, addr, data, bb);
        b -= bb;
        addr += bb;
        data += bb;
        if(sts == 0) return 0;
    }

    while(b >= I2C_ROM_PAGE_SIZE){
        sts = i2c_eeprom_read_1block(i2caddr,  addr, data, I2C_ROM_PAGE_SIZE);
//        idle_time_ms(15);
        if(sts == 0) return 0;
        addr += I2C_ROM_PAGE_SIZE;
        data += I2C_ROM_PAGE_SIZE;
        b -= I2C_ROM_PAGE_SIZE;
    }

    if(b != 0){
        sts = i2c_eeprom_read_1block(i2caddr,  addr, data, b);
    }
    return sts;
}
