/*
 * OpenI2CRADIO
 * EEPROM utils
 * Copyright (C) 2013-06-10 K.Ohta <whatisthis.sowhat ai gmail.com>
 * License: GPL2+LE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2,
 *  or (at your option) any later version.
 *  This library / program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; see the file COPYING. If not, write to the
 *  Free Software Foundation, 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301, USA.
 *
 *  As a special exception, if you link this(includeed from sdcc) library
 *  with other files, some of which are compiled with SDCC,
 *  to produce an executable, this library does not by itself cause
 *  the resulting executable to be covered by the GNU General Public License.
 *  This exception does not however invalidate any other reasons why
 *  the executable file might be covered by the GNU General Public License.
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(__SDCC)
#include <sdcc-lib.h>
#include <pic18fregs.h> /* ONLY FOR PIC18x */
#include <delay.h>
#else
#include <xc.h>
#endif
#include <signal.h>

#include "commondef.h"
#include "iodef.h"
#include "idle.h"
#include "i2c_io.h"
#include "i2c_eeprom.h"
#include "akc6955.h"
#include "lcd_acm1602.h"
#include "ui.h"
#include "eeprom.h"
#include "ioports.h"
#include "menu.h"
#include "power.h"
#include "adc_int.h"

void save_eeprom(void)
{
    unsigned int p[1];
    unsigned int sum[1];
        unsigned char i;
    unsigned char j;
    unsigned char *q = (char *)(&setup);

    *p = 0;
    *sum = 0;
    // Magic word
    writeword_eeprom(p, sum, EEPROM_INTSET_MAGIC);
    // amfreq
    for(i = 0; i < sizeof(__radioset_t); i++) {
        writebyte_eeprom(p, sum, *q);
        q++;
    }

    // Write checksum
    eeprom_writebyte(*p, sum[0] >> 8);
    eeprom_writebyte(*p + 1, sum[0] & 0xff);
}


unsigned int calcsum_frequencies(__freqset_t *p, unsigned int start)
{
    unsigned int sum = start;
    unsigned char *q = (unsigned char *)p;
    unsigned char c;
    unsigned char i;

    for(i = 0; i < (sizeof(__freqset_t) - 2); i++){
        c = *q;
        q++;
        sum = sum + c;
    }
    return sum;
}
/*
  * External I2C Map:
 *  0x0000
 *  0x01FF : User settings(Reserved not use)
 *  0x0100
 *  0x01ff : User bands
 *  0x0200
 *  0x0fff : User frequencies ( 12*2*28 = 672chs Max.)
 *  0x1000- : Reserve.
 */
int load_frequencies(unsigned int page, unsigned char check_only)
{

    __freqset_t *p;
    unsigned int sum;
    unsigned int addr = (page << 7) + EEPROM_FREQSET_ADDR;
    unsigned char *q;
    unsigned char *r;
    unsigned char c;
    unsigned int i;

//    if(addr > 0x0fff) return -1; // Address error
    if(page > USER_MEMORY_BANKS) return -1; // Address Error
    p = &freqset_temp;
    c = i2c_eeprom_burstread(I2CEEPROM_ADDR, addr, (unsigned char *)p, sizeof(__freqset_t));
    // Pass 1 Read OK?
    if(c == 0x00) return -2;
    // Pass 1 check magic
    if(freqset_temp.magic != EEPROM_FREQSET_MAGIC) return -3;
    // Pass 2 check sum
    sum = calcsum_frequencies(&freqset_temp, 0xa5a5);
    if(sum != freqset_temp.checksum) return -4;
    // Move from
    if(check_only == 0x00){
        q = (unsigned char *)(&freqset_temp);
        r = (unsigned char *)(&freqset);
        for(i = 0; i < sizeof(__freqset_t); i++){
            *r = *q;
            q++;
            r++;
        }
        setup.pagenum = page;
    }
    return 0;
}



void save_frequencies(void)
{
    unsigned char c;
    unsigned int addr = (setup.pagenum <<7) + EEPROM_FREQSET_ADDR;
    __freqset_t *p = &freqset;
    unsigned char i;
    unsigned int sum;

    if(setup.pagenum > USER_MEMORY_BANKS) return; // Address Error
    // Pass 0 Calc checksum
    freqset.magic = EEPROM_FREQSET_MAGIC;
    freqset.version = 0x0001;
    sum = calcsum_frequencies(&freqset, 0xa5a5);
    freqset.checksum = sum;
    // Pass 1 write to rom.
    c = i2c_eeprom_burstwrite(I2CEEPROM_ADDR, addr, (unsigned char *)p, sizeof(__freqset_t));
    return;
}

char save_frequencies_page(unsigned char page)
{
    unsigned char c;
    unsigned int addr = (page <<7) + EEPROM_FREQSET_ADDR;
    __freqset_t *p = &freqset_temp;
    unsigned char i;
    unsigned int sum;

    if(page > USER_MEMORY_BANKS) return -3; // Address Error
    // Pass 0 Calc checksum
    if(p->magic != EEPROM_FREQSET_MAGIC) return -1; // Data error
    sum = calcsum_frequencies(p, 0xa5a5);
    if(p->checksum != sum) return -2; // Sum error
    // Pass 1 write to rom.
    c = i2c_eeprom_burstwrite(I2CEEPROM_ADDR, addr, (unsigned char *)p, sizeof(__freqset_t));
    return 0;
}


void format_frequencies(unsigned int page)
{
    unsigned char i;
    unsigned int sum;

    freqset_temp.magic = EEPROM_FREQSET_MAGIC;
    freqset_temp.version = 0x0001;
    for(i = 0; i < USER_MEMORY_NUM; i++){
        freqset_temp.memoryfreqs[i].band = AKC6955_BAND_MW2;
        freqset_temp.memoryfreqs[i].fm = 0;
        freqset_temp.memoryfreqs[i].freq = 594;
    }
    freqset_temp.pagenum = page;
    sum = calcsum_frequencies(&freqset_temp, 0xa5a5);
    freqset_temp.checksum = sum;

    i = i2c_eeprom_burstwrite(I2CEEPROM_ADDR, (page << 7)+ EEPROM_FREQSET_ADDR, (unsigned char *)(&freqset_temp), sizeof(__freqset_t));
}

void check_frequencies(void)
{
    unsigned int i;
    _CLS();
    for(i = 0; i < USER_MEMORY_BANKS; i++){
        printstr("Check ExtROM:");
        print_numeric_nosupress(i, 2);
       _HOME();
        
        if(load_frequencies(i, 0xff) < 0){
//            printstr("Format:");
//            print_numeric_nosupress(i, 2);
            _HOME();
            format_frequencies(i);
        }
    }
   if(load_userbands() < 0) {
      format_userbands();
   }
}

static unsigned int calcsum_userband(__userband_t_t *p, unsigned int start)
{
    unsigned int sum = start;
    unsigned char *q = (unsigned char *)p;
    unsigned char c;
    unsigned char i;

    for(i = 0; i < (sizeof(__userband_t_t) - 2); i++){
        c = *q;
        q++;
        sum = sum + c;
    }
    return sum;
}

int load_userbands(void)
{

    __userband_t_t *p = &userband;
    unsigned int sum;
    unsigned int addr = EEPROM_BANDSET_ADDR;
    unsigned char c;

//    if(addr > 0x0fff) return -1; // Address error
   c = i2c_eeprom_burstread(I2CEEPROM_ADDR, addr, (unsigned char *)p, sizeof(__userband_t_t));
    // Pass 1 Read OK?
   if(c == 0x00) return -2;
    // Pass 1 check magic
   if(p->magic != EEPROM_BANDSET_MAGIC) return -3;
    // Pass 2 check sum
   sum = calcsum_userband(p, 0x5a5a);
   if(sum != p->checksum) return -4;
   return 0;
}

void save_userbands(void)
{
    unsigned char c;
    unsigned int addr = EEPROM_BANDSET_ADDR;
    __userband_t_t *p = &userband;
    unsigned char i;
    unsigned int sum;

    // Pass 0 Calc checksum
    userband.magic = EEPROM_BANDSET_MAGIC;
    userband.version = 0x0001;
    sum = calcsum_userband(&userband, 0x5a5a);
    userband.checksum = sum;
    // Pass 1 write to rom.
    c = i2c_eeprom_burstwrite(I2CEEPROM_ADDR, addr, (unsigned char *)p, sizeof(__userband_t_t));
    return;
}

void format_userbands(void)
{
   unsigned char i;
   for(i = 0; i < USER_BAND_NUM; i++) {
	userband.am_usrbands[i].start = 0x19;
        userband.am_usrbands[i].stop = 0x32;
        userband.am_usrbands[i].freq = 5600; //(0x32 - (0x32-0x19) / 2 *32) * 5
        userband.am_usrbands[i].mode3k = 0;
        //
	userband.fm_usrbands[i].start = 0x19;
        userband.fm_usrbands[i].stop = 0x32;
        userband.fm_usrbands[i].freq = 8600;
        userband.fm_usrbands[i].mode3k = 0;
   }
   save_userbands(); // Re-init
}


unsigned char load_eeprom(void)
{
    unsigned int p[1];
    unsigned int sum[1];
    unsigned char i;
    unsigned char *q = (unsigned char *)(&setup);
    unsigned char j;
    unsigned int magic;

    p[0] = 0;
    sum[0] = 0;
    // Magic word
    magic = readword_eeprom(p, sum);
    if(magic != EEPROM_INTSET_MAGIC) return 0x01; // NO MAGICWORD
    // amfreq
    for(i = 0; i < sizeof(__radioset_t); i++ ){
        *q = readbyte_eeprom(p, sum);
        q++;
    }
    magic = (eeprom_readbyte(p[0]) << 8) + eeprom_readbyte(p[0] + 1);

    if(sum[0] != magic) return 0x00;
    return 0xff;
}

/*
 * Check eeprom, and format/restore.
 */
static void check_eeprom_sub(void)
{
 _CLS();
 _LOCATE_0_0();
 printstr("Formatting...  ");
 format_eeprom(2,250);
 _LOCATE_0_0();
 printstr("Save defaults  ");
 setdefault();
 save_eeprom();
}


void check_eeprom(void)
{
    unsigned char c;

    switch(load_eeprom()) {
        case 0x01: // No magic-word
//            idle_time_ms(2000);
            c = printhelp_2lines("EEPROM FORMAT", "Press any key");
            check_eeprom_sub();
            break;
        case 0x00: // Checksum error
//           idle_time_ms(2000);
            c = printhelp_2lines("X-) Sum error", "Press any key");
            check_eeprom_sub();
            break;
        case 0xff: // Success
            check_frequencies();
            load_frequencies(setup.pagenum, 0);
            return;
            break;
        default: // Unknown error
            break;
    }
    valinit();
    check_frequencies();
    load_frequencies(setup.pagenum, 0);
}
