/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;

import jp.ossc.nimbus.service.log.DefaultLoggerService;
import jp.ossc.nimbus.service.message.DefaultMessageRecordFactoryService;

import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

/**
 * {@link GenericsServiceProxy}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class GenericsServiceProxyTest{
    
    
    @DataProvider(name = "GenericsServiceProxy")
    public Object[][] dataGenericsFactoryServiceProxy(){
        return new Object[][]{
            {new MyServiceBaseSupport()},
            {new MyServiceBaseSupport2()}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.GenericsServiceProxy#GenericsServiceProxy(jp.ossc.nimbus.core.ServiceBaseSupport)",
                    "jp.ossc.nimbus.core.GenericsServiceProxy#getMBeanInfo()",
                    "jp.ossc.nimbus.core.GenericsServiceProxy#setAttribute(javax.management.Attribute)",
                    "jp.ossc.nimbus.core.GenericsServiceProxy#getAttribute(java.lang.String)",
                    "jp.ossc.nimbus.core.GenericsServiceProxy#getAttributes(java.lang.String[])",
                    "jp.ossc.nimbus.core.GenericsServiceProxy#setAttributes(javax.management.AttributeList)",
                    "jp.ossc.nimbus.core.GenericsServiceProxy#invoke(String,java.lang.Object[],java.lang.String[])",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#.*"},
          dataProvider = "GenericsServiceProxy")
    public void testGenericsServiceProxy(ServiceBaseSupport support) throws Exception{
        GenericsServiceProxy service = new GenericsServiceProxy(support);
        assertNotNull(service.getLogger());
        assertNotNull(service.getMessageRecordFactory());
        assertNull(service.getServiceManager());
        assertNull(service.getServiceManagerName());
        assertNull(service.getServiceName());
        assertNull(service.getServiceNameObject());
        assertEquals(service.getState(), Service.State.DESTROYED);
        assertNull(service.getSystemLoggerServiceName());
        assertNull(service.getSystemMessageRecordFactoryServiceName());
        assertEquals(service.getTarget(), support);
        
        try{
            service.setServiceManagerName("test");
            service.setServiceName("Service1");
            ServiceManagerFactory.registerManager("test");
            DefaultLoggerService logger = new DefaultLoggerService();
            logger.setServiceManagerName("test");
            logger.setServiceName("Logger");
            logger.create();
            logger.start();
            DefaultMessageRecordFactoryService message = new DefaultMessageRecordFactoryService();
            message.setServiceManagerName("test");
            message.setServiceName("Message");
            message.create();
            message.start();
            MBeanInfo info = service.getMBeanInfo();
            assertNotNull(info);
            MBeanAttributeInfo[] attributeInfos = info.getAttributes();
            assertNotNull(attributeInfos);
            if(support instanceof MyServiceBaseSupport2){
                assertEquals(attributeInfos.length, 9);
            }else{
                assertEquals(attributeInfos.length, 6);
            }
            for(MBeanAttributeInfo attributeInfo : attributeInfos){
                if("ServiceName".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "java.lang.String");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertEquals(service.getAttribute(attributeInfo.getName()), "Service1");
                    service.setAttribute(new Attribute(attributeInfo.getName(), "Service2"));
                    assertEquals(service.getAttribute(attributeInfo.getName()), "Service2");
                    service.setAttribute(new Attribute(attributeInfo.getName(), "Service1"));
                }else if("ServiceManagerName".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "java.lang.String");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertEquals(service.getAttribute(attributeInfo.getName()), "test");
                    service.setAttribute(new Attribute(attributeInfo.getName(), "test2"));
                    assertEquals(service.getAttribute(attributeInfo.getName()), "test2");
                    service.setAttribute(new Attribute(attributeInfo.getName(), "test"));
                }else if("ServiceNameObject".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "jp.ossc.nimbus.core.ServiceName");
                    assertTrue(attributeInfo.isReadable());
                    assertFalse(attributeInfo.isWritable());
                    assertEquals(service.getAttribute(attributeInfo.getName()), new ServiceName("test", "Service1"));
                }else if("State".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "jp.ossc.nimbus.core.Service$State");
                    assertTrue(attributeInfo.isReadable());
                    assertFalse(attributeInfo.isWritable());
                    assertEquals(service.getAttribute(attributeInfo.getName()), Service.State.DESTROYED);
                }else if("SystemLoggerServiceName".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "jp.ossc.nimbus.core.ServiceName");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertNull(service.getAttribute(attributeInfo.getName()));
                    service.setAttribute(new Attribute(attributeInfo.getName(), new ServiceName("test", "Logger")));
                    assertEquals(service.getAttribute(attributeInfo.getName()), new ServiceName("test", "Logger"));
                    service.setAttribute(new Attribute(attributeInfo.getName(), null));
                }else if("SystemMessageRecordFactoryServiceName".equals(attributeInfo.getName())){
                    assertEquals(attributeInfo.getType(), "jp.ossc.nimbus.core.ServiceName");
                    assertTrue(attributeInfo.isReadable());
                    assertTrue(attributeInfo.isWritable());
                    assertNull(service.getAttribute(attributeInfo.getName()));
                    service.setAttribute(new Attribute(attributeInfo.getName(), new ServiceName("test", "Message")));
                    assertEquals(service.getAttribute(attributeInfo.getName()), new ServiceName("test", "Message"));
                    service.setAttribute(new Attribute(attributeInfo.getName(), null));
                }else{
                    if(support instanceof MyServiceBaseSupport2){
                        if("Hoge".equals(attributeInfo.getName())){
                            assertEquals(attributeInfo.getType(), "java.lang.String");
                            assertTrue(attributeInfo.isReadable());
                            assertTrue(attributeInfo.isWritable());
                            assertNull(service.getAttribute(attributeInfo.getName()));
                            service.setAttribute(new Attribute(attributeInfo.getName(), "hoge"));
                            assertEquals(service.getAttribute(attributeInfo.getName()), "hoge");
                        }else if("Fuga".equals(attributeInfo.getName())){
                            assertEquals(attributeInfo.getType(), "int");
                            assertTrue(attributeInfo.isReadable());
                            assertFalse(attributeInfo.isWritable());
                            assertEquals(service.getAttribute(attributeInfo.getName()), new Integer(100));
                        }else if("Bar".equals(attributeInfo.getName())){
                            assertEquals(attributeInfo.getType(), "java.lang.Object");
                            assertTrue(attributeInfo.isWritable());
                            service.setAttribute(new Attribute(attributeInfo.getName(), "bar"));
                            assertEquals(((MyServiceBaseSupport2)support).bar, "bar");
                        }else{
                            throw new Exception("Failed. unexpected attribute." + attributeInfo);
                        }
                    }else{
                        throw new Exception("Failed. unexpected attribute." + attributeInfo);
                    }
                }
            }
            AttributeList attributeList = service.getAttributes(new String[]{"ServiceName", "ServiceManagerName"});
            for(Object attribute : attributeList){
                Attribute attr = (Attribute)attribute;
                if("ServiceName".equals(attr.getName())){
                    assertEquals(attr.getValue(), "Service1");
                }else if("ServiceManagerName".equals(attr.getName())){
                    assertEquals(attr.getValue(), "test");
                }
            }
            attributeList = new AttributeList();
            attributeList.add(new Attribute("ServiceName", "Service2"));
            attributeList.add(new Attribute("ServiceManagerName", "test2"));
            attributeList = service.setAttributes(attributeList);
            for(Object attribute : attributeList){
                Attribute attr = (Attribute)attribute;
                if("ServiceName".equals(attr.getName())){
                    assertEquals(attr.getValue(), "Service2");
                    assertEquals(service.getServiceName(), "Service2");
                }else if("ServiceManagerName".equals(attr.getName())){
                    assertEquals(attr.getValue(), "test2");
                    assertEquals(service.getServiceManagerName(), "test2");
                }
            }
            service.setServiceManagerName("test");
            service.setServiceName("Service1");
            if(support instanceof MyServiceBaseSupport2){
                attributeList = service.getAttributes(new String[]{"Hoge", "Fuga"});
                for(Object attribute : attributeList){
                    Attribute attr = (Attribute)attribute;
                    if("Hoge".equals(attr.getName())){
                        assertEquals(attr.getValue(), "hoge");
                    }else if("Fuga".equals(attr.getName())){
                        assertEquals(attr.getValue(), new Integer(100));
                    }
                }
                attributeList = new AttributeList();
                attributeList.add(new Attribute("Hoge", "hoge2"));
                attributeList.add(new Attribute("Bar", "bar2"));
                attributeList = service.setAttributes(attributeList);
                for(Object attribute : attributeList){
                    Attribute attr = (Attribute)attribute;
                    if("Hoge".equals(attr.getName())){
                        assertEquals(attr.getValue(), "hoge2");
                        assertEquals(((MyServiceBaseSupport2)support).getHoge(), "hoge2");
                    }else if("Bar".equals(attr.getName())){
                        assertEquals(attr.getValue(), "bar2");
                        assertEquals(((MyServiceBaseSupport2)support).bar, "bar2");
                    }
                }
            }
            
            MBeanOperationInfo[] operationInfos = info.getOperations();
            assertNotNull(operationInfos);
            if(support instanceof MyServiceBaseSupport2){
                assertEquals(operationInfos.length, 6);
            }else{
                assertEquals(operationInfos.length, 5);
            }
            for(MBeanOperationInfo operationInfo : operationInfos){
                if("create".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), "void");
                }else if("start".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), "void");
                }else if("stop".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), "void");
                }else if("destroy".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), "void");
                }else if("restart".equals(operationInfo.getName())){
                    assertEquals(operationInfo.getSignature().length, 0);
                    assertEquals(operationInfo.getReturnType(), "void");
                }else{
                    if(support instanceof MyServiceBaseSupport2){
                        if("foo".equals(operationInfo.getName())){
                            assertEquals(operationInfo.getSignature().length, 1);
                            assertEquals(operationInfo.getSignature()[0].getType(), "int");
                            assertEquals(operationInfo.getReturnType(), "java.lang.String");
                        }else{
                            throw new Exception("Failed. unexpected operation." + operationInfo);
                        }
                    }else{
                        throw new Exception("Failed. unexpected operation." + operationInfo);
                    }
                }
            }
            if(support instanceof MyServiceBaseSupport2){
                assertEquals(service.invoke("foo", new Object[]{100}, new String[]{"int"}), "foo100");
            }
            assertEquals(((MyServiceBaseSupport)support).calledSetServiceBase, 1);
            assertNull(service.invoke("create", null, null));
            assertEquals(service.getState(), Service.State.CREATED);
            if(support instanceof MyServiceBaseSupport){
                assertEquals(((MyServiceBaseSupport)support).calledSetServiceBase, 1);
                assertEquals(((MyServiceBaseSupport)support).calledCreate, 1);
                assertEquals(((MyServiceBaseSupport)support).calledStart, 0);
                assertEquals(((MyServiceBaseSupport)support).calledStop, 0);
                assertEquals(((MyServiceBaseSupport)support).calledDestroy, 0);
            }
            assertNull(service.invoke("start", null, null));
            assertEquals(service.getState(), Service.State.STARTED);
            if(support instanceof MyServiceBaseSupport){
                assertEquals(((MyServiceBaseSupport)support).calledSetServiceBase, 1);
                assertEquals(((MyServiceBaseSupport)support).calledCreate, 1);
                assertEquals(((MyServiceBaseSupport)support).calledStart, 1);
                assertEquals(((MyServiceBaseSupport)support).calledStop, 0);
                assertEquals(((MyServiceBaseSupport)support).calledDestroy, 0);
            }
            assertNull(service.invoke("restart", null, null));
            assertEquals(service.getState(), Service.State.STARTED);
            if(support instanceof MyServiceBaseSupport){
                assertEquals(((MyServiceBaseSupport)support).calledSetServiceBase, 1);
                assertEquals(((MyServiceBaseSupport)support).calledCreate, 1);
                assertEquals(((MyServiceBaseSupport)support).calledStart, 2);
                assertEquals(((MyServiceBaseSupport)support).calledStop, 1);
                assertEquals(((MyServiceBaseSupport)support).calledDestroy, 0);
            }
            assertNull(service.invoke("stop", null, null));
            assertEquals(service.getState(), Service.State.STOPPED);
            if(support instanceof MyServiceBaseSupport){
                assertEquals(((MyServiceBaseSupport)support).calledSetServiceBase, 1);
                assertEquals(((MyServiceBaseSupport)support).calledCreate, 1);
                assertEquals(((MyServiceBaseSupport)support).calledStart, 2);
                assertEquals(((MyServiceBaseSupport)support).calledStop, 2);
                assertEquals(((MyServiceBaseSupport)support).calledDestroy, 0);
            }
            assertNull(service.invoke("destroy", null, null));
            assertEquals(service.getState(), Service.State.DESTROYED);
            if(support instanceof MyServiceBaseSupport){
                assertEquals(((MyServiceBaseSupport)support).calledSetServiceBase, 1);
                assertEquals(((MyServiceBaseSupport)support).calledCreate, 1);
                assertEquals(((MyServiceBaseSupport)support).calledStart, 2);
                assertEquals(((MyServiceBaseSupport)support).calledStop, 2);
                assertEquals(((MyServiceBaseSupport)support).calledDestroy, 1);
            }
        }finally{
            ServiceManagerFactory.unloadManager("test");
        }
    }
    
    private static class MyServiceBaseSupport implements ServiceBaseSupport{
        
        public int calledSetServiceBase;
        public int calledCreate;
        public int calledDestroy;
        public int calledStart;
        public int calledStop;

        @Override
        public void createService() throws Exception{
            calledCreate++;
        }

        @Override
        public void destroyService() throws Exception{
            calledDestroy++;
        }

        @Override
        public void startService() throws Exception{
            calledStart++;
        }

        @Override
        public void stopService() throws Exception{
            calledStop++;
        }

        @Override
        public void setServiceBase(ServiceBase service){
            calledSetServiceBase++;
        }
        
    }
    
    public static class MyServiceBaseSupport2 implements ServiceBaseSupport, MyServiceBaseSupport2MBean{
        
        public int calledSetServiceBase;
        public int calledCreate;
        public int calledDestroy;
        public int calledStart;
        public int calledStop;
        
        @Override
        public void createService() throws Exception{
            calledCreate++;
        }

        @Override
        public void destroyService() throws Exception{
            calledDestroy++;
        }

        @Override
        public void startService() throws Exception{
            calledStart++;
        }

        @Override
        public void stopService() throws Exception{
            calledStop++;
        }

        @Override
        public String foo(int index){
            return "foo" + index;
        }

        @Override
        public int getFuga(){
            return 100;
        }
        
        private String hoge;

        @Override
        public String getHoge(){
            return hoge;
        }

        @Override
        public void setHoge(String val){
            hoge = val;
        }
        
        Object bar;
        
        @Override
        public void setBar(Object obj){
            bar = obj;
        }

        @Override
        public void setServiceBase(ServiceBase service){
            calledSetServiceBase++;
        }
    }
    
    private static interface MyServiceBaseSupport2MBean{
        
        public void setHoge(String val);
        public String getHoge();
        
        public int getFuga();
        
        public void setBar(Object obj);
        
        public String foo(int index);
    }

}
