/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.semaphore;

import java.util.*;

import jp.ossc.nimbus.core.ServiceBase;
import jp.ossc.nimbus.util.SynchronizeMonitor;
import jp.ossc.nimbus.util.WaitSynchronizeMonitor;

/**
 * Z}tHB<p>
 *
 * @author H.Nakano
 */
public class DefaultSemaphoreService extends ServiceBase
 implements Semaphore, DefaultSemaphoreServiceMBean, java.io.Serializable{
    
    private static final long serialVersionUID = -408553618405283847L;
    
    //## o[ϐ錾 ##
    
    /** Z}tH\[X */
    protected volatile int resourceCnt = 1;
    
    /** Z}tH */
    protected volatile int initialResource = 1;
    
    /** ItO */
    protected volatile boolean forceEndFlg = true;
    
    /** Z}tHlj^ */
    protected transient SynchronizeMonitor getMonitor = new WaitSynchronizeMonitor();
    
    /** Z}tHlXbhW */
    protected transient Set<Thread> usedThreads = Collections.synchronizedSet(new HashSet<Thread>());
    
    /** Z}tHlXbhW */
    protected transient Map<Thread, Object> threadTasks = Collections.synchronizedMap(new HashMap<Thread, Object>());
    
    /** l҂XbhSleep[ms] */
    protected long sleepTime = 10000;
    
    /** \[XĎԊu */
    protected long checkInterval = -1;
    
    /** \[XĎ^XN */
    protected transient ResourceChecker checker;
    
    /** ő僊\[Xgp */
    protected int maxUsedResource;
    
    /** ő僊\[Xl҂ */
    protected int maxWaitedCount;
    
    /** J^C}[ */
    protected transient Timer forceFreeTimer;
    
    protected long forceFreeTimeoutMillis = -1L;
    
    protected int maxWaitCount = -1;
    
    protected long timeoutMillis = -1L;
    
    protected boolean isThreadBinding = true;
    
    // SemaphoreJavaDoc
    public long getForceFreeTimeoutMillis(){
        return forceFreeTimeoutMillis ;
    }

    // SemaphoreJavaDoc
    public void setForceFreeTimeoutMillis(long timeout){
        forceFreeTimeoutMillis = timeout;
    }

    // SemaphoreJavaDoc
    public int getMaxWaitCount(){
        return maxWaitCount;
    }

    // SemaphoreJavaDoc
    public long getTimeoutMillis(){
        return timeoutMillis;
    }

    // SemaphoreJavaDoc
    public void setMaxWaitCount(int count){
        maxWaitCount = count;
    }

    // SemaphoreJavaDoc
    public void setTimeoutMillis(long timeout){
        timeoutMillis = timeout;
    }
    
    // SemaphoreJavaDoc
    public boolean getResource(){
        return this.getResource(timeoutMillis) ;
    }
    
    // SemaphoreJavaDoc
    public boolean getResource(int maxWaitCount){
        return getResource(timeoutMillis, maxWaitCount) ;
    }
    
    // SemaphoreJavaDoc
    public boolean getResource(long timeOutMiliSecond){
        return getResource(timeOutMiliSecond, maxWaitCount);
    }
    
    // SemaphoreJavaDoc
    public boolean getResource(long timeOutMiliSecond, int maxWaitCount){
        return getResource(timeOutMiliSecond, maxWaitCount, forceFreeTimeoutMillis);
    }
    
    // SemaphoreJavaDoc
    @SuppressWarnings("unchecked")
    public boolean getResource(
        long timeOutMiliSecond,
        int maxWaitCount,
        long forceFreeMiliSecond
    ){
        final Thread current = Thread.currentThread();
        if((resourceCnt <= 0 && maxWaitCount > 0 && getMonitor.getWaitCount() > maxWaitCount)
            || forceEndFlg){
            return false;
        }
        getMonitor.initMonitor();
        long timeOutMs = -1 ;
        if(timeOutMiliSecond >= 0){
            timeOutMs = timeOutMiliSecond;
        }
        long processTime = 0;
        try{
            // IłȂꍇ
            while(!forceEndFlg){
                // \[X]Ăꍇ
                if(resourceCnt > 0){
                    boolean isGet = false;
                    synchronized(getMonitor){
                        if(resourceCnt > 0){
                            // \[Xl
                            resourceCnt--;
                            getMonitor.releaseMonitor();
                            final int nowUsed = initialResource - resourceCnt;
                            if(nowUsed > maxUsedResource){
                                maxUsedResource = nowUsed;
                            }
                            isGet = true;
                        }
                    }
                    if(isGet){
                        if(isThreadBinding){
                            // JԂw肳Ăꍇ
                            TimerTask task = null;
                            if(forceFreeMiliSecond > 0){
                                
                                // J^XN^C}[ɓo^
                                task = new ForceFreeTimerTask(current);
                                forceFreeTimer.schedule(task, forceFreeMiliSecond);
                            }
                            
                            // \[XgpXbhɓo^
                            usedThreads.add(current);
                            
                            // ^XNǗɃ^XNo^
                            if(threadTasks.containsKey(current)){
                                final Object tasks = threadTasks.get(current);
                                List<Object> taskList = null;
                                if(tasks instanceof List){
                                    taskList = (List<Object>)tasks;
                                }else{
                                    taskList = new ArrayList<Object>();
                                    taskList.add(tasks);
                                    threadTasks.put(current, taskList);
                                }
                                taskList.add(task);
                            }else{
                                threadTasks.put(current, task);
                            }
                        }
                        if(resourceCnt > 0 && getMonitor.isWait()){
                            getMonitor.notifyMonitor();
                        }
                        return true;
                    }
                }else{
                    final int nowWaited = getMonitor.getWaitCount();
                    if(nowWaited > maxWaitedCount){
                        maxWaitedCount = nowWaited;
                    }
                }
                long proc = 0;
                synchronized(current){
                    // \[X]ĂȂꍇ
                    // ܂́ÃXbhOɑ҂ĂXbhꍇ
                    
                    // I܂̓^CAEg̏ꍇ
                    if(forceEndFlg || (timeOutMs >= 0 && timeOutMs <= processTime)){
                        break;
                    }
                    
                    // ^CAEgw肪ꍇ́A^CAEg܂sleep
                    // ^CAEgw肪Ȃꍇ́AsleepTimesleepĂ݂
                    if(timeOutMs >= 0){
                        proc = System.currentTimeMillis();
                    }
                }
                int priority = current.getPriority();
                try{
                    if(resourceCnt <= 0){
                        if(priority < Thread.MAX_PRIORITY){
                            try{
                                current.setPriority(priority + 1);
                            }catch(SecurityException e){
                            }
                        }
                        long curSleepTime = timeOutMs >= 0 ? timeOutMs - processTime : sleepTime;
                        if(curSleepTime > 0 && resourceCnt <= 0){
                            getMonitor.waitMonitor(curSleepTime);
                        }
                    }
                }catch(InterruptedException e){
                    if(!getMonitor.isNotify()){
                        return false;
                    }
                }finally{
                    try{
                        current.setPriority(priority);
                    }catch(SecurityException e){
                    }
                }
                if(timeOutMs >= 0){
                    proc = System.currentTimeMillis() - proc;
                    processTime += proc;
                }
            }
            
            // I܂̓^CAEg̏ꍇ
            return false;
        }finally{
            getMonitor.releaseMonitor();
        }
    }
    
    // SemaphoreJavaDoc
    public void freeResource(){
        freeResource(Thread.currentThread());
    }
    
    @SuppressWarnings("unchecked")
    protected void freeResource(Thread usedThread){
        boolean isUsed = false;
        if(isThreadBinding){
            synchronized(usedThread){
                if(usedThreads.contains(usedThread)){
                    isUsed = true;
                    final Object tasks = threadTasks.get(usedThread);
                    if(tasks instanceof List){
                        final List<TimerTask> taskList = (List<TimerTask>)tasks;
                        final TimerTask task = (TimerTask)taskList.remove(0);
                        if(task != null){
                            task.cancel();
                        }
                        if(taskList.size() == 0){
                            threadTasks.remove(usedThread);
                            usedThreads.remove(usedThread);
                        }
                    }else{
                        final TimerTask task = (TimerTask)tasks;
                        if(task != null){
                            task.cancel();
                        }
                        threadTasks.remove(usedThread);
                        usedThreads.remove(usedThread);
                    }
                }
            }
        }
        synchronized(getMonitor){
            if((isThreadBinding && isUsed || !isThreadBinding) && resourceCnt < initialResource){
                if(resourceCnt < initialResource){
                    resourceCnt++;
                }
            }
        }
        if(forceEndFlg || resourceCnt > 0){
            getMonitor.notifyMonitor();
        }
    }
    
    // SemaphoreJavaDoc
    public int getResourceCapacity(){
        return initialResource;
    }
    
    // SemaphoreJavaDoc
    public void setResourceCapacity(int capa) {
        if(forceEndFlg){
            initialResource = capa ;
            resourceCnt = capa ;
        }
    }
    
    // SemaphoreJavaDoc
    public int getResourceRemain() {
        return resourceCnt;
    }
    
    // SemaphoreJavaDoc
    public int getWaitingCount(){
        return getMonitor.getWaitCount();
    }
    
    /**
     * Z}tHɑ΂Ė擾҂Xbhsleep鎞Ԃݒ肷B<p>
     * Z}tH҂̐擪łȂꍇ́AĂsleepB<br>
     * ftHǵA10bB
     *
     * @param millis Z}tHɑ΂Ė擾҂Xbhsleep鎞[ms]
     */
    public void setSleepTime(long millis){
        sleepTime = millis;
    }
    
    // SemaphoreJavaDoc
    public long getSleepTime(){
        return sleepTime;
    }
    
    // SemaphoreJavaDoc
    public void setCheckInterval(long millis){
        checkInterval = millis;
    }
    
    // SemaphoreJavaDoc
    public long getCheckInterval(){
        return checkInterval;
    }
    
    // Semaphore JavaDoc
    public void setThreadBinding(boolean isBinding){
        isThreadBinding = isBinding;
    }
    
    // Semaphore JavaDoc
    public boolean isThreadBinding(){
        return isThreadBinding;
    }
    
    public void startService() throws Exception{
        accept();
    }
    
    public void stopService() throws Exception{
        release();
    }
    
    // SemaphoreJavaDoc
    public synchronized void release(){
        if(forceEndFlg){
            return;
        }
        if(checker != null){
            checker.isStop = true;
            checker = null;
        }
        forceEndFlg = true;
        while(getMonitor.isWait()){
            getMonitor.notifyMonitor();
        }
        while(usedThreads.size() != 0){
            Object[] threads = usedThreads.toArray();
            for(int i = 0; i < threads.length; i++){
                freeResource((Thread)threads[i]);
            }
        }
        resourceCnt = initialResource;
        forceFreeTimer.cancel();
    }
    
    // SemaphoreJavaDoc
    public synchronized void accept(){
        if(!forceEndFlg){
            return;
        }
        forceEndFlg = false;
        if(checkInterval > 0){
            checker = new ResourceChecker();
            checker.start();
        }
        forceFreeTimer = new Timer("Semaphore ForceFreeTimerThread of " + getServiceNameObject(), true);
    }
    
    // SemaphoreJavaDoc
    public int getMaxUsedResource(){
        return maxUsedResource;
    }
    
    // SemaphoreJavaDoc
    public int getMaxWaitedCount(){
        return maxWaitedCount;
    }
    
    /**
     * fVACYsB<p>
     *
     * @param in fVACY̌ƂȂXg[
     * @exception IOException ǂݍ݂Ɏsꍇ
     * @exception ClassNotFoundException fVACY悤ƂIuWFNg̃NXȂꍇ
     */
    private void readObject(java.io.ObjectInputStream in)
     throws java.io.IOException, ClassNotFoundException{
        in.defaultReadObject();
        getMonitor = new WaitSynchronizeMonitor();
        usedThreads = Collections.synchronizedSet(new HashSet<Thread>());
        threadTasks = Collections.synchronizedMap(new HashMap<Thread, Object>());
        forceFreeTimer = new Timer("Semaphore ForceFreeTimerThread of " + getServiceNameObject(), true);
    }
    
    protected class ForceFreeTimerTask extends TimerTask{
        
        protected Thread usedThread;
        
        public ForceFreeTimerTask(Thread usedThread){
            this.usedThread = usedThread;
        }
        
        public void run(){
            freeResource(usedThread);
        }
    }
    
    protected class ResourceChecker extends Thread{
        
        public boolean isStop;
        
        public ResourceChecker(){
            super("Nimbus SemaphoreResourceCheckDaemon");
            setDaemon(true);
        }
        
        public void run(){
            while(!isStop && checkInterval > 0){
                try{
                    Thread.sleep(checkInterval);
                }catch(InterruptedException e){}
                if(resourceCnt > 0){
                    getMonitor.notifyMonitor();
                }
            }
        }
    }
}
