/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import java.io.*;
import java.util.*;
import org.w3c.dom.*;

/**
 * T[rX`&lt;manager&gt;vf^f[^B<p>
 * T[rX`t@C&lt;manager&gt;vfɋLqꂽei[郁^f[^ReiłB<p>
 *
 * @author M.Takata
 * @see <a href="nimbus-service_2_0.xsd">T[rX`t@CXL[}`</a>
 */
public class ManagerMetaData extends MetaData implements Serializable{
    
    private static final long serialVersionUID = 710269838167389543L;
    
    /**
     * &lt;manager&gt;vf̗vfB<p>
     */
    public static final String MANAGER_TAG_NAME = "manager";
    
    /**
     * &lt;manager&gt;vf̎qvf&lt;repository&gt;vf̗vfB<p>
     */
    private static final String REPOSITORY_TAG_NAME = "repository";
    
    /**
     * &lt;manager&gt;vf̎qvf&lt;log&gt;vf̗vfB<p>
     */
    private static final String LOG_TAG_NAME = "log";
    
    /**
     * &lt;manager&gt;vf̎qvf&lt;message&gt;vf̗vfB<p>
     */
    private static final String MESSAGE_TAG_NAME = "message";
    
    /**
     * &lt;manager&gt;vfnamȇB<p>
     */
    private static final String NAME_ATTRIBUTE_NAME = "name";
    
    /**
     * &lt;manager&gt;vfcodȇB<p>
     */
    private static final String CODE_ATTRIBUTE_NAME = "code";
    
    /**
     * &lt;manager&gt;vfshutdown-hook̑B<p>
     */
    private static final String SHUTDOWN_HOOK_ATTRIBUTE_NAME = "shutdown-hook";
    
    /**
     * nameB<p>
     *
     * @see #getName()
     */
    private String name;
    
    /**
     * codeB<p>
     *
     * @see #getCode()
     */
    private String code;
    
    /**
     * shutdown-hookB<p>
     *
     * @see #isExistShutdownHook()
     */
    private boolean isExistShutdownHook;
    
    /**
     * &lt;manager&gt;vf̎qvfƂĒ`ꂽ&lt;service&gt;vf̃^f[^i[}bvB<p>
     *
     * @see #getService(String)
     * @see #getServices()
     * @see #addService(ServiceMetaData)
     */
    private final Map<String, ServiceMetaData> services = new LinkedHashMap<String, ServiceMetaData>();
    
    /**
     * &lt;repository&gt;vf̃^f[^B<p>
     *
     * @see #getRepository()
     */
    private ServiceNameMetaData repository;
    
    /**
     * &lt;log&gt;vfŎw肳ꂽLogger̃^f[^B<p>
     *
     * @see #getLog()
     */
    private ServiceNameMetaData log;
    
    /**
     * &lt;message&gt;vfŎw肳ꂽMessageRecordFactorỹ^f[^B<p>
     *
     * @see #getMessage()
     */
    private ServiceNameMetaData message;
    
    /**
     * [hServiceLoader
     */
    private ServiceLoader myLoader;
    
    /**
     * &lt;manager-property&gt;vfŎw肳ꂽvpeBB<p>
     */
    private Map<String, ManagerPropertyMetaData> properties = new LinkedHashMap<String, ManagerPropertyMetaData>();
    
    private List<IfDefMetaData> ifDefMetaDataList;
    
    /**
     * ̃CX^X𐶐B<p>
     */
    public ManagerMetaData(){
    }
    
    /**
     * evf̃^f[^CX^X𐶐B<p>
     * ManagerMetaData̐evf́A&lt;nimbus&gt;vf\ServerMetaDatałB<br>
     * 
     * @param loader [hServiceLoader
     * @param parent evf̃^f[^
     * @see NimbusMetaData
     */
    public ManagerMetaData(ServiceLoader loader, MetaData parent){
        super(parent);
        myLoader = loader;
    }
    
    /**
     * [h{@link ServiceLoader}擾B<p>
     *
     * @return [hServiceLoader 
     */
    public ServiceLoader getServiceLoader(){
        return myLoader;
    }
    
    /**
     * [h{@link ServiceLoader}ݒ肷B<p>
     *
     * @param loader [hServiceLoader 
     */
    public void setServiceLoader(ServiceLoader loader){
        myLoader = loader;
    }
    
    /**
     * &lt;manager&gt;vfname̒l擾B<p>
     * nameȗĂꍇ́A{@link ServiceManager#DEFAULT_NAME}ԂB<br>
     * 
     * @return name̒l
     */
    public String getName(){
        if(name == null){
            return ServiceManager.DEFAULT_NAME;
        }
        return name;
    }
    
    /**
     * &lt;manager&gt;vfname̒lݒ肷B<p>
     * 
     * @param name name̒l
     */
    public void setName(String name){
        this.name = name;
    }
    
    /**
     * &lt;manager&gt;vfcode̒l擾B<p>
     * 
     * @return code̒l
     */
    public String getCode(){
        return code;
    }
    
    /**
     * &lt;manager&gt;vfcode̒lݒ肷B<p>
     * 
     * @param code code̒l
     */
    public void setCode(String code){
        this.code = code;
    }
    
    /**
     * &lt;manager&gt;vfshutdown-hook̒l擾B<p>
     * truȅꍇ́A&lt;manager&gt;vf\{@link ServiceManager}̔psShutdownHook𐶐Đݒ肷Bfalsȅꍇ́AShutdownHook𐶐ȂB<br>
     * shutdown-hookȗĂꍇ́AfalseԂB<br>
     * 
     * @return shutdown-hook̒l
     */
    public boolean isExistShutdownHook(){
        return isExistShutdownHook;
    }
    
    /**
     * &lt;manager&gt;vfshutdown-hook̒lݒ肷B<p>
     * 
     * @param flg shutdown-hook̒l
     */
    public void setExistShutdownHook(boolean flg){
        isExistShutdownHook = flg;
    }
    
    /**
     * &lt;manager&gt;vf̎qvf&lt;repository&gt;vfɎw肳ꂽ|Wg̃^f[^擾B<p>
     * &lt;repository&gt;vfw肳ĂȂꍇ́AnullԂB̏ꍇARepository̓T[rXƂċNꂸAMap{@link jp.ossc.nimbus.service.repository.Repository Repository}gpB<br>
     *
     * @return &lt;repository&gt;vfɎw肳ꂽ|Wg̃^f[^
     * @see jp.ossc.nimbus.service.repository.Repository Repository
     */
    public ServiceNameMetaData getRepository(){
        return repository;
    }
    
    /**
     * &lt;manager&gt;vf̎qvf&lt;repository&gt;vfɎw肳ꂽ|Wg̃^f[^ݒ肷B<p>
     *
     * @param data &lt;repository&gt;vfɎw肳ꂽ|Wg̃^f[^
     */
    public void setRepository(ServiceNameMetaData data){
        repository = data;
    }
    
    /**
     * &lt;log&gt;vfŎw肳ꂽLogger̃^f[^擾B<p>
     * &lt;log&gt;vfw肳ĂȂꍇ́AnullԂB<br>
     *
     * @return &lt;log&gt;vfŎw肳ꂽLogger̃^f[^
     */
    public ServiceNameMetaData getLog(){
        return log;
    }
    
    /**
     * &lt;log&gt;vfŎw肳ꂽLogger̃^f[^ݒ肷B<p>
     *
     * @param data &lt;log&gt;vfŎw肳ꂽLogger̃^f[^
     */
    public void setLog(ServiceNameMetaData data){
        log = data;
    }
    
    /**
     * &lt;message&gt;vfŎw肳ꂽMessageRecordFactorỹ^f[^擾B<p>
     * &lt;message&gt;vfw肳ĂȂꍇ́AnullԂB<br>
     *
     * @return &lt;log&gt;vfŎw肳ꂽLogger̃^f[^
     */
    public ServiceNameMetaData getMessage(){
        return message;
    }
    
    /**
     * &lt;message&gt;vfŎw肳ꂽMessageRecordFactorỹ^f[^ݒ肷B<p>
     *
     * @param data &lt;log&gt;vfŎw肳ꂽLogger̃^f[^
     */
    public void setMessage(ServiceNameMetaData data){
        message = data;
    }
    
    /**
     * o^ꂽ&lt;service&gt;vf̃^f[^̖OW擾B<p>
     *
     * @return o^ꂽ&lt;service&gt;vf̃^f[^̖ȌW
     */
    public Set<String> getServiceNameSet(){
        return services.keySet();
    }
    
    /**
     * &lt;manager&gt;vf̎qvfAw肳ꂽT[rX&lt;service&gt;vf̃^f[^擾B<p>
     * w肳ꂽT[rX&lt;service&gt;vfA&lt;manager&gt;vf̎qvfɒ`ĂȂꍇ́AnullԂB<br>
     *
     * @param name T[rX
     * @return &lt;service&gt;vf̃^f[^
     */
    public ServiceMetaData getService(String name){
        return services.get(name);
    }
    
    /**
     * &lt;manager&gt;vf̎qvfƂĒ`Ă&lt;service&gt;vf̃}bsO擾B<p>
     * &lt;manager&gt;vf̎qvf&lt;service&gt;vfP`ĂȂꍇ́AMapCX^XԂB<br>
     * ܂A擾MapCX^X́AManagerMetaDataCX^X̓ɕێMap̎QƂłB̂߁AMapɑ΂ύX́ÃCX^XɉeyڂB<br>
     * 
     * @return &lt;service&gt;vf̃}bsO
     */
    public Map<String, ServiceMetaData> getServices(){
        return services;
    }
    
    /**
     * &lt;manager&gt;vf̎qvfƂĒ`Ă&lt;service&gt;vf̃^f[^o^B<p>
     *
     * @param service &lt;service&gt;vf̃^f[^
     */
    public void addService(ServiceMetaData service){
        if(service == null){
            return;
        }
        service.setServiceLoader(getServiceLoader());
        service.setManager(this);
        ServiceMetaData removed = services.put(service.getName(), service);
        if(removed != null && removed.getManager() == this){
            removed.setManager(null);
        }
    }
    
    /**
     * &lt;manager&gt;vf̎qvfƂĒ`Ă&lt;service&gt;vf̃^f[^폜B<p>
     *
     * @param name &lt;service&gt;vfname̒l
     */
    public void removeService(String name){
        ServiceMetaData removed = services.remove(name);
        if(removed != null && removed.getManager() == this){
            removed.setManager(null);
        }
    }
    
    /**
     * &lt;manager&gt;vf̎qvfƂĒ`Ă&lt;service&gt;vf̃^f[^Sč폜B<p>
     */
    public void clearServices(){
        for(Object name : services.keySet().toArray()){
            removeService((String)name);
        }
    }
    
    /**
     * vpeB̏W擾B<p>
     *
     * @return &lt;manager-property&gt;vfŎw肳ꂽvpeB̏W
     */
    public Set<String> getPropertyNameSet(){
        return properties.keySet();
    }
    
    /**
     * w肳ꂽvpeB&lt;manager-property&gt;vfŎw肳ꂽvpeBl擾B<p>
     * YvpeB&lt;manager-property&gt;vfw肳ĂȂꍇ́AnullԂB<br>
     *
     * @param property vpeB
     * @return &lt;manager-property&gt;vfŎw肳ꂽvpeBl
     */
    public String getProperty(String property){
        ManagerPropertyMetaData propData = properties.get(property);
        return propData == null ? null : propData.getValue();
    }
    
    /**
     * &lt;manager-property&gt;vfŎw肳ꂽvpeB擾B<p>
     * &lt;manager-property&gt;vfw肳ĂȂꍇ́APropertiesԂB<br>
     *
     * @return &lt;manager-property&gt;vfŎw肳ꂽvpeB
     */
    public Map<String, String> getProperties(){
        final Map<String, String> props = new HashMap<String, String>();
        for(ManagerPropertyMetaData propData : properties.values()){
            props.put(propData.getName(), propData.getValue());
        }
        return props;
    }
    
    /**
     * w肳ꂽvpeB&lt;manager-property&gt;vfŎw肳ꂽvpeBlݒ肷B<p>
     *
     * @param property vpeB
     * @param value &lt;manager-property&gt;vfŎw肳ꂽvpeBl
     */
    public void setProperty(String property, String value){
        ManagerPropertyMetaData propData = properties.get(property);
        if(propData == null){
            propData = new ManagerPropertyMetaData(this);
        }
        propData.setName(property);
        propData.setValue(value);
        properties.put(property, propData);
    }
    
    /**
     * w肳ꂽvpeB&lt;manager-property&gt;vfŎw肳ꂽvpeB폜B<p>
     *
     * @param property vpeB
     */
    public void removeProperty(String property){
        properties.remove(property);
    }
    
    /**
     * &lt;manager-property&gt;vfŎw肳ꂽSẴvpeB폜B<p>
     */
    public void clearProperties(){
        properties.clear();
    }
    
    /**
     * &lt;repository&gt;vf𐶐B<p>
     *
     * @param managerName |WgT[rXo^Ă}l[W
     * @param serviceName |WgT[rX̃T[rX
     * @return &lt;repository&gt;vf̃^f[^
     */
    public ServiceNameMetaData createRepositoryMetaData(
        String managerName,
        String serviceName
    ){
        return new ServiceNameMetaData(
            this,
            REPOSITORY_TAG_NAME,
            managerName,
            serviceName
        );
    }
    
    /**
     * &lt;log&gt;vf𐶐B<p>
     *
     * @param managerName OT[rXo^Ă}l[W
     * @param serviceName OT[rX̃T[rX
     * @return &lt;log&gt;vf̃^f[^
     */
    public ServiceNameMetaData createLogMetaData(
        String managerName,
        String serviceName
    ){
        return new ServiceNameMetaData(
            this,
            LOG_TAG_NAME,
            managerName,
            serviceName
        );
    }
    
    /**
     * &lt;message&gt;vf𐶐B<p>
     *
     * @param managerName bZ[WT[rXo^Ă}l[W
     * @param serviceName bZ[WT[rX̃T[rX
     * @return &lt;message&gt;vf̃^f[^
     */
    public ServiceNameMetaData createMessageMetaData(
        String managerName,
        String serviceName
    ){
        return new ServiceNameMetaData(
            this,
            MESSAGE_TAG_NAME,
            managerName,
            serviceName
        );
    }
    
    /**
     * &lt;manager&gt;vfElementp[XāAg̏Ayюqvf̃^f[^̐sB<p>
     *
     * @param element &lt;manager&gt;vfElement
     * @exception DeploymentException &lt;manager&gt;vf̉́Ǎʂɂ郁^f[^̐Ɏsꍇ
     */
    @Override
    public void importXML(Element element) throws DeploymentException{
        super.importXML(element);
        
        if(!element.getTagName().equals(MANAGER_TAG_NAME)){
            throw new DeploymentException(
                "Tag must be " + MANAGER_TAG_NAME + " : "
                 + element.getTagName()
            );
        }
        
        name = getOptionalAttribute(element, NAME_ATTRIBUTE_NAME);
        final String shutdownHook
             = getOptionalAttribute(element, SHUTDOWN_HOOK_ATTRIBUTE_NAME);
        if(shutdownHook != null && shutdownHook.length() != 0){
            isExistShutdownHook = Boolean.valueOf(shutdownHook).booleanValue();
        }
        
        code = getOptionalAttribute(element, CODE_ATTRIBUTE_NAME);
        
        importXMLInner(element, null);
        
        final Iterator<Element> ifDefElements = getChildrenByTagName(
            element,
            IfDefMetaData.IFDEF_TAG_NAME
        );
        while(ifDefElements.hasNext()){
            if(ifDefMetaDataList == null){
                ifDefMetaDataList = new ArrayList<IfDefMetaData>();
            }
            final IfDefMetaData ifdefData
                 = new IfDefMetaData(this);
            ifdefData.importXML((Element)ifDefElements.next());
            ifDefMetaDataList.add(ifdefData);
        }
        
        if(ifDefMetaDataList == null || ifDefMetaDataList.size() == 0){
            return;
        }
        
        for(int i = 0, imax = ifDefMetaDataList.size(); i < imax; i++){
            IfDefMetaData ifdefData = ifDefMetaDataList.get(i);
            Element ifDefElement = ifdefData.getElement();
            if(ifDefElement == null){
                continue;
            }
            
            importXMLInner(ifDefElement, ifdefData);
            
            ifdefData.setElement(null);
        }
    }
    
    protected void importXMLInner(Element element, IfDefMetaData ifdefData) throws DeploymentException{
        
        final boolean ifdefMatch
            = ifdefData == null ? true : ifdefData.isMatch();
        
        final Iterator<Element> propElements = getChildrenByTagName(
            element,
            ManagerPropertyMetaData.MANAGER_PROPERTY_TAG_NAME
        );
        while(propElements.hasNext()){
            ManagerPropertyMetaData propData
                = new ManagerPropertyMetaData(this);
            if(ifdefData != null){
                propData.setIfDefMetaData(ifdefData);
                ifdefData.addChild(propData);
            }
            propData.importXML((Element)propElements.next());
            if(ifdefMatch){
                properties.put(
                    propData.getName(),
                    propData
                );
                String prop = propData.getValue();
                // VXevpeB̒u
                prop = Utility.replaceSystemProperty(prop, false);
                // T[rX[_\vpeB̒u
                prop = Utility.replaceServiceLoderConfig(
                    myLoader == null ? null : myLoader.getConfig(),
                    prop,
                    false
                );
                // T[ovpeB̒u
                prop = Utility.replaceServerProperty(prop, true);
                propData.setValue(prop);
            }
        }
        
        final Element repositoryElement
             = getOptionalChild(element, REPOSITORY_TAG_NAME);
        if(repositoryElement != null){
            if(ifdefMatch && repository != null){
                throw new DeploymentException("Element of " + REPOSITORY_TAG_NAME + " is duplicated.");
            }
            ServiceNameMetaData tmp = new ServiceNameMetaData(this, getName());
            if(ifdefData != null){
                tmp.setIfDefMetaData(ifdefData);
                ifdefData.addChild(tmp);
            }
            tmp.importXML(repositoryElement);
            if(ifdefMatch){
                repository = tmp;
            }
        }
        
        final Element logElement
             = getOptionalChild(element, LOG_TAG_NAME);
        if(logElement != null){
            if(ifdefMatch && log != null){
                throw new DeploymentException("Element of " + LOG_TAG_NAME + " is duplicated.");
            }
            ServiceNameMetaData tmp = new ServiceNameMetaData(this, getName());
            if(ifdefData != null){
                tmp.setIfDefMetaData(ifdefData);
                ifdefData.addChild(tmp);
            }
            tmp.importXML(logElement);
            if(ifdefMatch){
                log = tmp;
            }
        }
        
        final Element messageElement = getOptionalChild(
            element,
            MESSAGE_TAG_NAME
        );
        if(messageElement != null){
            if(ifdefMatch && message != null){
                throw new DeploymentException("Element of " + MESSAGE_TAG_NAME + " is duplicated.");
            }
            ServiceNameMetaData tmp = new ServiceNameMetaData(this, getName());
            if(ifdefData != null){
                tmp.setIfDefMetaData(ifdefData);
                ifdefData.addChild(tmp);
            }
            tmp.importXML(messageElement);
            if(ifdefMatch){
                message = tmp;
            }
        }
        
        final Iterator<Element> serviceElements = getChildrenByTagName(
            element,
            ServiceMetaData.SERVICE_TAG_NAME
        );
        while(serviceElements.hasNext()){
            final ServiceMetaData serviceData
                 = new ServiceMetaData(myLoader, this, this);
            if(ifdefData != null){
                serviceData.setIfDefMetaData(ifdefData);
                ifdefData.addChild(serviceData);
            }
            serviceData.importXML((Element)serviceElements.next());
            if(ifdefMatch && services.containsKey(serviceData.getName())){
                throw new DeploymentException(
                    "Dupulicated service name : " + serviceData.getName()
                );
            }
            if(ifdefMatch){
                addService(serviceData);
                addDependsService(serviceData);
            }
        }
    }
    
    protected void addDependsService(ServiceMetaData serviceData) throws DeploymentException{
        List<ServiceMetaData.DependsMetaData> dependsList = serviceData.getDepends();
        for(int i = 0; i < dependsList.size(); i++){
            ServiceMetaData.DependsMetaData dependsData = dependsList.get(i);
            ServiceMetaData dependsServiceData = dependsData.getService();
            if(dependsServiceData != null){
                if(services.containsKey(dependsServiceData.getName())){
                    throw new DeploymentException(
                        "Dupulicated service name : " + dependsServiceData.getName()
                    );
                }
                addService(dependsServiceData);
                addDependsService(dependsServiceData);
            }
        }
    }
    
    @Override
    public StringBuilder toXML(StringBuilder buf){
        appendComment(buf);
        buf.append('<').append(MANAGER_TAG_NAME);
        if(name != null){
            buf.append(' ').append(NAME_ATTRIBUTE_NAME)
                .append("=\"").append(name).append("\"");
        }
        if(code != null){
            buf.append(' ').append(CODE_ATTRIBUTE_NAME)
                .append("=\"").append(code).append("\"");
        }
        if(isExistShutdownHook){
            buf.append(' ').append(SHUTDOWN_HOOK_ATTRIBUTE_NAME)
                .append("=\"").append(isExistShutdownHook).append("\"");
        }
        buf.append('>');
        if(properties.size() != 0){
            buf.append(LINE_SEPARATOR);
            final StringBuilder subBuf = new StringBuilder();
            Iterator<ManagerPropertyMetaData> props = properties.values().iterator();
            while(props.hasNext()){
                ManagerPropertyMetaData propData = props.next();
                if(propData.getIfDefMetaData() != null){
                    continue;
                }
                propData.toXML(subBuf);
                if(props.hasNext()){
                    subBuf.append(LINE_SEPARATOR);
                }
            }
            buf.append(addIndent(subBuf));
        }
        if(repository != null && repository.getIfDefMetaData() == null){
            buf.append(LINE_SEPARATOR);
            buf.append(
                addIndent(repository.toXML(new StringBuilder()))
            );
        }
        if(log != null && log.getIfDefMetaData() == null){
            buf.append(LINE_SEPARATOR);
            buf.append(
                addIndent(log.toXML(new StringBuilder()))
            );
        }
        if(message != null && message.getIfDefMetaData() == null){
            buf.append(LINE_SEPARATOR);
            buf.append(
                addIndent(message.toXML(new StringBuilder()))
            );
        }
        if(services.size() != 0){
            buf.append(LINE_SEPARATOR);
            Iterator<ServiceMetaData> datas = services.values().iterator();
            while(datas.hasNext()){
                ServiceMetaData data = datas.next();
                if(data.getIfDefMetaData() != null){
                    continue;
                }
                buf.append(
                    addIndent(data.toXML(new StringBuilder()))
                );
                if(datas.hasNext()){
                    buf.append(LINE_SEPARATOR);
                }
            }
        }
        buf.append(LINE_SEPARATOR);
        if(ifDefMetaDataList != null && ifDefMetaDataList.size() != 0){
            for(int i = 0, imax = ifDefMetaDataList.size(); i < imax; i++){
                IfDefMetaData ifdefData = ifDefMetaDataList.get(i);
                buf.append(
                    addIndent(ifdefData.toXML(new StringBuilder()))
                );
                buf.append(LINE_SEPARATOR);
            }
        }
        buf.append("</").append(MANAGER_TAG_NAME).append('>');
        return buf;
    }
    
    /**
     * ̃CX^X̕\擾B<p>
     *
     * @return \
     */
    @Override
    public String toString(){
        final StringBuilder buf = new StringBuilder();
        buf.append(super.toString());
        buf.append('{');
        if(getName() != null){
            buf.append(NAME_ATTRIBUTE_NAME);
            buf.append('=');
            buf.append(getName());
            buf.append(',');
        }
        buf.append(SHUTDOWN_HOOK_ATTRIBUTE_NAME);
        buf.append('=');
        buf.append(isExistShutdownHook);
        buf.append('}');
        return buf.toString();
    }
}
