/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.beans.dataset;

import java.io.*;
import java.util.*;
import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.xml.sax.*;

import jp.ossc.nimbus.beans.StringArrayEditor;
import jp.ossc.nimbus.core.MetaData;
import jp.ossc.nimbus.core.NimbusEntityResolver;
import jp.ossc.nimbus.core.ServiceManagerFactory;
import jp.ossc.nimbus.core.DeploymentException;

/**
 * f[^Zbg\[XR[hB<p>
 * f[^ZbgAR[hXgAR[hȂǂ̔ėpBeanpRN[gBeañ\[X𐶐B<br>
 *
 * @author M.Takata
 */
public class DataSetCodeGenerator{
    
    static{
        NimbusEntityResolver.registerEntity(
            "-//Nimbus//DTD Nimbus DataSet generation 2.0//JA",
            "jp/ossc/nimbus/beans/dataset/dataset_2_0.dtd"
        );
    }
    
    private static final String USAGE_RESOURCE
         = "jp/ossc/nimbus/beans/dataset/DataSetCodeGeneratorUsage.txt";
    
    private DataSetsMetaData dataSetsData;
    private boolean isValidate = true;
    private String encoding;
    
    private DataSetCodeGenerator(){
    }
    
    public File[] generate(File definition, File outDir){
        Set<File> files = new LinkedHashSet<File>();
        try{
            final InputSource inputSource = new InputSource(new FileInputStream(definition));
            final DocumentBuilderFactory domFactory
                 = DocumentBuilderFactory.newInstance();
            domFactory.setValidating(isValidate);
            final DocumentBuilder builder = domFactory.newDocumentBuilder();
            final NimbusEntityResolver resolver = new NimbusEntityResolver();
            builder.setEntityResolver(resolver);
            final MyErrorHandler handler = new MyErrorHandler();
            builder.setErrorHandler(handler);
            final Document doc = builder.parse(inputSource);
            if(handler.isError()){
                ServiceManagerFactory.getLogger().write("DSCG_00004", definition);
                return new File[0];
            }
            dataSetsData = new DataSetsMetaData();
            dataSetsData.importXML(doc.getDocumentElement());
            dataSetsData.generate(outDir, files);
            return files.toArray(new File[files.size()]);
        }catch(Exception e){
            ServiceManagerFactory.getLogger().write("DSCG_00004", e, definition);
            return files.toArray(new File[files.size()]);
        }
    }
    
    private class MyErrorHandler implements ErrorHandler{
        
        private boolean isError;
        
        public void warning(SAXParseException e) throws SAXException{
            ServiceManagerFactory.getLogger().write("DSCG_00001", e.getMessage());
        }
        public void error(SAXParseException e) throws SAXException{
            isError = true;
            ServiceManagerFactory.getLogger().write("DSCG_00002", e.getMessage());
        }
        public void fatalError(SAXParseException e) throws SAXException{
            isError = true;
            ServiceManagerFactory.getLogger().write("DSCG_00003", e.getMessage());
        }
        public boolean isError(){
            return isError;
        }
    }
    
    private class DataSetsMetaData extends MetaData{
        
        private static final long serialVersionUID = 3876649968668319651L;
        
        public static final String TAG_NAME = "dataSets";
        
        public Map<String, PropertyMetaData> properties;
        public List<RecordMetaData> records;
        public List<HeaderMetaData> headers;
        public List<RecordListMetaData> recordLists;
        public List<DataSetMetaData> dataSets;
        
        public DataSetsMetaData(){
            super();
        }
        
        public void importXML(Element element) throws DeploymentException{
            super.importXML(element);
            
            if(!element.getTagName().equals(TAG_NAME)){
                throw new DeploymentException(
                    "Root tag must be " + TAG_NAME + " : " + element.getTagName()
                );
            }
            
            final Iterator<Element> propertyElements = getChildrenByTagName(
                element,
                PropertyMetaData.TAG_NAME
            );
            while(propertyElements.hasNext()){
                if(properties == null){
                    properties = new LinkedHashMap<String, PropertyMetaData>();
                }
                final PropertyMetaData propertyData = new PropertyMetaData(DataSetsMetaData.this);
                propertyData.importXML(propertyElements.next());
                if(properties.containsKey(propertyData.getName())){
                    throw new DeploymentException(
                        "Property is duplicated : " + propertyData.getName()
                    );
                }
                properties.put(
                    propertyData.getName(),
                    propertyData
                );
            }
            
            final Iterator<Element> recordElements = getChildrenByTagName(
                element,
                RecordMetaData.TAG_NAME
            );
            while(recordElements.hasNext()){
                if(records == null){
                    records = new ArrayList<RecordMetaData>();
                }
                final RecordMetaData recordData = new RecordMetaData(DataSetsMetaData.this);
                recordData.importXML(recordElements.next());
                records.add(recordData);
            }
            
            final Iterator<Element> headerElements = getChildrenByTagName(
                element,
                HeaderMetaData.TAG_NAME
            );
            while(headerElements.hasNext()){
                if(headers == null){
                    headers = new ArrayList<HeaderMetaData>();
                }
                final HeaderMetaData headerData = new HeaderMetaData(DataSetsMetaData.this);
                headerData.importXML(headerElements.next());
                headers.add(headerData);
            }
            
            final Iterator<Element> recordListElements = getChildrenByTagName(
                element,
                RecordListMetaData.TAG_NAME
            );
            while(recordListElements.hasNext()){
                if(recordLists == null){
                    recordLists = new ArrayList<RecordListMetaData>();
                }
                final RecordListMetaData recordListData = new RecordListMetaData(DataSetsMetaData.this);
                recordListData.importXML(recordListElements.next());
                recordLists.add(recordListData);
            }
            
            final Iterator<Element> dataSetElements = getChildrenByTagName(
                element,
                DataSetMetaData.TAG_NAME
            );
            while(dataSetElements.hasNext()){
                if(dataSets == null){
                    dataSets = new ArrayList<DataSetMetaData>();
                }
                final DataSetMetaData dataSetData = new DataSetMetaData(DataSetsMetaData.this);
                dataSetData.importXML(dataSetElements.next());
                dataSets.add(dataSetData);
            }
        }
        
        private void writeCode(File file, CodeGenerator generator, Set<File> files) throws Exception{
            if(file.getParentFile() != null && !file.getParentFile().exists()){
                file.getParentFile().mkdirs();
            }
            if(files.contains(file)){
                ServiceManagerFactory.getLogger().write("DSCG_00005", file);
            }
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            generator.writeCode(pw);
            pw.flush();
            String source = sw.toString();
            FileOutputStream fos = new FileOutputStream(file);
            try{
                @SuppressWarnings("resource")
                OutputStreamWriter osw = encoding == null ? new OutputStreamWriter(fos) : new OutputStreamWriter(fos, encoding);
                osw.write(source, 0, source.length());
                osw.flush();
            }finally{
                fos.close();
            }
            files.add(file);
        }
        
        public void generate(File outDir, Set<File> files) throws Exception{
            if(records != null){
                for(int i = 0; i < records.size(); i++){
                    RecordMetaData data = records.get(i);
                    File file = data.getFile(outDir);
                    writeCode(file, data, files);
                }
            }
            if(headers != null){
                for(int i = 0; i < headers.size(); i++){
                    HeaderMetaData data = headers.get(i);
                    File file = data.getFile(outDir);
                    writeCode(file, data, files);
                }
            }
            if(recordLists != null){
                for(int i = 0; i < recordLists.size(); i++){
                    RecordListMetaData data = recordLists.get(i);
                    File file = data.getFile(outDir);
                    writeCode(file, data, files);
                }
            }
            if(dataSets != null){
                for(int i = 0; i < dataSets.size(); i++){
                    DataSetMetaData data = dataSets.get(i);
                    File file = data.getFile(outDir);
                    writeCode(file, data, files);
                    if(data.headers != null){
                        Iterator<HeaderMetaData> itr = data.headers.values().iterator();
                        while(itr.hasNext()){
                            HeaderMetaData headerData = itr.next();
                            if(headerData.schema != null){
                                file = headerData.getFile(outDir);
                                writeCode(file, headerData, files);
                            }
                        }
                    }
                    if(data.recordLists != null){
                        Iterator<RecordListMetaData> itr = data.recordLists.values().iterator();
                        while(itr.hasNext()){
                            RecordListMetaData recordListData = itr.next();
                            if(recordListData.recordClassName != null){
                                file = recordListData.getFile(outDir);
                                writeCode(file, recordListData, files);
                            }
                        }
                    }
                }
            }
        }
    }
    
    private class PropertyMetaData extends jp.ossc.nimbus.core.PropertyMetaData{
        
        private static final long serialVersionUID = 7835225760732994370L;
        
        public static final String TAG_NAME = "property";
        
        public PropertyMetaData(MetaData parent){
            super(parent);
        }
        protected String getTagName(){
            return TAG_NAME;
        }
    }
    
    private interface CodeGenerator{
        public File getFile(File dir);
        public void writeCode(PrintWriter pw);
    }
    
    private class RecordMetaData extends MetaData implements CodeGenerator{
        
        private static final long serialVersionUID = 5515496152289298757L;
        
        public static final String TAG_NAME = "record";
        public static final String TAG_NAME_SCHEMA = "schema";
        public static final String ATTRIBUTE_NAME_CODE = "code";
        public static final String ATTRIBUTE_NAME_EXTENDS = "extends";
        public static final String ATTRIBUTE_NAME_TYPE = "type";
        
        public String packageName;
        public String className;
        public String superClass = Record.class.getName();
        public String schema;
        public String schemaType;
        
        public RecordMetaData(MetaData parent){
            super(parent);
        }
        
        public void importXML(Element element) throws DeploymentException{
            super.importXML(element);
            
            if(!element.getTagName().equals(TAG_NAME)){
                throw new DeploymentException(
                    "Root tag must be " + TAG_NAME + " : " + element.getTagName()
                );
            }
            String code = Utility.replaceProperty(dataSetsData, getUniqueAttribute(element, ATTRIBUTE_NAME_CODE));
            className = code;
            if(code.indexOf('.') != -1){
                className = code.substring(code.lastIndexOf('.') + 1);
                packageName = code.substring(0, code.lastIndexOf('.'));
            }
            superClass = Utility.replaceProperty(dataSetsData, getOptionalAttribute(element, ATTRIBUTE_NAME_EXTENDS, superClass));
            
            final Element schemaElement = getUniqueChild(element, TAG_NAME_SCHEMA);
            schema = Utility.replaceProperty(dataSetsData, getElementContent(schemaElement));
            schemaType = Utility.replaceProperty(dataSetsData, getOptionalAttribute(schemaElement, ATTRIBUTE_NAME_TYPE, "set"));
        }
        
        public File getFile(File dir){
            String filePath = null;
            if(packageName == null){
                filePath = className + ".java";
            }else{
                filePath = packageName.replaceAll("\\.", "/") + '/' + className + ".java";
            }
            return new File(dir, filePath);
        }
        
        public void writeCode(PrintWriter pw){
            if(packageName != null){
                pw.println("package " + packageName + ';');
            }
            pw.println();
            pw.println("public class " + className + " extends " + superClass + "{");
            RecordSchema recordSchema = RecordSchema.getInstance(schema);
            for(int i = 0; i < recordSchema.getPropertySize(); i++){
                PropertySchema propSchema = recordSchema.getPropertySchema(i);
                pw.println("    public static final String " + propSchema.getName().toUpperCase() + " = \"" + propSchema.getName() + "\";");
                pw.println("    public static final int " + propSchema.getName().toUpperCase() + "_INDEX = " + i + ";");
            }
            pw.println("    ");
            pw.println("    public " + className + "(){");
            if("set".equals(schemaType)){
                pw.println("        super(\"" + Utility.escapeLineSeparator(schema) + "\");");
            }else if("replace".equals(schemaType)){
                pw.println("        replaceSchema(\"" + Utility.escapeLineSeparator(schema) + "\");");
            }else if("append".equals(schemaType)){
                pw.println("        appendSchema(\"" + Utility.escapeLineSeparator(schema) + "\");");
            }
            pw.println("    }");
            pw.println("    ");
            for(int i = 0, imax = recordSchema.getPropertySize(); i < imax; i++){
                PropertySchema propSchema = recordSchema.getPropertySchema(i);
                pw.println("    public " + propSchema.getType().getName() + " " + Utility.createGetterName(propSchema.getName()) + "(){");
                if(propSchema.getType().isPrimitive()){
                    if(Boolean.TYPE.equals(propSchema.getType())){
                        pw.println("        return getBooleanProperty(" + propSchema.getName().toUpperCase() + ");");
                    }else if(Byte.TYPE.equals(propSchema.getType())){
                        pw.println("        return getByteProperty(" + propSchema.getName().toUpperCase() + ");");
                    }else if(Short.TYPE.equals(propSchema.getType())){
                        pw.println("        return getShortProperty(" + propSchema.getName().toUpperCase() + ");");
                    }else if(Integer.TYPE.equals(propSchema.getType())){
                        pw.println("        return getIntProperty(" + propSchema.getName().toUpperCase() + ");");
                    }else if(Long.TYPE.equals(propSchema.getType())){
                        pw.println("        return getLongProperty(" + propSchema.getName().toUpperCase() + ");");
                    }else if(Float.TYPE.equals(propSchema.getType())){
                        pw.println("        return getFloatProperty(" + propSchema.getName().toUpperCase() + ");");
                    }else if(Double.TYPE.equals(propSchema.getType())){
                        pw.println("        return getDoubleProperty(" + propSchema.getName().toUpperCase() + ");");
                    }else if(Character.TYPE.equals(propSchema.getType())){
                        pw.println("        Character c = ((Character)getProperty(" + propSchema.getName().toUpperCase() + "));");
                        pw.println("        return c == null ? '' : c.charValue();");
                    }
                }else{
                    pw.println("        return (" + propSchema.getType().getName() + ")getProperty(" + propSchema.getName().toUpperCase() + ");");
                }
                pw.println("    }");
                pw.println("    ");
                pw.println("    public void " + Utility.createSetterName(propSchema.getName()) + "(" + propSchema.getType().getName() + " val){");
                pw.println("        setProperty(" + propSchema.getName().toUpperCase() + ", val);");
                pw.println("    }");
                pw.println("    ");
            }
            pw.println("}");
        }
    }
    
    private class HeaderMetaData extends MetaData implements CodeGenerator{
        
        private static final long serialVersionUID = 7980846939747068932L;
        
        public static final String TAG_NAME = "header";
        public static final String TAG_NAME_SCHEMA = "schema";
        public static final String ATTRIBUTE_NAME_CODE = "code";
        public static final String ATTRIBUTE_NAME_EXTENDS = "extends";
        public static final String ATTRIBUTE_NAME_NAME = "name";
        public static final String ATTRIBUTE_NAME_TYPE = "type";
        
        public String name;
        public String packageName;
        public String className;
        public String superClass = Header.class.getName();
        public String schema;
        public String schemaType;
        
        public HeaderMetaData(MetaData parent){
            super(parent);
        }
        
        public void importXML(Element element) throws DeploymentException{
            super.importXML(element);
            
            if(!element.getTagName().equals(TAG_NAME)){
                throw new DeploymentException(
                    "Root tag must be " + TAG_NAME + " : " + element.getTagName()
                );
            }
            String code = Utility.replaceProperty(dataSetsData, getUniqueAttribute(element, ATTRIBUTE_NAME_CODE));
            className = code;
            if(code.indexOf('.') != -1){
                className = code.substring(code.lastIndexOf('.') + 1);
                packageName = code.substring(0, code.lastIndexOf('.'));
            }
            superClass = Utility.replaceProperty(dataSetsData, getOptionalAttribute(element, ATTRIBUTE_NAME_EXTENDS, superClass));
            name = Utility.replaceProperty(dataSetsData, getOptionalAttribute(element, ATTRIBUTE_NAME_NAME));
            
            final Element schemaElement = (getParent() instanceof DataSetsMetaData) ? getUniqueChild(element, TAG_NAME_SCHEMA) : getOptionalChild(element, TAG_NAME_SCHEMA);
            if(schemaElement != null){
                schema = Utility.replaceProperty(dataSetsData, getElementContent(schemaElement));
                schemaType = Utility.replaceProperty(dataSetsData, getOptionalAttribute(schemaElement, ATTRIBUTE_NAME_TYPE, "set"));
            }
        }
        
        public File getFile(File dir){
            String filePath = null;
            if(packageName == null){
                filePath = className + ".java";
            }else{
                filePath = packageName.replaceAll("\\.", "/") + '/' + className + ".java";
            }
            return new File(dir, filePath);
        }
        
        public String getFullClassName(){
            return packageName == null ? className : (packageName + '.' + className);
        }
        
        public void writeCode(PrintWriter pw){
            if(schema == null){
                pw.println("        setHeaderClass(\"" + name + "\", " + getFullClassName() + ".class);");
            }else{
                if(packageName != null){
                    pw.println("package " + packageName + ';');
                }
                pw.println();
                pw.println("public class " + className + " extends " + superClass + "{");
                RecordSchema recordSchema = RecordSchema.getInstance(schema);
                for(int i = 0; i < recordSchema.getPropertySize(); i++){
                    PropertySchema propSchema = recordSchema.getPropertySchema(i);
                    pw.println("    public static final String " + propSchema.getName().toUpperCase() + " = \"" + propSchema.getName() + "\";");
                    pw.println("    public static final int " + propSchema.getName().toUpperCase() + "_INDEX = " + i + ";");
                }
                pw.println("    ");
                pw.println("    public " + className + "(){");
                if("set".equals(schemaType)){
                    pw.println("        super(\"" + name + "\", \"" + Utility.escapeLineSeparator(schema) + "\");");
                }else if("replace".equals(schemaType)){
                    if(name != null){
                        pw.println("        setName(\"" + name + "\");");
                    }
                    pw.println("        replaceSchema(\"" + Utility.escapeLineSeparator(schema) + "\");");
                }else if("append".equals(schemaType)){
                    if(name != null){
                        pw.println("        setName(\"" + name + "\");");
                    }
                    pw.println("        appendSchema(\"" + Utility.escapeLineSeparator(schema) + "\");");
                }
                pw.println("    }");
                pw.println("    ");
                pw.println("    public " + className + "(String name){");
                if("set".equals(schemaType)){
                    pw.println("        super(name, \"" + Utility.escapeLineSeparator(schema) + "\");");
                }else if("replace".equals(schemaType)){
                    pw.println("        setName(name);");
                    pw.println("        replaceSchema(\"" + Utility.escapeLineSeparator(schema) + "\");");
                }else if("append".equals(schemaType)){
                    pw.println("        setName(name);");
                    pw.println("        appendSchema(\"" + Utility.escapeLineSeparator(schema) + "\");");
                }
                pw.println("    }");
                pw.println("    ");
                for(int i = 0, imax = recordSchema.getPropertySize(); i < imax; i++){
                    PropertySchema propSchema = recordSchema.getPropertySchema(i);
                    pw.println("    public " + propSchema.getType().getName() + " " + Utility.createGetterName(propSchema.getName()) + "(){");
                    if(propSchema.getType().isPrimitive()){
                        if(Boolean.TYPE.equals(propSchema.getType())){
                            pw.println("        return getBooleanProperty(" + propSchema.getName().toUpperCase() + ");");
                        }else if(Byte.TYPE.equals(propSchema.getType())){
                            pw.println("        return getByteProperty(" + propSchema.getName().toUpperCase() + ");");
                        }else if(Short.TYPE.equals(propSchema.getType())){
                            pw.println("        return getShortProperty(" + propSchema.getName().toUpperCase() + ");");
                        }else if(Integer.TYPE.equals(propSchema.getType())){
                            pw.println("        return getIntProperty(" + propSchema.getName().toUpperCase() + ");");
                        }else if(Long.TYPE.equals(propSchema.getType())){
                            pw.println("        return getLongProperty(" + propSchema.getName().toUpperCase() + ");");
                        }else if(Float.TYPE.equals(propSchema.getType())){
                            pw.println("        return getFloatProperty(" + propSchema.getName().toUpperCase() + ");");
                        }else if(Double.TYPE.equals(propSchema.getType())){
                            pw.println("        return getDoubleProperty(" + propSchema.getName().toUpperCase() + ");");
                        }else if(Character.TYPE.equals(propSchema.getType())){
                            pw.println("        Character c = ((Character)getProperty(" + propSchema.getName().toUpperCase() + "));");
                            pw.println("        return c == null ? '' : c.charValue();");
                        }
                    }else{
                        pw.println("        return (" + propSchema.getType().getName() + ")getProperty(" + propSchema.getName().toUpperCase() + ");");
                    }
                    pw.println("    }");
                    pw.println("    ");
                    pw.println("    public void " + Utility.createSetterName(propSchema.getName()) + "(" + propSchema.getType().getName() + " val){");
                    pw.println("        setProperty(" + propSchema.getName().toUpperCase() + ", val);");
                    pw.println("    }");
                    pw.println("    ");
                }
                pw.println("}");
            }
        }
    }
    
    private class RecordListMetaData extends MetaData implements CodeGenerator{
        
        private static final long serialVersionUID = 8369293626987943298L;
        
        public static final String TAG_NAME = "recordList";
        public static final String TAG_NAME_STOCK_KEY = "keyIndex";
        public static final String TAG_NAME_CONDITION_INDEX = "conditionIndex";
        public static final String ATTRIBUTE_NAME_CODE = "code";
        public static final String ATTRIBUTE_NAME_EXTENDS = "extends";
        public static final String ATTRIBUTE_NAME_RECORD_CODE = "recordCode";
        public static final String ATTRIBUTE_NAME_NAME = "name";
        
        public String name;
        public String packageName;
        public String className;
        public String superClass = RecordList.class.getName();
        public String recordPackageName;
        public String recordClassName;
        public Map<String, String[]> keyIndexMap;
        public Map<String, String> conditionIndexMap;
        
        public RecordListMetaData(MetaData parent){
            super(parent);
        }
        
        public void importXML(Element element) throws DeploymentException{
            super.importXML(element);
            
            if(!element.getTagName().equals(TAG_NAME)){
                throw new DeploymentException(
                    "Root tag must be " + TAG_NAME + " : " + element.getTagName()
                );
            }
            String code = Utility.replaceProperty(dataSetsData, getUniqueAttribute(element, ATTRIBUTE_NAME_CODE));
            className = code;
            if(code.indexOf('.') != -1){
                className = code.substring(code.lastIndexOf('.') + 1);
                packageName = code.substring(0, code.lastIndexOf('.'));
            }
            String recordCode = Utility.replaceProperty(
                dataSetsData,
                (getParent() instanceof DataSetsMetaData) ? getUniqueAttribute(element, ATTRIBUTE_NAME_RECORD_CODE) : getOptionalAttribute(element, ATTRIBUTE_NAME_RECORD_CODE)
            );
            recordClassName = recordCode;
            if(recordCode != null && recordCode.indexOf('.') != -1){
                recordClassName = recordCode.substring(recordCode.lastIndexOf('.') + 1);
                recordPackageName = recordCode.substring(0, recordCode.lastIndexOf('.'));
            }
            superClass = Utility.replaceProperty(dataSetsData, getOptionalAttribute(element, ATTRIBUTE_NAME_EXTENDS, superClass));
            name = Utility.replaceProperty(dataSetsData, getOptionalAttribute(element, ATTRIBUTE_NAME_NAME));
            
            final Iterator<Element> keyIndexElements = getChildrenByTagName(
                element,
                TAG_NAME_STOCK_KEY
            );
            while(keyIndexElements.hasNext()){
                if(keyIndexMap == null){
                    keyIndexMap = new HashMap<String, String[]>();
                }
                final Element keyIndexElement = keyIndexElements.next();
                if(keyIndexElement != null){
                    String keyIndexName = getUniqueAttribute(keyIndexElement, ATTRIBUTE_NAME_NAME);
                    String keyIndexStr = Utility.replaceProperty(dataSetsData, getElementContent(keyIndexElement));
                    StringArrayEditor editor = new StringArrayEditor();
                    editor.setAsText(keyIndexStr);
                    String[] keyIndex = (String[])editor.getValue();
                    keyIndexMap.put(keyIndexName, keyIndex);
                }
            }
            
            final Iterator<Element> conditionIndexElements = getChildrenByTagName(
                element,
                TAG_NAME_CONDITION_INDEX
            );
            while(conditionIndexElements.hasNext()){
                if(conditionIndexMap == null){
                    conditionIndexMap = new HashMap<String, String>();
                }
                final Element conditionIndexElement = conditionIndexElements.next();
                if(conditionIndexElement != null){
                    String conditionIndexName = getUniqueAttribute(conditionIndexElement, ATTRIBUTE_NAME_NAME);
                    String conditionIndex = Utility.escapeLineSeparator(Utility.replaceProperty(dataSetsData, getElementContent(conditionIndexElement)));
                    conditionIndexMap.put(conditionIndexName, conditionIndex);
                }
            }
        }
        
        public File getFile(File dir){
            String filePath = null;
            if(packageName == null){
                filePath = className + ".java";
            }else{
                filePath = packageName.replaceAll("\\.", "/") + '/' + className + ".java";
            }
            return new File(dir, filePath);
        }
        
        public String getFullClassName(){
            return packageName == null ? className : (packageName + '.' + className);
        }
        
        public String getFullRecordClassName(){
            return recordPackageName == null ? recordClassName : (recordPackageName + '.' + recordClassName);
        }
        
        public void writeCode(PrintWriter pw){
            if(recordClassName == null){
                pw.println("        setRecordListClass(\"" + name + "\", " + getFullClassName() + ".class);");
            }else{
                if(packageName != null){
                    pw.println("package " + packageName + ';');
                }
                pw.println();
                pw.println("public class " + className + " extends " + superClass + "{");
                pw.println("    public " + className + "(){");
                pw.println("        this(\"" + name + "\");");
                pw.println("    }");
                pw.println("    ");
                pw.println("    public " + className + "(String name){");
                pw.println("        super(name, " + getFullRecordClassName() + ".class);");
                if(keyIndexMap != null){
                    Iterator<Map.Entry<String, String[]>> entries = keyIndexMap.entrySet().iterator();
                    while(entries.hasNext()){
                        Map.Entry<String, String[]> entry = entries.next();
                        pw.print("        setKeyIndex(\"" + entry.getKey() + "\", new String[]{");
                        String[] props = entry.getValue();
                        for(int i = 0, imax = props.length; i < imax; i++){
                            pw.print(props[i] == null ? "null" : ('"' + props[i] + '"'));
                            if(i != imax - 1){
                                pw.print(", ");
                            }
                        }
                        pw.println("});");
                    }
                }
                if(conditionIndexMap != null){
                    Iterator<Map.Entry<String, String>> entries = conditionIndexMap.entrySet().iterator();
                    while(entries.hasNext()){
                        Map.Entry<String, String> entry = entries.next();
                        pw.println("        setConditionIndex(\"" + entry.getKey() + "\", "
                            + (entry.getValue() == null ? "null)" : ('"' + entry.getValue() + "\")")));
                    }
                }
                pw.println("    }");
                pw.println("    ");
                pw.println("    public " + getFullRecordClassName() + ' ' + Utility.createCreaterName(recordClassName) + "(){");
                pw.println("        return (" + getFullRecordClassName() + ")createRecord();");
                pw.println("    }");
                pw.println("}");
            }
        }
    }
    
    private class NestedRecordListMetaData extends MetaData implements CodeGenerator{
        
        private static final long serialVersionUID = -547306924656306470L;
        
        public static final String TAG_NAME = "nestedRecordList";
        public static final String ATTRIBUTE_NAME_CODE = "code";
        public static final String ATTRIBUTE_NAME_NAME = "name";
        
        public String name;
        public String code;
        
        public NestedRecordListMetaData(MetaData parent){
            super(parent);
        }
        
        public void importXML(Element element) throws DeploymentException{
            super.importXML(element);
            
            if(!element.getTagName().equals(TAG_NAME)){
                throw new DeploymentException(
                    "Root tag must be " + TAG_NAME + " : " + element.getTagName()
                );
            }
            code = Utility.replaceProperty(dataSetsData, getUniqueAttribute(element, ATTRIBUTE_NAME_CODE));
            name = Utility.replaceProperty(dataSetsData, getUniqueAttribute(element, ATTRIBUTE_NAME_NAME));
        }
        
        public File getFile(File dir){
            return null;
        }
        
        public void writeCode(PrintWriter pw){
            pw.println("        setNestedRecordListClass(\"" + name + "\", " + code + ".class);");
        }
    }
    
    private class NestedRecordMetaData extends MetaData implements CodeGenerator{
        
        private static final long serialVersionUID = 5542852046600369680L;
        
        public static final String TAG_NAME = "nestedRecord";
        public static final String ATTRIBUTE_NAME_CODE = "code";
        public static final String ATTRIBUTE_NAME_NAME = "name";
        
        public String name;
        public String code;
        
        public NestedRecordMetaData(MetaData parent){
            super(parent);
        }
        
        public void importXML(Element element) throws DeploymentException{
            super.importXML(element);
            
            if(!element.getTagName().equals(TAG_NAME)){
                throw new DeploymentException(
                    "Root tag must be " + TAG_NAME + " : " + element.getTagName()
                );
            }
            code = Utility.replaceProperty(dataSetsData, getUniqueAttribute(element, ATTRIBUTE_NAME_CODE));
            name = Utility.replaceProperty(dataSetsData, getUniqueAttribute(element, ATTRIBUTE_NAME_NAME));
        }
        
        public File getFile(File dir){
            return null;
        }
        
        public void writeCode(PrintWriter pw){
            pw.println("        setNestedRecordClass(\"" + name + "\", " + code + ".class);");
        }
    }
    
    private class DataSetMetaData extends MetaData implements CodeGenerator{
        
        private static final long serialVersionUID = 6664682114622861929L;
        
        public static final String TAG_NAME = "dataSet";
        public static final String ATTRIBUTE_NAME_CODE = "code";
        public static final String ATTRIBUTE_NAME_NAME = "name";
        public static final String ATTRIBUTE_NAME_EXTENDS = "extends";
        
        public String packageName;
        public String className;
        public String name;
        public String superClass = DataSet.class.getName();
        public Map<String, HeaderMetaData> headers;
        public Map<String, RecordListMetaData> recordLists;
        public Map<String, NestedRecordMetaData> nestedRecords;
        public Map<String, NestedRecordListMetaData> nestedRecordLists;
        
        public DataSetMetaData(MetaData parent){
            super(parent);
        }
        
        public void importXML(Element element) throws DeploymentException{
            super.importXML(element);
            
            if(!element.getTagName().equals(TAG_NAME)){
                throw new DeploymentException(
                    "Root tag must be " + TAG_NAME + " : " + element.getTagName()
                );
            }
            String code = Utility.replaceProperty(dataSetsData, getUniqueAttribute(element, ATTRIBUTE_NAME_CODE));
            className = code;
            if(code.indexOf('.') != -1){
                className = code.substring(code.lastIndexOf('.') + 1);
                packageName = code.substring(0, code.lastIndexOf('.'));
            }
            superClass = Utility.replaceProperty(dataSetsData, getOptionalAttribute(element, ATTRIBUTE_NAME_EXTENDS, superClass));
            name = Utility.replaceProperty(dataSetsData, getOptionalAttribute(element, ATTRIBUTE_NAME_NAME));
            
            final Iterator<Element> headerElements = getChildrenByTagName(
                element,
                HeaderMetaData.TAG_NAME
            );
            while(headerElements.hasNext()){
                if(headers == null){
                    headers = new LinkedHashMap<String, HeaderMetaData>();
                }
                final HeaderMetaData headerData = new HeaderMetaData(DataSetMetaData.this);
                headerData.importXML(headerElements.next());
                if(headers.containsKey(headerData.name)){
                    throw new DeploymentException(
                        "Header is duplicated : " + headerData.name
                    );
                }
                headers.put(headerData.name, headerData);
            }
            
            final Iterator<Element> recordListElements = getChildrenByTagName(
                element,
                RecordListMetaData.TAG_NAME
            );
            while(recordListElements.hasNext()){
                if(recordLists == null){
                    recordLists = new LinkedHashMap<String, RecordListMetaData>();
                }
                final RecordListMetaData recordListData = new RecordListMetaData(DataSetMetaData.this);
                recordListData.importXML(recordListElements.next());
                if(recordLists.containsKey(recordListData.name)){
                    throw new DeploymentException(
                        "RecordList is duplicated : " + recordListData.name
                    );
                }
                recordLists.put(recordListData.name, recordListData);
            }
            
            final Iterator<Element> nestedRecordElements = getChildrenByTagName(
                element,
                NestedRecordMetaData.TAG_NAME
            );
            while(nestedRecordElements.hasNext()){
                if(nestedRecords == null){
                    nestedRecords = new LinkedHashMap<String, NestedRecordMetaData>();
                }
                final NestedRecordMetaData nestedRecordData = new NestedRecordMetaData(DataSetMetaData.this);
                nestedRecordData.importXML(nestedRecordElements.next());
                if(nestedRecords.containsKey(nestedRecordData.name)){
                    throw new DeploymentException(
                        "NestedRecord is duplicated : " + nestedRecordData.name
                    );
                }
                nestedRecords.put(nestedRecordData.name, nestedRecordData);
            }
            
            final Iterator<Element> nestedRecordListElements = getChildrenByTagName(
                element,
                NestedRecordListMetaData.TAG_NAME
            );
            while(nestedRecordListElements.hasNext()){
                if(nestedRecordLists == null){
                    nestedRecordLists = new LinkedHashMap<String, NestedRecordListMetaData>();
                }
                final NestedRecordListMetaData nestedRecordListData = new NestedRecordListMetaData(DataSetMetaData.this);
                nestedRecordListData.importXML(nestedRecordListElements.next());
                if(nestedRecordLists.containsKey(nestedRecordListData.name)){
                    throw new DeploymentException(
                        "NestedRecordList is duplicated : " + nestedRecordListData.name
                    );
                }
                nestedRecordLists.put(nestedRecordListData.name, nestedRecordListData);
            }
        }
        
        public File getFile(File dir){
            String filePath = null;
            if(packageName == null){
                filePath = className + ".java";
            }else{
                filePath = packageName.replaceAll("\\.", "/") + '/' + className + ".java";
            }
            return new File(dir, filePath);
        }
        
        public void writeCode(PrintWriter pw){
            if(packageName != null){
                pw.println("package " + packageName + ';');
            }
            pw.println();
            pw.println("public class " + className + " extends " + superClass + "{");
            pw.println("    ");
            if(headers != null){
                Iterator<String> itr = headers.keySet().iterator();
                while(itr.hasNext()){
                    String name = itr.next();
                    pw.println("    public static final String HEADER_" + name.toUpperCase() + " = \"" + name + "\";");
                }
            }
            if(recordLists != null){
                Iterator<String> itr = recordLists.keySet().iterator();
                while(itr.hasNext()){
                    String name = itr.next();
                    pw.println("    public static final String RECORD_LIST_" + name.toUpperCase() + " = \"" + name + "\";");
                }
            }
            if(nestedRecords != null){
                Iterator<String> itr = nestedRecords.keySet().iterator();
                while(itr.hasNext()){
                    String name = itr.next();
                    pw.println("    public static final String NESTED_RECORD_" + name.toUpperCase() + " = \"" + name + "\";");
                }
            }
            if(nestedRecordLists != null){
                Iterator<String> itr = nestedRecordLists.keySet().iterator();
                while(itr.hasNext()){
                    String name = itr.next();
                    pw.println("    public static final String NESTED_RECORD_LIST_" + name.toUpperCase() + " = \"" + name + "\";");
                }
            }
            pw.println("    public " + className + "(){");
            pw.println("        super(\"" + name + "\");");
            if(headers != null){
                Iterator<HeaderMetaData> itr = headers.values().iterator();
                while(itr.hasNext()){
                    HeaderMetaData headerData = itr.next();
                    if(headerData.schema != null){
                        String tmpSchema = headerData.schema;
                        headerData.schema = null;
                        headerData.writeCode(pw);
                        headerData.schema = tmpSchema;
                    }else{
                        headerData.writeCode(pw);
                    }
                }
            }
            if(recordLists != null){
                Iterator<RecordListMetaData> itr = recordLists.values().iterator();
                while(itr.hasNext()){
                    RecordListMetaData recordListData = itr.next();
                    if(recordListData.recordClassName != null){
                        String tmpRecordCode = recordListData.recordClassName;
                        recordListData.recordClassName = null;
                        recordListData.writeCode(pw);
                        recordListData.recordClassName = tmpRecordCode;
                    }else{
                        recordListData.writeCode(pw);
                    }
                }
            }
            if(nestedRecords != null){
                Iterator<NestedRecordMetaData> itr = nestedRecords.values().iterator();
                while(itr.hasNext()){
                    NestedRecordMetaData nestedRecordData = itr.next();
                    nestedRecordData.writeCode(pw);
                }
            }
            if(nestedRecordLists != null){
                Iterator<NestedRecordListMetaData> itr = nestedRecordLists.values().iterator();
                while(itr.hasNext()){
                    NestedRecordListMetaData nestedRecordListData = itr.next();
                    nestedRecordListData.writeCode(pw);
                }
            }
            pw.println("    }");
            if(headers != null){
                Iterator<HeaderMetaData> itr = headers.values().iterator();
                while(itr.hasNext()){
                    HeaderMetaData headerData = itr.next();
                    if(headerData.name == null){
                        continue;
                    }
                    pw.println("    public " + headerData.getFullClassName() + " " + Utility.createGetterName(headerData.name + "Header") + "(){");
                    pw.println("        return (" + headerData.getFullClassName() + ")getHeader(HEADER_" + headerData.name.toUpperCase() + ");");
                    pw.println("    }");
                }
            }
            if(recordLists != null){
                Iterator<RecordListMetaData> itr = recordLists.values().iterator();
                while(itr.hasNext()){
                    RecordListMetaData recordListData = itr.next();
                    if(recordListData.name == null){
                        continue;
                    }
                    pw.println("    public " + recordListData.getFullClassName() + " " + Utility.createGetterName(recordListData.name + "RecordList") + "(){");
                    pw.println("        return (" + recordListData.getFullClassName() + ")getRecordList(RECORD_LIST_" + recordListData.name.toUpperCase() + ");");
                    pw.println("    }");
                }
            }
            if(nestedRecords != null){
                Iterator<NestedRecordMetaData> itr = nestedRecords.values().iterator();
                while(itr.hasNext()){
                    NestedRecordMetaData nestedRecordData = itr.next();
                    pw.println("    public " + nestedRecordData.code + " " + Utility.createCreaterName(nestedRecordData.name + "NestedRecord") + "(){");
                    pw.println("        return (" + nestedRecordData.code + ")createNestedRecord(NESTED_RECORD_" + nestedRecordData.name.toUpperCase() + ");");
                    pw.println("    }");
                }
            }
            if(nestedRecordLists != null){
                Iterator<NestedRecordListMetaData> itr = nestedRecordLists.values().iterator();
                while(itr.hasNext()){
                    NestedRecordListMetaData nestedRecordListData = itr.next();
                    pw.println("    public " + nestedRecordListData.code + " " + Utility.createCreaterName(nestedRecordListData.name + "NestedRecordList") + "(){");
                    pw.println("        return (" + nestedRecordListData.code + ")createNestedRecordList(NESTED_RECORD_LIST_" + nestedRecordListData.name.toUpperCase() + ");");
                    pw.println("    }");
                }
            }
            pw.println("}");
        }
    }
    
    private static class Utility extends jp.ossc.nimbus.core.Utility{
        private static final String CREATE_METHOD_PREFIX = "create";
        private static final String GET_METHOD_PREFIX = "get";
        private static final String SET_METHOD_PREFIX = "set";
        
        private Utility(){}
        public static String replaceProperty(DataSetsMetaData metaData, String str){
            String result = str;
            result = Utility.replaceSystemProperty(result, true);
            if(result == null){
                return null;
            }
            final int startIndex = result.indexOf(SYSTEM_PROPERTY_START);
            if(startIndex == -1){
                return result;
            }
            final int endIndex = result.indexOf(SYSTEM_PROPERTY_END);
            if(endIndex == -1 || startIndex > endIndex){
                return result;
            }
            String propStr = result.substring(
                startIndex + SYSTEM_PROPERTY_START.length(),
                endIndex
            );
            final int defaultValueIndex = propStr.indexOf(SYSTEM_PROPERTY_DEFAULT_VALUE_SEPARATOR);
            String defaultValue = null;
            if(defaultValueIndex != -1){
                propStr = propStr.substring(0, defaultValueIndex);
                defaultValue = propStr.substring(defaultValueIndex + 1);
            }
            String prop = defaultValue;
            if(propStr != null && propStr.length() != 0){
                PropertyMetaData propData = (PropertyMetaData)metaData.properties.get(propStr);
                if(propData != null){
                    prop = propData.getValue();
                }
            }
            if(prop == null){
                return result.substring(0, endIndex + SYSTEM_PROPERTY_END.length())
                 + replaceProperty(
                     metaData,
                     result.substring(endIndex + SYSTEM_PROPERTY_END.length())
                 );
            }else{
                result = result.substring(0, startIndex) + prop
                     + result.substring(endIndex + SYSTEM_PROPERTY_END.length());
            }
            if(result.indexOf(SYSTEM_PROPERTY_START) != -1){
                return replaceProperty(metaData, result);
            }
            return result;
        }
        
        public static String escapeLineSeparator(String str){
            if(str == null || str.length() == 0){
                return str;
            }
            if(str.indexOf("\r\n") != -1){
                str = str.replaceAll("\r\n", "\\\\n");
            }
            if(str.indexOf("\n") != -1){
                str = str.replaceAll("\n", "\\\\n");
            }
            if(str.indexOf("\r") != -1){
                str = str.replaceAll("\r", "\\\\n");
            }
            return str;
        }
        
        public static String createCreaterName(String property){
            StringBuilder result = new StringBuilder(property);
            final int len = result.length();
            if(len != 0 && !Character.isUpperCase(result.charAt(0))){
                char capital = Character.toUpperCase(result.charAt(0));
                result.deleteCharAt(0).insert(0, capital);
            }
            return result.insert(0, CREATE_METHOD_PREFIX).toString();
        }
        
        public static String createGetterName(String property){
            StringBuilder result = new StringBuilder(property);
            final int len = result.length();
            if(len != 0 && !Character.isUpperCase(result.charAt(0))){
                char capital = Character.toUpperCase(result.charAt(0));
                result.deleteCharAt(0).insert(0, capital);
            }
            return result.insert(0, GET_METHOD_PREFIX).toString();
        }
        
        public static String createSetterName(String property){
            StringBuilder result = new StringBuilder(property);
            final int len = result.length();
            if(len != 0 && !Character.isUpperCase(result.charAt(0))){
                char capital = Character.toUpperCase(result.charAt(0));
                result.deleteCharAt(0).insert(0, capital);
            }
            return result.insert(0, SET_METHOD_PREFIX).toString();
        }
    }
    
    /**
     * gp@Wo͂ɕ\B<p>
     */
    private static void usage(){
        try{
            System.out.println(
                getResourceString(USAGE_RESOURCE)
            );
        }catch(IOException e){
            e.printStackTrace();
        }
    }
    /**
     * \[X𕶎ƂēǂݍށB<p>
     *
     * @param name \[X
     * @exception IOException \[X݂Ȃꍇ
     */
    private static String getResourceString(String name) throws IOException{
        
        // \[X̓̓Xg[擾
        InputStream is = DataSetCodeGenerator.class.getClassLoader()
            .getResourceAsStream(name);
        
        // bZ[W̓ǂݍ
        StringBuilder buf = new StringBuilder();
        BufferedReader reader = null;
        final String separator = System.getProperty("line.separator");
        try{
            reader = new BufferedReader(new InputStreamReader(is));
            String line = null;
            while((line = reader.readLine()) != null){
                buf.append(line).append(separator);
            }
        }finally{
            if(reader != null){
                try{
                    reader.close();
                }catch(IOException e){
                }
            }
        }
        return unicodeConvert(buf.toString());
    }
    
    /**
     * jR[hGXP[v܂ł\̂镶ftHgGR[fBO̕ɕϊB<p>
     *
     * @param str jR[hGXP[v܂ł\̂镶
     * @return ftHgGR[fBO̕
     */
    private static String unicodeConvert(String str){
        char c;
        int len = str.length();
        StringBuilder buf = new StringBuilder(len);
        
        for(int i = 0; i < len; ){
            c = str.charAt(i++);
            if(c == '\\'){
                c = str.charAt(i++);
                if(c == 'u'){
                    int value = 0;
                    for(int j = 0; j < 4; j++){
                        c = str.charAt(i++);
                        switch(c){
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                            value = (value << 4) + (c - '0');
                            break;
                        case 'a':
                        case 'b':
                        case 'c':
                        case 'd':
                        case 'e':
                        case 'f':
                            value = (value << 4) + 10 + (c - 'a');
                            break;
                        case 'A':
                        case 'B':
                        case 'C':
                        case 'D':
                        case 'E':
                        case 'F':
                            value = (value << 4) + 10 + (c - 'A');
                            break;
                        default:
                            throw new IllegalArgumentException(
                                "Failed to convert unicode : " + c
                            );
                        }
                    }
                    buf.append((char)value);
                }else{
                    switch(c){
                    case 't':
                        c = '\t';
                        break;
                    case 'r':
                        c = '\r';
                        break;
                    case 'n':
                        c = '\n';
                        break;
                    case 'f':
                        c = '\f';
                        break;
                    default:
                    }
                    buf.append(c);
                }
            }else{
                buf.append(c);
            }
        }
        return buf.toString();
    }
    
    public static void main(String[] args) throws Exception{
        
        if(args.length == 0 || args[0].equals("-help")){
            // gp@\
            usage();
            return;
        }
        
        boolean option = false;
        String key = null;
        File dest = new File(".");
        String encoding = null;
        boolean verbose = false;
        boolean validate = true;
        final List<File> definitionFiles = new ArrayList<File>();
        for(int i = 0; i < args.length; i++){
            if(option){
                if(key.equals("-d")){
                    dest = new File(args[i]);
                }else if(key.equals("-validate")){
                    validate = Boolean.valueOf(args[i]).booleanValue();
                }else if(key.equals("-encoding")){
                    encoding = args[i];
                }
                option = false;
                key = null;
            }else{
                if(args[i].equals("-d")){
                    option = true;
                    key = args[i];
                }else if(args[i].equals("-validate")){
                    option = true;
                    key = args[i];
                }else if(args[i].equals("-encoding")){
                    option = true;
                    key = args[i];
                }else if(args[i].equals("-v")){
                    verbose = true;
                }else{
                  definitionFiles.add(new File(args[i]));
                }
            }
        }
        
        final DataSetCodeGenerator generator = new DataSetCodeGenerator();
        generator.isValidate = validate;
        generator.encoding = encoding;
        for(int i = 0; i < definitionFiles.size(); i++){
            File[] files = generator.generate((File)definitionFiles.get(i), dest);
            if(verbose){
                for(int j = 0; j < files.length; j++){
                    System.out.println(files[j]);
                }
            }
        }
        Thread.sleep(500);
    }
}
