/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.servlet;

import java.io.*;
import java.util.*;
import java.net.URLEncoder;
import javax.servlet.*;
import javax.servlet.http.*;

import jp.ossc.nimbus.beans.*;
import jp.ossc.nimbus.beans.dataset.*;
import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.service.context.*;
import jp.ossc.nimbus.service.interpreter.*;

public class SharedContextServlet extends HttpServlet{
    
    private static final long serialVersionUID = -6992362984683159336L;

    /**
     * ΏۂƂ鋤LReLXg̃T[rXw肷邽߂̏p[^B<p>
     */
    protected static final String INIT_PARAM_NAME_SERVICE_NAMES = "ServiceNames";
    
    /**
     * C^v^̃T[rXw肷邽߂̏p[^B<p>
     */
    protected static final String INIT_PARAM_NAME_INTERPRETER_SERVICE_NAME = "InterpreterServiceName";
    
    /**
     * ǉLɂ邩ǂw肷邽߂̏p[^B<p>
     */
    protected static final String INIT_PARAM_NAME_PUT_ENABLED = "PutEnabled";
    
    /**
     * XVLɂ邩ǂw肷邽߂̏p[^B<p>
     */
    protected static final String INIT_PARAM_NAME_UPDATE_ENABLED = "UpdateEnabled";
    
    /**
     * 폜Lɂ邩ǂw肷邽߂̏p[^B<p>
     */
    protected static final String INIT_PARAM_NAME_REMOVE_ENABLED = "RemoveEnabled";
    
    /**
     * S폜Lɂ邩ǂw肷邽߂̏p[^B<p>
     */
    protected static final String INIT_PARAM_NAME_CLEAR_ENABLED = "ClearEnabled";
    
    /**
     * Ǎ݂Lɂ邩ǂw肷邽߂̏p[^B<p>
     */
    protected static final String INIT_PARAM_NAME_LOAD_ENABLED = "LoadEnabled";
    
    /**
     * ݂Lɂ邩ǂw肷邽߂̏p[^B<p>
     */
    protected static final String INIT_PARAM_NAME_SAVE_ENABLED = "SaveEnabled";
    
    private Interpreter interpreter;
    
    private ServiceName[] getServiceNames(){
        final ServletConfig config = getServletConfig();
        final String serviceNamesStr = config.getInitParameter(INIT_PARAM_NAME_SERVICE_NAMES);
        if(serviceNamesStr == null){
            return null;
        }
        final ServiceNameArrayEditor editor = new ServiceNameArrayEditor();
        editor.setAsText(serviceNamesStr);
        return (ServiceName[])editor.getValue();
    }
    
    private ServiceName getInterpreterServiceName(){
        final ServletConfig config = getServletConfig();
        final String serviceNameStr = config.getInitParameter(INIT_PARAM_NAME_INTERPRETER_SERVICE_NAME);
        if(serviceNameStr == null){
            return null;
        }
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        return (ServiceName)editor.getValue();
    }
    
    private boolean isPutEnabled(){
        final ServletConfig config = getServletConfig();
        final String isEnabled = config.getInitParameter(INIT_PARAM_NAME_PUT_ENABLED);
        return isEnabled == null ? false : Boolean.valueOf(isEnabled).booleanValue();
    }
    
    private boolean isUpdateEnabled(){
        final ServletConfig config = getServletConfig();
        final String isEnabled = config.getInitParameter(INIT_PARAM_NAME_UPDATE_ENABLED);
        return isEnabled == null ? false : Boolean.valueOf(isEnabled).booleanValue();
    }
    
    private boolean isRemoveEnabled(){
        final ServletConfig config = getServletConfig();
        final String isEnabled = config.getInitParameter(INIT_PARAM_NAME_REMOVE_ENABLED);
        return isEnabled == null ? false : Boolean.valueOf(isEnabled).booleanValue();
    }
    
    private boolean isClearEnabled(){
        final ServletConfig config = getServletConfig();
        final String isEnabled = config.getInitParameter(INIT_PARAM_NAME_CLEAR_ENABLED);
        return isEnabled == null ? false : Boolean.valueOf(isEnabled).booleanValue();
    }
    
    private boolean isLoadEnabled(){
        final ServletConfig config = getServletConfig();
        final String isEnabled = config.getInitParameter(INIT_PARAM_NAME_LOAD_ENABLED);
        return isEnabled == null ? false : Boolean.valueOf(isEnabled).booleanValue();
    }
    
    private boolean isSaveEnabled(){
        final ServletConfig config = getServletConfig();
        final String isEnabled = config.getInitParameter(INIT_PARAM_NAME_SAVE_ENABLED);
        return isEnabled == null ? false : Boolean.valueOf(isEnabled).booleanValue();
    }
    
    /**
     * T[ubg̏sB<p>
     * T[rX`̃[hyу[h`FbNsB
     *
     * @exception ServletException T[ubg̏Ɏsꍇ
     */
    public synchronized void init() throws ServletException{
        ServiceName interpreterServiceName = getInterpreterServiceName();
        if(interpreterServiceName == null){
            BeanShellInterpreterService bshInterpreter = new BeanShellInterpreterService();
            try{
                bshInterpreter.create();
                bshInterpreter.start();
            }catch(Exception e){
                throw new ServletException(e);
            }
            interpreter = bshInterpreter;
        }else{
            interpreter = (Interpreter)ServiceManagerFactory.getServiceObject(interpreterServiceName);
        }
    }
    
    /**
     * POSTNGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void doPost(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        process(req, resp);
    }
    
    /**
     * GETNGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void doGet(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        process(req, resp);
    }
    
    /**
     * NGXgsB<p>
     * ǗR\[sB
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void process(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        req.setCharacterEncoding("UTF-8");
        
        final String action = req.getParameter("action");
        if(action == null){
            processIndexResponse(req, resp);
        }else if(action.equals("context")){
            processContextResponse(req, resp);
        }else if(action.equals("get")){
            processGetResponse(req, resp);
        }else if(action.equals("containsKey")){
            processContainsKeyResponse(req, resp);
        }else if(action.equals("keySet")){
            processKeySetResponse(req, resp);
        }else if(action.equals("query")){
            processQueryResponse(req, resp);
        }else if(action.equals("put")){
            if(!isPutEnabled()){
                resp.sendError(HttpServletResponse.SC_NOT_ACCEPTABLE);
                return;
            }
            processPutResponse(req, resp);
        }else if(action.equals("update")){
            if(!isUpdateEnabled()){
                resp.sendError(HttpServletResponse.SC_NOT_ACCEPTABLE);
                return;
            }
            processUpdateResponse(req, resp);
        }else if(action.equals("remove")){
            if(!isRemoveEnabled()){
                resp.sendError(HttpServletResponse.SC_NOT_ACCEPTABLE);
                return;
            }
            processRemoveResponse(req, resp);
        }else if(action.equals("clear")){
            if(!isClearEnabled()){
                resp.sendError(HttpServletResponse.SC_NOT_ACCEPTABLE);
                return;
            }
            processClearResponse(req, resp);
        }else if(action.equals("load")){
            if(!isLoadEnabled()){
                resp.sendError(HttpServletResponse.SC_NOT_ACCEPTABLE);
                return;
            }
            processLoadResponse(req, resp);
        }else if(action.equals("loadKey")){
            if(!isLoadEnabled()){
                resp.sendError(HttpServletResponse.SC_NOT_ACCEPTABLE);
                return;
            }
            processLoadKeyResponse(req, resp);
        }else if(action.equals("save")){
            if(!isSaveEnabled()){
                resp.sendError(HttpServletResponse.SC_NOT_ACCEPTABLE);
                return;
            }
            processSaveResponse(req, resp);
        }else{
            resp.sendError(HttpServletResponse.SC_NOT_FOUND);
        }
    }
    
    private String getCurrentPath(HttpServletRequest req){
        String path = req.getServletPath();
        if(path.endsWith("/")){
            return "." + (req.getPathInfo() == null ? "" : req.getPathInfo());
        }else{
            return "." + path;
        }
    }
    
    /**
     * CfbNXʃNGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processIndexResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        resp.setContentType("text/html;charset=UTF-8");
        final StringBuilder buf = new StringBuilder();
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContexts</title></head>");
        buf.append("<body>");
        
        ServiceName[] contextServiceNames = getServiceNames();
        if(contextServiceNames == null){
            ServiceManager[] managers = ServiceManagerFactory.findManagers();
            List<ServiceName> names = new ArrayList<ServiceName>();
            for(int i = 0; i < managers.length; i++){
                Set<Service> services = managers[i].serviceSet();
                Iterator<Service> itr = services.iterator();
                while(itr.hasNext()){
                    Service service = itr.next();
                    if(service instanceof SharedContext){
                        names.add(service.getServiceNameObject());
                    }
                }
            }
            contextServiceNames = (ServiceName[])names.toArray(new ServiceName[names.size()]);
        }
        
        buf.append("<b>Contexts</b><br>");
        buf.append("<table border=\"1\" cellspacing=\"0\" cellpadding=\"3\" width=\"90%\">");
        buf.append("<tr bgcolor=\"#cccccc\"><th>name</th><th>replicated/distributed</th><th>client/server</th><th>size</th></tr>");
        Arrays.sort(contextServiceNames);
        final StringBuilder url = new StringBuilder();
        for(int i = 0; i < contextServiceNames.length; i++){
            final ServiceName serviceName = contextServiceNames[i];
            Service service = ServiceManagerFactory.getService(serviceName);
            boolean isDistributed = false;
            if(service instanceof DistributedSharedContext){
                isDistributed = true;
            }else if(service instanceof SharedContext){
                isDistributed = false;
            }else{
                continue;
            }
            SharedContext<?> context = (SharedContext<?>)service;
            url.setLength(0);
            url.append(getCurrentPath(req))
               .append("?action=context&name=")
               .append(URLEncoder.encode(serviceName.toString(), "UTF-8"));
            buf.append("<tr>");
            buf.append("<td><a href=\"")
               .append(resp.encodeURL(url.toString()))
               .append("\">");
            buf.append(serviceName).append("</a></td>");
            buf.append("<td>").append(isDistributed ? "distributed" : "replicated").append("</td>");
            buf.append("<td>").append(context.isClient() ? "client" : ("server" + (context.isMain() ? "(main)" : "(sub)"))).append("</td>");
            buf.append("<td>").append(context.size() + (context.isClient() || isDistributed ? ("(" + context.sizeLocal() + ")") : "")).append("</td>");
            buf.append("</tr>");
        }
        buf.append("</table>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXgʃNGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processContextResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        SharedContext<?> context = (SharedContext<?>)ServiceManagerFactory.getService(serviceName);
        boolean isDistributed = context instanceof DistributedSharedContext ? true : false;
        
        resp.setContentType("text/html;charset=UTF-8");
        final StringBuilder buf = new StringBuilder();
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + "</title></head>");
        buf.append("<body>");
        
        buf.append("<b>Context " + serviceName + "</b><br>");
        buf.append("<table border=\"1\" cellspacing=\"0\" cellpadding=\"3\" width=\"90%\">");
        buf.append("<tr><th bgcolor=\"#cccccc\">replicated/distributed</th><td colspan=\"2\">").append(isDistributed ? "distributed" : "replicated").append("</td></tr>");
        buf.append("<tr><th bgcolor=\"#cccccc\">client/server</th><td colspan=\"2\">").append(context.isClient() ? "client" : ("server" + (context.isMain() ? "(main)" : "(sub)"))).append("</td></tr>");
        buf.append("<tr><th bgcolor=\"#cccccc\">size</th><td colspan=\"2\">").append(context.size() + (context.isClient() || isDistributed ? ("(" + context.sizeLocal() + ")") : "")).append("</td></tr>");
        
        buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
        buf.append("<input type=\"hidden\" name=\"action\" value=\"get\"/>");
        buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
        buf.append("<tr><th bgcolor=\"#cccccc\"><input type=\"submit\" value=\"get\"/></th><td colspan=\"2\"><table><tr><td>key:</td><td><textarea name=\"key\" cols=\"40\" rows=\"4\"></textarea></td></tr></table></td></tr>");
        buf.append("</form>");
        
        buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
        buf.append("<input type=\"hidden\" name=\"action\" value=\"containsKey\"/>");
        buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
        buf.append("<tr><th bgcolor=\"#cccccc\"><input type=\"submit\" value=\"containsKey\"/></th><td colspan=\"2\"><table><tr><td>key:</td><td><textarea name=\"key\" cols=\"40\" rows=\"4\"></textarea></td></tr></table></td></tr>");
        buf.append("</form>");
        
        buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
        buf.append("<input type=\"hidden\" name=\"action\" value=\"keySet\"/>");
        buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
        buf.append("<tr><th bgcolor=\"#cccccc\"><input type=\"submit\" value=\"keySet\"/></th><td colspan=\"2\">&nbsp;</td></tr>");
        buf.append("</form>");
        
        buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
        buf.append("<input type=\"hidden\" name=\"action\" value=\"query\"/>");
        buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
        buf.append("<tr><th bgcolor=\"#cccccc\"><input type=\"submit\" value=\"query\"/></th><td colspan=\"2\"><table><tr><td>query:</td><td><textarea name=\"query\" cols=\"40\" rows=\"10\"></textarea></td></tr></table></td></tr>");
        buf.append("</form>");
        
        if(isRemoveEnabled()){
            buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
            buf.append("<input type=\"hidden\" name=\"action\" value=\"remove\"/>");
            buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
            buf.append("<tr><th bgcolor=\"#cccccc\"><input type=\"submit\" value=\"remove\"/></th><td colspan=\"2\"><table><tr><td>key:</td><td><textarea name=\"key\" cols=\"40\" rows=\"4\"></textarea></td></tr></table></td></tr>");
            buf.append("</form>");
        }
        if(isPutEnabled()){
            buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
            buf.append("<input type=\"hidden\" name=\"action\" value=\"put\"/>");
            buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
            buf.append("<tr><th bgcolor=\"#cccccc\"><input type=\"submit\" value=\"put\"/></th><td><table><tr><td>key:</td><td><textarea name=\"key\" cols=\"40\" rows=\"4\"></textarea></td></tr></table></td><td><table><tr><td>value:</td><td><textarea name=\"value\"></textarea></td></tr></table></td></tr>");
            buf.append("</form>");
        }
        if(isClearEnabled()){
            buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
            buf.append("<input type=\"hidden\" name=\"action\" value=\"clear\"/>");
            buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
            buf.append("<tr><th bgcolor=\"#cccccc\"><input type=\"submit\" value=\"clear\"/></th><td colspan=\"2\"><input type=\"radio\" name=\"type\" value=\"local\" checked>local<input type=\"radio\" name=\"type\" value=\"all\">all</td></tr>");
            buf.append("</form>");
        }
        if(isLoadEnabled()){
            buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
            buf.append("<input type=\"hidden\" name=\"action\" value=\"load\"/>");
            buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
            buf.append("<tr><th bgcolor=\"#cccccc\"><input type=\"submit\" value=\"load\"/></th><td colspan=\"2\"><table><tr><td>key:</td><td><textarea name=\"key\" cols=\"40\" rows=\"4\"></textarea></td></tr></table></td></tr>");
            buf.append("</form>");
            buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
            buf.append("<input type=\"hidden\" name=\"action\" value=\"loadKey\"/>");
            buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
            buf.append("<tr><th bgcolor=\"#cccccc\"><input type=\"submit\" value=\"loadKey\"/></th><td colspan=\"2\">&nbsp;</td></tr>");
            buf.append("</form>");
        }
        if(isSaveEnabled()){
            buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
            buf.append("<input type=\"hidden\" name=\"action\" value=\"save\"/>");
            buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
            buf.append("<tr><th bgcolor=\"#cccccc\"><input type=\"submit\" value=\"save\"/></th><td colspan=\"2\">&nbsp;</td></tr>");
            buf.append("</form>");
        }
        buf.append("</table>")
        ;
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXg̒l擾郊NGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processGetResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        SharedContext<?> context = (SharedContext<?>)ServiceManagerFactory.getService(serviceName);
        String keyStr = req.getParameter("key");
        if(keyStr.indexOf("\n") == -1
            && keyStr.indexOf("\r") == -1){
            keyStr = "return " + keyStr;
        }
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " Get </title></head>");
        buf.append("<body>");
        
        try{
            Object key = interpreter.evaluate(keyStr);
            if(context.containsKey(key)){
                Object value = context.get(key);
                writeValue(req, resp, serviceNameStr, keyStr, buf, value, true);
            }else{
                buf.append("not contains key : ").append(key);
            }
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    private StringBuilder writeThrowable(StringBuilder buf, Throwable th){
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        th.printStackTrace(pw);
        pw.flush();
        String stackTrace = sw.toString();
        return buf.append("<pre>").append(stackTrace).append("</pre>");
    }
    
    private StringBuilder writeValue(
        HttpServletRequest req,
        HttpServletResponse resp,
        String contextName,
        String key,
        StringBuilder buf,
        Object value,
        boolean isLink
    ) throws IOException{
        if(value == null){
            buf.append("null");
        }else if(value instanceof Record){
            Record record = (Record)value;
            boolean isUpdateSupport = isLink && (record instanceof SharedContextRecord) && isUpdateEnabled();
            RecordSchema schema = record.getRecordSchema();
            PropertySchema[] propSchemata = schema.getPropertySchemata();
            if(isUpdateSupport){
                buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
                buf.append("<input type=\"hidden\" name=\"action\" value=\"update\"/>");
                buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(contextName).append("\"/>");
                buf.append("<textarea name=\"key\" hidden>").append(key).append("</textarea>");
            }
            buf.append("<table border=\"1\" cellspacing=\"0\" cellpadding=\"3\" width=\"90%\">");
            buf.append("<tr><th bgcolor=\"#cccccc\">type</th><td colspan=\"4\">").append(value.getClass().getName()).append("</td></tr>");
            if(isUpdateSupport){
                buf.append("<tr bgcolor=\"#cccccc\"><th rowspan=\"2\">name</th><th rowspan=\"2\">type</th><th colspan=\"2\">value</th></tr>");
                buf.append("<tr bgcolor=\"#cccccc\"><th>current</th><th>new</th></tr>");
            }else{
                buf.append("<tr bgcolor=\"#cccccc\"><th>name</th><th>type</th><th>value</th></tr>");
            }
            for(int i = 0; i < propSchemata.length; i++){
                buf.append("<tr><th bgcolor=\"#cccccc\">").append(propSchemata[i].getName()).append("</th>")
                   .append("<td>").append(propSchemata[i].getType().getName()).append("</td>")
                   .append("<td>").append(record.getProperty(i)).append("</td>");
                if(isUpdateSupport){
                    buf.append("<td><table><tr><td></td><td><textarea name=\"value\" cols=\"40\" rows=\"4\"></textarea></td></tr></table></td>");
                }
                buf.append("</tr>");
            }
            buf.append("</table>");
            if(isUpdateSupport){
                buf.append("<input type=\"submit\" value=\"update\">");
                buf.append("</form>");
            }
        }else if(value instanceof RecordList){
            RecordList recordList = (RecordList)value;
            boolean isUpdateSupport = isLink && (recordList instanceof SharedContextRecordList) && isUpdateEnabled();
            RecordSchema schema = recordList.getRecordSchema();
            PropertySchema[] propSchemata = schema.getPropertySchemata();
            if(isUpdateSupport){
                buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
                buf.append("<input type=\"hidden\" name=\"action\" value=\"update\"/>");
                buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(contextName).append("\"/>");
                buf.append("<textarea name=\"key\" hidden>").append(key).append("</textarea>");
            }
            buf.append("<table border=\"1\" cellspacing=\"0\" cellpadding=\"3\" width=\"90%\">");
            if(isUpdateSupport){
                buf.append("<tr><th bgcolor=\"#cccccc\" colspan=\"2\">type</th><td colspan=\"")
                   .append(propSchemata.length).append("\">")
                   .append(value.getClass().getName()).append("</td></tr>");
            }else{
                buf.append("<tr><th bgcolor=\"#cccccc\">type</th><td colspan=\"")
                   .append(propSchemata.length).append("\">")
                   .append(value.getClass().getName()).append("</td></tr>");
            }
            buf.append("<tr bgcolor=\"#cccccc\">");
            if(isUpdateSupport){
                buf.append("<th rowspan=\"2\" colspan=\"2\">Index</th>");
            }else{
                buf.append("<th rowspan=\"2\">Index</th>");
            }
            for(int i = 0; i < propSchemata.length; i++){
                buf.append("<th>").append(propSchemata[i].getName()).append("</th>");
            }
            buf.append("</tr>");
            buf.append("<tr bgcolor=\"#cccccc\">");
            for(int i = 0; i < propSchemata.length; i++){
                buf.append("<th>").append(propSchemata[i].getType().getName()).append("</th>");
            }
            buf.append("</tr>");
            for(int i = 0; i < recordList.size(); i++){
                Record record = (Record)recordList.get(i);
                if(isUpdateSupport){
                    buf.append("<tr><td rowspan=\"2\">").append(i).append("</td>").append("<td>current</td>");
                }else{
                    buf.append("<tr><td>").append(i).append("</td>");
                }
                for(int j = 0; j < propSchemata.length; j++){
                    buf.append("<td>").append(record.getProperty(j)).append("</td>");
                }
                buf.append("</tr>");
                if(isUpdateSupport){
                    buf.append("<tr>").append("<td>new</td>");
                    for(int j = 0; j < propSchemata.length; j++){
                        buf.append("<td><table><tr><td style=\"text-align:center;\"></td></tr><tr><td><textarea name=\"value_").append(i).append("\" cols=\"40\" rows=\"4\"></textarea></td></tr></table></td>");
                    }
                    buf.append("</tr>");
                }
            }
            buf.append("</table>");
            if(isUpdateSupport){
                buf.append("<input type=\"submit\" value=\"update\">");
                buf.append("</form>");
            }
        }else{
            if(isLink && isPutEnabled()){
                buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\">");
                buf.append("<input type=\"hidden\" name=\"action\" value=\"put\"/>");
                buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(contextName).append("\"/>");
                buf.append("<textarea name=\"key\" hidden>").append(key).append("</textarea>");
            }
            buf.append("<table><tr><td>").append(value.toString()).append("</td>");
            if(isLink && isPutEnabled()){
                buf.append("<td></td><td><textarea name=\"value\" cols=\"40\" rows=\"4\"></textarea></td><td><input type=\"submit\" value=\"put\"></td></tr></table>");
                buf.append("</form>");
            }else{
                buf.append("</tr></table>");
            }
        }
        return buf;
    }
    
    /**
     * ReLXg̃L[݂邩mF郊NGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processContainsKeyResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        SharedContext<?> context = (SharedContext<?>)ServiceManagerFactory.getService(serviceName);
        String keyStr = req.getParameter("key");
        if(keyStr.indexOf("\n") == -1
            && keyStr.indexOf("\r") == -1){
            keyStr = "return " + keyStr;
        }
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " ContainsKey</title></head>");
        buf.append("<body>");
        
        try{
            Object key = interpreter.evaluate(keyStr);
            if(context.containsKey(key)){
                buf.append("contains key : ").append(key);
                if(context.containsKeyLocal(key)){
                    buf.append(" on local");
                }
            }else{
                buf.append("not contains key : ").append(key);
            }
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXg̃L[W擾郊NGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    @SuppressWarnings("unchecked")
    protected void processKeySetResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        SharedContext<?> context = (SharedContext<?>)ServiceManagerFactory.getService(serviceName);
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " KeySet</title></head>");
        buf.append("<body>");
        
        try{
            List<?> keys = new ArrayList<Object>(context.keySet());
            if(keys.size() != 0){
                if(keys.get(0) instanceof Comparable){
                    Collections.sort((List<Comparable<Object>>)keys);
                }else{
                    Collections.sort(
                        keys,
                        new Comparator<Object>(){
                            public int compare(Object o1, Object o2){
                                if(o1 == null && o2 == null){
                                    return 0;
                                }else if(o1 == null && o2 != null){
                                    return -1;
                                }else if(o1 != null && o2 == null){
                                    return 1;
                                }
                                return o1.toString().compareTo(o2.toString());
                            }
                        }
                    );
                }
            }
            buf.append("<table border=\"1\" cellspacing=\"0\" cellpadding=\"3\" width=\"90%\">");
            buf.append("<tr bgcolor=\"#cccccc\"><th colspan=\"2\">key</th><th rowspan=\"2\">local/remote</th><th rowspan=\"2\">&nbsp;</th></tr>");
            buf.append("<tr bgcolor=\"#cccccc\"><th>value</th><th>type</th></tr>");
            for(int i = 0; i < keys.size(); i++){
                Object key = keys.get(i);
                
                buf.append("<tr><td>").append(key).append("</td>");
                buf.append("<td>").append(key == null ? "&nbsp;" : key.getClass().getName()).append("</td>");
                buf.append("<td>").append(context.containsKeyLocal(key) ? "local" : "remote").append("</td>");
                buf.append("<form method=\"POST\" action=\"").append(getCurrentPath(req)).append("\" name=\"key_").append(i).append("\">");
                buf.append("<input type=\"hidden\" name=\"action\" value=\"\"/>");
                buf.append("<input type=\"hidden\" name=\"name\" value=\"").append(serviceNameStr).append("\"/>");
                buf.append("<td><table><tr><td>key:</td><td><textarea name=\"key\" cols=\"40\" rows=\"4\"></textarea></td>");
                buf.append("<td><input type=\"submit\" value=\"get\" onclick=\"document.key_").append(i).append(".action.value='get'\"/></td>");
                if(isRemoveEnabled()){
                    buf.append("<td><input type=\"submit\" value=\"remove\" onclick=\"document.key_").append(i).append(".action.value='remove'\"/></td>");
                }
                buf.append("</tr></table></td></form>");
                buf.append("</tr>");
            }
            buf.append("</table>");
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXgɃNGs郊NGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processQueryResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        SharedContext<?> context = (SharedContext<?>)ServiceManagerFactory.getService(serviceName);
        String query = req.getParameter("query");
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " Query</title></head>");
        buf.append("<body>");
        
        try{
            Object ret = context.executeInterpretQuery(query, null);
            writeValue(req, resp, null, null, buf, ret, false);
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXg̒l폜郊NGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processRemoveResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        SharedContext<?> context = (SharedContext<?>)ServiceManagerFactory.getService(serviceName);
        String keyStr = req.getParameter("key");
        if(keyStr.indexOf("\n") == -1
            && keyStr.indexOf("\r") == -1){
            keyStr = "return " + keyStr;
        }
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " Remove</title></head>");
        buf.append("<body>");
        
        try{
            Object key = interpreter.evaluate(keyStr);
            context.remove(key);
            buf.append("removed key : ").append(key);
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXgɒlǉ郊NGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processPutResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        @SuppressWarnings("unchecked")
        SharedContext<Object> context = (SharedContext<Object>)ServiceManagerFactory.getService(serviceName);
        String keyStr = req.getParameter("key");
        if(keyStr.indexOf("\n") == -1
            && keyStr.indexOf("\r") == -1){
            keyStr = "return " + keyStr;
        }
        String valueStr = req.getParameter("value");
        if(valueStr.indexOf("\n") == -1
            && valueStr.indexOf("\r") == -1){
            valueStr = "return " + valueStr;
        }
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " Put</title></head>");
        buf.append("<body>");
        
        try{
            Object key = interpreter.evaluate(keyStr);
            Object value = interpreter.evaluate(valueStr);
            context.put(key, value);
            buf.append("put key : ").append(key).append(" value : ").append(value);
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXg̒lS폜郊NGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processClearResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        SharedContext<?> context = (SharedContext<?>)ServiceManagerFactory.getService(serviceName);
        String type = req.getParameter("type");
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " Clear</title></head>");
        buf.append("<body>");
        
        try{
            if("local".equals(type)){
                context.clearLocal();
            }else{
                context.clear();
            }
            buf.append("clear ").append(type);
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXgǂݍރNGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processLoadResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        SharedContext<?> context = (SharedContext<?>)ServiceManagerFactory.getService(serviceName);
        String keyStr = req.getParameter("key");
        if(keyStr != null
            && keyStr.length() != 0
            && keyStr.indexOf("\n") == -1
            && keyStr.indexOf("\r") == -1){
            keyStr = "return " + keyStr;
        }
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " Load</title></head>");
        buf.append("<body>");
        
        try{
            if(keyStr != null && keyStr.length() != 0){
                Object key = interpreter.evaluate(keyStr);
                context.load(key);
                buf.append("load ").append(key).append(" complete");
            }else{
                context.load();
                buf.append("load complete");
            }
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXg̃L[ǂݍރNGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processLoadKeyResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        SharedContext<?> context = (SharedContext<?>)ServiceManagerFactory.getService(serviceName);
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " LoadKey</title></head>");
        buf.append("<body>");
        
        try{
            context.loadKey();
            buf.append("load key complete");
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXgރNGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processSaveResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        SharedContext<?> context = (SharedContext<?>)ServiceManagerFactory.getService(serviceName);
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " Save</title></head>");
        buf.append("<body>");
        
        try{
            context.save();
            buf.append("save complete");
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
    
    /**
     * ReLXg̒lXV郊NGXgsB<p>
     *
     * @param req HTTPNGXg
     * @param resp HTTPX|X
     * @exception ServletException 
     * @exception IOException 
     */
    protected void processUpdateResponse(
        HttpServletRequest req,
        HttpServletResponse resp
    ) throws ServletException, IOException{
        final String serviceNameStr = req.getParameter("name");
        final ServiceNameEditor editor = new ServiceNameEditor();
        editor.setAsText(serviceNameStr);
        final ServiceName serviceName = (ServiceName)editor.getValue();
        @SuppressWarnings("unchecked")
        SharedContext<Object> context = (SharedContext<Object>)ServiceManagerFactory.getService(serviceName);
        String keyStr = req.getParameter("key");
        
        final StringBuilder buf = new StringBuilder();
        resp.setContentType("text/html;charset=UTF-8");
        buf.append("<html>");
        buf.append("<head><title>Nimbus SharedContext " + serviceName + " Update</title></head>");
        buf.append("<body>");
        
        try{
            Object key = interpreter.evaluate(keyStr);
            Object value = context.get(key);
            SharedContextValueDifference diff = null;
            if(value instanceof SharedContextRecord){
                SharedContextRecord record = (SharedContextRecord)value;
                String[] values = req.getParameterValues("value");
                for(int i = 0; i < values.length; i++){
                    if(values[i] != null && values[i].length() != 0){
                        String valueStr = values[i];
                        if(valueStr.indexOf("\n") == -1
                            && valueStr.indexOf("\r") == -1){
                            valueStr = "return " + valueStr;
                        }
                        Object updateValue = interpreter.evaluate(valueStr);
                        diff = record.updateProperty(i, updateValue, diff);
                    }
                }
            }else if(value instanceof SharedContextRecordList){
                SharedContextRecordList recordList = (SharedContextRecordList)value;
                for(int i = 0, imax = recordList.size(); i < imax; i++){
                    SharedContextRecord record = (SharedContextRecord)recordList.get(i);
                    String[] vals = req.getParameterValues("value_" + i);
                    if(vals != null && vals.length != 0){
                        for(int j = 0; j < vals.length; j++){
                            if(vals[j] != null && vals[j].length() != 0){
                                String valueStr = vals[j];
                                if(valueStr.indexOf("\n") == -1
                                    && valueStr.indexOf("\r") == -1){
                                    valueStr = "return " + valueStr;
                                }
                                Object updateValue = interpreter.evaluate(valueStr);
                                diff = record.updateProperty(j, updateValue, diff);
                            }
                        }
                    }
                }
            }
            if(diff != null){
                context.update(key, diff);
                buf.append("update " + key + " complete");
            }else{
                buf.append("no difference " + key);
            }
        }catch(Exception e){
            writeThrowable(buf, e);
        }
        
        buf.append("<hr>");
        buf.append("<a href=\"").append(getCurrentPath(req))
            .append("\">Contexts</a>");
        
        buf.append("</body>");
        buf.append("</html>");
        resp.getWriter().println(buf.toString());
    }
}
