/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.jmx;

import java.util.Collection;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.TreeMap;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.Iterator;
import java.util.Date;
import java.util.Set;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.io.Serializable;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.text.ParseException;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.MalformedObjectNameException;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXServiceURL;

import org.apache.commons.jexl2.Expression;
import org.apache.commons.jexl2.JexlEngine;
import org.apache.commons.jexl2.JexlContext;
import org.apache.commons.jexl2.MapContext;

import jp.ossc.nimbus.beans.PropertyAccess;
import jp.ossc.nimbus.core.ServiceName;
import jp.ossc.nimbus.core.ServiceBase;
import jp.ossc.nimbus.core.ServiceManagerFactory;
import jp.ossc.nimbus.daemon.Daemon;
import jp.ossc.nimbus.daemon.DaemonControl;
import jp.ossc.nimbus.daemon.DaemonRunnable;
import jp.ossc.nimbus.service.jndi.JndiFinder;
import jp.ossc.nimbus.service.log.Logger;
import jp.ossc.nimbus.service.writer.Category;
import jp.ossc.nimbus.service.writer.MessageWriteException;
import jp.ossc.nimbus.util.converter.Converter;

/**
 * MBeanĎT[rXB<p>
 *
 * @author M.Takata
 */
public class MBeanWatcherService extends ServiceBase implements DaemonRunnable<Map<String,Object>>, MBeanWatcherServiceMBean{
    
    private static final long serialVersionUID = -1421073056315791503L;
    
    protected ServiceName jndiFinderServiceName;
    protected JndiFinder jndiFinder;
    protected String rmiAdaptorName = DEFAULT_JMX_RMI_ADAPTOR_NAME;
    private String serviceURL;
    private Map<String, ?> jmxConnectorEnvironment;
    private long interval;
    private ServiceName categoryServiceName;
    private Map<String, Target> targetMap;
    private boolean isConnectOnStart;
    private String getValueErrorMessageId = MSG_ID_GET_VALUE_ERROR;
    private String connectErrorMessageId = MSG_ID_CONNECT_ERROR;
    private String writeErrorMessageId = MSG_ID_WRITE_ERROR;
    private boolean isMBeanSet;
    private ObjectName objectName;
    private QueryExp queryExp;
    
    private JMXConnector connector;
    private Category category;
    private Daemon watcher;
    private long startTime;
    private ThreadLocal<Map<String,Object>> threadContext = new ThreadLocal<Map<String,Object>>(){
        protected synchronized Map<String,Object> initialValue(){
            return new HashMap<String,Object>();
        }
    };
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setJndiFinderServiceName(ServiceName name){
        jndiFinderServiceName = name;
    }
    // MBeanWatcherServiceMBeanJavaDoc
    public ServiceName getJndiFinderServiceName(){
        return jndiFinderServiceName;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setRMIAdaptorName(String name){
        rmiAdaptorName = name;
    }
    // MBeanWatcherServiceMBeanJavaDoc
    public String getRMIAdaptorName(){
        return rmiAdaptorName;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setServiceURL(String url){
        serviceURL = url;
    }
    // MBeanWatcherServiceMBeanJavaDoc
    public String getServiceURL(){
        return serviceURL;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setJMXConnectorEnvironment(Map<String,?> env){
        jmxConnectorEnvironment = env;
    }
    // MBeanWatcherServiceMBeanJavaDoc
    public Map<String, ?> getJMXConnectorEnvironment(){
        return jmxConnectorEnvironment;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setInterval(long interval){
        this.interval = interval;
    }
    // MBeanWatcherServiceMBeanJavaDoc
    public long getInterval(){
        return interval;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setCategoryServiceName(ServiceName name){
        categoryServiceName = name;
    }
    // MBeanWatcherServiceMBeanJavaDoc
    public ServiceName getCategoryServiceName(){
        return categoryServiceName;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setConnectOnStart(boolean isConnect){
        isConnectOnStart = isConnect;
    }
    // MBeanWatcherServiceMBeanJavaDoc
    public boolean isConnectOnStart(){
        return isConnectOnStart;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setGetValueErrorMessageId(String id){
        getValueErrorMessageId = id;
    }
    // MBeanWatcherServiceMBeanJavaDoc
    public String getGetValueErrorMessageId(){
        return getValueErrorMessageId;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setConnectErrorMessageId(String id){
        connectErrorMessageId = id;
    }
    // MBeanWatcherServiceMBeanJavaDoc
    public String getConnectErrorMessageId(){
        return connectErrorMessageId;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setWriteErrorMessageId(String id){
        writeErrorMessageId = id;
    }
    // MBeanWatcherServiceMBeanJavaDoc
    public String getWriteErrorMessageId(){
        return writeErrorMessageId;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setMBeanSet(boolean isSet){
        isMBeanSet = isSet;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public boolean isMBeanSet(){
        return isMBeanSet;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setObjectName(String name) throws MalformedObjectNameException{
        objectName = new ObjectName(name);
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public String getObjectName(){
        return objectName == null ? null : objectName.toString();
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void setQueryExp(QueryExp exp){
        queryExp = exp;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public QueryExp getQueryExp(){
        return queryExp;
    }
    
    public void addTarget(Target target){
        if(targetMap == null){
            targetMap = new LinkedHashMap<String, Target>();
        }
        targetMap.put(target.getKey(), target);
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public Map<String, Target> getTargetMap(){
        return targetMap;
    }
    
    // MBeanWatcherServiceMBeanJavaDoc
    public void reset(){
        Iterator<Target> targets = targetMap.values().iterator();
        while(targets.hasNext()){
            Target target = targets.next();
            target.reset();
        }
    }
    
    public void startService() throws Exception{
        if(jndiFinderServiceName != null){
            jndiFinder = (JndiFinder)ServiceManagerFactory.getServiceObject(jndiFinderServiceName);
        }else if(serviceURL != null){
            if(isConnectOnStart){
                connector = JMXConnectorFactory.newJMXConnector(
                    new JMXServiceURL(serviceURL),
                    jmxConnectorEnvironment
                );
                connector.connect();
            }
        }else{
            throw new IllegalArgumentException("ServiceURL or jndiFinderServiceName must be specified.");
        }
        
        if(categoryServiceName != null){
            category = (Category)ServiceManagerFactory.getServiceObject(categoryServiceName);
        }
        for(Target target : targetMap.values()){
            target.setWatcherServiceName(getServiceNameObject());
            target.setWatcherService(this);
            target.setLogger(getLogger());
            target.start();
        }
        if(interval > 0){
            watcher = new Daemon(this);
            watcher.setName(
                "Nimbus MBeanWatcher " + getServiceNameObject()
            );
            watcher.setDaemon(true);
            watcher.start();
        }
    }
    
    public void stopService() throws Exception{
        if(watcher != null){
            watcher.stopNoWait();
            watcher = null;
        }
        if(connector != null){
            connector.close();
            connector = null;
        }
        for(Target target : targetMap.values()){
            target.stop();
        }
    }
    
    public Map<String,Object> watch() throws Exception{
        JMXConnector tmpConnector = null;
        try{
            MBeanServerConnection connection = null;
            if(jndiFinder != null){
                connection = (MBeanServerConnection)jndiFinder.lookup(rmiAdaptorName);
            }else{
                if(connector == null){
                    tmpConnector = JMXConnectorFactory.newJMXConnector(
                        new JMXServiceURL(serviceURL),
                        jmxConnectorEnvironment
                    );
                    tmpConnector.connect();
                    connection = tmpConnector.getMBeanServerConnection();
                }else{
                    connection = connector.getMBeanServerConnection();
                }
            }
            Map<String,Object> out = new LinkedHashMap<String,Object>();
            Iterator<Map.Entry<String, Target>> entries = targetMap.entrySet().iterator();
            while(entries.hasNext()){
                Map.Entry<String, Target> entry = entries.next();
                String key = entry.getKey();
                Target target = entry.getValue();
                try{
                    out.put(key, target.getValue(connection));
                }catch(Exception e){
                    if(getValueErrorMessageId != null){
                        getLogger().write(
                            getValueErrorMessageId,
                            target,
                            e
                        );
                    }
                }
            }
            return out;
        }finally{
            if(tmpConnector != null){
                try{
                    tmpConnector.close();
                }catch(IOException e){}
            }
        }
    }
    
    public void write() throws Exception{
        write(watch());
    }
    
    private void write(Map<String, Object> out) throws MessageWriteException{
        if(category != null){
            category.write(out);
        }
    }
    
    public boolean onStart(){return true;}
    public boolean onStop(){return true;}
    public boolean onSuspend(){return true;}
    public boolean onResume(){return true;}
    public Map<String, Object> provide(DaemonControl ctrl) throws Throwable{
        startTime = System.currentTimeMillis();
        try{
            return watch();
        }catch(Throwable e){
            if(connectErrorMessageId != null){
                getLogger().write(
                    connectErrorMessageId,
                    serviceURL,
                    e
                );
            }
            return null;
        }
    }
    public void consume(Map<String, Object> paramObj, DaemonControl ctrl) throws Throwable{
        if(paramObj == null){
            return;
        }
        try{
            write(paramObj);
        }catch(Throwable e){
            if(writeErrorMessageId != null){
                getLogger().write(
                    writeErrorMessageId,
                    paramObj,
                    e
                );
            }
        }
        final long processTime = System.currentTimeMillis() - startTime;
        final long sleepTime = interval - processTime;
        if(sleepTime > 0){
            Thread.sleep(sleepTime);
        }
    }
    public void garbage(){}
    
    protected void setContextValue(String name, Object value){
        Map<String,Object> context = threadContext.get();
        context.put(name, value);
    }
    
    protected Object getContextValue(String name){
        Map<String,Object> context = threadContext.get();
        return context.get(name);
    }
    
    /**
     * ĎΏہB<p>
     *
     * @author M.Takata
     */
    public static abstract class Target implements Serializable{
        
        private static final long serialVersionUID = 3184726605262675808L;
        
        protected String key;
        protected transient Logger logger;
        protected transient MBeanWatcherService watcher;
        protected ServiceName watcherServiceName;
        protected String contextKey;
        
        /**
         * {@link MBeanWatcherService}̃T[rXݒ肷B<p>
         *
         * @param name MBeanWatcherServicẽT[rX
         */
        protected void setWatcherServiceName(ServiceName name){
            this.watcherServiceName = name;
        }
        
        /**
         * {@link MBeanWatcherService}̃T[rX擾B<p>
         *
         * @return MBeanWatcherServicẽT[rX
         */
        protected ServiceName getWatcherServiceName(){
            return watcherServiceName;
        }
        
        /**
         * {@link MBeanWatcherService}ݒ肷B<p>
         *
         * @param watcher MBeanWatcherService
         */
        protected void setWatcherService(MBeanWatcherService watcher){
            this.watcher = watcher;
        }
        
        /**
         * {@link MBeanWatcherService}擾B<p>
         *
         * @return MBeanWatcherService
         */
        protected MBeanWatcherService getWatcherService(){
            return watcher;
        }
        
        /**
         * {@link Logger}ݒ肷B<p>
         *
         * @param logger Logger
         */
        protected void setLogger(Logger logger){
            this.logger = logger;
        }
        
        /**
         * {@link Logger}擾B<p>
         *
         * @return Logger
         */
        protected Logger getLogger(){
            return logger == null ? ServiceManagerFactory.getLogger() : logger;
        }
        
        /**
         * o͂ۂ̃L[擾B<p>
         *
         * @return L[
         */
        public String getKey(){
            return key;
        }
        
        /**
         * o͂ۂ̃L[ݒ肷B<p>
         *
         * @param key L[
         */
        public void setKey(String key){
            this.key = key;
        }
        
        /**
         * ReLXgɏo͂ۂ̃L[擾B<p>
         *
         * @return L[
         */
        public String getContextKey(){
            return contextKey;
        }
        
        /**
         * ReLXgɏo͂ۂ̃L[ݒ肷B<p>
         *
         * @param key L[
         */
        public void setContextKey(String key){
            contextKey = key;
        }
        
        /**
         * ĎΏۂ̒l擾B<p>
         *
         * @param connection JMXڑ
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        public abstract Object getValue(MBeanServerConnection connection) throws Exception;
        
        /**
         * lBigDecimalɕϊB<p>
         *
         * @param value l
         * @param isNullToZero lnullꍇɃ[Ƃ݂Ȃꍇ́AtrueBfalsȅꍇ́Alnull̏ꍇAnullԂB
         * @return ϊꂽBigDecimall
         * @exception NumberFormatException ϊɎsꍇ
         */
        protected static BigDecimal toBigDecimal(Object value, boolean isNullToZero) throws NumberFormatException{
            BigDecimal result = null;
            if(value == null){
                result = isNullToZero ? new BigDecimal(0.0d) : null;
            }else if(value instanceof BigDecimal){
                result = (BigDecimal)value;
            }else if(value instanceof BigInteger){
                result = new BigDecimal((BigInteger)value);
            }else if(value instanceof Double || value instanceof Float){
                result = new BigDecimal(((Number)value).doubleValue());
            }else if(value instanceof Number){
                result = BigDecimal.valueOf(((Number)value).longValue());
            }else{
                result = new BigDecimal(value.toString());
            }
            return result;
        }
        
        /**
         * ZbgB<p>
         */
        public void reset(){
        }
        
        /**
         * JnB<p>
         */
        public void start(){
        }
        
        /**
         * IB<p>
         */
        public void stop(){
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.append('{');
            buf.append("key=").append(getKey());
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * ^CX^vB<p>
     * Ď̃^CX^v擾B<br>
     *
     * @author M.Takata
     */
    public static class Timestamp extends Target{
        
        private static final long serialVersionUID = 3632167440398869434L;
        
        /**
         * L[̃ftHglB<p>
         */
        public static final String DEFAULT_KEY = "Timestamp";
        
        private String format;
        
        /**
         * ^CX^v𕶎ɕҏWꍇ̃tH[}bg擾B<p>
         *
         * @return tH[}bg
         */
        public String getFormat(){
            return format;
        }
        
        /**
         * ^CX^v𕶎ɕҏWꍇ̃tH[}bgݒ肷B<p>
         * ftHg͂ȂŁA^CX^vƂjava.util.DateIuWFNgԂB<br>
         *
         * @param format tH[}bg
         */
        public void setFormat(String format){
            this.format = format;
        }
        
        /**
         * o͂ۂ̃L[擾B<p>
         * {@link #setKey(String)}Őݒ肵ĂȂꍇ́A{@link #DEFAULT_KEY}ԂB<br>
         *
         * @return L[
         */
        public String getKey(){
            return super.getKey() != null ? super.getKey() : DEFAULT_KEY;
        }
        
        /**
         * ^CX^v擾B<p>
         * {@link #setFormat(String)}ŃtH[}bgw肵ĂȂꍇ́AĂяo_łjava.util.DateIuWFNgԂB<br>
         * {@link #setFormat(String)}ŃtH[}bgw肵Ăꍇ́AĂяo_łjava.util.DateIuWFNgtH[}bgĕԂB<br>
         *
         * @param connection JMXڑ
         * @return ^CX^v
         * @exception Exception tH[}bgɎsꍇ
         */
        public Object getValue(MBeanServerConnection connection) throws Exception{
            Object value = format == null ? (Object)new Date() : (Object)new SimpleDateFormat(format).format(new Date());
            if(value != null && contextKey != null){
                watcher.setContextValue(contextKey, value);
            }
            return value;
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",format=").append(format);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * ReLXgΏہB<p>
     * ReLXgl擾B<br>
     *
     * @author M.Takata
     */
    public static class Context extends Target{
        
        private static final long serialVersionUID = 566760345569101974L;
        
        /**
         * ReLXgl擾B<p>
         *
         * @param connection JMXڑ
         * @return ReLXg擾l
         * @exception Exception tH[}bgɎsꍇ
         */
        public Object getValue(MBeanServerConnection connection) throws Exception{
            return watcher.getContextValue(contextKey);
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",contextKey=").append(contextKey);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * Managed BeanĎΏہB<p>
     * Managed BeanĎΏۂƂĎΏۊNXB<br>
     *
     * @author M.Takata
     */
    public static abstract class MBeanTarget extends Target{
        private static final long serialVersionUID = -5180685937237509600L;
        protected boolean isMBeanSet;
        protected ObjectName objectName;
        protected QueryExp queryExp;
        
        /**
         * ĎΏۂManaged BeanWƂĈǂݒ肷B<p>
         * ftHǵAfalseŁAĎΏۂ́AӂManaged BeanB<br>
         *
         * @param isSet ĎΏۂManaged BeanWƂĈꍇtrue
         */
        public void setMBeanSet(boolean isSet){
            isMBeanSet = isSet;
        }
        
        /**
         * ĎΏۂManaged BeanWƂĈǂ𔻒肷B<p>
         *
         * @return truȅꍇAĎΏۂManaged BeanWƂĈ
         */
        public boolean isMBeanSet(){
            return isMBeanSet;
        }
        
        /**
         * ĎΏۂManaged Bean̖Oݒ肷B<p>
         * {@link #setMBeanSet(boolean) setMBeanSet(false)}Ɛݒ肵Ăꍇ́AManaged Beanӂɓ肷銮Sw肷B<br>
         * {@link #setMBeanSet(boolean) setMBeanSet(true)}Ɛݒ肵Ăꍇ́AManaged Bean̏W肷IuWFNgw肷B<br>
         *
         * @param name Managed Bean̖OJMX̃IuWFNg`Ŏw肷
         * @exception MalformedObjectNameException IuWFNgsȏꍇ
         */
        public void setObjectName(String name) throws MalformedObjectNameException{
            this.objectName = new ObjectName(name);
        }
        
        /**
         * ĎΏۂManaged Bean̖O擾B<p>
         *
         * @return Managed Bean̖OJMX̃IuWFNg`Ŏw肷
         */
        public String getObjectName(){
            return objectName == null ? null : objectName.toString();
        }
        
        /**
         * ĎΏۂManaged Beani荞ޏݒ肷B<p>
         * {@link #setMBeanSet(boolean) setMBeanSet(true)}̏ꍇ̂ݗLB<br>
         *
         * @param exp 
         */
        public void setQueryExp(QueryExp exp){
            queryExp = exp;
        }
        
        /**
         * ĎΏۂManaged Beani荞ޏ擾B<p>
         *
         * @return 
         */
        public QueryExp getQueryExp(){
            return queryExp;
        }
        
        /**
         * ĎΏۂManaged BeanĎΏۂ̒l擾B<p>
         * ĎΏۂManaged BeanW̏ꍇAL[IuWFNgAlĎΏۂ̒lƂȂ}bvԂB<br>
         * 
         * @param connection JMXڑ
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        public Object getValue(MBeanServerConnection connection) throws Exception{
            boolean tmpIsMBeanSet = isMBeanSet;
            ObjectName tmpObjectName = objectName;
            QueryExp tmpQueryExp = queryExp;
            if(tmpObjectName == null){
                tmpIsMBeanSet = watcher.isMBeanSet;
                tmpObjectName = watcher.objectName;
                tmpQueryExp = watcher.queryExp;
            }
            Object value = null;
            if(!tmpIsMBeanSet){
                value = getValue(connection, tmpObjectName);
            }else{
                Set<ObjectName> objectNameSet = connection.queryNames(tmpObjectName, tmpQueryExp);
                Map<String, Object> map = new LinkedHashMap<String, Object>();
                if(objectNameSet != null && objectNameSet.size() != 0){
                    Iterator<ObjectName> itr = objectNameSet.iterator();
                    while(itr.hasNext()){
                        ObjectName name = itr.next();
                        map.put(name.toString(), getValue(connection, name));
                    }
                }
                value = map;
            }
            if(value != null && contextKey != null){
                watcher.setContextValue(contextKey, value);
            }
            return value;
        }
        
        /**
         * w肳ꂽIuWFNgManaged BeanĎΏۂ̒l擾B<p>
         * 
         * @param connection JMXڑ
         * @param objectName IuWFNg
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        protected abstract Object getValue(MBeanServerConnection connection, ObjectName objectName) throws Exception;
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",objectName=").append(getObjectName());
            buf.append(",isMBeanSet=").append(isMBeanSet);
            buf.append(",queryExp=").append(queryExp);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * Managed Bean̑ĎΏۂƂ{@link MBeanWatcherService.MBeanTarget}̎NXB<p>
     *
     * @author M.Takata
     */
    public static class Attribute extends MBeanTarget{
        
        private static final long serialVersionUID = 253836219685470254L;
        private String name;
        
        /**
         * ĎΏۂƂManaged Bean̑ݒ肷B<p>
         *
         * @param name 
         */
        public void setName(String name){
            this.name = name;
        }
        
        /**
         * ĎΏۂƂManaged Bean̑擾B<p>
         *
         * @return 
         */
        public String getName(){
            return name;
        }
        
        /**
         * o͂ۂ̃L[擾B<p>
         * {@link #setKey(String)}Őݒ肵ĂȂꍇ́AIuWFNg#ԂB<br>
         *
         * @return L[
         */
        public String getKey(){
            return super.getKey() != null ? super.getKey() : getObjectName() + '#' + getName();
        }
        
        /**
         * w肳ꂽIuWFNgManaged BeanĎΏۂ̑l擾B<p>
         * 
         * @param connection JMXڑ
         * @param objectName IuWFNg
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        public Object getValue(MBeanServerConnection connection, ObjectName objectName) throws Exception{
            return connection.getAttribute(objectName, name);
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",name=").append(name);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * Managed BeañIy[VĎΏۂƂ{@link MBeanWatcherService.MBeanTarget}̎NXB<p>
     *
     * @author M.Takata
     */
    public static class Operation extends MBeanTarget{
        private static final long serialVersionUID = 8874947184358756744L;
        private String name;
        private List<Object> params;
        private String[] signiture;
        
        /**
         * ĎΏۂƂManaged BeañIy[Vݒ肷B<p>
         *
         * @param name Iy[V
         */
        public void setName(String name){
            this.name = name;
        }
        
        /**
         * ĎΏۂƂManaged BeañIy[V擾B<p>
         *
         * @return Iy[V
         */
        public String getName(){
            return name;
        }
        
        /**
         * ĎΏۂƂManaged BeañIy[ṼVOj`ݒ肷B<p>
         *
         * @param sgn Iy[ṼVOj`
         */
        public void setSigniture(String[] sgn){
            signiture = sgn;
        }
        
        /**
         * ĎΏۂƂManaged BeañIy[ṼVOj`擾B<p>
         *
         * @return Iy[ṼVOj`
         */
        public String[] getSigniture(){
            return signiture;
        }
        
        /**
         * ĎΏۂƂManaged BeañIy[V̈ݒ肷B<p>
         *
         * @param params Iy[V̈̔z
         */
        public void setParameters(Object[] params){
            if(this.params == null){
                this.params = new ArrayList<Object>();
            }
            for(int i = 0; i < params.length; i++){
                this.params.add(params[i]);
            }
        }
        
        /**
         * ĎΏۂƂManaged BeañIy[V̈ǉB<p>
         *
         * @param param Iy[V̈
         */
        public void addParameter(Object param){
            if(params == null){
                params = new ArrayList<Object>();
            }
            params.add(param);
        }
        
        /**
         * ĎΏۂƂManaged BeañIy[V̈Xg擾B<p>
         *
         * @return Iy[V̈Xg
         */
        public List<Object> getParameterList(){
            return params;
        }
        
        /**
         * o͂ۂ̃L[擾B<p>
         * {@link #setKey(String)}Őݒ肵ĂȂꍇ́AIuWFNg#Iy[V([p[^1,p[^2,....])ԂB<br>
         *
         * @return L[
         */
        public String getKey(){
            return super.getKey() != null ? super.getKey() : getObjectName() + '#' + getName() + '(' + (params == null ? "" : params.toString()) + ')';
        }
        
        /**
         * w肳ꂽIuWFNgManaged BeanĎΏۂ̃Iy[V̖߂l擾B<p>
         * 
         * @param connection JMXڑ
         * @param objectName IuWFNg
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        public Object getValue(MBeanServerConnection connection, ObjectName objectName) throws Exception{
            return connection.invoke(objectName, name, params == null ? null : params.toArray(), signiture);
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",name=").append(name);
            buf.append(",params=").append(params);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * bvĎΏہB<p>
     * ĎΏۂbvē̏tĎΏۊNXB<br>
     *
     * @author M.Takata
     */
    public abstract static class WrapTarget extends Target{
        
        private static final long serialVersionUID = 6038448327332359108L;
        protected Target target;
        
        /**
         * {@link MBeanWatcherService}̃T[rXݒ肷B<p>
         * bvĂĎΏۂɂMBeanWatcherServicẽT[rXݒ肷B<br>
         *
         * @param name MBeanWatcherServicẽT[rX
         */
        public void setWatcherServiceName(ServiceName name){
            super.setWatcherServiceName(name);
            target.setWatcherServiceName(name);
        }
        
        /**
         * {@link MBeanWatcherService}ݒ肷B<p>
         * bvĂĎΏۂɂMBeanWatcherServiceݒ肷B<br>
         *
         * @param watcher MBeanWatcherService
         */
        protected void setWatcherService(MBeanWatcherService watcher){
            super.setWatcherService(watcher);
            target.setWatcherService(watcher);
        }
        
        /**
         * {@link Logger}ݒ肷B<p>
         * bvĂĎΏۂɂLoggerݒ肷B<br>
         * 
         * @param logger Logger
         */
        public void setLogger(Logger logger){
            super.setLogger(logger);
            target.setLogger(logger);
        }
        
        /**
         * bvĎΏ{@link Logger}ݒ肷B<p>
         * 
         * @param target ĎΏ
         */
        public void setTarget(Target target){
            this.target = target;
        }
        
        /**
         * bvĎΏ{@link Logger}擾B<p>
         * 
         * @return ĎΏ
         */
        public Target getTarget(){
            return target;
        }
        
        /**
         * ZbgB<p>
         * bvĂĎΏۂZbgB<br>
         */
        public void reset(){
            super.reset();
            target.reset();
        }
        
        /**
         * JnB<p>
         * bvĂĎΏۂJnB<br>
         */
        public void start(){
            super.start();
            target.start();
        }
        
        /**
         * IB<p>
         * bvĂĎΏۂIB<br>
         */
        public void stop(){
            target.stop();
            super.stop();
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",target=").append(target);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * bvĎΏۂ̒l`FbN{@link MBeanWatcherService.WrapTarget}̎NXB<br>
     *
     * @author M.Takata
     */
    public static class Check extends WrapTarget{
        
        private static final long serialVersionUID = -8821714143031466189L;
        private boolean isNullToZero;
        private ServiceName loggerServiceName;
        private List<Condition> checkConditions = new ArrayList<Condition>();
        
        /**
         * ĎΏۂl^null̏ꍇɁA[Ƃ݂Ȃǂݒ肷B<p>
         * ftHǵAfalseŃ[Ƃ݂ȂȂB<br>
         *
         * @return [Ƃ݂ȂꍇAtrue
         */
        public void setNullToZero(boolean isNullToZero){
            this.isNullToZero = isNullToZero;
        }
        
        /**
         * ĎΏۂl^null̏ꍇɁA[Ƃ݂Ȃǂ𔻒肷B<p>
         *
         * @return truȅꍇA[Ƃ݂Ȃ
         */
        public boolean isNullToZero(){
            return isNullToZero;
        }
        
        /**
         * `FbNG[̃Oo͂{@link Logger}T[rX̃T[rXݒ肷B<p>
         * w肵Ȃꍇ́ÃT[rX{@link ServiceBase#getLogger()}Ŏ擾LoggerŃOo͂sB<br>
         *
         * @param name LoggerT[rX̃T[rX
         */
        public void setLoggerServiceName(ServiceName name){
            loggerServiceName = name;
        }
        
        /**
         * `FbNG[̃Oo͂{@link Logger}T[rX̃T[rX擾B<p>
         *
         * @return LoggerT[rX̃T[rX
         */
        public ServiceName getLoggerServiceName(){
            return loggerServiceName;
        }
        
        /**
         * {@link Logger}擾B<p>
         * {@link #setLoggerServiceName(ServiceName)}ݒ肳Ăꍇ́ALoggerԂB<br>
         *
         * @return Logger
         */
        public Logger getLogger(){
            if(loggerServiceName == null){
                return super.getLogger();
            }else{
                return (Logger)ServiceManagerFactory.getServiceObject(loggerServiceName);
            }
        }
        
        /**
         * `FbNǉB<p>
         * ǉꂽԂɃ`FbNāA`FbNG[ɂȂƌ㑱̏̓`FbNɃZbgB<br>
         *
         * @param condition `FbN
         */
        public void addCheckCondition(Condition condition){
            checkConditions.add(condition);
        }
        
        /**
         * `FbÑXg擾B<p>
         *
         * @return `FbÑXg
         */
        public List<Condition> getCheckConditionList(){
            return checkConditions;
        }
        
        /**
         * bvĎΏۂ̃Iy[V̖߂l擾A`FbNsă`FbNG[̏ꍇ̓Oo͂B<p>
         * `FbNǉꂽԂɃ`FbNāA`FbNG[ɂȂƌ㑱̏̓`FbNɃZbgB<br>
         * 
         * @param connection JMXڑ
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        public Object getValue(MBeanServerConnection connection) throws Exception{
            Object value = target.getValue(connection);
            if(isNullToZero){
                value = Target.toBigDecimal(value, isNullToZero);
            }
            boolean checkError = false;
            for(int i = 0, imax = checkConditions.size(); i < imax; i++){
                Condition condition = (Condition)checkConditions.get(i);
                if(checkError){
                    condition.reset();
                }else if(!condition.check(value, getLogger(), getWatcherServiceName(), getKey())){
                    checkError = true;
                }
            }
            if(value != null && contextKey != null){
                watcher.setContextValue(contextKey, value);
            }
            return value;
        }
        
        /**
         * ZbgB<p>
         * bvĂĎΏۋyсAĎΏۂ̃`FbNG[ԂZbgB<br>
         */
        public void reset(){
            super.reset();
            for(int i = 0, imax = checkConditions.size(); i < imax; i++){
                Condition condition = (Condition)checkConditions.get(i);
                condition.reset();
            }
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append("checkConditions=").append(checkConditions);
            buf.append(",isNullToZero=").append(isNullToZero);
            buf.append(",loggerServiceName=").append(loggerServiceName);
            buf.append('}');
            return buf.toString();
        }
        
        private static class ErrorCountComparator implements Comparator<Integer>, Serializable{
            
            private static final long serialVersionUID = -8997521049688710362L;
            
            public int compare(Integer o1, Integer o2){
                return o2.compareTo(o1);
            }
        }
        
        /**
         * `FbNB<br>
         *
         * @author M.Takata
         */
        public static class Condition implements Serializable{
            
            private static final long serialVersionUID = 451166286622455937L;
            
            public static final String VALUE = "value";
            public static final String PROP_FUNCTION_NAME = "prop";
            
            private Expression checkExpression;
            private String expression;
            private Map<Integer,String> idMap = new TreeMap<Integer,String>(new ErrorCountComparator());
            private int errorCount;
            private List<CheckTime> checkTimes;
            private List<Term> checkTerms;
            
            /**
             * ĎΏۂ`FbNݒ肷B<p>
             * ́AThe Apache Jakarta Project Commons Jexl(http://jakarta.apache.org/commons/jexl/)gpB<br>
             * ĎΏےl\"value"L[[h܂߁AĎΏےl]booleanԂƂB<br>
             * ̕]trueɂȂƁA`FbNG[Ƃ݂ȂB<br>
             * 
             * @param expression 
             * @exception Exception sȏꍇ
             */
            public void setCheckExpression(String expression) throws Exception{
                JexlEngine jexl = new JexlEngine();
                jexl.setSilent(true);
                Map<String, Object> funcs = new HashMap<String, Object>();
                PropertyAccess propAccess = new PropertyAccess();
                propAccess.setIgnoreNullProperty(true);
                funcs.put(PROP_FUNCTION_NAME, propAccess);
                jexl.setFunctions(funcs);
                checkExpression = jexl.createExpression(expression);
                this.expression = expression;
            }
            
            /**
             * ĎΏۂ`FbN擾B<p>
             * 
             * @return 
             */
            public String getCheckExpression(){
                return expression;
            }
            
            /**
             * `FbNG[ɏo͂郍ÕbZ[WIDݒ肷B<p>
             * {@link #setLogMessageIdByErrorCount(int, String) setLogMessageIdByErrorCount(1, id)}ŌĂяôƓB<br>
             *
             * @param id ÕbZ[WID
             */
            public void setLogMessageId(String id){
                setLogMessageIdByErrorCount(1, id);
            }
            
            /**
             * `FbNG[w肵񐔘AĔɏo͂郍ÕbZ[WIDݒ肷B<p>
             * `FbNG[̉񐔂́A`FbNG[AĔȂꍇƁAw肳ꂽ`FbNG[񐔂̍ő̒l̉񐔂܂őƃZbgB<br>
             * Oo͂̍ۂɂ́AT[rXAL[AAĎΏۂ̒lAAĔ`FbNG[̉񐔂Oo͂̈ƂēnB<br>
             * ėp̃ObZ[WIDƂāA{@link #MSG_ID_CHECK_WARN}A{@link #MSG_ID_CHECK_ERROR}A{@link #MSG_ID_CHECK_FATAL}pӂĂB<br>
             *
             * @param errorCount `FbNG[
             * @param id ÕbZ[WID
             */
            public void setLogMessageIdByErrorCount(int errorCount, String id){
                idMap.put(new Integer(errorCount), id);
            }
            
            /**
             * `FbNs鎞ݒ肷B<p>
             *
             * @param times HH:mm:ss܂́AHH:mm:ss.SSS̎z
             * @exception ParseException w肳ꂽ̃p[XɎsꍇ
             */
            public void setCheckTimes(String[] times) throws ParseException{
                if(times == null || times.length == 0){
                    checkTimes = null;
                }else{
                    checkTimes = new ArrayList<CheckTime>();
                    for(int i = 0; i < times.length; i++){
                        CheckTime ct = new CheckTime(times[i]);
                        checkTimes.add(ct);
                    }
                    Collections.sort(checkTimes);
                }
            }
            
            /**
             * `FbNs鎞擾B<p>
             *
             * @return HH:mm:ss܂́AHH:mm:ss.SSS̎z
             */
            public String[] getCheckTimes(){
                if(checkTimes == null){
                    return null;
                }
                String[] result = new String[checkTimes.size()];
                for(int i = 0; i < checkTimes.size(); i++){
                    result[i] = checkTimes.get(i).toString();
                }
                return result;
            }
            
            /**
             * `FbNsԂݒ肷B<p>
             *
             * @param terms HH:mm:ss܂́AHH:mm:ss.SSS̎-ŘAJn-I̕zBI͊ԂɊ܂܂ȂB
             * @exception ParseException w肳ꂽ̃p[XɎsꍇ
             */
            public void setCheckTerms(String[] terms) throws ParseException{
                if(terms == null || terms.length == 0){
                    checkTerms = null;
                }else{
                    checkTerms = new ArrayList<Term>();
                    for(int i = 0; i < terms.length; i++){
                        Term term = new Term(terms[i]);
                        checkTerms.add(term);
                    }
                    Collections.sort(checkTerms);
                }
            }
            
            /**
             * `FbNsԂ擾B<p>
             *
             * @return HH:mm:ss܂́AHH:mm:ss.SSS̎-ŘAJn-I̕zBI͊ԂɊ܂܂ȂB
             */
            public String[] getCheckTerms(){
                if(checkTerms == null){
                    return null;
                }
                String[] result = new String[checkTerms.size()];
                for(int i = 0; i < checkTerms.size(); i++){
                    result[i] = checkTerms.get(i).toString();
                }
                return result;
            }
            
            /**
             * w肳ꂽl̃`FbNsă`FbNG[ƃOo͂B<p>
             *
             * @param value `FbNΏۂ̒l
             * @param logger Oo͂sLogger
             * @param watcherServiceName ĎT[rX̃T[rX
             * @param key ĎΏۂ̃L[
             * @return `FbNʁB`FbNG[̏ꍇfalse
             * @exception Exception `FbNɎsꍇ
             */
            protected boolean check(Object value, Logger logger, ServiceName watcherServiceName, String key) throws Exception{
                List<CheckTime> currentCheckTimes = null;
                Day nowDay = null;
                Time nowTime = null;
                if(checkTimes != null){
                    Calendar now = Calendar.getInstance();
                    nowDay = new Day(now);
                    nowTime = new Time(now);
                    for(int i = 0; i < checkTimes.size(); i++){
                        CheckTime ct = checkTimes.get(i);
                        if(ct.isChecked(nowDay) || nowTime.compareTo(ct) < 0){
                            continue;
                        }
                        if(currentCheckTimes == null){
                            currentCheckTimes = new ArrayList<CheckTime>();
                        }
                        currentCheckTimes.add(ct);
                    }
                    if(currentCheckTimes == null){
                        return true;
                    }
                }
                if(currentCheckTimes == null && checkTerms != null){
                    if(nowTime == null){
                        nowTime = new Time(Calendar.getInstance());
                    }
                    boolean check = false;
                    for(int i = 0; i < checkTerms.size(); i++){
                        Term term = checkTerms.get(i);
                        if(!term.contains(nowTime)){
                            continue;
                        }
                        check = true;
                    }
                    if(!check){
                        return true;
                    }
                }
                JexlContext jexlContext = new MapContext();
                jexlContext.set(VALUE, value);
                Boolean error = (Boolean)checkExpression.evaluate(jexlContext);
                if(error.booleanValue()){
                    errorCount++;
                    boolean isFirst = true;
                    Iterator<Map.Entry<Integer, String>> entries = idMap.entrySet().iterator();
                    while(entries.hasNext()){
                        Map.Entry<Integer, String> entry = entries.next();
                        Integer count = (Integer)entry.getKey();
                        if(errorCount >= count.intValue()){
                            if(isFirst || errorCount == count.intValue()){
                                logger.write((String)entry.getValue(), watcherServiceName, key, expression, value, count);
                            }
                            if(isFirst){
                                errorCount = 0;
                                if(currentCheckTimes != null){
                                    for(int i = 0; i < currentCheckTimes.size(); i++){
                                        CheckTime ct = currentCheckTimes.get(i);
                                        ct.setCheckDay(nowDay);
                                    }
                                }
                            }
                            return false;
                        }
                        isFirst = false;
                    }
                }else{
                    errorCount = 0;
                }
                if(currentCheckTimes != null){
                    for(int i = 0; i < currentCheckTimes.size(); i++){
                        CheckTime ct = currentCheckTimes.get(i);
                        ct.setCheckDay(nowDay);
                    }
                }
                return true;
            }
            
            /**
             * ĎΏۂ̃`FbNG[񐔂ZbgB<br>
             */
            protected void reset(){
                errorCount = 0;
            }
            
            /**
             * ̊ĎΏۂ̕\擾B<p>
             *
             * @return \
             */
            public String toString(){
                StringBuilder buf = new StringBuilder();
                buf.append("Condition{");
                buf.append(",expression=").append(expression);
                buf.append(",idMap=").append(idMap);
                buf.append('}');
                return buf.toString();
            }
            
            private class Day implements Serializable, Comparable<Day>{
                
                private static final long serialVersionUID = -3482088973138104434L;
                private int day;
                
                public Day(Calendar now){
                    day += (now.get(Calendar.YEAR) * 1000);
                    day += now.get(Calendar.DAY_OF_YEAR);
                }
                
                public int hashCode(){
                    return day;
                }
                
                public boolean equals(Object obj){
                    if(obj == this){
                        return true;
                    }
                    if(obj == null || !(obj instanceof Day)){
                        return false;
                    }
                    Day cmp = (Day)obj;
                    return day == cmp.day;
                }
                
                public int compareTo(Day cmp){
                    if(day == cmp.day){
                        return 0;
                    }else{
                        return day > cmp.day ? 1 : -1;
                    }
                }
                
                public String toString(){
                    return Integer.toString(day);
                }
            }
            
            private class Time implements Serializable, Comparable<Time>{
                
                private static final long serialVersionUID = 5738491787638885636L;
                private static final String TIME_FORMAT1 = "HH:mm:ss";
                private static final String TIME_FORMAT2 = "HH:mm:ss.SSS";
                
                private int time;
                
                public Time(Calendar now){
                    time += (now.get(Calendar.HOUR_OF_DAY) * 10000000);
                    time += (now.get(Calendar.MINUTE) * 100000);
                    time += (now.get(Calendar.SECOND) * 1000);
                    time += now.get(Calendar.MILLISECOND);
                }
                
                public Time(String time) throws ParseException{
                    Date date = time.length() == 8 ? new SimpleDateFormat(TIME_FORMAT1).parse(time) : new SimpleDateFormat(TIME_FORMAT2).parse(time);
                    Calendar cal = Calendar.getInstance();
                    cal.setTime(date);
                    this.time += (cal.get(Calendar.HOUR_OF_DAY) * 10000000);
                    this.time += (cal.get(Calendar.MINUTE) * 100000);
                    this.time += (cal.get(Calendar.SECOND) * 1000);
                    this.time += cal.get(Calendar.MILLISECOND);
                }
                
                public int hashCode(){
                    return time;
                }
                
                public boolean equals(Object obj){
                    if(obj == this){
                        return true;
                    }
                    if(obj == null || !(obj instanceof Time)){
                        return false;
                    }
                    Time cmp = (Time)obj;
                    return time == cmp.time;
                }
                
                public int compareTo(Time cmp){
                    if(time == cmp.time){
                        return 0;
                    }else{
                        return time > cmp.time ? 1 : -1;
                    }
                }
                
                public String toString(){
                    StringBuilder buf = new StringBuilder();
                    buf.append(time);
                    int length = buf.length();
                    buf.insert(length - 3, '.');
                    buf.insert(length - 5, ':');
                    buf.insert(length - 7, ':');
                    return buf.toString();
                }
            }
            
            private class Term implements Serializable, Comparable<Term>{
                private static final long serialVersionUID = 4874635307721480812L;
                private Time from;
                private Time to;
                public Term(String term) throws ParseException{
                    if(term.indexOf('-') == -1){
                        throw new ParseException("Format is 'from-to' : " + term, term.length());
                    }
                    final String[] times = term.split("-");
                    if(times.length != 2){
                        throw new ParseException("Format is 'from-to' : " + term, term.length());
                    }
                    from = new Time(times[0]);
                    to = new Time(times[1]);
                }
                
                public boolean contains(Time time){
                    return time.compareTo(from) >= 0 && time.compareTo(to) < 0;
                }
                
                public int hashCode(){
                    return from.hashCode() + to.hashCode();
                }
                
                public boolean equals(Object obj){
                    if(obj == this){
                        return true;
                    }
                    if(obj == null || !(obj instanceof Term)){
                        return false;
                    }
                    Term cmp = (Term)obj;
                    return from.equals(cmp.from) && to.equals(cmp.to);
                }
                
                public int compareTo(Term cmp){
                    return from.compareTo(cmp.from);
                }
                
                public String toString(){
                    StringBuilder buf = new StringBuilder();
                    return buf.append(from).append('-').append(to).toString();
                }
            }
            
            private class CheckTime extends Time{
                private static final long serialVersionUID = 5309699914961118492L;
                private Day checkDay;
                public CheckTime(String time) throws ParseException{
                    super(time);
                }
                
                public boolean isChecked(Day currentDay){
                    if(checkDay == null){
                        return false;
                    }
                    return currentDay.compareTo(checkDay) <= 0;
                }
                
                public void setCheckDay(Day day){
                    checkDay = day;
                }
            }
        }
    }
    
    /**
     * ҏWĎΏہB<p>
     * ĎΏۂbvāAĎΏۂ̒lɕҏWsĎΏۊNXB<br>
     *
     * @author M.Takata
     */
    public abstract static class EditTarget extends WrapTarget{
        
        private static final long serialVersionUID = -3292286495172239503L;
        
        protected boolean isElementEdit;
        
        /**
         * ĎΏۂ̒lWz̏ꍇɁÅevfɑ΂ĕҏWsǂݒ肷B<p>
         * ftHǵAfalseŗvfɑ΂ҏW͍sȂB<br>
         *
         * @param isElement evfɑ΂ĕҏWsꍇtrue
         */
        public void setElementEdit(boolean isElement){
            isElementEdit = isElement;
        }
        
        /**
         * ĎΏۂ̒lWz̏ꍇɁÅevfɑ΂ĕҏWsǂ𔻒肷B<p>
         *
         * @return truȅꍇAevfɑ΂ĕҏWs
         */
        public boolean isElementEdit(){
            return isElementEdit;
        }
        
        /**
         * bvĎΏۂ̒lҏWĎ擾B<p>
         * 
         * @param connection JMXڑ
         * @param objectName IuWFNg
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        public Object getValue(MBeanServerConnection connection) throws Exception{
            Object value = target.getValue(connection);
            if(isElementEdit){
                Object[] array = null;
                if(value instanceof Map<?, ?>){
                    array = ((Map<?, ?>)value).values().toArray();
                }else if(value instanceof Collection<?>){
                    array = ((Collection<?>)value).toArray();
                }else if(value.getClass().isArray()){
                    array = (Object[])value;
                }else{
                    value = edit(value);
                }
                if(array != null){
                    for(int i = 0; i < array.length; i++){
                        array[i] = edit(array[i]);
                    }
                    value = array;
                }
            }else{
                value = edit(value);
            }
            if(value != null && contextKey != null){
                watcher.setContextValue(contextKey, value);
            }
            return value;
        }
        
        /**
         * w肳ꂽlҏWB<p>
         *
         * @param value ҏWΏۂ̒l
         * @return ҏWʂ̒l
         * @exception Exception ҏWɎsꍇ
         */
        protected abstract Object edit(Object value) throws Exception;
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",isElementEdit=").append(isElementEdit);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * bvĎΏۂ̒l̃vpeB擾{@link MBeanWatcherService.EditTarget}̎NXB<p>
     *
     * @author M.Takata
     */
    public static class Property extends EditTarget{
        
        private static final long serialVersionUID = 5990303499820230369L;
        private jp.ossc.nimbus.beans.Property property;
        private boolean isIgnoreNullProperty;
        
        /**
         * bvĎΏۂ̒l擾vpeBݒ肷B<p>
         *
         * @param property vpeB
         */
        public void setProperty(jp.ossc.nimbus.beans.Property property){
            property.setIgnoreNullProperty(isIgnoreNullProperty);
            this.property = property;
        }
        
        /**
         * bvĎΏۂ̒l擾vpeB擾B<p>
         *
         * @return vpeB
         */
        public jp.ossc.nimbus.beans.Property getProperty(){
            return property;
        }
        
        /**
         * nullQƂ̃vpeB擾gpƂꍇɁAOthrow邩ǂݒ肷B<p>
         * ftHǵAfalseB<br>
         *
         * @param isIgnore nullQƂ̎ɗOthrowȂꍇtrue
         */
        public void setIgnoreNullProperty(boolean isIgnore){
            isIgnoreNullProperty = isIgnore;
            if(property != null){
                property.setIgnoreNullProperty(isIgnoreNullProperty);
            }
        }
        
        /**
         * nullQƂ̃vpeB擾gpƂꍇɁAOthrow邩ǂ𔻒肷B<p>
         *
         * @return truȅꍇAnullQƂ̎ɗOthrowȂ
         */
        public boolean isIgnoreNullProperty(){
            return isIgnoreNullProperty;
        }
        
        /**
         * w肳ꂽlݒ肳ꂽvpeB擾B<p>
         *
         * @param value ҏWΏۂ̒l
         * @return ҏWʂ̒l
         * @exception Exception ҏWɎsꍇ
         */
        protected Object edit(Object value) throws Exception{
            return property.getProperty(value);
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",property=").append(property);
            buf.append(",isIgnoreNullProperty=").append(isIgnoreNullProperty);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * bvĎΏۂ̒l{@link Converter}ŕϊ{@link MBeanWatcherService.EditTarget}̎NXB<p>
     *
     * @author M.Takata
     */
    public static class Convert extends EditTarget{
        
        private static final long serialVersionUID = 1737702373890781789L;
        private ServiceName converterServiceName;
        private Converter converter;
        
        /**
         * bvĎΏۂ̒lϊ{@link Converter}ݒ肷B<p>
         *
         * @param converter Converter
         */
        public void setConverter(Converter converter){
            this.converter = converter;
        }
        
        /**
         * bvĎΏۂ̒lϊ{@link Converter}擾B<p>
         *
         * @return Converter
         */
        public Converter getConverter(){
            return converter;
        }
        
        /**
         * bvĎΏۂ̒lϊ{@link Converter}T[rX̃T[rXݒ肷B<p>
         *
         * @param name ConverterT[rX̃T[rX
         */
        public void setConverterServiceName(ServiceName name){
            converterServiceName = name;
        }
        
        /**
         * bvĎΏۂ̒lϊ{@link Converter}T[rX̃T[rX擾B<p>
         *
         * @return ConverterT[rX̃T[rX
         */
        public ServiceName getConverterServiceName(){
            return converterServiceName;
        }
        
        /**
         * w肳ꂽlݒ肳ꂽ{@link Converter}ŕϊB<p>
         *
         * @param value ҏWΏۂ̒l
         * @return ҏWʂ̒l
         * @exception Exception ҏWɎsꍇ
         */
        protected Object edit(Object value) throws Exception{
            Converter conv = converter;
            if(converterServiceName != null){
                conv = (Converter)ServiceManagerFactory.getServiceObject(converterServiceName);
            }
            return conv.convert(value);
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",converter=").append(converter);
            buf.append(",converterServiceName=").append(converterServiceName);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * bvĎΏۂ̒lƑO̒l̕ω擾{@link MBeanWatcherService.WrapTarget}̎NXB<p>
     *
     * @author M.Takata
     */
    public static class Change extends WrapTarget{
        
        private static final long serialVersionUID = 315703698395044669L;
        private boolean isNullToZero;
        private BigDecimal lastValue;
        
        /**
         * bvĎΏۂ̒ll^null̏ꍇɁA[Ƃ݂Ȃǂݒ肷B<p>
         * ftHǵAfalseŃ[Ƃ݂ȂȂB<br>
         *
         * @return [Ƃ݂ȂꍇAtrue
         */
        public void setNullToZero(boolean isNullToZero){
            this.isNullToZero = isNullToZero;
        }
        
        /**
         * bvĎΏۂ̒ll^null̏ꍇɁA[Ƃ݂Ȃǂ𔻒肷B<p>
         *
         * @return truȅꍇA[Ƃ݂Ȃ
         */
        public boolean isNullToZero(){
            return isNullToZero;
        }
        
        /**
         * bvĎΏۂ̒lO̒lO񍷂擾B<p>
         * 
         * @param connection JMXڑ
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        public Object getValue(MBeanServerConnection connection) throws Exception{
            BigDecimal value = Target.toBigDecimal(target.getValue(connection), isNullToZero);
            if(lastValue == null){
                lastValue = value;
                value = isNullToZero ? new BigDecimal(0d) : null;
            }else{
                BigDecimal result = value.subtract(lastValue);
                lastValue = value;
                value = result;
            }
            if(value != null && contextKey != null){
                watcher.setContextValue(contextKey, value);
            }
            return value;
        }
        
        /**
         * ZbgB<p>
         * bvĂĎΏۋyсAO̒lZbgB<br>
         */
        public void reset(){
            super.reset();
            lastValue = null;
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",isNullToZero=").append(isNullToZero);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * bvĎΏۂ̒lԊuŎ擾ȀW擾{@link MBeanWatcherService.WrapTarget}̎NXB<p>
     *
     * @author M.Takata
     */
    public static class Period extends WrapTarget implements DaemonRunnable<Object>{
        
        private static final long serialVersionUID = 6105816116626297923L;
        private long interval = 1000l;
        private int count = 60;
        private List<Object> valueList = new ArrayList<Object>();
        private Daemon periodicGetter;
        private long startTime;
        
        /**
         * bvĎΏۂ̒l擾Ԋu[ms]ݒ肷B<p>
         * ftHǵA1000[ms]B<br>
         *
         * @param millis l擾Ԋu[ms]
         */
        public void setInterval(long millis){
            interval = millis;
        }
        
        /**
         * bvĎΏۂ̒l擾Ԋu[ms]擾B<p>
         *
         * @return l擾Ԋu[ms]
         */
        public long getInterval(){
            return interval;
        }
        
        /**
         * bvĎΏۂ̒l擾錏ݒ肷B<p>
         * ftHǵA60B<br>
         *
         * @param cnt l擾錏
         */
        public void setCount(int cnt){
            count = cnt;
        }
        
        /**
         * bvĎΏۂ̒l擾錏擾B<p>
         *
         * @return l擾錏
         */
        public int getCount(){
            return count;
        }
        
        /**
         * bvĎΏۂ̒lԊuŎ擾W擾B<p>
         * 
         * @param connection JMXڑ
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        public Object getValue(MBeanServerConnection connection) throws Exception{
            List<Object> value = null;
            synchronized(valueList){
                value = new ArrayList<Object>(valueList);
            }
            if(value != null && contextKey != null){
                watcher.setContextValue(contextKey, value);
            }
            return value;
        }
        
        /**
         * JnB<p>
         * l̎擾JnB<br>
         */
        public void start(){
            super.start();
            periodicGetter = new Daemon(this);
            periodicGetter.setName(
                "Nimbus MBeanWatcher periodic value getter " + getWatcherServiceName() + target
            );
            periodicGetter.setDaemon(true);
            periodicGetter.start();
        }
        
        /**
         * IB<p>
         * l̎擾~B<br>
         */
        public void stop(){
            if(periodicGetter != null){
                periodicGetter.stopNoWait();
                periodicGetter = null;
            }
            synchronized(valueList){
                valueList.clear();
            }
            super.stop();
        }
        
        /**
         * ZbgB<p>
         * bvĂĎΏۋyсA擾lZbgB<br>
         */
        public void reset(){
            super.reset();
            synchronized(valueList){
                valueList.clear();
            }
        }
        public boolean onStart(){return true;}
        public boolean onStop(){return true;}
        public boolean onSuspend(){return true;}
        public boolean onResume(){return true;}
        public Object provide(DaemonControl ctrl) throws Throwable{
            startTime = System.currentTimeMillis();
            JMXConnector tmpConnector = null;
            try{
                MBeanServerConnection connection = null;
                if(watcher.jndiFinder != null){
                    connection = (MBeanServerConnection)watcher.jndiFinder.lookup(watcher.rmiAdaptorName);
                }else{
                    if(watcher.connector == null){
                        tmpConnector = JMXConnectorFactory.newJMXConnector(
                            new JMXServiceURL(watcher.serviceURL),
                            watcher.jmxConnectorEnvironment
                        );
                        tmpConnector.connect();
                        connection = tmpConnector.getMBeanServerConnection();
                    }else{
                        connection = watcher.connector.getMBeanServerConnection();
                    }
                }
                return target.getValue(connection);
            }catch(Exception e){
                return null;
            }finally{
                if(tmpConnector != null){
                    try{
                        tmpConnector.close();
                    }catch(IOException e){}
                }
            }
        }
        public void consume(Object paramObj, DaemonControl ctrl) throws Throwable{
            synchronized(valueList){
                valueList.add(paramObj);
                if(valueList.size() > count){
                    for(int i = 0, imax = count - valueList.size(); i < imax; i++){
                        valueList.remove(0);
                    }
                }
            }
            final long processTime = System.currentTimeMillis() - startTime;
            final long sleepTime = interval - processTime;
            if(sleepTime > 0){
                Thread.sleep(sleepTime);
            }
        }
        public void garbage(){}
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",interval=").append(interval);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * 񍀉ZĎΏہB<p>
     * Q̊ĎΏۂbvāAQ̊ĎΏۂ̒lgē񍀉ZsĎΏۊNXB<br>
     *
     * @author M.Takata
     */
    public abstract static class BinaryOperation extends WrapTarget{
        
        private static final long serialVersionUID = 7253736292096568710L;
        protected Target secondTarget;
        protected boolean isNullToZero;
        
        /**
         * {@link MBeanWatcherService}̃T[rXݒ肷B<p>
         * Q̊ĎΏۂɂMBeanWatcherServicẽT[rXݒ肷B<br>
         *
         * @param name MBeanWatcherServicẽT[rX
         */
        public void setWatcherServiceName(ServiceName name){
            super.setWatcherServiceName(name);
            secondTarget.setWatcherServiceName(name);
        }
        
        /**
         * {@link MBeanWatcherService}ݒ肷B<p>
         * Q̊ĎΏۂɂMBeanWatcherServiceݒ肷B<br>
         *
         * @param watcher MBeanWatcherService
         */
        protected void setWatcherService(MBeanWatcherService watcher){
            super.setWatcherService(watcher);
            secondTarget.setWatcherService(watcher);
        }
        
        /**
         * {@link Logger}ݒ肷B<p>
         * Q̊ĎΏۂɂLoggerݒ肷B<br>
         * 
         * @param logger Logger
         */
        public void setLogger(Logger logger){
            super.setLogger(logger);
            secondTarget.setLogger(logger);
        }
        
        /**
         * 񍀉Z̑񍀂ƂȂĎΏۂݒ肷B<p>
         *
         * @param target 񍀂ƂȂĎΏ
         */
        public void setSecondTarget(Target target){
            secondTarget = target;
        }
        
        /**
         * 񍀉Z̑񍀂ƂȂĎΏۂ擾B<p>
         *
         * @return 񍀂ƂȂĎΏ
         */
        public Target getSecondTarget(){
            return secondTarget;
        }
        
        /**
         * bvĎΏۂ̒ll^null̏ꍇɁA[Ƃ݂Ȃǂݒ肷B<p>
         * ftHǵAfalseŃ[Ƃ݂ȂȂB<br>
         *
         * @return [Ƃ݂ȂꍇAtrue
         */
        public void setNullToZero(boolean isNullToZero){
            this.isNullToZero = isNullToZero;
        }
        
        /**
         * bvĎΏۂ̒ll^null̏ꍇɁA[Ƃ݂Ȃǂ𔻒肷B<p>
         *
         * @return truȅꍇA[Ƃ݂Ȃ
         */
        public boolean isNullToZero(){
            return isNullToZero;
        }
        
        /**
         * bvQ̊ĎΏۂ̒l񍀉Zsʂ擾B<p>
         * 
         * @param connection JMXڑ
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        public Object getValue(MBeanServerConnection connection) throws Exception{
            BigDecimal first = Target.toBigDecimal(target.getValue(connection), isNullToZero);
            BigDecimal second = Target.toBigDecimal(secondTarget.getValue(connection), isNullToZero);
            if(first == null || second == null){
                return null;
            }
            Object value = calculate(first, second);
            if(value != null && contextKey != null){
                watcher.setContextValue(contextKey, value);
            }
            return value;
        }
        
        /**
         * 񍀉ZsB<p>
         *
         * @param first ꍀ
         * @param second 
         * @return Z
         * @exception Exception ZɎsꍇ
         */
        protected abstract BigDecimal calculate(BigDecimal first, BigDecimal second) throws Exception;
        
        /**
         * ZbgB<p>
         * bvĂĎΏۋyсAQ̊ĎΏۂZbgB<br>
         */
        public void reset(){
            super.reset();
            secondTarget.reset();
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",secondTarget=").append(secondTarget);
            buf.append(",isNullToZero=").append(isNullToZero);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * ZĎΏہB<p>
     * Q̊ĎΏۂ̒lZ{@link MBeanWatcherService.BinaryOperation}̎NXB<br>
     *
     * @author M.Takata
     */
    public static class AddOperation extends BinaryOperation{
        
        private static final long serialVersionUID = -5086486388715849637L;
        
        /**
         * ZsB<p>
         *
         * @param first ꍀ
         * @param second 
         * @return Z
         * @exception Exception ZɎsꍇ
         */
        protected BigDecimal calculate(BigDecimal first, BigDecimal second) throws Exception{
            return first.add(second);
        }
    }
    
    /**
     * ZĎΏہB<p>
     * Q̊ĎΏۂ̒lZ{@link MBeanWatcherService.BinaryOperation}̎NXB<br>
     *
     * @author M.Takata
     */
    public static class SubtractOperation extends BinaryOperation{
        
        private static final long serialVersionUID = -2343694151274725683L;
        
        /**
         * ZsB<p>
         *
         * @param first ꍀ
         * @param second 
         * @return Z
         * @exception Exception ZɎsꍇ
         */
        protected BigDecimal calculate(BigDecimal first, BigDecimal second) throws Exception{
            return first.subtract(second);
        }
    }
    
    /**
     * ZĎΏہB<p>
     * Q̊ĎΏۂ̒lZ{@link MBeanWatcherService.BinaryOperation}̎NXB<br>
     *
     * @author M.Takata
     */
    public static class MultiplyOperation extends BinaryOperation{
        
        private static final long serialVersionUID = -5651446587284471517L;
        
        /**
         * ZsB<p>
         *
         * @param first ꍀ
         * @param second 
         * @return Z
         * @exception Exception ZɎsꍇ
         */
        protected BigDecimal calculate(BigDecimal first, BigDecimal second) throws Exception{
            return first.multiply(second);
        }
    }
    
    /**
     * ZĎΏہB<p>
     * Q̊ĎΏۂ̒lZ{@link MBeanWatcherService.BinaryOperation}̎NXB<br>
     *
     * @author M.Takata
     */
    public static class DivideOperation extends BinaryOperation{
        private static final long serialVersionUID = -5331257610256153532L;
        private static final BigDecimal ZERO = new BigDecimal(0.0d);
        private int roundingMode = BigDecimal.ROUND_HALF_EVEN;
        private int scale = -1;
        private boolean isReturnNullOnZeroDivide = true;
        
        /**
         * ZʂɓKpۂ߃[hݒ肷B<p>
         * ftHǵA{@link BigDecimal#ROUND_HALF_EVEN}B<br>
         *
         * @param mode ۂ߃[h
         */
        public void setRoundingMode(int mode){
            this.roundingMode = mode;
        }
        
        /**
         * ZʂɓKpۂ߃[h擾B<p>
         *
         * @return ۂ߃[h
         */
        public int getRoundingMode(){
            return roundingMode;
        }
        
        /**
         * ZʂɓKp鏬_ȉݒ肷B<p>
         * w肵Ȃꍇ́Aꍀ̌ɈˑB<br>
         *
         * @param scale _ȉ
         */
        public void setScale(int scale){
            this.scale = scale;
        }
        
        /**
         * ZʂɓKp鏬_ȉ擾B<p>
         *
         * @return _ȉ
         */
        public int getScale(){
            return scale;
        }
        
        /**
         * [ZƂȂꍇɁAnullԂǂݒ肷B<p>
         * ftHǵAfalseŁA[Zɂ͗OB<br>
         *
         * @param isReturnNull nullԂꍇtrue
         */
        public void setReturnNullOnZeroDivide(boolean isReturnNull){
            this.isReturnNullOnZeroDivide = isReturnNull;
        }
        
        /**
         * [ZƂȂꍇɁAnullԂǂ𔻒肷B<p>
         *
         * @return truȅꍇAnullԂ
         */
        public boolean isReturnNullOnZeroDivide(){
            return isReturnNullOnZeroDivide;
        }
        
        /**
         * ZsB<p>
         *
         * @param first ꍀ
         * @param second 
         * @return Z
         * @exception Exception ZɎsꍇ
         */
        protected BigDecimal calculate(BigDecimal first, BigDecimal second) throws Exception{
            if(ZERO.equals(second) && isReturnNullOnZeroDivide){
                return null;
            }
            return scale >= 0 ? first.divide(second, scale, roundingMode) : first.divide(second, roundingMode);
        }
    }
    
    /**
     * ől񍀉ZĎΏہB<p>
     * Q̊ĎΏۂ̒l̍ől擾{@link MBeanWatcherService.BinaryOperation}̎NXB<br>
     *
     * @author M.Takata
     */
    public static class MaxOperation extends BinaryOperation{
        
        private static final long serialVersionUID = 7889957820603303173L;
        
        /**
         * őlԂB<p>
         *
         * @param first ꍀ
         * @param second 
         * @return Z
         * @exception Exception ZɎsꍇ
         */
        protected BigDecimal calculate(BigDecimal first, BigDecimal second) throws Exception{
            return first.max(second);
        }
    }
    
    /**
     * ŏl񍀉ZĎΏہB<p>
     * Q̊ĎΏۂ̒l̍ŏl擾{@link MBeanWatcherService.BinaryOperation}̎NXB<br>
     *
     * @author M.Takata
     */
    public static class MinOperation extends BinaryOperation{
        
        private static final long serialVersionUID = -4042120812950659730L;
        
        /**
         * ŏlԂB<p>
         *
         * @param first ꍀ
         * @param second 
         * @return Z
         * @exception Exception ZɎsꍇ
         */
        protected BigDecimal calculate(BigDecimal first, BigDecimal second) throws Exception{
            return first.min(second);
        }
    }
    
    /**
     * WZĎΏہB<p>
     * ĎΏۂbvāAĎΏۂ̒lɑ΂ďWZsĎΏۊNXB<br>
     *
     * @author M.Takata
     */
    public static abstract class SetOperation extends EditTarget{
        
        private static final long serialVersionUID = -7200261696645994257L;
        
        /**
         * ĎΏۂ̒lWz̏ꍇɁÅevfɑ΂ĕҏWsǂ𔻒肷B<p>
         *
         * @return KtrueԂ
         */
        public boolean isElementEdit(){
            return true;
        }
        
        /**
         * bvĎΏۂ̒lɑ΂ďWZsʂ擾B<p>
         * 
         * @param connection JMXڑ
         * @return ĎΏۂ̒l
         * @exception Exception ĎΏۂ̒l̎擾Ɏsꍇ
         */
        public Object getValue(MBeanServerConnection connection) throws Exception{
            Object value = target.getValue(connection);
            if(value == null){
                value = new BigDecimal(0.0d);
            }else{
                Object[] array = null;
                if(value instanceof Map<?, ?>){
                    array = ((Map<?, ?>)value).values().toArray();
                }else if(value instanceof Collection<?>){
                    array = ((Collection<?>)value).toArray();
                }else if(value.getClass().isArray()){
                    array = (Object[])value;
                }
                if(array != null){
                    if(array.length == 0){
                        value = new BigDecimal(0.0d);
                    }else{
                        List<BigDecimal> list = new ArrayList<BigDecimal>();
                        for(int i = 0; i < array.length; i++){
                            if(array[i] != null && array[i] instanceof Number){
                                BigDecimal target = Target.toBigDecimal(array[i], true);
                                list.add(target);
                            }
                        }
                        if(list.size() == 0){
                            value = new BigDecimal(0.0d);
                        }else{
                            value = edit(list);
                        }
                    }
                }
            }
            if(value != null && contextKey != null){
                watcher.setContextValue(contextKey, value);
            }
            return value;
        }
        
        /**
         * w肳ꂽlɑ΂ďWZsB<p>
         *
         * @param value ҏWΏۂ̒l
         * @return ҏWʂ̒l
         * @exception Exception ҏWɎsꍇ
         */
        @SuppressWarnings("unchecked")
        protected Object edit(Object value) throws Exception{
            List<BigDecimal> list = (List<BigDecimal>)value;
            return calculate(list.toArray(new BigDecimal[list.size()]));
        }
        
        /**
         * w肳ꂽlɑ΂ďWZsB<p>
         *
         * @param numbers WZ̑ΏۂƂȂ鐔lW
         * @return WZ̒l
         * @exception Exception WZɎsꍇ
         */
        protected abstract BigDecimal calculate(BigDecimal[] numbers) throws Exception;
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",isElementEdit=").append(isElementEdit());
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * aWZs{@link MBeanWatcherService.SetOperation}̎NXB<p>
     *
     * @author M.Takata
     */
    public static class Sum extends SetOperation{
        
        private static final long serialVersionUID = 2277773295544965565L;
        
        /**
         * W̑asB<p>
         *
         * @param numbers WZ̑ΏۂƂȂ鐔lW
         * @return WZ̒l
         * @exception Exception WZɎsꍇ
         */
        protected BigDecimal calculate(BigDecimal[] numbers) throws Exception{
            BigDecimal result = new BigDecimal(0.0d);
            for(int i = 0; i < numbers.length; i++){
                result = result.add(numbers[i]);
            }
            return result;
        }
    }
    
    /**
     * ϏWZs{@link MBeanWatcherService.SetOperation}̎NXB<p>
     *
     * @author M.Takata
     */
    public static class Average extends Sum{
        
        private static final long serialVersionUID = -7262525074283633937L;
        private int roundingMode = BigDecimal.ROUND_HALF_EVEN;
        
        /**
         * ς߂ۂ̏ZʂɓKpۂ߃[hݒ肷B<p>
         * ftHǵA{@link BigDecimal#ROUND_HALF_EVEN}B<br>
         *
         * @param mode ۂ߃[h
         */
        public void setRoundingMode(int mode){
            this.roundingMode = mode;
        }
        
        /**
         * ς߂ۂ̏ZʂɓKpۂ߃[h擾B<p>
         *
         * @return ۂ߃[h
         */
        public int getRoundingMode(){
            return roundingMode;
        }
        
        /**
         * W̕ϒl擾B<p>
         *
         * @param numbers WZ̑ΏۂƂȂ鐔lW
         * @return WZ̒l
         * @exception Exception WZɎsꍇ
         */
        protected BigDecimal calculate(BigDecimal[] numbers) throws Exception{
            return super.calculate(numbers).divide(new BigDecimal((double)numbers.length), roundingMode);
        }
        
        /**
         * ̊ĎΏۂ̕\擾B<p>
         *
         * @return \
         */
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.deleteCharAt(buf.length() - 1);
            buf.append(",roundingMode=").append(roundingMode);
            buf.append('}');
            return buf.toString();
        }
    }
    
    /**
     * őWZs{@link MBeanWatcherService.SetOperation}̎NXB<p>
     *
     * @author M.Takata
     */
    public static class Max extends SetOperation{
        
        private static final long serialVersionUID = 3152818165408079349L;
        
        /**
         * W̍ől擾B<p>
         *
         * @param numbers WZ̑ΏۂƂȂ鐔lW
         * @return WZ̒l
         * @exception Exception WZɎsꍇ
         */
        protected BigDecimal calculate(BigDecimal[] numbers) throws Exception{
            BigDecimal result = null;
            for(int i = 0; i < numbers.length; i++){
                if(result == null){
                    result = numbers[i];
                }else{
                    result = result.max(numbers[i]);
                }
            }
            return result;
        }
    }
    
    /**
     * ŏWZs{@link MBeanWatcherService.SetOperation}̎NXB<p>
     *
     * @author M.Takata
     */
    public static class Min extends SetOperation{
        
        private static final long serialVersionUID = -8356445551875453616L;
        
        /**
         * W̍ŏl擾B<p>
         *
         * @param numbers WZ̑ΏۂƂȂ鐔lW
         * @return WZ̒l
         * @exception Exception WZɎsꍇ
         */
        protected BigDecimal calculate(BigDecimal[] numbers) throws Exception{
            BigDecimal result = null;
            for(int i = 0; i < numbers.length; i++){
                if(result == null){
                    result = numbers[i];
                }else{
                    result = result.min(numbers[i]);
                }
            }
            return result;
        }
    }
}
