/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.http.httpclient;

import java.io.*;
import java.util.*;
import java.util.zip.*;

import org.apache.commons.httpclient.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.service.http.*;
import jp.ossc.nimbus.util.converter.*;

/**
 * Jakarta HttpClientgHTTPX|XNXB<p>
 *
 * @author M.Takata
 */
public class HttpResponseImpl implements HttpResponse, Cloneable{
    /** wb_[ : Content-Type */
    protected static final String HEADER_CONTENT_TYPE = "Content-Type";
    /** wb_[ : charset */
    protected static final String HEADER_CHARSET = "charset";
    /** wb_[ : Content-Encoding */
    protected static final String HEADER_CONTENT_ENCODING = "Content-Encoding";
    /** wb_[ : Connection */
    protected static final String HEADER_CONNECTION = "Connection";
    /** wb_[ : Content-Length */
    protected static final String HEADER_CONTENT_LENGTH = "Content-Length";
    /** wb_[ : Transfer-Encoding */
    protected static final String HEADER_TRANSFER_ENCODING = "Transfer-Encoding";
    /** Content-Encoding : deflate */
    protected static final String CONTENT_ENCODING_DEFLATE = "deflate";
    /** Content-Encoding : gzip */
    protected static final String CONTENT_ENCODING_GZIP = "gzip";
    /** Content-Encoding : x-zip */
    protected static final String CONTENT_ENCODING_X_GZIP = "x-gzip";
    /** Connection : close */
    protected static final String CONNECTION_CLOSE = "close";
    /** Transfer-Encoding : chunked */
    protected static final String TRANSFER_ENCODING_CHUNKED = "chunked";
    /** ftHgX|Xcharset */
    protected static final String DEFAULT_RESPONSE_CHARSET = "ISO8859_1";
    
    protected int statusCode;
    protected String statusMessage;
    protected HttpMethodBase method;
    protected Map<String, String[]> headerMap;
    protected InputStream inputStream;
    protected Object outputObject;
    protected ServiceName streamConverterServiceName;
    protected StreamConverter streamConverter;
    protected byte[] outputBytes;
    
    /**
     * HTTP\bhݒ肷B<p>
     *
     * @param method HTTP\bh
     * @exception IOException X|XXg[̓ǂݍ݂Ɏsꍇ
     */
    public void setHttpMethod(HttpMethodBase method)throws IOException{
        this.method = method;
        statusMessage = method.getStatusText();
        // X|XkĂΉ
        inputStream = decompress(
            method.getResponseBodyAsStream()
        );
    }
    
    /**
     * HTTPX|X̃Xg[o̓IuWFNgXg[ɕϊ{@link jp.ossc.nimbus.util.converter.StreamConverter StreamConverter}T[rX̃T[rXݒ肷B<p>
     *
     * @param name StreamConverterT[rX̃T[rX
     */
    public void setStreamConverterServiceName(ServiceName name){
        streamConverterServiceName = name;
    }
    
    /**
     * HTTPX|X̃Xg[o̓IuWFNgXg[ɕϊ{@link jp.ossc.nimbus.util.converter.StreamConverter StreamConverter}T[rX̃T[rX擾B<p>
     *
     * @return StreamConverterT[rX̃T[rX
     */
    public ServiceName getStreamConverterServiceName(){
        return streamConverterServiceName;
    }
    
    /**
     * HTTPX|X̃Xg[o̓IuWFNgɕϊ{@link jp.ossc.nimbus.util.converter.StreamConverter StreamConverter}ݒ肷B<p>
     *
     * @param converter StreamConverter
     */
    public void setStreamConverter(StreamConverter converter){
        streamConverter = converter;
    }
    
    /**
     * HTTPX|X̃Xg[o̓IuWFNgɕϊ{@link jp.ossc.nimbus.util.converter.StreamConverter StreamConverter}擾B<p>
     *
     * @return StreamConverter
     */
    public StreamConverter getStreamConverter(){
        return streamConverter;
    }
    
    /**
     * ̓Xg[̈kB<p>
     * (Content-EncodingɎw肳ꂽtŉ)
     * 
     * @param is ̓Xg[
     * @return kꂽ̓Xg[
     * @throws IOException T|[gĂȂk`(deflate, gzipȊO)w肳ꂽꍇ
     */
    protected InputStream decompress(InputStream is) throws IOException {
        if(is == null){
            return null;
        }
        // wb_[[Content-Encoding]̒l擾
        String encode = getHeader(HEADER_CONTENT_ENCODING);
        InputStream in = is;
        if(encode != null){
            
            if(encode.indexOf(CONTENT_ENCODING_DEFLATE) != -1){
                // deflatek
                in = new InflaterInputStream(in);
            }else if(encode.indexOf(CONTENT_ENCODING_GZIP) != -1
                        || encode.indexOf(CONTENT_ENCODING_X_GZIP) != -1){
                // gzipk
                in = new GZIPInputStream(in);
            }else{
                throw new IOException("Can not decompress. [" + encode + "]");
            }
        }
        String transferEncoding = getHeader(HEADER_TRANSFER_ENCODING);
        if(isConnectionClose()
            && (getContentLength() > 0
                || (transferEncoding != null && transferEncoding.indexOf(TRANSFER_ENCODING_CHUNKED) != -1)) 
        ){
            final ByteArrayOutputStream baos = new ByteArrayOutputStream();
            final byte[] bytes = new byte[1024];
            int length = 0;
            while((length = in.read(bytes)) != -1){
                baos.write(bytes, 0, length);
            }
            outputBytes = baos.toByteArray();
            final ByteArrayInputStream bais
                 = new ByteArrayInputStream(outputBytes);
            return bais;
        }else{
            return in;
        }
    }
    
    // HttpResponseJavaDoc
    @Override
    public Set<String> getHeaderNameSet(){
        return getHeaderMap().keySet();
    }
    
    // HttpResponseJavaDoc
    @Override
    public String getHeader(String name){
        if(headerMap != null){
            String[] vals = headerMap.get(name);
            if(vals == null){
                for(Map.Entry<String, String[]> entry : headerMap.entrySet()){
                    String headerName = entry.getKey();
                    if(headerName.equalsIgnoreCase(name)){
                        vals = entry.getValue();
                        break;
                    }
                }
            }
            return vals == null || vals.length == 0 ? null : vals[0];
        }
        final Header header = method.getResponseHeader(name);
        return header == null ? null : header.getValue();
    }
    
    // HttpResponseJavaDoc
    @Override
    public String[] getHeaders(String name){
        if(headerMap != null){
            String[] vals = (String[])headerMap.get(name);
            if(vals == null){
                for(Map.Entry<String, String[]> entry : headerMap.entrySet()){
                    String headerName = (String)entry.getKey();
                    if(headerName.equalsIgnoreCase(name)){
                        vals = (String[])entry.getValue();
                        break;
                    }
                }
            }
            return vals;
        }
        final Header[] headers = method.getResponseHeaders(name);
        if(headers == null){
            return null;
        }
        final String[] vals = new String[headers.length];
        for(int i = 0; i < headers.length; i++){
            vals[i] = headers[i].getValue();
        }
        return vals;
    }
    
    public int getContentLength(){
        final String lenStr = getHeader(HEADER_CONTENT_LENGTH);
        int length = 0;
        if(lenStr != null && lenStr.length() != 0){
            try{
                length = Integer.parseInt(lenStr);
            }catch(NumberFormatException e){
            }
        }
        return length;
    }
    
    /**
     * HTTPwb_̃}bv擾B<p>
     *
     * @return HTTPwb_̃}bv
     */
    public Map<String, String[]> getHeaderMap(){
        if(headerMap == null){
            headerMap = new HashMap<String, String[]>();
            final Header[] headers = method.getResponseHeaders();
            if(headers != null){
                for(int i = 0; i < headers.length; i++){
                    String name = headers[i].getName();
                    String value = headers[i].getValue();
                    String[] vals = (String[])headerMap.get(name);
                    if(vals == null){
                        vals = new String[]{value};
                        headerMap.put(name, vals);
                    }else{
                        final String[] newVals = new String[vals.length + 1];
                        System.arraycopy(vals, 0, newVals, 0, vals.length);
                        newVals[newVals.length - 1] = value;
                        headerMap.put(name, newVals);
                    }
                }
            }
        }
        return headerMap;
    }
    
    /**
     * HTTPwb_̃}bvݒ肷B<p>
     *
     * @param map HTTPwb_̃}bv
     */
    public void setHeaderMap(Map<String, String[]> map){
        headerMap = map;
    }
    
    public void addHeader(String name, String value){
        if(headerMap == null){
            headerMap = new HashMap<String, String[]>();
        }
        String[] vals = (String[])headerMap.get(name);
        if(vals == null){
            vals = new String[]{value};
            headerMap.put(name, vals);
        }else{
            final String[] newVals = new String[vals.length + 1];
            System.arraycopy(vals, 0, newVals, 0, vals.length);
            newVals[newVals.length - 1] = value;
            headerMap.put(name, newVals);
        }
    }
    
    // HttpResponseJavaDoc
    @Override
    public InputStream getInputStream() throws IOException{
        return inputStream;
    }
    
    /**
     * X|XXg[ݒ肷B<p>
     *
     * @param in X|XXg[
     */
    public void setInputStream(InputStream in){
        inputStream = in;
    }
    
    // HttpResponseJavaDoc
    @Override
    public <T> T getObject() throws ConvertException{
        return getObject(null);
    }
    
    // HttpResponseJavaDoc
    @SuppressWarnings("unchecked")
    @Override
    public <T> T getObject(T bind) throws ConvertException{
        if(outputObject == null
             && (streamConverter != null || streamConverterServiceName != null)){
            StreamConverter converter = streamConverter;
            if(streamConverterServiceName != null){
                converter = (StreamConverter)ServiceManagerFactory
                    .getServiceObject(streamConverterServiceName);
            }
            if(converter instanceof StreamStringConverter){
                ((StreamStringConverter)converter).setCharacterEncodingToObject(
                    getCharacterEncoding()
                );
            }
            if(inputStream != null){
                try{
                    if(bind != null && converter instanceof BindingStreamConverter){
                        outputObject = ((BindingStreamConverter)converter).convertToObject(inputStream, bind);
                    }else{
                        outputObject = converter.convertToObject(inputStream);
                    }
                }finally{
                    try{
                        inputStream.reset();
                    }catch(IOException e){}
                }
            }
        }
        return (T)outputObject;
    }
    
    /**
     * IuWFNgݒ肷B<p>
     *
     * @param object IuWFNg
     */
    public void setObject(Object object){
        outputObject = object;
    }
    
    // HttpResponseJavaDoc
    @Override
    public String getCharacterEncoding(){
        final String contentType = getHeader(HEADER_CONTENT_TYPE);
        if(contentType == null){
            return DEFAULT_RESPONSE_CHARSET;
        }
        
        final int index = contentType.indexOf(HEADER_CHARSET);
        if(index == -1){
            return DEFAULT_RESPONSE_CHARSET;
        }else{
            return contentType.substring(
                index + HEADER_CHARSET.length() + 1
            );
        }
    }
    
    // HttpResponseJavaDoc
    @Override
    public int getStatusCode(){
        return statusCode;
    }
    
    /**
     * X|XHTTPXe[^Xݒ肷B<p>
     *
     * @param code HTTPXe[^X
     */
    public void setStatusCode(int code){
        statusCode = code;
    }
    
    // HttpResponseJavaDoc
    @Override
    public String getStatusMessage(){
        return statusMessage;
    }
    
    /**
     * X|XHTTPXe[^XbZ[Wݒ肷B<p>
     *
     * @param message HTTPXe[^XbZ[W
     */
    public void setStatusMessage(String message){
        statusMessage = message;
    }
    
    /**
     * X|XXg[o̓IuWFNgɕϊۂ̃oCgz擾B<p>
     *
     * @return X|XXg[o̓IuWFNgɕϊۂ̃oCgz
     */
    public byte[] getOutputBytes(){
        return outputBytes;
    }
    
    /**
     * 𐶐B<p>
     *
     * @return 
     * @exception CloneNotSupportedException Ɏsꍇ
     */
    public Object clone() throws CloneNotSupportedException{
        return (HttpResponseImpl)super.clone();
    }
    
    @Override
    public void close(){
        if(method != null){
            method.releaseConnection();
        }
    }
    
    /**
     * ڑ؂ėǂfB<p>
     *
     * @return Connectionwb_close݂͑Ȃꍇ́AtrueB<p>
     */
    public boolean isConnectionClose(){
        String connection = getHeader(HEADER_CONNECTION);
        return connection == null || CONNECTION_CLOSE.equalsIgnoreCase(connection);
    }
}