/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link NimbusMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class NimbusMetaDataTest{

    @DataProvider(name = "NimbusMetaData")
    public Object[][] dataNimbusMetaData() throws MalformedURLException{
        return new Object[][]{
            {null, null},
            {null, new File("service-definition.xml").toURI().toURL()},
            {new DefaultServiceLoaderService(), null},
            {new DefaultServiceLoaderService(), new File("service-definition.xml").toURI().toURL()},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#NimbusMetaData(jp.ossc.nimbus.core.ServiceLoader,java.net.URL)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getURL()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "NimbusMetaData")
    public void testNimbusMetaData(ServiceLoader loader, URL url){
        NimbusMetaData metaData = new NimbusMetaData(loader, url);
        assertNull(metaData.getParent());
        assertEquals(metaData.getServiceLoader(), loader);
        assertEquals(metaData.getURL(), url);
        assertEquals(metaData.getManagers().size(), 0);
        assertEquals(metaData.getPropertyEditors().size(), 0);
        assertEquals(metaData.getPropertyNameSet().size(), 0);
        assertEquals(metaData.getProperties().size(), 0);
        assertNull(metaData.getDefaultLog());
        assertNull(metaData.getRepository());
        assertNull(metaData.getLog());
        assertNull(metaData.getMessage());
        assertNull(metaData.getDocType());
        assertNull(metaData.getEncoding());
    }

    @DataProvider(name = "SetServiceLoader")
    public Object[][] dataSetServiceLoader(){
        return new Object[][]{
            {null},
            {new DefaultServiceLoaderService()}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#setServiceLoader(jp.ossc.nimbus.core.ServiceLoader)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getServiceLoader()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "SetServiceLoader")
    public void testSetServiceLoader(ServiceLoader loader){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        metaData.setServiceLoader(loader);
        assertEquals(metaData.getServiceLoader(), loader);
    }

    @DataProvider(name = "SetDocType")
    public Object[][] dataSetDocType(){
        return new Object[][]{
            {null},
            {"<!DOCTYPE nimbus PUBLIC \"-//Nimbus//DTD Nimbus 2.0//JA\" \"http://nimbus2.sourceforge.jp/dtd/nimbus-service_2_0.dtd\">"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#setDocType(java.lang.String)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getDocType()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "SetDocType")
    public void testSetDocType(String docType){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        metaData.setDocType(docType);
        assertEquals(metaData.getDocType(), docType);
    }

    @DataProvider(name = "SetEncoding")
    public Object[][] dataSetEncoding(){
        return new Object[][]{
            {null},
            {"UTF-8"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ManagerMetaData#setEncoding(java.lang.String)",
                    "jp.ossc.nimbus.core.ManagerMetaData#getEncoding()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "SetEncoding")
    public void testSetEncoding(String encoding){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        metaData.setEncoding(encoding);
        assertEquals(metaData.getEncoding(), encoding);
    }

    @DataProvider(name = "SetRepository")
    public Object[][] dataSetRepository(){
        return new Object[][]{
            {null},
            {new ServiceNameMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#setRepository(jp.ossc.nimbus.core.ServiceNameMetaData)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getRepository()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "SetRepository")
    public void testSetRepository(ServiceNameMetaData data){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        metaData.setRepository(data);
        assertEquals(metaData.getRepository(), data);
    }

    @DataProvider(name = "SetLog")
    public Object[][] dataSetLog(){
        return new Object[][]{
            {null},
            {new ServiceNameMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#setLog(jp.ossc.nimbus.core.ServiceNameMetaData)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getLog()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "SetLog")
    public void testSetLog(ServiceNameMetaData data){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        metaData.setLog(data);
        assertEquals(metaData.getLog(), data);
    }

    @DataProvider(name = "SetMessage")
    public Object[][] dataSetMessage(){
        return new Object[][]{
            {null},
            {new ServiceNameMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#setMessage(jp.ossc.nimbus.core.ServiceNameMetaData)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getMessage()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "SetMessage")
    public void testSetMessage(ServiceNameMetaData data){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        metaData.setMessage(data);
        assertEquals(metaData.getMessage(), data);
    }

    @DataProvider(name = "SetDefaultLog")
    public Object[][] dataSetDefaultLog(){
        return new Object[][]{
            {null},
            {new DefaultLogMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#setDefaultLog(jp.ossc.nimbus.core.DefaultLogMetaData)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getDefaultLog()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "SetDefaultLog")
    public void testSetDefaultLog(DefaultLogMetaData data){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        metaData.setDefaultLog(data);
        assertEquals(metaData.getDefaultLog(), data);
    }

    @DataProvider(name = "SetProperty")
    public Object[][] dataSetProperty(){
        return new Object[][]{
            {null},
            {new String[][]{{"prop1", "1"}}},
            {new String[][]{{"prop1", "1"}, {"prop2", "2"}, {"prop3", "3"}, {"prop4", "4"}, {"prop5", "5"}, {"prop6", "6"}, {"prop7", "7"}, {"prop8", "8"}, {"prop9", "9"}, {"prop10", "10"}}},
            {new String[][]{{"prop1", null}, {null, "2"}, {null, null}}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#setProperty(java.lang.String,java.lang.String)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getPropertyNameSet()",
                    "jp.ossc.nimbus.core.NimbusMetaData#getProperty(java.lang.String)",
                    "jp.ossc.nimbus.core.NimbusMetaData#removeProperty(java.lang.String)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getProperties()",
                    "jp.ossc.nimbus.core.NimbusMetaData#clearProperties()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "SetProperty")
    public void testSetProperty(String[][] propertySet){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        if(propertySet == null || propertySet.length == 0){
            assertEquals(metaData.getProperties().size(), 0);
        }else{
            Map<String, String> properties = new HashMap<String, String>();
            for(String[] property : propertySet){
                metaData.setProperty(property[0], property[1]);
                properties.put(property[0], property[1]);
            }
            for(String name : metaData.getPropertyNameSet()){
                assertEquals(metaData.getProperty(name), properties.get(name));
            }
            for(int i = 0, imax = propertySet.length; i < imax; i+=2){
                metaData.removeProperty(propertySet[i][0]);
                properties.remove(propertySet[i][0]);
            }
            Map<String, String> props = metaData.getProperties();
            assertEquals(props, properties);
            metaData.clearProperties();
            props = metaData.getProperties();
            assertEquals(props.size(), 0);
        }
    }
    
    @DataProvider(name = "AddPropertyEditor")
    public Object[][] dataAddPropertyEditor(){
        return new Object[][]{
            new Object[]{null},
            new Object[]{new String[][]{{"java.lang.String", "test.StringPropertyEditor"}}},
            new Object[]{new String[][]{{"java.lang.String", "test.StringPropertyEditor"}, {"java.lang.Byte", "test.BytePropertyEditor"}, {"java.lang.Short", "test.ShortPropertyEditor"}, {"java.lang.Character", "test.CharacterPropertyEditor"}, {"java.lang.Integer", "test.IntegerPropertyEditor"}, {"java.lang.Long", "test.LongPropertyEditor"}, {"java.lang.Float", "test.FloatPropertyEditor"}, {"java.lang.Double", "test.DoublePropertyEditor"}}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#addPropertyEditor(java.lang.String,java.lang.String)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getPropertyEditors()",
                    "jp.ossc.nimbus.core.NimbusMetaData#removePropertyEditor(java.lang.String)",
                    "jp.ossc.nimbus.core.NimbusMetaData#clearPropertyEditors()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "AddPropertyEditor")
    public void testAddPropertyEditor(String[][] editors){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        if(editors == null || editors.length == 0){
            assertTrue(metaData.getPropertyEditors().size() == 0);
        }else{
            for(String[] typeAndEditor : editors){
                metaData.addPropertyEditor(typeAndEditor[0], typeAndEditor[1]);
            }
            Map<String, String> editorsMap = metaData.getPropertyEditors();
            assertEquals(editorsMap.size(), editors.length);
            for(String[] typeAndEditor : editors){
                assertEquals(editorsMap.get(typeAndEditor[0]), typeAndEditor[1]);
            }
            Set<String> types = editorsMap.keySet();
            for(int i = 0, imax = editors.length; i < imax; i+=2){
                metaData.removePropertyEditor(editors[i][0]);
                types.remove(editors[i][0]);
            }
            editorsMap = metaData.getPropertyEditors();
            assertEquals(editorsMap.keySet(), types);
            metaData.clearPropertyEditors();
            assertEquals(metaData.getPropertyEditors().size(), 0);
        }
    }

    @DataProvider(name = "AddManager")
    public Object[][] dataAddManager() throws Exception{
        return new Object[][]{
            {null},
            {new ManagerMetaData[]{new ManagerMetaData(null, null)}},
            {new ManagerMetaData[]{importXML(new ManagerMetaData(null, null), "<manager name=\"manager1\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager2\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager3\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager4\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager5\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager6\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager7\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager8\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager9\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager10\"/>")}},
            {new ManagerMetaData[]{null, new ManagerMetaData(null, null)}}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#addManager(jp.ossc.nimbus.core.ManagerMetaData)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getManagerNameSet()",
                    "jp.ossc.nimbus.core.NimbusMetaData#getManager(java.lang.String)",
                    "jp.ossc.nimbus.core.NimbusMetaData#getManagers()",
                    "jp.ossc.nimbus.core.NimbusMetaData#removeManager(java.lang.String)",
                    "jp.ossc.nimbus.core.NimbusMetaData#clearManagers()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "AddManager")
    public void testAddManager(ManagerMetaData[] managers){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        if(managers == null || managers.length == 0){
            assertEquals(metaData.getManagers().size(), 0);
        }else{
            Map<String, ManagerMetaData> managerMap = new HashMap<String, ManagerMetaData>();
            for(ManagerMetaData manager : managers){
                metaData.addManager(manager);
                if(manager != null){
                    managerMap.put(manager.getName(), manager);
                }
            }
            for(String name : metaData.getManagerNameSet()){
                assertEquals(metaData.getManager(name), managerMap.get(name));
                assertEquals(metaData.getManager(name).getParent(), metaData);
            }
            
            Collection<ManagerMetaData> coll = metaData.getManagers();
            assertEquals(coll.size(), managerMap.size());
            assertTrue(coll.containsAll(managerMap.values()));
            
            for(int i = 0, imax = managers.length; i < imax; i+=2){
                if(managers[i] != null){
                    metaData.removeManager(managers[i].getName());
                    assertNull(managers[i].getParent());
                    managerMap.remove(managers[i].getName());
                }
            }
            assertEquals(coll.size(), managerMap.size());
            assertTrue(coll.containsAll(managerMap.values()));
            metaData.clearManagers();
            assertEquals(coll.size(), 0);
            for(int i = 0, imax = managers.length; i < imax; i++){
                if(managers[i] != null){
                    assertNull(managers[i].getParent());
                }
            }
        }
    }

    @DataProvider(name = "CreateRepositoryMetaData")
    public Object[][] dataCreateRepositoryMetaData(){
        return new Object[][]{
            {null, null, ServiceManager.DEFAULT_NAME},
            {null, "service", ServiceManager.DEFAULT_NAME},
            {"manager", null, "manager"},
            {"manager", "service", "manager"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#createRepositoryMetaData(java.lang.String,java.lang.String)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "CreateRepositoryMetaData")
    public void testCreateRepositoryMetaData(String manager, String service, String expectedManager){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        ServiceNameMetaData nameData = metaData.createRepositoryMetaData(manager, service);
        assertNotNull(nameData);
        assertEquals(nameData.tagName, "manager-repository");
        assertEquals(nameData.getParent(), metaData);
        assertEquals(nameData.getManagerName(), expectedManager);
        assertEquals(nameData.getServiceName(), service);
    }

    @DataProvider(name = "CreateLogMetaData")
    public Object[][] dataCreateLogMetaData(){
        return new Object[][]{
            {null, null, ServiceManager.DEFAULT_NAME},
            {null, "service", ServiceManager.DEFAULT_NAME},
            {"manager", null, "manager"},
            {"manager", "service", "manager"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#createLogMetaData(java.lang.String,java.lang.String)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "CreateLogMetaData")
    public void testCreateLogMetaData(String manager, String service, String expectedManager){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        ServiceNameMetaData nameData = metaData.createLogMetaData(manager, service);
        assertNotNull(nameData);
        assertEquals(nameData.tagName, "log");
        assertEquals(nameData.getParent(), metaData);
        assertEquals(nameData.getManagerName(), expectedManager);
        assertEquals(nameData.getServiceName(), service);
    }

    @DataProvider(name = "CreateMessageMetaData")
    public Object[][] dataCreateMessageMetaData(){
        return new Object[][]{
            {null, null, ServiceManager.DEFAULT_NAME},
            {null, "service", ServiceManager.DEFAULT_NAME},
            {"manager", null, "manager"},
            {"manager", "service", "manager"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#createMessageMetaData(java.lang.String,java.lang.String)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          dataProvider = "CreateMessageMetaData")
    public void testCreateMessageMetaData(String manager, String service, String expectedManager){
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        ServiceNameMetaData nameData = metaData.createMessageMetaData(manager, service);
        assertNotNull(nameData);
        assertEquals(nameData.tagName, "message");
        assertEquals(nameData.getParent(), metaData);
        assertEquals(nameData.getManagerName(), expectedManager);
        assertEquals(nameData.getServiceName(), service);
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<nimbus/>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><default-log><debug output=\"true\"/></default-log></nimbus>"), new Object[]{importXML(new DefaultLogMetaData(null), "<default-log><debug output=\"true\"/></default-log>"), null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><default-log><debug output=\"true\"/></default-log></ifdef></nimbus>"), new Object[]{importXML(new DefaultLogMetaData(null), "<default-log><debug output=\"true\"/></default-log>"), null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"false\"><default-log><debug output=\"true\"/></default-log></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><manager-repository>Repository</manager-repository></nimbus>"), new Object[]{null, importXML(new ServiceNameMetaData(null), "<manager-repository>Repository</manager-repository>"), null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><manager-repository>Repository</manager-repository></ifdef></nimbus>"), new Object[]{null, importXML(new ServiceNameMetaData(null), "<manager-repository>Repository</manager-repository>"), null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"false\"><manager-repository>Repository</manager-repository></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><log>Log</log></nimbus>"), new Object[]{null, null, importXML(new ServiceNameMetaData(null), "<log>Log</log>"), null, null, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><log>Log</log></ifdef></nimbus>"), new Object[]{null, null, importXML(new ServiceNameMetaData(null), "<log>Log</log>"), null, null, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"false\"><log>Log</log></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><message>Message</message></nimbus>"), new Object[]{null, null, null, importXML(new ServiceNameMetaData(null), "<message>Message</message>"), null, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><message>Message</message></ifdef></nimbus>"), new Object[]{null, null, null, importXML(new ServiceNameMetaData(null), "<message>Message</message>"), null, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"false\"><message>Message</message></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><server-property name=\"PROP1\">1</server-property></nimbus>"), new Object[]{null, null, null, null, new String[][]{{"PROP1", "1"}}, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><server-property name=\"PROP1\">1</server-property></ifdef></nimbus>"), new Object[]{null, null, null, null, new String[][]{{"PROP1", "1"}}, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"false\"><server-property name=\"PROP1\">1</server-property></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><server-property name=\"PROP1\">1</server-property><server-property name=\"PROP2\">2</server-property></nimbus>"), new Object[]{null, null, null, null, new String[][]{{"PROP1", "1"},{"PROP2", "2"}}, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><server-property name=\"PROP1\">1</server-property><server-property name=\"PROP2\">2</server-property></ifdef></nimbus>"), new Object[]{null, null, null, null, new String[][]{{"PROP1", "1"},{"PROP2", "2"}}, null, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"false\"><server-property name=\"PROP1\">1</server-property><server-property name=\"PROP2\">2</server-property></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></property-editors></nimbus>"), new Object[]{null, null, null, null, null, new String[][]{{"java.lang.String", "test.StringPropertyEditor"}}, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></property-editors></ifdef></nimbus>"), new Object[]{null, null, null, null, null, new String[][]{{"java.lang.String", "test.StringPropertyEditor"}}, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"false\"><property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></property-editors></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor><property-editor type=\"java.lang.Byte\">test.BytePropertyEditor</property-editor></property-editors></nimbus>"), new Object[]{null, null, null, null, null, new String[][]{{"java.lang.String", "test.StringPropertyEditor"}, {"java.lang.Byte", "test.BytePropertyEditor"}}, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor><property-editor type=\"java.lang.Byte\">test.BytePropertyEditor</property-editor></property-editors></ifdef></nimbus>"), new Object[]{null, null, null, null, null, new String[][]{{"java.lang.String", "test.StringPropertyEditor"}, {"java.lang.Byte", "test.BytePropertyEditor"}}, new MetaData[0]}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"false\"><property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor><property-editor type=\"java.lang.Byte\">test.BytePropertyEditor</property-editor></property-editors></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><manager/></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[]{importXML(new ManagerMetaData(null, null), "<manager/>")}}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><manager/></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[]{importXML(new ManagerMetaData(null, null), "<manager/>")}}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"false\"><manager/></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus><manager name=\"manager1\"/></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[]{importXML(new ManagerMetaData(null, null), "<manager name=\"manager1\"/>")}}},
            {createElement("<nimbus><manager name=\"manager1\"/><manager name=\"manager2\"/></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[]{importXML(new ManagerMetaData(null, null), "<manager name=\"manager1\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager2\"/>")}}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><manager name=\"manager1\"/><manager name=\"manager2\"/></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[]{importXML(new ManagerMetaData(null, null), "<manager name=\"manager1\"/>"), importXML(new ManagerMetaData(null, null), "<manager name=\"manager2\"/>")}}},
            {createElement("<nimbus><ifdef name=\"test\" value=\"false\"><manager name=\"manager1\"/><manager name=\"manager2\"/></ifdef></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
            {createElement("<nimbus unknown=\"hoge\"><unknown/></nimbus>"), new Object[]{null, null, null, null, null, null, new MetaData[0]}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#getProperties\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#getProperty\\(java\\.lang\\.String\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#getPropertyEditors\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#getDefaultLog\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#getRepository\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#getLog\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#getMessage\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#getManagers\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        
        System.setProperty("test", "true");
        try{
            NimbusMetaData metaData = new NimbusMetaData(null, null);
            metaData.importXML(element);
            assertMetaDataEquals(metaData.getDefaultLog(), (MetaData)assertValues[0]);
            assertMetaDataEquals(metaData.getRepository(), (MetaData)assertValues[1]);
            assertMetaDataEquals(metaData.getLog(), (MetaData)assertValues[2]);
            assertMetaDataEquals(metaData.getMessage(), (MetaData)assertValues[3]);
            String[][] props = (String[][])assertValues[4];
            if(props == null || props.length == 0){
                assertEquals(metaData.getProperties().size(), 0);
            }else{
                assertEquals(metaData.getProperties().size(), props.length);
                for(String[] propAndValue : props){
                    assertEquals(metaData.getProperty(propAndValue[0]), propAndValue[1]);
                }
            }
            String[][] editors = (String[][])assertValues[5];
            Map<String, String> propEditors = metaData.getPropertyEditors();
            if(editors == null || editors.length == 0){
                assertEquals(propEditors.size(), 0);
            }else{
                assertEquals(propEditors.size(), editors.length);
                for(String[] typeAndEditor : editors){
                    assertEquals(propEditors.get(typeAndEditor[0]), typeAndEditor[1]);
                }
            }
            Collection<ManagerMetaData> managers = metaData.getManagers();
            MetaData[] mngs = (MetaData[])assertValues[6];
            if(mngs.length == 0){
                assertTrue(managers.size() == 0);
            }else{
                assertEquals(managers.size(), mngs.length);
                for(int i = 0; i < mngs.length; i++){
                    assertMetaDataEquals(metaData.getManager(((ManagerMetaData)mngs[i]).getName()), mngs[i]);
                }
            }
        }finally{
            System.clearProperty("test");
        }
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<nimbus><server-property name=\"PROP1\">1</server-property><server-property name=\"PROP1\">1</server-property></nimbus>")},
            {createElement("<nimbus><server-property name=\"PROP1\">1</server-property><ifdef name=\"test\" value=\"true\"><server-property name=\"PROP1\">1</server-property></ifdef></nimbus>")},
            {createElement("<nimbus><default-log/><default-log/></nimbus>")},
            {createElement("<nimbus><property-editors/><property-editors/></nimbus>")},
            {createElement("<nimbus><property-editors/><ifdef name=\"test\" value=\"true\"><property-editors/></ifdef></nimbus>")},
            {createElement("<nimbus><default-log/><ifdef name=\"test\" value=\"true\"><default-log/></ifdef></nimbus>")},
            {createElement("<nimbus><manager-repository>Repository</manager-repository><manager-repository>Repository</manager-repository></nimbus>")},
            {createElement("<nimbus><manager-repository>Repository</manager-repository><ifdef name=\"test\" value=\"true\"><manager-repository>Repository</manager-repository></ifdef></nimbus>")},
            {createElement("<nimbus><log>Log</log><log>Log</log></nimbus>")},
            {createElement("<nimbus><log>Log</log><ifdef name=\"test\" value=\"true\"><log>Log</log></ifdef></nimbus>")},
            {createElement("<nimbus><message>Message</message><message>Message</message></nimbus>")},
            {createElement("<nimbus><message>Message</message><ifdef name=\"test\" value=\"true\"><message>Message</message></ifdef></nimbus>")},
            {createElement("<nimbus><manager name=\"manager1\"/><manager name=\"manager1\"/></nimbus>")},
            {createElement("<nimbus><manager name=\"manager1\"/><ifdef name=\"test\" value=\"true\"><manager name=\"manager1\"/></ifdef></nimbus>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        System.setProperty("test", "true");
        try{
            NimbusMetaData metaData = new NimbusMetaData(null, null);
            metaData.importXML(element);
        }finally{
            System.clearProperty("test");
        }
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<nimbus/>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><manager-repository>Repository</manager-repository></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <manager-repository>" + ServiceManager.DEFAULT_NAME + "#Repository</manager-repository>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><log>Log</log></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <log>" + ServiceManager.DEFAULT_NAME + "#Log</log>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><default-log><debug output=\"true\"/></default-log></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <default-log>" + System.getProperty("line.separator") + "        <debug output=\"true\"/>" + System.getProperty("line.separator") + "    </default-log>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></property-editors></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <property-editors>" + System.getProperty("line.separator") + "        <property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>" + System.getProperty("line.separator") + "    </property-editors>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><server-property name=\"PROP1\">1</server-property></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <server-property name=\"PROP1\">1</server-property>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><message>Message</message></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <message>" + ServiceManager.DEFAULT_NAME + "#Message</message>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><manager name=\"manager1\"/></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <manager name=\"manager1\">" + System.getProperty("line.separator") + "    </manager>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><manager name=\"manager1\"/><manager name=\"manager2\"/></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <manager name=\"manager1\">" + System.getProperty("line.separator") + "    </manager>" + System.getProperty("line.separator") + "    <manager name=\"manager2\">" + System.getProperty("line.separator") + "    </manager>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><manager-repository>Repository</manager-repository></ifdef></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <manager-repository>" + ServiceManager.DEFAULT_NAME + "#Repository</manager-repository>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><log>Log</log></ifdef></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <log>" + ServiceManager.DEFAULT_NAME + "#Log</log>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><message>Message</message></ifdef></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <message>" + ServiceManager.DEFAULT_NAME + "#Message</message>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><manager name=\"manager1\"/></ifdef></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <manager name=\"manager1\">" + System.getProperty("line.separator") + "        </manager>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><default-log><debug output=\"true\"/></default-log></ifdef></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <default-log>" + System.getProperty("line.separator") + "            <debug output=\"true\"/>" + System.getProperty("line.separator") + "        </default-log>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</nimbus>"},
            {createElement("<nimbus><ifdef name=\"test\" value=\"true\"><property-editors><property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor></property-editors></ifdef></nimbus>"), "<?xml version=\"1.0\"?>" + System.getProperty("line.separator") + "<nimbus>" + System.getProperty("line.separator") + "    <ifdef name=\"test\" value=\"true\">" + System.getProperty("line.separator") + "        <property-editors>" + System.getProperty("line.separator") + "            <property-editor type=\"java.lang.String\">test.StringPropertyEditor</property-editor>" + System.getProperty("line.separator") + "        </property-editors>" + System.getProperty("line.separator") + "    </ifdef>" + System.getProperty("line.separator") + "</nimbus>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.NimbusMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#NimbusMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ServiceLoader,java\\.net\\.URL\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.NimbusMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        NimbusMetaData metaData = new NimbusMetaData(null, null);
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
