/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.util.List;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link ConstructorMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class ConstructorMetaDataTest{

    @DataProvider(name = "ConstructorMetaData")
    public Object[][] dataConstructorMetaData(){
        return new Object[][]{
            {null},
            {new ObjectMetaData()},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ConstructorMetaData#ConstructorMetaData(jp.ossc.nimbus.core.ObjectMetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "ConstructorMetaData")
    public void testConstructorMetaData(ObjectMetaData parent){
        ConstructorMetaData metaData = new ConstructorMetaData(parent);
        assertEquals(metaData.getParent(), parent);
    }

    @DataProvider(name = "SetInvoke")
    public Object[][] dataSetInvoke(){
        return new Object[][]{
            {null},
            {new InvokeMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ConstructorMetaData#setInvoke(jp.ossc.nimbus.core.InvokeMetaData)",
                    "jp.ossc.nimbus.core.ConstructorMetaData#getInvoke()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          dataProvider = "SetInvoke")
    public void testSetInvoke(InvokeMetaData invoke){
        ConstructorMetaData metaData = new ConstructorMetaData(null);
        metaData.setInvoke(invoke);
        assertEquals(metaData.getInvoke(), invoke);
    }

    @DataProvider(name = "SetStaticInvoke")
    public Object[][] dataSetStaticInvoke(){
        return new Object[][]{
            {null},
            {new StaticInvokeMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ConstructorMetaData#setStaticInvoke(jp.ossc.nimbus.core.StaticInvokeMetaData)",
                    "jp.ossc.nimbus.core.ConstructorMetaData#getStaticInvoke()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          dataProvider = "SetStaticInvoke")
    public void testSetStaticInvoke(StaticInvokeMetaData invoke){
        ConstructorMetaData metaData = new ConstructorMetaData(null);
        metaData.setStaticInvoke(invoke);
        assertEquals(metaData.getStaticInvoke(), invoke);
    }

    @DataProvider(name = "SetStaticFieldRef")
    public Object[][] dataSetStaticFieldRef(){
        return new Object[][]{
            {null},
            {new StaticFieldRefMetaData(null)}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ConstructorMetaData#setStaticFieldRef(jp.ossc.nimbus.core.StaticFieldRefMetaData)",
                    "jp.ossc.nimbus.core.ConstructorMetaData#getStaticFieldRef()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          dataProvider = "SetStaticFieldRef")
    public void testSetStaticFieldRef(StaticFieldRefMetaData field){
        ConstructorMetaData metaData = new ConstructorMetaData(null);
        metaData.setStaticFieldRef(field);
        assertEquals(metaData.getStaticFieldRef(), field);
    }

    @DataProvider(name = "AddArgument")
    public Object[][] dataAddArgument(){
        return new Object[][]{
            {null},
            {new ArgumentMetaData[]{new ArgumentMetaData(null, null)}},
            {new ArgumentMetaData[]{new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null)}},
            {new ArgumentMetaData[]{null, new ArgumentMetaData(null, null)}}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ConstructorMetaData#addArgument(jp.ossc.nimbus.core.ArgumentMetaData)",
                    "jp.ossc.nimbus.core.ConstructorMetaData#getArguments()",
                    "jp.ossc.nimbus.core.ConstructorMetaData#removeArgument(jp.ossc.nimbus.core.ArgumentMetaData)",
                    "jp.ossc.nimbus.core.ConstructorMetaData#clearArguments()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(.*\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          dataProvider = "AddArgument")
    public void testAddArgument(ArgumentMetaData[] arguments){
        ConstructorMetaData metaData = new ConstructorMetaData(null);
        if(arguments == null || arguments.length == 0){
            assertEquals(metaData.getArguments().size(), 0);
        }else{
            for(ArgumentMetaData argument : arguments){
                metaData.addArgument(argument);
            }
            List<ArgumentMetaData> list = metaData.getArguments();
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), arguments[i]);
            }
            for(int i = 0, imax = list.size(); i < imax; i+=2){
                metaData.removeArgument(arguments[i]);
            }
            list = metaData.getArguments();
            int index = 1;
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), arguments[index]);
                index += 2;
            }
            metaData.clearArguments();
            assertEquals(metaData.getArguments().size(), 0);
        }
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<constructor/>"), new Object[]{null, null, null, new MetaData[0]}},
            {
                createElement("<constructor><invoke name=\"toString\"><target><object code=\"java.lang.StringBuilder\"/></target></invoke></constructor>"),
                new Object[]{importXML(new InvokeMetaData(null), "<invoke name=\"toString\"><target><object code=\"java.lang.StringBuilder\"/></target></invoke>"), null, null, new MetaData[0]}
            },
            {
                createElement("<constructor><static-field-ref code=\"java.lang.System\" name=\"out\"/></constructor>"),
                new Object[]{null, importXML(new StaticFieldRefMetaData(null), "<static-field-ref code=\"java.lang.System\" name=\"out\"/>"), null, new MetaData[0]}
            },
            {
                createElement("<constructor><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/></constructor>"),
                new Object[]{null, null, importXML(new StaticInvokeMetaData(null), "<static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/>"), new MetaData[0]}
            },
            {
                createElement("<constructor><argument>hoge</argument></constructor>"),
                new Object[]{null, null, null, new MetaData[]{importXML(new ArgumentMetaData(null, null), "<argument>hoge</argument>")}}
            },
            {
                createElement("<constructor><argument>hoge</argument><argument>fuga</argument></constructor>"),
                new Object[]{null, null, null, new MetaData[]{importXML(new ArgumentMetaData(null, null), "<argument>hoge</argument>"), importXML(new ArgumentMetaData(null, null), "<argument>fuga</argument>")}}
            },
            {createElement("<constructor unknown=\"hoge\"/>"), new Object[]{null, null, null, new MetaData[0]}},
            {createElement("<constructor><unknown/></constructor>"), new Object[]{null, null, null, new MetaData[0]}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ConstructorMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(.*\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#toXML\\(java\\.lang\\.StringBuilder\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#getInvoke\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#getStaticFieldRef\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#getStaticInvoke\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#getArguments\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        ConstructorMetaData metaData = new ConstructorMetaData(null);
        metaData.importXML(element);
        assertMetaDataEquals(metaData.getInvoke(), (MetaData)assertValues[0]);
        assertMetaDataEquals(metaData.getStaticFieldRef(), (MetaData)assertValues[1]);
        assertMetaDataEquals(metaData.getStaticInvoke(), (MetaData)assertValues[2]);
        List<ArgumentMetaData> arguments = metaData.getArguments();
        MetaData[] args = (MetaData[])assertValues[3];
        if(args.length == 0){
            assertTrue(arguments.size() == 0);
        }else{
            assertEquals(arguments.size(), args.length);
            for(int i = 0; i < arguments.size(); i++){
                assertMetaDataEquals(arguments.get(i), args[i]);
            }
        }
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<constructor><invoke name=\"hoge\"/></constructor>")},
            {createElement("<constructor><invoke name=\"toString\"><target><object code=\"java.lang.StringBuilder\"/></target></invoke><invoke name=\"toString\"><target><object code=\"java.lang.StringBuilder\"/></target></invoke></constructor>")},
            {createElement("<constructor><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/></constructor>")},
            {createElement("<constructor><static-field-ref code=\"java.lang.System\" name=\"out\"/><static-field-ref code=\"java.lang.System\" name=\"out\"/></constructor>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ConstructorMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        ConstructorMetaData metaData = new ConstructorMetaData(null);
        metaData.importXML(element);
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<constructor/>"), "<constructor></constructor>"},
            {createElement("<constructor></constructor>"), "<constructor></constructor>"},
            {createElement("<constructor><invoke name=\"toString\"><target><object code=\"java.lang.StringBuilder\"/></target></invoke></constructor>"), "<constructor>" + System.getProperty("line.separator") + "    <invoke name=\"toString\">" + System.getProperty("line.separator") + "        <target>" + System.getProperty("line.separator") + "            <object code=\"java.lang.StringBuilder\"/>" + System.getProperty("line.separator") + "        </target>" + System.getProperty("line.separator") + "    </invoke>" + System.getProperty("line.separator") + "</constructor>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ConstructorMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#ConstructorMetaData\\(jp\\.ossc\\.nimbus\\.core\\.ObjectMetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ConstructorMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        ConstructorMetaData metaData = new ConstructorMetaData(null);
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
