/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 *
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.websocket.jetty;

import java.io.IOException;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import jp.ossc.nimbus.beans.ServiceNameEditor;
import jp.ossc.nimbus.core.ServiceManagerFactory;
import jp.ossc.nimbus.core.ServiceName;
import jp.ossc.nimbus.core.ServiceNotFoundException;
import jp.ossc.nimbus.service.websocket.jetty.WebSocketFactory;
import jp.ossc.nimbus.util.crypt.CryptParameters;
import jp.ossc.nimbus.util.crypt.FalsifiedParameterException;
import jp.ossc.nimbus.util.crypt.OverLimitExpiresException;

import org.eclipse.jetty.websocket.WebSocket;

/**
 * {@link WebSocketFactory}T[rXgāAWebSocket𐶐WebSocketServletB<p>
 * ̂ȂWebSocketڑs킹Ȃ߂ɁAڑF؂s@\B<br>
 * ڑF؂ł́AHTTPNGXgwb_܂̓NGXgNGŁAxN^AÍp[^AnbVl󂯎A{@link CryptParameters}gāAÍp[^𕜍āAʌoAg̒m鋤ʌƈv邩ŁAF؂sB<br>
 *
 * p[^ꗗ<br>
 * <table border="1">
 *   <tr bgcolor="#CCCCFF">
 *     <th>key</th>
 *     <th></th>
 *     <th>K{</th>
 *     <th>ftHg</th>
 *   </tr>
 *   <tr>
 *     <td>WebSocketFactoryServiceName</td>
 *     <td>{@link WebSocketFactory}T[rX̃T[rX</td>
 *     <td align="center"></td>
 *     <td>K{ݒ</td>
 *   </tr>
 *   <tr>
 *     <td>Origin</td>
 *     <td>HTTPwb_Origin̒l</td>
 *     <td align="center"></td>
 *     <td>ݒiHTTPwb_Origin͌؂Ȃj</td>
 *   </tr>
 *   <tr>
 *     <td>Authenticate</td>
 *     <td>ڑF؃tOitrue,falsej</td>
 *     <td align="center"></td>
 *     <td>ݒiڑF؂Ȃj</td>
 *   </tr>
 *   <tr>
 *     <td>CryptKey</td>
 *     <td>ڑF؂ŁAÍp[^𕜍ۂ̈Í</td>
 *     <td align="center"></td>
 *     <td>ڑF؂sꍇ́AK{ݒ</td>
 *   </tr>
 *   <tr>
 *     <td>CryptAlgorithm</td>
 *     <td>ڑF؂ŁAÍp[^𕜍ۂ̈ÍASY</td>
 *     <td align="center"></td>
 *     <td>{@link CryptParameters#DEFAULT_SECRET_KEY_ALGORITHM}</td>
 *   </tr>
 *   <tr>
 *     <td>CryptTransformation</td>
 *     <td>ڑF؂ŁAÍp[^𕜍ۂ̈Íϊ</td>
 *     <td align="center"></td>
 *     <td>{@link CryptParameters#DEFAULT_TRANSFORMATION}</td>
 *   </tr>
 *   <tr>
 *     <td>IVLength</td>
 *     <td>ڑF؂ŁAÍp[^𕜍ۂ̏xN^</td>
 *     <td align="center"></td>
 *     <td>{@link CryptParameters#DEFAULT_INITIAL_VECTOR_LENGTH}</td>
 *   </tr>
 *   <tr>
 *     <td>CryptProvider</td>
 *     <td>ڑF؂ŁAÍp[^𕜍ۂ̈ÍvoC_</td>
 *     <td align="center"></td>
 *     <td>ݒ</td>
 *   </tr>
 *   <tr>
 *     <td>HashKey</td>
 *     <td>ڑF؂ŁAₖh~̃nbVl𐶐ۂ̃nbVʌ</td>
 *     <td align="center"></td>
 *     <td>ڑF؂sꍇŉₖh~@\gpꍇ́AK{ݒ</td>
 *   </tr>
 *   <tr>
 *     <td>HashAlgorithm</td>
 *     <td>ڑF؂ŁAₖh~̃nbVl𐶐ۂ̃nbVASY</td>
 *     <td align="center"></td>
 *     <td>{@link CryptParameters#DEFAULT_HASH_ALGORITHM}</td>
 *   </tr>
 *   <tr>
 *     <td>HashProvider</td>
 *     <td>ڑF؂ŁAₖh~̃nbVl𐶐ۂ̃nbVvoC_</td>
 *     <td align="center"></td>
 *     <td>ݒ</td>
 *   </tr>
 *   <tr>
 *     <td>CryptExpires</td>
 *     <td>ڑF؂ŁAÍp[^𕜍ۂ̈Íp[^̗̂L[ms]</td>
 *     <td align="center"></td>
 *     <td>-1iLԃ`FbNsȂj</td>
 *   </tr>
 *   <tr>
 *     <td>CryptParameterName</td>
 *     <td>ڑF؂ŁAÍp[^NGXgNG擾ꍇ̃p[^</td>
 *     <td align="center"></td>
 *     <td>{@link #DEFAULT_PARAMETER_NAME_CRYPT}</td>
 *   </tr>
 *   <tr>
 *     <td>IVParameterName</td>
 *     <td>ڑF؂ŁAxN^NGXgNG擾ꍇ̃p[^</td>
 *     <td align="center"></td>
 *     <td>{@link #DEFAULT_PARAMETER_NAME_IV}</td>
 *   </tr>
 *   <tr>
 *     <td>HashParameterName</td>
 *     <td>ڑF؂ŁAnbVlNGXgNG擾ꍇ̃p[^</td>
 *     <td align="center"></td>
 *     <td>{@link #DEFAULT_PARAMETER_NAME_HASH}</td>
 *   </tr>
 *   <tr>
 *     <td>CryptHttpHeaderName</td>
 *     <td>ڑF؂ŁAÍp[^HTTPNGXgwb_擾ꍇ̃wb_</td>
 *     <td align="center"></td>
 *     <td>{@link #DEFAULT_HTTP_HEADER_NAME_CRYPT}</td>
 *   </tr>
 *   <tr>
 *     <td>IVHttpHeaderName</td>
 *     <td>ڑF؂ŁAxN^HTTPNGXgwb_擾ꍇ̃wb_</td>
 *     <td align="center"></td>
 *     <td>{@link #DEFAULT_HTTP_HEADER_NAME_IV}</td>
 *   </tr>
 *   <tr>
 *     <td>HashHttpHeaderName</td>
 *     <td>ڑF؂ŁAnbVlHTTPNGXgwb_擾ꍇ̃wb_</td>
 *     <td align="center"></td>
 *     <td>{@link #DEFAULT_HTTP_HEADER_NAME_HASH}</td>
 *   </tr>
 *   <tr>
 *     <td>CommonKey</td>
 *     <td>ڑF؂ŁAÍp[^𕜍ē鋤ʌ</td>
 *     <td align="center"></td>
 *     <td>ڑF؂sꍇ́AK{ݒ</td>
 *   </tr>
 *   <tr>
 *     <td>CommonKeyMapKey</td>
 *     <td>ڑF؂ŁAÍp[^̋ʌ̃L[</td>
 *     <td align="center"></td>
 *     <td>{@link #DEFAULT_MAP_KEY_COMMON_KEY}</td>
 *   </tr>
 *   <tr>
 *     <td>bufferSize</td>
 *     <td>WebSocketŎgpobt@TCY</td>
 *     <td align="center"></td>
 *     <td>64 * 1024</td>
 *   </tr>
 *   <tr>
 *     <td>maxIdleTime</td>
 *     <td>WebSocket̍őACh^C</td>
 *     <td align="center"></td>
 *     <td>300000</td>
 *   </tr>
 *   <tr>
 *     <td>maxTextMessageSize</td>
 *     <td>WebSocketŁAeLXgf[^MۂMaxTCY</td>
 *     <td align="center"></td>
 *     <td>16 * 1024</td>
 *   </tr>
 *   <tr>
 *     <td>maxBinaryMessageSize</td>
 *     <td>WebSocketŁAoCif[^MۂMaxTCY</td>
 *     <td align="center"></td>
 *     <td>-1</td>
 *   </tr>
 * </table>
 *
 * @author M.Takata
 */
public class WebSocketServlet extends org.eclipse.jetty.websocket.WebSocketServlet{

    private static final long serialVersionUID = -4339303201578450133L;

    /**
     * p[^F{@link WebSocketFactory}T[rX̃T[rXB<p>
     * K{ݒłB<br>
     */
    public static final String INIT_PARAM_NAME_WEB_SOCKET_FACTORY_SERVICE_NAME = "WebSocketFactoryServiceName";

    /**
     * p[^FHTTPwb_Origin̒lB<p>
     * ̃p[^ݒ肷ƁAK\HTTPwb_Origin̒l؂łB<br>
     * ftHgł́AHTTPwb_Origin͌؂ȂB<br>
     */
    public static final String INIT_PARAM_NAME_ORIGIN = "Origin";

    /**
     * p[^FڑF؃tOB<p>
     * trueݒ肷ƁAWebSocket̐ڑɔF؂sB<br>
     * ftHgł́AfalseŁAڑF؂ȂB<br>
     */
    public static final String INIT_PARAM_NAME_AUTHENTICATE = "Authenticate";

    /**
     * p[^FÍB<p>
     * ڑF؂ŁAÍp[^𕜍ۂ̈Íݒ肷B<br>
     * ڑF؂sꍇ́AK{ݒłB<br>
     */
    public static final String INIT_PARAM_NAME_CRYPT_KEY = "CryptKey";

    /**
     * p[^FÍASYB<p>
     * ڑF؂ŁAÍp[^𕜍ۂ̈ÍASYݒ肷B<br>
     * ftHǵA{@link CryptParameters#DEFAULT_SECRET_KEY_ALGORITHM}B<br>
     */
    public static final String INIT_PARAM_NAME_CRYPT_ALGORITHM = "CryptAlgorithm";

    /**
     * p[^FÍϊB<p>
     * ڑF؂ŁAÍp[^𕜍ۂ̈Íϊݒ肷B<br>
     * ftHǵA{@link CryptParameters#DEFAULT_TRANSFORMATION}B<br>
     */
    public static final String INIT_PARAM_NAME_CRYPT_TRANSFORMATION = "CryptTransformation";

    /**
     * p[^FxN^B<p>
     * ڑF؂ŁAÍp[^𕜍ۂ̏xN^ݒ肷B<br>
     * ftHǵA{@link CryptParameters#DEFAULT_INITIAL_VECTOR_LENGTH}B<br>
     */
    public static final String INIT_PARAM_NAME_IV_LENGTH = "IVLength";

    /**
     * p[^FÍvoC_B<p>
     * ڑF؂ŁAÍp[^𕜍ۂ̈ÍvoC_ݒ肷B<br>
     * ftHǵAÍvoC_w肵ȂB<br>
     */
    public static final String INIT_PARAM_NAME_CRYPT_PROVIDER = "CryptProvider";

    /**
     * p[^FnbVʌB<p>
     * ڑF؂ŁAₖh~̃nbVl𐶐ۂ̃nbVʌݒ肷B<br>
     * ڑF؂sꍇŉₖh~@\gpꍇ́AK{ݒłB<br>
     */
    public static final String INIT_PARAM_NAME_HASH_KEY = "HashKey";

    /**
     * p[^FnbVASYB<p>
     * ڑF؂ŁAₖh~̃nbVl𐶐ۂ̃nbVASYݒ肷B<br>
     * ftHǵA{@link CryptParameters#DEFAULT_HASH_ALGORITHM}B<br>
     */
    public static final String INIT_PARAM_NAME_HASH_ALGORITHM = "HashAlgorithm";

    /**
     * p[^FnbVvoC_B<p>
     * ڑF؂ŁAₖh~̃nbVl𐶐ۂ̃nbVvoC_ݒ肷B<br>
     * ftHǵAnbVvoC_w肵ȂB<br>
     */
    public static final String INIT_PARAM_NAME_HASH_PROVIDER = "HashProvider";

    /**
     * p[^FÍp[^LԁB<p>
     * ڑF؂ŁAÍp[^𕜍ۂ̈Íp[^̗̂L[ms]ݒ肷B<br>
     * ftHǵA-1ŗLԃ`FbNsȂB<br>
     */
    public static final String INIT_PARAM_NAME_CRYPT_EXPIRES = "CryptExpires";

    /**
     * p[^FÍp[^̃NGXgp[^B<p>
     * ڑF؂ŁAÍp[^NGXgNG擾ꍇ̃p[^ݒ肷B<br>
     * ftHǵA{@link #DEFAULT_PARAMETER_NAME_CRYPT}B<br>
     */
    public static final String INIT_PARAM_NAME_CRYPT_PARAMETER_NAME = "CryptParameterName";

    /**
     * p[^FxN^̃NGXgp[^B<p>
     * ڑF؂ŁAxN^NGXgNG擾ꍇ̃p[^ݒ肷B<br>
     * ftHǵA{@link #DEFAULT_PARAMETER_NAME_IV}B<br>
     */
    public static final String INIT_PARAM_NAME_IV_PARAMETER_NAME = "IVParameterName";

    /**
     * p[^FnbVl̃NGXgp[^B<p>
     * ڑF؂ŁAnbVlNGXgNG擾ꍇ̃p[^ݒ肷B<br>
     * ftHǵA{@link #DEFAULT_PARAMETER_NAME_HASH}B<br>
     */
    public static final String INIT_PARAM_NAME_HASH_PARAMETER_NAME = "HashParameterName";

    /**
     * p[^FÍp[^HTTPNGXgwb_B<p>
     * ڑF؂ŁAÍp[^HTTPNGXgwb_擾ꍇ̃wb_ݒ肷B<br>
     * ftHǵA{@link #DEFAULT_HTTP_HEADER_NAME_CRYPT}B<br>
     */
    public static final String INIT_PARAM_NAME_CRYPT_HTTP_HEADER_NAME_NAME = "CryptHttpHeaderName";

    /**
     * p[^FxN^HTTPNGXgwb_B<p>
     * ڑF؂ŁAxN^HTTPNGXgwb_擾ꍇ̃wb_ݒ肷B<br>
     * ftHǵA{@link #DEFAULT_HTTP_HEADER_NAME_IV}B<br>
     */
    public static final String INIT_PARAM_NAME_IV_HTTP_HEADER_NAME_NAME = "IVHttpHeaderName";

    /**
     * p[^FnbVlHTTPNGXgwb_B<p>
     * ڑF؂ŁAnbVlHTTPNGXgwb_擾ꍇ̃wb_ݒ肷B<br>
     * ftHǵA{@link #DEFAULT_HTTP_HEADER_NAME_HASH}B<br>
     */
    public static final String INIT_PARAM_NAME_HASH_HTTP_HEADER_NAME_NAME = "HashHttpHeaderName";

    /**
     * p[^FʌB<p>
     * ڑF؂ŁAÍp[^𕜍ē鋤ʌݒ肷B<br>
     * ڑF؂sꍇ́AK{ݒłB<br>
     */
    public static final String INIT_PARAM_NAME_COMMON_KEY = "CommonKey";

    /**
     * p[^FÍp[^̋ʌ̃L[B<p>
     * ڑF؂ŁAÍp[^̋ʌ̃L[ݒ肷B<br>
     * ftHǵA{@link #DEFAULT_MAP_KEY_COMMON_KEY}B<br>
     */
    public static final String INIT_PARAM_NAME_COMMON_KEY_MAP_KEY = "CommonKeyMapKey";

    /**
     * ڑF؂ŁAÍp[^NGXgNG擾ꍇ̃p[^̃ftHglB<p>
     */
    public static final String DEFAULT_PARAMETER_NAME_CRYPT = "crypt";

    /**
     * ڑF؂ŁAxN^NGXgNG擾ꍇ̃p[^̃ftHglB<p>
     */
    public static final String DEFAULT_PARAMETER_NAME_IV = "iv";

    /**
     * ڑF؂ŁAnbVlNGXgNG擾ꍇ̃p[^̃ftHglB<p>
     */
    public static final String DEFAULT_PARAMETER_NAME_HASH = "hash";

    /**
     * ڑF؂ŁAʌÍp[^擾L[̃ftHglB<p>
     */
    public static final String DEFAULT_MAP_KEY_COMMON_KEY = "commonKey";

    /**
     * ڑF؂ŁAÍp[^HTTPNGXgwb_擾ꍇ̃wb_̃ftHglB<p>
     */
    public static final String DEFAULT_HTTP_HEADER_NAME_CRYPT = "x-websocket-crypt";

    /**
     * ڑF؂ŁAxN^HTTPNGXgwb_擾ꍇ̃wb_̃ftHglB<p>
     */
    public static final String DEFAULT_HTTP_HEADER_NAME_IV = "x-websocket-iv";

    /**
     * ڑF؂ŁAnbVlHTTPNGXgwb_擾ꍇ̃wb_̃ftHglB<p>
     */
    public static final String DEFAULT_HTTP_HEADER_NAME_HASH = "x-websocket-hash";

    protected WebSocketFactory webSocketFactory;
    protected Pattern originPattern;
    protected boolean isAutheticate = false;
    protected long cryptExpires = -1;
    protected CryptParameters crypt;
    protected String cryptParameterName = DEFAULT_PARAMETER_NAME_CRYPT;
    protected String ivParameterName = DEFAULT_PARAMETER_NAME_IV;
    protected String hashParameterName = DEFAULT_PARAMETER_NAME_HASH;
    protected String cryptHttpHeaderName = DEFAULT_HTTP_HEADER_NAME_CRYPT;
    protected String ivHttpHeaderName = DEFAULT_HTTP_HEADER_NAME_IV;
    protected String hashHttpHeaderName = DEFAULT_HTTP_HEADER_NAME_HASH;
    protected boolean isCheckFalsified;
    protected String commonKeyMapKey = DEFAULT_MAP_KEY_COMMON_KEY;
    protected String commonKey;

    @Override
    public void init() throws ServletException{
        super.init();

        final ServletConfig config = getServletConfig();
        final ServiceNameEditor editor = new ServiceNameEditor();

        String name = config.getInitParameter(
            INIT_PARAM_NAME_WEB_SOCKET_FACTORY_SERVICE_NAME
        );
        if(name == null){
            throw new ServletException("init-param '" + INIT_PARAM_NAME_WEB_SOCKET_FACTORY_SERVICE_NAME + "' must be specified.");
        }
        editor.setAsText(name);
        ServiceName webSocketFactoryServiceName = (ServiceName)editor.getValue();
        try{
            webSocketFactory = ServiceManagerFactory.getServiceObject(webSocketFactoryServiceName);
        }catch(ServiceNotFoundException e){
            throw new ServletException(e);
        }

        String origin = config.getInitParameter(
            INIT_PARAM_NAME_ORIGIN
        );
        if(origin != null){
            try{
                originPattern = Pattern.compile(origin);
            }catch(PatternSyntaxException e){
                throw new ServletException(e);
            }
        }

        String tmpCryptParameterName = config.getInitParameter(
            INIT_PARAM_NAME_CRYPT_PARAMETER_NAME
        );
        if(tmpCryptParameterName != null){
            cryptParameterName = tmpCryptParameterName;
        }

        String tmpIVParameterName = config.getInitParameter(
            INIT_PARAM_NAME_IV_PARAMETER_NAME
        );
        if(tmpIVParameterName != null){
            ivParameterName = tmpIVParameterName;
        }

        String tmpHashParameterName = config.getInitParameter(
            INIT_PARAM_NAME_HASH_PARAMETER_NAME
        );
        if(tmpHashParameterName != null){
            hashParameterName = tmpHashParameterName;
        }

        String tmpCryptHttpHeaderName = config.getInitParameter(
            INIT_PARAM_NAME_CRYPT_HTTP_HEADER_NAME_NAME
        );
        if(tmpCryptHttpHeaderName != null){
            cryptHttpHeaderName = tmpCryptHttpHeaderName;
        }

        String tmpIVHttpHeaderName = config.getInitParameter(
            INIT_PARAM_NAME_IV_HTTP_HEADER_NAME_NAME
        );
        if(tmpIVHttpHeaderName != null){
            ivHttpHeaderName = tmpIVHttpHeaderName;
        }

        String tmpHashHttpHeaderName = config.getInitParameter(
            INIT_PARAM_NAME_HASH_HTTP_HEADER_NAME_NAME
        );
        if(tmpHashHttpHeaderName != null){
            hashHttpHeaderName = tmpHashHttpHeaderName;
        }

        String authenticate = config.getInitParameter(
            INIT_PARAM_NAME_AUTHENTICATE
        );
        if(authenticate != null){
            isAutheticate = Boolean.valueOf(authenticate);
        }
        if(isAutheticate){
            String tmpCommonKey = config.getInitParameter(
                INIT_PARAM_NAME_COMMON_KEY
            );
            if(tmpCommonKey == null){
                throw new ServletException("init-param '" + INIT_PARAM_NAME_COMMON_KEY + "' must be specified.");
            }
            commonKey = tmpCommonKey;

            String tmpCommonKeyMapKey = config.getInitParameter(
                INIT_PARAM_NAME_COMMON_KEY_MAP_KEY
            );
            if(tmpCommonKeyMapKey != null){
                commonKeyMapKey = tmpCommonKeyMapKey;
            }

            String tmpCryptKey = config.getInitParameter(
                INIT_PARAM_NAME_CRYPT_KEY
            );
            if(tmpCryptKey == null){
                throw new ServletException("init-param '" + INIT_PARAM_NAME_CRYPT_KEY + "' must be specified.");
            }
            byte[] cryptKey = tmpCryptKey.getBytes();

            String tmpCryptAlgorithm = config.getInitParameter(
                INIT_PARAM_NAME_CRYPT_ALGORITHM
            );
            String cryptAlgorithm = CryptParameters.DEFAULT_SECRET_KEY_ALGORITHM;
            if(tmpCryptAlgorithm != null){
                cryptAlgorithm = tmpCryptAlgorithm;
            }

            String tmpCryptTransformation = config.getInitParameter(
                INIT_PARAM_NAME_CRYPT_TRANSFORMATION
            );
            String cryptTransformation = CryptParameters.DEFAULT_TRANSFORMATION;
            if(tmpCryptTransformation != null){
                cryptTransformation = tmpCryptTransformation;
            }

            String tmpIVLength = config.getInitParameter(
                INIT_PARAM_NAME_IV_LENGTH
            );
            int ivLength = CryptParameters.DEFAULT_INITIAL_VECTOR_LENGTH;
            if(tmpIVLength != null){
                try{
                    ivLength = Integer.parseInt(tmpIVLength);
                }catch(NumberFormatException e){
                    throw new ServletException("init-param '" + INIT_PARAM_NAME_IV_LENGTH + "' is illegal. value=" + tmpIVLength);
                }
            }

            String cryptProvider = config.getInitParameter(
                INIT_PARAM_NAME_CRYPT_PROVIDER
            );

            String hashKey = config.getInitParameter(
                INIT_PARAM_NAME_HASH_KEY
            );
            String hashAlgorithm = CryptParameters.DEFAULT_HASH_ALGORITHM;
            String hashProvider = null;
            if(hashKey != null){
                String tmpHashAlgorithm = config.getInitParameter(
                    INIT_PARAM_NAME_HASH_ALGORITHM
                );
                if(tmpHashAlgorithm != null){
                    hashAlgorithm = tmpHashAlgorithm;
                }

                hashProvider = config.getInitParameter(
                    INIT_PARAM_NAME_HASH_PROVIDER
                );
                isCheckFalsified = true;
            }
            try{
                crypt = new CryptParameters(
                    cryptKey,
                    cryptAlgorithm,
                    cryptTransformation,
                    ivLength,
                    cryptProvider,
                    hashKey
                );
                if(isCheckFalsified){
                    if(hashProvider == null){
                        crypt.setHashAlgorithm(hashAlgorithm);
                    }else{
                        crypt.setHashAlgorithm(hashAlgorithm, hashProvider);
                    }
                }
            }catch(Exception e){
                throw new ServletException("CryptParameters instansiate error.", e);
            }

            String cryptExpires = config.getInitParameter(
                INIT_PARAM_NAME_CRYPT_EXPIRES
            );
            if(cryptExpires != null){
                try{
                    this.cryptExpires = Long.parseLong(cryptExpires);
                }catch(NumberFormatException e){
                    throw new ServletException("init-param '" + INIT_PARAM_NAME_CRYPT_EXPIRES + "' is illegal. value=" + cryptExpires);
                }
            }

        }
    }

    @Override
    protected void service(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException{
        if(isAutheticate){
            String params = request.getParameter(cryptParameterName);
            boolean isHttpHeader = false;
            if(params == null){
                params = request.getHeader(cryptHttpHeaderName);
                if(params != null){
                    isHttpHeader = true;
                }
            }
            String iv = isHttpHeader ? request.getHeader(ivHttpHeaderName) : request.getParameter(ivParameterName);
            if(params == null || iv == null){
                response.sendError(HttpServletResponse.SC_FORBIDDEN);
                return;
            }
            Map<String, Object> paramMap = null;
            if(isCheckFalsified){
                String hash = isHttpHeader ? request.getHeader(hashHttpHeaderName) : request.getParameter(hashParameterName);
                try{
                    paramMap = crypt.decrypt(iv, params, hash, cryptExpires);
                }catch(FalsifiedParameterException e){
                    response.sendError(HttpServletResponse.SC_FORBIDDEN);
                    return;
                }catch(OverLimitExpiresException e){
                    response.sendError(HttpServletResponse.SC_FORBIDDEN);
                    return;
                }
            }else{
                try{
                    paramMap = crypt.decrypt(iv, params, cryptExpires);
                }catch(FalsifiedParameterException e){
                    response.sendError(HttpServletResponse.SC_FORBIDDEN);
                    return;
                }catch(OverLimitExpiresException e){
                    response.sendError(HttpServletResponse.SC_FORBIDDEN);
                    return;
                }
            }
            String commonKey = (String)paramMap.get(commonKeyMapKey);
            if(!this.commonKey.equals(commonKey)){
                response.sendError(HttpServletResponse.SC_FORBIDDEN);
                return;
            }
        }
        super.service(request, response);
    }

    @Override
    public boolean checkOrigin(javax.servlet.http.HttpServletRequest request, String origin){
        return originPattern == null ? true : (origin == null ? false : originPattern.matcher(origin).matches());
    }

    @Override
    public WebSocket doWebSocketConnect(HttpServletRequest request, String protocol){
        return webSocketFactory.createWebSocket(request, protocol);
    }
}