/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 *
 * Copyright 2011 The Nimbus2 Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.util.converter;

import static org.testng.Assert.assertEquals;

import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

/**
 * {@link DateFormatConverter}eXgP[XB<p>
 *
 * @author M.Ishida
 */
public class DateFormatConverterTest {

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.DateFormatConverter#DateFormatConverter()",
            "Normal" })
    public void testDateFormatConverter() {
        DateFormatConverter convertter = new DateFormatConverter();
        assertEquals(convertter.getConvertType(),DateFormatConverter.DATE_TO_STRING);
        assertEquals(convertter.getFormat(),"yyyy/MM/dd HH:mm:ss.SSS");
        assertEquals(convertter.getNullString(),"");
    }

    @DataProvider(name = "DateFormatConverter")
    public Object[][] dataDateFormatConverter() {
        return new Object[][] {
                { DateFormatConverter.DATE_TO_STRING, "yyyy/MM/dd HH:mm:ss.SSS"},
                { DateFormatConverter.STRING_TO_DATE, "yyyy/MM/dd"},
            };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.DateFormatConverter#DateFormatConverter(int, java.lang.String)",
            "Normal" },
            dataProvider = "DateFormatConverter")
    public void testDateFormatConverter(int type, String format) {
        DateFormatConverter convertter = new DateFormatConverter(type, format);
        assertEquals(convertter.getConvertType(), type);
        assertEquals(convertter.getFormat(), format);
    }

    @DataProvider(name = "DateFormatConverterError")
    public Object[][] dataDateFormatConverterError() {
        return new Object[][] {
                { -1 ,""},
                { 0  ,""},
                { 3  ,""}
            };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.DateFormatConverter#DateFormatConverter(int, java.lang.String)",
            "Error" },
            expectedExceptions = { IllegalArgumentException.class },
            dataProvider = "DateFormatConverterError")
    public void testDateFormatConverterError(int type, String format)
            throws IllegalArgumentException {
        new DateFormatConverter(type, format);
    }

    @DataProvider(name = "SetConvertType")
    public Object[][] dataSetConvertType() {
        return new Object[][] { { DateFormatConverter.DATE_TO_STRING },
                { DateFormatConverter.STRING_TO_DATE } };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.DateFormatConverter#setConvertType(int)",
            "jp.ossc.nimbus.util.converter.DateFormatConverter#getConvertType()",
            "Normal" },
            dataProvider = "SetConvertType")
    public void testSetConvertType(int type) {
        DateFormatConverter convertter = new DateFormatConverter();
        convertter.setConvertType(type);
        assertEquals(convertter.getConvertType(), type);
    }

    @DataProvider(name = "SetConvertTypeError")
    public Object[][] dataSetConvertTypeError() {
        return new Object[][] { { 0},
                { -1},
                { 3 },
                { 100 } };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.DateFormatConverter#setConvertType(int)",
            "Error" },
            expectedExceptions = { IllegalArgumentException.class },
            dataProvider = "SetConvertTypeError")
    public void testSetConvertTypeError(int type) {
        DateFormatConverter convertter = new DateFormatConverter();
        convertter.setConvertType(type);
    }


    @DataProvider(name = "SetFormat")
    public Object[][] dataSetFormat() {
        return new Object[][] { { "yyyy/MM/dd" },
                                { "yyMMddHHmmssZ" }
                            };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.DateFormatConverter#setFormat(java.lang.String)",
            "jp.ossc.nimbus.util.converter.DateFormatConverter#getFormat()",
            "Normal" },
            dataProvider = "SetFormat")
    public void testSetFormat(String format) {
        DateFormatConverter convertter = new DateFormatConverter();
        convertter.setFormat(format);
        assertEquals(convertter.getFormat(), format);
    }

    @DataProvider(name = "SetFormatError")
    public Object[][] dataSetFormatError() {
        return new Object[][] { { "test" },
                            { null }
                            };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.DateFormatConverter#setFormat(java.lang.String)",
            "Error" },
            expectedExceptions = { IllegalArgumentException.class },
            dataProvider = "SetFormatError")
    public void testSetFormatError(String format) {
        DateFormatConverter convertter = new DateFormatConverter();
        convertter.setFormat(format);
    }

    @DataProvider(name = "SetNullString")
    public Object[][] dataSetNullString() {
        return new Object[][] { { "" },
                { "null" },
                { null }};
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.DateFormatConverter#setNullString(java.lang.String)",
            "jp.ossc.nimbus.util.converter.DateFormatConverter#getNullString()",
            "Normal" },
            dataProvider = "SetNullString")
    public void testSetNullObject(String str) {
        DateFormatConverter convertter = new DateFormatConverter();
        convertter.setNullString(str);
        assertEquals(convertter.getNullString(), str);
    }

    @DataProvider(name = "Convert")
    public Object[][] dataConvert() {
        return new Object[][] {
                { DateFormatConverter.DATE_TO_STRING ,"yyyy/MM/dd HH:mm:ss.SSS", null, null, null},
                { DateFormatConverter.DATE_TO_STRING ,"yyyy/MM/dd HH:mm:ss.SSS", "Null", null, "Null"},
                { DateFormatConverter.DATE_TO_STRING ,"yyyy/MM/dd HH:mm:ss.SSS", "", new java.util.Date(1000000000000l), "2001/09/09 10:46:40.000"},
                { DateFormatConverter.DATE_TO_STRING ,"yyyy/MM/dd", "", new java.util.Date(1000000000000l), "2001/09/09"},
                { DateFormatConverter.STRING_TO_DATE ,"yyyy/MM/dd HH:mm:ss.SSS", null, null, null},
                { DateFormatConverter.STRING_TO_DATE ,"yyyy/MM/dd HH:mm:ss.SSS", null, "", null},
                { DateFormatConverter.STRING_TO_DATE ,"yyyy/MM/dd HH:mm:ss.SSS", "Null", null, null},
                { DateFormatConverter.STRING_TO_DATE ,"yyyy/MM/dd HH:mm:ss.SSS", null, "2001/09/09 10:46:40.000", new java.util.Date(1000000000000l)},
                };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.DateFormatConverter#convert(java.lang.Object)",
            "Normal" },
            dataProvider = "Convert")
    public void testConvert(int type, String format, String nullStr, Object targetObj, Object returnObj) {
        DateFormatConverter convertter = new DateFormatConverter();
        convertter.setConvertType(type);
        convertter.setFormat(format);
        convertter.setNullString(nullStr);
        assertEquals(convertter.convert(targetObj), returnObj);
    }

    @DataProvider(name = "ConvertError")
    public Object[][] dataConvertError() {
        return new Object[][] {
                { DateFormatConverter.STRING_TO_DATE ,"yyyy/MM/dd HH:mm:ss.SSS", null, "test"},
                };
    }

    @Test(groups = {
            "jp.ossc.nimbus.util.converter.DateFormatConverter#convert(java.lang.Object)",
            "Error" },
            expectedExceptions = { ConvertException.class },
            dataProvider = "ConvertError")
    public void testConvertError(int type, String format, String nullStr, Object targetObj) {
        DateFormatConverter convertter = new DateFormatConverter();
        convertter.setConvertType(type);
        convertter.setFormat(format);
        convertter.setNullString(nullStr);
        convertter.convert(targetObj);
    }

}
