/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.validator;

import java.util.*;
import java.sql.*;
import java.lang.reflect.InvocationTargetException;

import jp.ossc.nimbus.beans.*;
import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.beans.dataset.Record;
import jp.ossc.nimbus.beans.dataset.RecordList;
import jp.ossc.nimbus.beans.dataset.PropertySchema;
import jp.ossc.nimbus.service.sql.ConnectionFactory;
import jp.ossc.nimbus.service.sql.ConnectionFactoryException;
import jp.ossc.nimbus.service.sql.PersistentManager;
import jp.ossc.nimbus.service.sql.PersistentException;
import jp.ossc.nimbus.service.context.Context;
import jp.ossc.nimbus.service.codemaster.CodeMasterFinder;
import jp.ossc.nimbus.util.validator.Validator;
import jp.ossc.nimbus.util.validator.ValidateException;

/**
 * }X^of[^B<p>
 * DB̃}X^Ɋ܂܂Ă邩؂B<br>
 * }X^̎擾@ƂĂQނA}X^̍XVpxɉđI\łB<br>
 * <p>
 * }X^̍XVpxꍇ́A؂̓sx}X^KvB<br>
 * ̂悤ȏꍇ́A{@link #setQuery(String)}A{@link #setConnectionFactoryServiceName(ServiceName)}A{@link #setPersistentManagerServiceName(ServiceName)}ݒ肷B<br>
 * <p>
 * }X^̍XVpxႢꍇ́A؂̓sx}X^KvȂB<br>
 * ̂悤ȃ}X^gpꍇANimbusł͒ʏ{@link CodeMasterFinder R[h}X^}ɁA}X^RecordListo^ĂŁA}X^𖈉񌟍Ȃ悤ɂB<br>
 * ̋@\𗘗p鎖ŁA؂̓sx}X^ɁAR[h}X^̃}X^RecordListɑ΂ēIs؂邱ƂłB<br>
 * R[h}X^̎擾@ɂ͂QނAǂݎѐۏႷꍇ́AXbhReLXg擾B̏ꍇA{@link #setThreadContextServiceName(ServiceName)}A{@link #setCodeMasterThreadContextKey(String)}ݒ肷B<br>
 * ܂AǂݎѐۏKvȂꍇ́A{@link CodeMasterFinder}擾鎖\łB̏ꍇA{@link #setCodeMasterFinderServiceName(ServiceName)}ݒ肷B<br>
 * 擾R[h}X^}X^RecordList肷邽߂ɁA{@link #setCodeMasterName(String)}ݒ肷B<br>
 * }X^RecordList瓮Isۂ̌́AؒlCӂ̏ɍv郌R[h邩ǂ؂鎖\ŁȀꍇ́A{@link #setSearchCondition(String)}A{@link #setBindDataMap(String, String)}ݒ肷B<br>
 * ؑΏۂ̒lA}X^RecordList̃vC}L[̒lŁAv郌R[h邩ǂ؂ꍇ́AL̐ݒ͕svłB<br>
 * 
 * @author M.Takata
 */
public class MasterValidatorService extends ServiceBase
 implements Validator, MasterValidatorServiceMBean{
    
    private static final long serialVersionUID = 3833661471756025996L;
    private ServiceName connectionFactoryServiceName;
    private ConnectionFactory connectionFactory;
    private ServiceName persistentManagerServiceName;
    private PersistentManager persistentManager;
    private String query;
    
    private ServiceName codeMasterFinderServiceName;
    private CodeMasterFinder codeMasterFinder;
    
    private ServiceName threadContextServiceName;
    private Context<Object, Object> threadContext;
    private String codeMasterThreadContextKey
         = jp.ossc.nimbus.service.aop.interceptor.ThreadContextKey.CODEMASTER;
    
    private String codeMasterName;
    private String searchCondition;
    private Map<String, Property> bindDataMap;
    
    // MasterValidatorServiceMBeanJavaDoc
    public void setConnectionFactoryServiceName(ServiceName name){
        connectionFactoryServiceName = name;
    }
    // MasterValidatorServiceMBeanJavaDoc
    public ServiceName getConnectionFactoryServiceName(){
        return connectionFactoryServiceName;
    }
    
    // MasterValidatorServiceMBeanJavaDoc
    public void setPersistentManagerServiceName(ServiceName name){
        persistentManagerServiceName = name;
    }
    // MasterValidatorServiceMBeanJavaDoc
    public ServiceName getPersistentManagerServiceName(){
        return persistentManagerServiceName;
    }
    
    // MasterValidatorServiceMBeanJavaDoc
    public void setQuery(String query){
        this.query = query;
    }
    // MasterValidatorServiceMBeanJavaDoc
    public String getQuery(){
        return query;
    }
    
    // MasterValidatorServiceMBeanJavaDoc
    public void setCodeMasterFinderServiceName(ServiceName name){
        codeMasterFinderServiceName = name;
    }
    // MasterValidatorServiceMBeanJavaDoc
    public ServiceName getCodeMasterFinderServiceName(){
        return codeMasterFinderServiceName;
    }
    
    // MasterValidatorServiceMBeanJavaDoc
    public void setThreadContextServiceName(ServiceName name){
        threadContextServiceName = name;
    }
    // MasterValidatorServiceMBeanJavaDoc
    public ServiceName getThreadContextServiceName(){
        return threadContextServiceName;
    }
    
    // MasterValidatorServiceMBeanJavaDoc
    public void setCodeMasterThreadContextKey(String key){
        codeMasterThreadContextKey = key;
    }
    // MasterValidatorServiceMBeanJavaDoc
    public String getCodeMasterThreadContextKey(){
        return codeMasterThreadContextKey;
    }
    
    // MasterValidatorServiceMBeanJavaDoc
    public void setCodeMasterName(String name){
        codeMasterName = name;
    }
    // MasterValidatorServiceMBeanJavaDoc
    public String getCodeMasterName(){
        return codeMasterName;
    }
    
    // MasterValidatorServiceMBeanJavaDoc
    public void setSearchCondition(String name){
        searchCondition = name;
    }
    // MasterValidatorServiceMBeanJavaDoc
    public String getSearchCondition(){
        return searchCondition;
    }
    
    // MasterValidatorServiceMBeanJavaDoc
    public void setBindDataMap(String key, String valueKey){
        if(bindDataMap == null){
            bindDataMap = new HashMap<String, Property>();
        }
        if(!valueKey.startsWith(BIND_DATA_VALUE_KEY)){
            throw new IllegalArgumentException("ValueKey must start with 'VALUE'.");
        }
        if(valueKey.equals(BIND_DATA_VALUE_KEY)){
            bindDataMap.put(key, null);
        }else{
            valueKey = valueKey.substring(BIND_DATA_VALUE_KEY.length());
            if(valueKey.charAt(0) == '.'){
                valueKey = valueKey.substring(1);
            }
            bindDataMap.put(key, PropertyFactory.createProperty(valueKey));
        }
    }
    // MasterValidatorServiceMBeanJavaDoc
    public String getBindDataMap(String key){
        if(bindDataMap == null){
            return null;
        }
        final Property prop = (Property)bindDataMap.get(key);
        return prop == null ? BIND_DATA_VALUE_KEY
             : BIND_DATA_VALUE_KEY + '.' + prop.getPropertyName();
    }
    
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception T[rX̊JnɎsꍇ
     */
    public void startService() throws Exception{
        if(connectionFactoryServiceName != null){
            connectionFactory = ServiceManagerFactory
                .getServiceObject(connectionFactoryServiceName);
            if(persistentManagerServiceName == null){
                throw new IllegalArgumentException("PersistentManagerServiceName must be specified.");
            }
            persistentManager = ServiceManagerFactory
                .getServiceObject(persistentManagerServiceName);
            
            if(query == null){
                throw new IllegalArgumentException("Query must be specified.");
            }
        }else{
            if(codeMasterFinderServiceName != null){
                codeMasterFinder = ServiceManagerFactory
                    .getServiceObject(codeMasterFinderServiceName);
            }
            if(threadContextServiceName != null){
                threadContext = ServiceManagerFactory
                    .getServiceObject(threadContextServiceName);
            }
            if(codeMasterFinder == null && threadContext == null){
                throw new IllegalArgumentException("It is necessary to set either of CodeMasterFinder or ThreadContext.");
            }
            if(codeMasterName == null){
                throw new IllegalArgumentException("CodeMasterName must be specified.");
            }
            if(searchCondition != null){
                if(bindDataMap == null || bindDataMap.size() == 0){
                    throw new IllegalArgumentException("BindDataMap must be specified.");
                }
            }
        }
    }
    
    /**
     * w肳ꂽIuWFNg}X^Ɋ܂܂Ă邩؂B<p>
     *
     * @param obj ؑΏۂ̃IuWFNg
     * @return ،ʁBؐ̏ꍇtrue
     * @exception ValidateException ؂Ɏsꍇ
     */
    @SuppressWarnings("unchecked")
    public boolean validate(Object obj) throws ValidateException{
        if(connectionFactory != null){
            Connection con = null;
            try{
                con = connectionFactory.getConnection();
                final List<Map<String, Object>> result = (List<Map<String, Object>>)persistentManager.loadQuery(con, query, obj, null);
                if(result.size() == 0){
                    return false;
                }
                final Collection<Object> values = result.get(0).values();
                if(values.size() == 0){
                    return false;
                }
                if(values.size() == 1){
                    final Object value = values.iterator().next();
                    if(value instanceof Boolean){
                        return ((Boolean)value).booleanValue();
                    }else if(value instanceof Number){
                        return ((Number)value).intValue() == 1;
                    }else{
                        return true;
                    }
                }else{
                    return true;
                }
            }catch(ConnectionFactoryException e){
                throw new ValidateException(e);
            }catch(PersistentException e){
                throw new ValidateException(e);
            }finally{
                if(con != null){
                    try{
                        con.close();
                    }catch(SQLException e){
                    }
                }
            }
        }else{
            Map<String, Object> codeMaster = null;
            if(threadContext != null){
                codeMaster = (Map<String, Object>)threadContext.get(codeMasterThreadContextKey);
            }
            if(codeMaster == null && codeMasterFinder != null){
                codeMaster = codeMasterFinder.getCodeMasters();
            }
            if(codeMaster == null){
                throw new ValidateException("CodeMaster is not found.");
            }
            final Object master = codeMaster.get(codeMasterName);
            if(master == null){
                throw new ValidateException("Master '" + codeMasterName + "' is not found.");
            }
            if(!(master instanceof RecordList)){
                throw new ValidateException("Master '" + codeMasterName + "' is not RecordList.");
            }
            final RecordList recordList = (RecordList)master;
            if(searchCondition != null){
                final Map<String, Object> params = new HashMap<String, Object>();
                try{
                    for(Map.Entry<String, Property> entry : bindDataMap.entrySet()){
                        final String key = entry.getKey();
                        final Property prop = entry.getValue();
                        if(prop == null){
                            params.put(key, obj);
                        }else{
                            params.put(key, prop.getProperty(obj));
                        }
                    }
                }catch(NoSuchPropertyException e){
                    throw new ValidateException(e);
                }catch(InvocationTargetException e){
                    throw new ValidateException(e.getCause());
                }
                try{
                    return recordList.searchByCondition(
                        searchCondition,
                        params
                    ).size() != 0;
                }catch(Exception e){
                    throw new ValidateException(e);
                }
            }else{
                PropertySchema[] schemata = recordList.getRecordSchema().getPrimaryKeyPropertySchemata();
                if(schemata == null || schemata.length != 1){
                    throw new ValidateException("Size of primary key property not equal 1.");
                }
                Record key = recordList.createRecord();
                key.setProperty(schemata[0].getName(), obj);
                return recordList.searchByPrimaryKey(key) != null;
            }
        }
    }
}