/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.queue;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.daemon.*;

/**
 * UQueueHandlerReiT[rXB<p>
 * 
 * @author M.Takata
 */
public class DistributedQueueHandlerContainerService<E> extends ServiceBase
 implements QueueHandlerContainer<E>, DistributedQueueHandlerContainerServiceMBean{
    
    private static final long serialVersionUID = 4594481433048573418L;
    
    protected ServiceName distributedQueueSelectorServiceName;
    protected DistributedQueueSelector<E> distributedQueueSelector;
    
    protected Daemon[] daemons;
    protected QueueReceiver<E>[] invokers;
    protected ServiceName queueHandlerServiceName;
    protected QueueHandler<E> queueHandler;
    protected boolean isDaemonQueueHandler = true;
    
    protected long waitTimeout = -1;
    protected int maxRetryCount = 0;
    protected long retryInterval = 1000;
    protected String handlingErrorMessageId = DEFAULT_HANDLING_ERROR_MESSAGE_ID;
    protected String retryOverErrorMessageId = DEFAULT_RETRY_OVER_ERROR_MESSAGE_ID;
    protected int queueHandlerThreadPriority = -1;
    protected boolean isReleaseQueue = true;
    protected boolean isSuspend;
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public void setDistributedQueueSelectorServiceName(ServiceName name){
        distributedQueueSelectorServiceName = name;
    }
    // DistributedQueueHandlerContainerServiceJavaDoc
    public ServiceName getDistributedQueueSelectorServiceName(){
        return distributedQueueSelectorServiceName;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public void setQueueHandlerServiceName(ServiceName name){
        if(queueHandlerServiceName == null){
            queueHandlerServiceName = name;
            if(daemons != null){
                for(Daemon daemon : daemons){
                    daemon.resume();
                }
            }
        }else{
            queueHandlerServiceName = name;
        }
    }
    // DistributedQueueHandlerContainerServiceJavaDoc
    public ServiceName getQueueHandlerServiceName(){
        return queueHandlerServiceName;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public void setReleaseQueue(boolean isRelease){
        isReleaseQueue = isRelease;
    }
    // DistributedQueueHandlerContainerServiceJavaDoc
    public boolean isReleaseQueue(){
        return isReleaseQueue;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public void setWaitTimeout(long timeout){
        waitTimeout = timeout;
    }
    // DistributedQueueHandlerContainerServiceJavaDoc
    public long getWaitTimeout(){
        return waitTimeout;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public void setMaxRetryCount(int count){
        maxRetryCount = count;
    }
    // DistributedQueueHandlerContainerServiceJavaDoc
    public int getMaxRetryCount(){
        return maxRetryCount;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public void setRetryInterval(long interval){
        retryInterval = interval;
    }
    // DistributedQueueHandlerContainerServiceJavaDoc
    public long getRetryInterval(){
        return retryInterval;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public void setHandlingErrorMessageId(String id){
        handlingErrorMessageId = id;
    }
    // DistributedQueueHandlerContainerServiceJavaDoc
    public String getHandlingErrorMessageId(){
        return handlingErrorMessageId;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public void setRetryOverErrorMessageId(String id){
        retryOverErrorMessageId = id;
    }
    // DistributedQueueHandlerContainerServiceJavaDoc
    public String getRetryOverErrorMessageId(){
        return retryOverErrorMessageId;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public int getQueueHandlerSize(){
        return invokers == null ? 0 : invokers.length;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public int getActiveQueueHandlerSize(){
        int count = 0;
        if(invokers == null){
            if(distributedQueueSelector != null){
                final Queue<E>[] queues = distributedQueueSelector.getQueues();
                if(queues[0] instanceof QueueHandlerContainer<?>){
                    for(int i = 0; i < queues.length; i++){
                        count += ((QueueHandlerContainer<E>)queues[i]).getActiveQueueHandlerSize();
                    }
                }
            }
        }else{
            for(QueueReceiver<E> invoker : invokers){
                if(invoker.isActive){
                    count++;
                }
            }
        }
        return count;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public int getStandbyQueueHandlerSize(){
        int count = 0;
        if(invokers == null){
            if(distributedQueueSelector != null){
                final Queue<E>[] queues = distributedQueueSelector.getQueues();
                if(queues[0] instanceof QueueHandlerContainer<?>){
                    for(int i = 0; i < queues.length; i++){
                        count += ((QueueHandlerContainer<E>)queues[i]).getStandbyQueueHandlerSize();
                    }
                }
            }
        }else{
            for(QueueReceiver<E> invoker : invokers){
                if(!invoker.isActive){
                    count++;
                }
            }
        }
        return count;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public void setDaemonQueueHandler(boolean isDaemon){
        isDaemonQueueHandler = isDaemon;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public boolean isDaemonQueueHandler(){
        return isDaemonQueueHandler;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public void setQueueHandlerThreadPriority(int newPriority){
        queueHandlerThreadPriority = newPriority;
    }
    // DistributedQueueHandlerContainerServiceJavaDoc
    public int getQueueHandlerThreadPriority(){
        return queueHandlerThreadPriority;
    }
    
    // DistributedQueueHandlerContainerServiceJavaDoc
    public long getAverageHandleProcessTime(){
        long time = 0;
        if(invokers == null){
            if(distributedQueueSelector != null){
                final Queue<E>[] queues = distributedQueueSelector.getQueues();
                if(queues[0] instanceof QueueHandlerContainer<?>){
                    for(int i = 0; i < queues.length; i++){
                        time += ((QueueHandlerContainer<E>)queues[i]).getAverageHandleProcessTime();
                    }
                    time /= queues.length;
                }
            }
        }else{
            if(invokers.length != 0){
                for(QueueReceiver<E> invoker : invokers){
                    time += invoker.getAverageReceiveProcessTime();
                }
                time /= invokers.length;
            }
        }
        return time;
    }
    
    public void setDistributedQueueSelector(DistributedQueueSelector<E> selector){
        distributedQueueSelector = selector;
    }
    
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception T[rX̊JnɎsꍇ
     */
    @SuppressWarnings("unchecked")
    public void startService() throws Exception{
        if(distributedQueueSelectorServiceName != null){
            distributedQueueSelector = ServiceManagerFactory.getServiceObject(distributedQueueSelectorServiceName);
        }
        
        if(distributedQueueSelector == null){
            throw new IllegalArgumentException("DistributedQueueSelector is null.");
        }
        // L[tJn
        accept();
        
        final Queue<E>[] queues = distributedQueueSelector.getQueues();
        if(!(queues[0] instanceof QueueHandlerContainer)){
            invokers = new QueueReceiver[queues.length];
            daemons = new Daemon[invokers.length];
            for(int i = 0; i < invokers.length; i++){
                invokers[i] = new QueueReceiver<E>(queues[i]);
                invokers[i].handler = getQueueHandler();
                
                daemons[i] = new Daemon(invokers[i]);
                daemons[i].setDaemon(isDaemonQueueHandler);
                daemons[i].setName(getServiceNameObject() + " QueueReceiver" + (i + 1));
                if(queueHandlerThreadPriority > 0){
                    daemons[i].setPriority(queueHandlerThreadPriority);
                }
                if(invokers[i].handler == null){
                    daemons[i].suspend();
                }
                daemons[i].start();
            }
        }
    }
    
    /**
     * T[rX̒~sB<p>
     *
     * @exception Exception T[rX̒~Ɏsꍇ
     */
    public void stopService() throws Exception{
        
        if(daemons != null){
            // f[~
            for(int i = 0; i < daemons.length; i++){
                daemons[i].stop();
                daemons[i] = null;
                invokers[i] = null;
            }
        }
        
        // L[t~
        if(isReleaseQueue){
            release();
        }
        distributedQueueSelector = null;
        daemons = null;
        invokers = null;
    }
    
    public synchronized void resume(){
        if(!isSuspend){
            return;
        }
        isSuspend = false;
        if(daemons != null){
            for(Daemon daemon : daemons){
                daemon.resume();
            }
        }
    }
    
    public synchronized void suspend(){
        if(isSuspend){
            return;
        }
        if(daemons != null){
            for(Daemon daemon : daemons){
                daemon.suspend();
            }
        }
        isSuspend = true;
    }
    
    public boolean isSuspend(){
        return isSuspend;
    }
    
    /**
     * QueueHandlerݒ肷B<p>
     *
     * @param handler QueueHandler
     */
    public void setQueueHandler(QueueHandler<E> handler){
        if(queueHandler == null){
            queueHandler = handler;
            if(daemons != null){
                for(int i = 0; i < daemons.length; i++){
                    daemons[i].resume();
                }
            }else if(distributedQueueSelector != null){
                final Queue<E>[] queues = distributedQueueSelector.getQueues();
                if(queues[0] instanceof QueueHandlerContainer<?>){
                    for(int i = 0; i < queues.length; i++){
                        ((QueueHandlerContainer<E>)queues[i]).setQueueHandler(handler);
                    }
                }
            }
        }else{
            queueHandler = handler;
        }
    }
    
    /**
     * QueueHandler擾B<p>
     *
     * @return QueueHandler
     */
    public QueueHandler<E> getQueueHandler(){
        if(queueHandler != null){
            return queueHandler;
        }
        if(queueHandlerServiceName != null){
            return ServiceManagerFactory
                .getServiceObject(queueHandlerServiceName);
        }
        return null;
    }
    
    protected class QueueReceiver<T> implements DaemonRunnable<T>{
        
        protected Queue<T> queue;
        
        protected QueueHandler<T> handler;
        
        protected long receiveCount;
        protected long receiveProcessTime;
        
        public QueueReceiver(Queue<T> queue){
            this.queue = queue;
        }
        
        public long getReceiveCount(){
            return receiveCount;
        }
        
        public long getAverageReceiveProcessTime(){
            return receiveCount == 0 ? 0 : (receiveProcessTime / receiveCount);
        }
        
        /**
         * sǂtOB<p>
         */
        public boolean isActive;
        
        /**
         * f[JnɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onStart() {
            return true;
        }
        
        /**
         * f[~ɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onStop() {
            return true;
        }
        
        /**
         * f[fɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onSuspend() {
            return true;
        }
        
        /**
         * f[ĊJɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onResume() {
            return true;
        }
        
        /**
         * L[PoĕԂB<p>
         * 
         * @param ctrl DaemonControlIuWFNg
         * @return ̓IuWFNg
         */
        @SuppressWarnings("unchecked")
        public T provide(DaemonControl ctrl){
            if(handler == null){
                handler = (QueueHandler<T>)getQueueHandler();
                if(handler == null){
                    return null;
                }
            }
            return queue.get(waitTimeout);
        }
        
        /**
         * dequeuedœnꂽIuWFNgQueueHandlerĂяoB<p>
         *
         * @param dequeued L[oꂽIuWFNg
         * @param ctrl DaemonControlIuWFNg
         */
        public void consume(T dequeued, DaemonControl ctrl){
            if(handler == null){
                return;
            }
            boolean isRetry = false;
            int retryCount = 0;
            receiveCount++;
            long start = System.currentTimeMillis();
            do{
                try{
                    isActive = true;
                    try{
                        handler.handleDequeuedObject(dequeued);
                        isRetry = false;
                    }catch(Throwable th){
                        if(maxRetryCount > 0){
                            if(retryCount >= maxRetryCount){
                                isRetry = false;
                                try{
                                    handler.handleRetryOver(dequeued, th);
                                }catch(Throwable th2){
                                    getLogger().write(
                                        retryOverErrorMessageId,
                                        th,
                                        dequeued
                                    );
                                }
                            }else{
                                isRetry = true;
                                try{
                                    isRetry = handler.handleError(dequeued, th);
                                }catch(Throwable th2){
                                    getLogger().write(
                                        handlingErrorMessageId,
                                        th,
                                        dequeued
                                    );
                                }
                            }
                        }else{
                            isRetry = false;
                            try{
                                handler.handleError(dequeued, th);
                            }catch(Throwable th2){
                                getLogger().write(
                                    handlingErrorMessageId,
                                    th,
                                    dequeued
                                );
                            }
                        }
                    }
                }finally{
                    isActive = false;
                }
                if(isRetry && retryInterval > 0){
                    try{
                        Thread.sleep(retryInterval);
                    }catch(InterruptedException e){
                        isRetry = false;
                    }
                }
                retryCount++;
            }while(isRetry);
            receiveProcessTime += (System.currentTimeMillis() - start);
        }
        
        /**
         * L[̒gfoB<p>
         */
        public void garbage(){
            if(queue != null){
                while(queue.size() > 0){
                    consume(queue.get(0), null);
                }
            }
        }
    }
    
    
    /**
     * K؂ȕUL[̂PɃf[^𓊓B<p>
     * 
     * @param item IuWFNg
     */
    public synchronized void push(E item){
        final Queue<E> queue = distributedQueueSelector.selectQueue(item);
        queue.push(item);
    }
    
    /**
     * L[f[^oB<p>
     * T|[g܂B<br>
     * 
     * @return L[擾IuWFNg
     */
    public E get(){
        throw new UnsupportedOperationException();
    }
    
    /**
     * L[f[^oB<p>
     * T|[g܂B<br>
     * 
     * @param timeOutMs ^CAEg[ms]
     * @return L[擾IuWFNg
     */
    public E get(long timeOutMs){
        throw new UnsupportedOperationException();
    }
    
    /**
     * L[f[^ǂށB<p>
     * T|[g܂B<br>
     * 
     * @return L[擾IuWFNg
     */
    public E peek(){
        throw new UnsupportedOperationException();
    }
    
    /**
     * L[f[^ǂށB<br>
     * T|[g܂B<br>
     * 
     * @param timeOutMs ^CAEg[ms]
     * @return L[擾IuWFNg
     */
    public E peek(long timeOutMs){
        throw new UnsupportedOperationException();
    }
    
    /**
     * L[w肵f[^폜B<p>
     * T|[g܂B<br>
     *
     * @param item 폜Ώۂ̃IuWFNg
     */
    public void remove(Object item){
        throw new UnsupportedOperationException();
    }
    
    /**
     * SĂ̕UL[B<p>
     */
    public void clear(){
        final Queue<E>[] queues = distributedQueueSelector.getQueues();
        if(queues != null){
            for(Queue<E> queue : queues){
                queue.clear();
            }
        }
    }
    
    /**
     * SĂ̕UL[̍vTCY擾B<p>
     * 
     * @return L[i[
     */
    public int size(){
        int size = 0;
        final Queue<E>[] queues = distributedQueueSelector.getQueues();
        if(queues != null){
            for(Queue<E> queue : queues){
                size += queue.size();
            }
        }
        return size;
    }
    
    /**
     * SĂ̕UL[ɓꂽ擾B<p>
     *
     * @return L[
     */
    public long getCount(){
        long count = 0;
        final Queue<E>[] queues = distributedQueueSelector.getQueues();
        if(queues != null){
            for(Queue<E> queue : queues){
                count += queue.getCount();
            }
        }
        return count;
    }
    
    /**
     * SĂ̕UL[̃L[擾҂JnB<p>
     * {@link #release()}ďoɁAL[擾҂󂯕t悤ɂB
     */
    public void accept(){
        final Queue<E>[] queues = distributedQueueSelector.getQueues();
        if(queues != null){
            for(Queue<E> queue : queues){
                queue.accept();
            }
        }
    }
    
    /**
     * SĂ̕UL[̃L[擾҂JAL[擾҂󂯕tȂ悤ɂB<p>
     */
    public void release(){
        final Queue<E>[] queues = distributedQueueSelector.getQueues();
        if(queues != null){
            for(Queue<E> queue : queues){
                queue.release();
            }
        }
    }
}
