/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.distribute;

import java.util.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.daemon.*;

/**
 * {@link KeepAliveCheckerSelector}ۃNXB<p>
 *
 * @author M.Takata
 */
public abstract class AbstractKeepAliveCheckerSelectorService
 extends ServiceBase
 implements KeepAliveCheckerSelector, ClusterListener, AbstractKeepAliveCheckerSelectorServiceMBean{
    
    private static final long serialVersionUID = 1452664941582086362L;
    
    /**
     * I\{@link KeepAliveChecker}T[rX̃T[rXzB<p>
     */
    protected ServiceName[] selectableCheckerServiceNames;
    
    /**
     * I\{@link KeepAliveChecker}T[rX̃T[rX̃XgB<p>
     */
    protected List<ServiceName> selectableCheckers;
    
    /**
     * I{@link KeepAliveChecker}ɐmFsԊu[ms]B<p>
     */
    protected long checkInterval = -1L;
    
    /**
     * Ă{@link KeepAliveChecker}̃T[rX̃XgB<p>
     */
    protected List<ServiceName> aliveCheckers;
    
    /**
     * I{@link KeepAliveChecker}ɐmFsf[XbhB<p>
     */
    protected Daemon daemon;
    
    protected String aliveLogMessageId = DEFAULT_ALIVE_LOG_MSG_ID;
    
    protected String deadLogMessageId = DEFAULT_DEAD_LOG_MSG_ID;
    
    protected boolean isOutputAliveLogMessage = true;
    
    protected boolean isOutputDeadLogMessage = true;
    
    protected boolean isKeepOrder = false;
    
    protected ServiceName clusterServiceName;
    protected ClusterService cluster;
    protected List<? extends Object> clusterMembers;
    
    // AbstractKeepAliveCheckerSelectorServiceMBean JavaDoc
    public void setSelectableCheckerServiceNames(ServiceName[] names){
        selectableCheckerServiceNames = names;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBean JavaDoc
    public ServiceName[] getSelectableCheckerServiceNames(){
        return selectableCheckerServiceNames;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBean JavaDoc
    public void setCheckInterval(long millis){
        checkInterval = millis;
    }
    // AbstractKeepAliveCheckerSelectorServiceMBean JavaDoc
    public long getCheckInterval(){
        return checkInterval;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public void setAliveLogMessageId(String id){
        aliveLogMessageId = id;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public String getAliveLogMessageId(){
        return aliveLogMessageId;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public void setDeadLogMessageId(String id){
        deadLogMessageId = id;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public String getDeadLogMessageId(){
        return deadLogMessageId;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public void setOutputAliveLogMessage(boolean isOutput){
        isOutputAliveLogMessage = isOutput;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public boolean isOutputAliveLogMessage(){
        return isOutputAliveLogMessage;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public void setOutputDeadLogMessage(boolean isOutput){
        isOutputDeadLogMessage = isOutput;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public boolean isOutputDeadLogMessage(){
        return isOutputDeadLogMessage;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public void setKeepOrder(boolean isKeep){
        isKeepOrder = isKeep;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public boolean isKeepOrder(){
        return isKeepOrder;
    }
    
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public void setClusterServiceName(ServiceName name){
        clusterServiceName = name;
    }
    // AbstractKeepAliveCheckerSelectorServiceMBeanJavaDoc
    public ServiceName getClusterServiceName(){
        return clusterServiceName;
    }
    
    public void createService() throws Exception{
        aliveCheckers = Collections.synchronizedList(new ArrayList<ServiceName>());
        selectableCheckers = new ArrayList<ServiceName>();
    }
    
    public void startService() throws Exception{
        if(clusterServiceName != null){
            cluster = ServiceManagerFactory.getServiceObject(clusterServiceName);
            cluster.addClusterListener(this);
        }else{
            if(checkInterval > 0){
                daemon = new Daemon(new KeepAliveCheckDaemon());
                daemon.setName(
                    "Nimbus KeepAliveCheckDaemon " + getServiceNameObject()
                );
                daemon.start();
            }
            if(selectableCheckerServiceNames != null){
                selectableCheckers.clear();
                for(int i = 0; i < selectableCheckerServiceNames.length; i++){
                    selectableCheckers.add(selectableCheckerServiceNames[i]);
                }
            }
        }
    }
    
    public void stopService() throws Exception{
        if(cluster != null){
            cluster.removeClusterListener(this);
            cluster = null;
        }else{
            
            if(daemon != null){
                // f[~
                daemon.stop();
            }
            aliveCheckers.clear();
        }
    }
    
    public void destroyService() throws Exception{
        aliveCheckers = null;
    }
    
    // KeepAliveCheckerSelector JavaDoc
    public KeepAliveChecker[] getSelectableCheckers(){
        if(cluster != null){
            List<? extends Object> members = clusterMembers;
            List<KeepAliveChecker> list = null;
            if(members == null || members.size() == 0){
                return new KeepAliveChecker[0];
            }
            if(list == null){
                list = new ArrayList<KeepAliveChecker>();
            }
            for(int i = 0, imax = members.size(); i < imax; i++){
                ClusterService.GlobalUID uid = (ClusterService.GlobalUID)members.get(i);
                Object option = uid.getOption();
                if(option != null && option instanceof KeepAliveChecker){
                    list.add((KeepAliveChecker)option);
                }
            }
            return list.toArray(new KeepAliveChecker[list.size()]);
        }else{
            if(checkInterval <= 0){
                updateChekerStates(false);
            }
            ServiceName[] names = (ServiceName[])aliveCheckers.toArray(
                new ServiceName[aliveCheckers.size()]
            );
            final List<KeepAliveChecker> list = new ArrayList<KeepAliveChecker>();
            KeepAliveChecker checker = null;
            for(int i = 0; i < names.length; i++){
                try{
                    Service service = ServiceManagerFactory.getService(names[i]);
                    if(service.getState() != State.STARTED){
                        continue;
                    }
                    checker = ServiceManagerFactory.getServiceObject(names[i]);
                    list.add(checker);
                }catch(ServiceNotFoundException e){
                    continue;
                }
            }
            return list.toArray(new KeepAliveChecker[list.size()]);
        }
    }
    
    public List<?> getAliveCheckers(){
        if(cluster != null){
            return clusterMembers;
        }else{
            return aliveCheckers;
        }
    }
    
    protected void updateChekerStates(boolean init){
        if(aliveCheckers == null){
            return;
        }
        final List<ServiceName> tmpAliveCheckers = Collections.synchronizedList(
            new ArrayList<ServiceName>(aliveCheckers)
        );
        for(int i = 0; i < selectableCheckerServiceNames.length; i++){
            try{
                Service service = ServiceManagerFactory
                    .getService(selectableCheckerServiceNames[i]);
                final KeepAliveChecker checker
                     = ServiceManagerFactory.getServiceObject(selectableCheckerServiceNames[i]);
                if(service.getState() == State.STARTED
                    && checker.isAlive()){
                    if(!tmpAliveCheckers.contains(selectableCheckerServiceNames[i])){
                        tmpAliveCheckers.add(selectableCheckerServiceNames[i]);
                        if(!init && isOutputAliveLogMessage){
                            getLogger().write(
                                aliveLogMessageId,
                                selectableCheckerServiceNames[i]
                            );
                        }
                    }
                }else{
                    if(tmpAliveCheckers.contains(selectableCheckerServiceNames[i])){
                        tmpAliveCheckers.remove(selectableCheckerServiceNames[i]);
                        if(!init && isOutputDeadLogMessage){
                            getLogger().write(
                                deadLogMessageId,
                                selectableCheckerServiceNames[i]
                            );
                        }
                    }
                }
            }catch(ServiceNotFoundException e){
                tmpAliveCheckers.remove(selectableCheckerServiceNames[i]);
            }
        }
        if(isKeepOrder){
            Collections.sort(
                tmpAliveCheckers,
                new SelectableCheckerComparator()
            );
        }
        aliveCheckers = tmpAliveCheckers;
    }
    
    // ClusterListenerJavaDoc
    @SuppressWarnings("unchecked")
    public void memberInit(Object myId, List<? extends Object> members){
        List<Comparable<Object>> tmpMembers = new ArrayList<Comparable<Object>>((List<Comparable<Object>>)members);
        if(isKeepOrder){
            Collections.sort(tmpMembers);
        }
        clusterMembers = tmpMembers;
    }
    
    // ClusterListenerJavaDoc
    @SuppressWarnings("unchecked")
    public void memberChange(List<? extends Object> oldMembers, List<? extends Object> newMembers){
        List<Object> addedMember = new ArrayList<Object>(newMembers);
        addedMember.removeAll(oldMembers);
        if(addedMember.size() != 0 && isOutputAliveLogMessage){
            for(int i = 0; i < addedMember.size(); i++){
                getLogger().write(aliveLogMessageId, addedMember.get(i));
            }
        }
        List<Object> removedMember = new ArrayList<Object>(oldMembers);
        removedMember.removeAll(newMembers);
        if(removedMember.size() != 0 && isOutputDeadLogMessage){
            for(int i = 0, imax = removedMember.size(); i < imax; i++){
                getLogger().write(
                    deadLogMessageId,
                    removedMember.get(i)
                );
            }
        }
        List<Comparable<Object>> tmpMembers = new ArrayList<Comparable<Object>>((List<Comparable<Object>>)newMembers);
        if(isKeepOrder){
            Collections.sort(tmpMembers);
        }
        clusterMembers = tmpMembers;
    }
    
    // ClusterListenerJavaDoc
    public void changeMain() throws Exception{}
    
    // ClusterListenerJavaDoc
    public void changeSub(){}
    
    protected class KeepAliveCheckDaemon implements DaemonRunnable<Object>{
        
        // DaemonRunnableJavaDoc
        public boolean onStart(){
            updateChekerStates(true);
            return true;
        }
        
        // DaemonRunnableJavaDoc
        public boolean onStop(){
            return true;
        }
        
        // DaemonRunnableJavaDoc
        public boolean onSuspend(){
            return true;
        }
        
        // DaemonRunnableJavaDoc
        public boolean onResume(){
            return true;
        }
        
        // DaemonRunnableJavaDoc
        public Object provide(DaemonControl ctrl) throws Exception{
            if(checkInterval >= 0){
                Thread.sleep(checkInterval);
            }else{
                ctrl.setRunning(false);
            }
            return null;
        }
        
        // DaemonRunnableJavaDoc
        public void consume(Object paramObj, DaemonControl ctrl) throws Exception{
            updateChekerStates(false);
        }
        
        // DaemonRunnableJavaDoc
        public void garbage(){
        }
    }
    
    protected class SelectableCheckerComparator implements Comparator<Object>{
        @SuppressWarnings("unchecked")
        public int compare(Object o1, Object o2){
            if(o1 == o2){
                return 0;
            }
            if(selectableCheckers == null || selectableCheckers.size() == 0){
                if(o1 != null && o1 instanceof Comparable){
                    return ((Comparable<Object>)o1).compareTo(o2);
                }else{
                    return 0;
                }
            }
            final int index1 = selectableCheckers.indexOf(o1);
            final int index2 = selectableCheckers.indexOf(o2);
            return index1 - index2;
        }
    }
}
