/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.io;

import java.io.*;
import java.util.List;

import jp.ossc.nimbus.util.converter.StringConverter;
import jp.ossc.nimbus.util.converter.ConvertException;

/**
 * CSV`WriterNXB<p>
 * <pre>
 * import java.io.*;
 * import jp.ossc.nimbus.io.CSVWriter;
 *
 * FileOutputStream fos = new FileOutputStream("sample.csv");
 * OutputStreamWriter osw = new OutputStreamWriter(fos, "UTF-8");
 * CSVWriter writer = new CSVWriter(osw);
 * String[] csv = new String[2];
 * try{
 *     csv[0] = "hoge";
 *     csv[1] = "100";
 *     writer.writeCSV(csv);
 *        :
 * }finally{
 *     writer.close();
 * }
 * </pre>
 * 
 * @author M.Takata
 */
public class CSVWriter extends BufferedWriter implements StringConverter{
    
    /**
     * ftHg̃Zp[^B<p>
     */
    public static final char DEFAULT_SEPARATOR = ',';
    
    /**
     * ftHg̃Zp[^̃GXP[vB<p>
     * GXP[vGXP[vꍇ́AGXP[vd˂B<br>
     */
    public static final char DEFAULT_SEPARATOR_ESCAPE = '\\';
    
    /**
     * ftHg̈͂ݕB<p>
     * ͂ݕGXP[vꍇ́A͂ݕd˂B<br>
     */
    public static final char DEFAULT_ENCLOSURE = '"';
    
    /**
     * ftHg̉sB<p>
     */
    public static final String DEFAULT_LINE_SEPARATOR
         = System.getProperty("line.separator");
    
    private static final String REPLACE_CR = "\\r";
    private static final String REPLACE_LF = "\\n";
    
    protected char separator = DEFAULT_SEPARATOR;
    protected char separatorEscape = DEFAULT_SEPARATOR_ESCAPE;
    protected char enclosure = DEFAULT_ENCLOSURE;
    protected boolean isEnclose;
    protected String lineSeparator = DEFAULT_LINE_SEPARATOR;
    protected boolean isAppendElement;
    protected String nullValue;
    protected boolean isEscapeLineSeparatorInEnclosure;
    
    protected WriterWrapper writerWrapper;
    
    /**
     * ftHg̏݃obt@TCYڑ̃CX^X𐶐B<p>
     */
    public CSVWriter(){
        super(new WriterWrapper());
        writerWrapper = (WriterWrapper)lock;
    }
    
    /**
     * ftHg̏݃obt@TCYCX^X𐶐B<p>
     *
     * @param writer ݐWriter
     */
    public CSVWriter(Writer writer){
        super(new WriterWrapper(writer));
        writerWrapper = (WriterWrapper)lock;
    }
    
    /**
     * w肳ꂽ݃obt@TCYڑ̃CX^X𐶐B<p>
     *
     * @param size ݃obt@TCY
     */
    public CSVWriter(int size){
        super(new WriterWrapper(), size);
        writerWrapper = (WriterWrapper)lock;
    }
    
    /**
     * w肳ꂽ݃obt@TCYCX^X𐶐B<p>
     *
     * @param writer ݐWriter
     * @param size ݃obt@TCY
     */
    public CSVWriter(Writer writer, int size){
        super(new WriterWrapper(writer), size);
        writerWrapper = (WriterWrapper)lock;
    }
    
    /**
     * Writerݒ肷B<p>
     *
     * @param writer Writer
     * @exception IOException Writerݒ肳Ăꍇ
     */
    public void setWriter(Writer writer) throws IOException{
        writerWrapper.setWriter(writer);
        isAppendElement = false;
    }
    
    /**
     * Zp[^ݒ肷B<p>
     *
     * @param separator Zp[^
     */
    public void setSeparator(char separator){
        this.separator = separator;
    }
    
    /**
     * Zp[^擾B<p>
     *
     * @return Zp[^
     */
    public char getSeparator(){
         return separator;
    }
    
    /**
     * Zp[^̃GXP[vݒ肷B<p>
     *
     * @param escape GXP[v
     */
    public void setSeparatorEscape(char escape){
        separatorEscape = escape;
    }
    
    /**
     * Zp[^̃GXP[v擾B<p>
     *
     * @return GXP[v
     */
    public char getSeparatorEscape(){
         return separatorEscape;
    }
    
    /**
     * sZp[^ݒ肷B<p>
     *
     * @param separator sZp[^
     */
    public void setLineSeparator(String separator){
        this.lineSeparator = separator;
    }
    
    /**
     * sZp[^擾B<p>
     *
     * @return sZp[^
     */
    public String getLineSeparator(){
         return lineSeparator;
    }
    
    /**
     * ͂ݕݒ肷B<p>
     *
     * @param enclosure ͂ݕ
     */
    public void setEnclosure(char enclosure){
        this.enclosure = enclosure;
    }
    
    /**
     * ͂ݕ擾B<p>
     *
     * @return ͂ݕ
     */
    public char getEnclosure(){
         return enclosure;
    }
    
    /**
     * CSV̗vf͂ݕň͂ނǂݒ肷B<p>
     * ftHǵAfalseň͂܂ȂB<br>
     *
     * @param isEnclose ͂ݕň͂ޏꍇtrue
     */
    public void setEnclose(boolean isEnclose){
        this.isEnclose = isEnclose;
    }
    
    /**
     * CSV̗vf͂ݕň͂ނǂ𔻒肷B<p>
     *
     * @return truȅꍇA͂ݕň͂
     */
    public boolean isEnclose(){
         return isEnclose;
    }
    
    /**
     * nullCSVvfƂďƂꍇɁAo͂镶ݒ肷B<p>
     * ݒ肵Ȃꍇ́ANullPointerExceptionB<br>
     *
     * @param value 
     */
    public void setNullValue(String value){
        nullValue = value;
    }
    
    /**
     * nullCSVvfƂďƂꍇɁAo͂镶擾B<p>
     *
     * @return 
     */
    public String getNullValue(){
        return nullValue;
    }
    
    /**
     * CSV̗vf͂ݕň͂ޏꍇɁAsGXP[v邩ǂݒ肷B<p>
     * ftHǵAfalseŃGXP[vȂB<br>
     * 
     * @param isEscape GXP[vꍇtrue
     */
    public void setEscapeLineSeparatorInEnclosure(boolean isEscape){
        isEscapeLineSeparatorInEnclosure = isEscape;
    }
    
    /**
     * CSV̗vf͂ݕň͂ޏꍇɁAsGXP[v邩ǂ𔻒肷B<p>
     * 
     * @return truȅꍇAGXP[v
     */
    public boolean isEscapeLineSeparatorInEnclosure(){
        return isEscapeLineSeparatorInEnclosure;
    }
    
    /**
     * s؂蕶ށB<p>
     * s؂蕶́A{@link #getLineSeparator()}gpB<br>
     * 
     * @exception IOException o̓G[ꍇ
     */
    public void newLine() throws IOException{
        super.write(lineSeparator);
        isAppendElement = false;
    }
    
    /**
     * CSVvfށB<p>
     * Zp[^̒ǉAZp[^܂܂Ăꍇ̃GXP[vA͂ݕł̈͂ݏōsB<br>
     * 
     * @param element CSVvf
     * @exception IOException o̓G[ꍇ
     */
    public void writeElement(String element) throws IOException{
        if(isAppendElement){
            super.write(separator);
        }
        if(isEnclose){
            super.write(enclosure);
        }
        super.write(escape(element));
        if(isEnclose){
            super.write(enclosure);
        }
        isAppendElement = true;
    }
    
    /**
     * CSVvfށB<p>
     * 
     * @param element CSVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(boolean element) throws IOException{
        writeElement(Boolean.toString(element));
    }
    
    /**
     * CSVvfށB<p>
     * 
     * @param element CSVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(byte element) throws IOException{
        writeElement(Byte.toString(element));
    }
    
    /**
     * CSVvfށB<p>
     * 
     * @param element CSVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(char element) throws IOException{
        writeElement(Character.toString(element));
    }
    
    /**
     * CSVvfށB<p>
     * 
     * @param element CSVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(short element) throws IOException{
        writeElement(Short.toString(element));
    }
    
    /**
     * CSVvfށB<p>
     * 
     * @param element CSVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(int element) throws IOException{
        writeElement(Integer.toString(element));
    }
    
    /**
     * CSVvfށB<p>
     * 
     * @param element CSVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(long element) throws IOException{
        writeElement(Long.toString(element));
    }
    
    /**
     * CSVvfށB<p>
     * 
     * @param element CSVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(float element) throws IOException{
        writeElement(Float.toString(element));
    }
    
    /**
     * CSVvfށB<p>
     * 
     * @param element CSVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(double element) throws IOException{
        writeElement(Double.toString(element));
    }
    
    /**
     * CSVvfށB<p>
     * 
     * @param element CSVvf
     * @exception IOException o̓G[ꍇ
     * @see #writeElement(String)
     */
    public void writeElement(Object element) throws IOException{
        writeElement(element == null ? (String)null : element.toString());
    }
    
    private String escape(String element){
        if(isEnclose){
            return escape(element, enclosure, nullValue, isEscapeLineSeparatorInEnclosure);
        }else{
            return escape(element, separator, separatorEscape, nullValue);
        }
    }
    
    private static String escape(
        String element,
        char separator,
        char separatorEscape,
        String nullValue
    ){
        if(element == null){
            return nullValue;
        }
        final int index1 = element.indexOf(separator);
        final int index2 = element.indexOf(separatorEscape);
        final int index3 = element.indexOf('\r');
        final int index4 = element.indexOf('\n');
        if(index1 == -1 && index2 == -1 && index3 == -1 && index4 == -1){
            return element;
        }
        int index = index1 == -1 ? index2
             : (index2 == -1 ? index1 : Math.min(index1, index2));
        index = index == -1 ? index3
             : (index3 == -1 ? index : Math.min(index, index3));
        index = index == -1 ? index4
             : (index4 == -1 ? index : Math.min(index, index4));
        
        final StringBuilder buf = new StringBuilder();
        for(int i = 0; i < index; i++){
            char c = element.charAt(i);
            buf.append(c);
        }
        for(int i = index, imax = element.length(); i < imax; i++){
            char c = element.charAt(i);
            if(c == separator || c == separatorEscape){
                buf.append(separatorEscape);
                buf.append(c);
            }else if(c == '\r'){
                buf.append(REPLACE_CR);
            }else if(c == '\n'){
                buf.append(REPLACE_LF);
            }else{
                buf.append(c);
            }
        }
        return buf.toString();
    }
    
    private static String escape(
        String element,
        char enclosure,
        String nullValue,
        boolean isEscapeLineSeparator
    ){
        if(element == null){
            return nullValue;
        }
        final int index1 = element.indexOf(enclosure);
        final int index2 = isEscapeLineSeparator ? element.indexOf('\r') : -1;
        final int index3 = isEscapeLineSeparator ? element.indexOf('\n') : -1;
        if(index1 == -1 && index2 == -1 && index3 == -1){
            return element;
        }
        int index = index1 == -1 ? index2
             : (index2 == -1 ? index1 : Math.min(index1, index2));
        index = index == -1 ? index3
             : (index3 == -1 ? index : Math.min(index, index3));
        final StringBuilder buf = new StringBuilder();
        for(int i = 0; i < index; i++){
            char c = element.charAt(i);
            buf.append(c);
        }
        for(int i = index, imax = element.length(); i < imax; i++){
            char c = element.charAt(i);
            if(c == enclosure){
                buf.append(enclosure);
                buf.append(c);
            }else if(isEscapeLineSeparator){
                if(c == '\r'){
                    buf.append(REPLACE_CR);
                }else if(c == '\n'){
                    buf.append(REPLACE_LF);
                }else{
                    buf.append(c);
                }
            }else{
                buf.append(c);
            }
        }
        return buf.toString();
    }
    
    /**
     * w肳ꂽzCSVƂďށB<p>
     * s̒ǉAZp[^̒ǉAZp[^܂܂Ăꍇ̃GXP[vA͂ݕł̈͂ݏōsB<br>
     *
     * @param elements CSV`ŏo͂镶z
     * @exception IOException o̓G[ꍇ
     */
    public void writeCSV(String... elements) throws IOException{
        for(int i = 0; i < elements.length; i++){
            writeElement(elements[i]);
        }
        newLine();
    }
    
    /**
     * w肳ꂽzCSVƂďށB<p>
     *
     * @param elements CSV`ŏo͂z
     * @exception IOException o̓G[ꍇ
     * @see #writeCSV(String[])
     */
    public void writeCSV(Object... elements) throws IOException{
        for(int i = 0; i < elements.length; i++){
            writeElement(elements[i]);
        }
        newLine();
    }
    
    /**
     * w肳ꂽXgCSVƂďށB<p>
     * s̒ǉAZp[^̒ǉAZp[^܂܂Ăꍇ̃GXP[vA͂ݕł̈͂ݏōsB<br>
     *
     * @param elements CSV`ŏo͂郊Xg
     * @exception IOException o̓G[ꍇ
     */
    public void writeCSV(List<?> elements) throws IOException{
        for(int i = 0, imax = elements.size(); i < imax; i++){
            writeElement(elements.get(i));
        }
        newLine();
    }
    
    /**
     * zw肳ꂽCSV`ɕϊB<p>
     *
     * @param separator Zp[^
     * @param escape GXP[v
     * @param nullValue nullCSVvfƂďƂꍇɁAo͂镶
     * @param elements CSV̗vfƂȂ镶z
     * @return CSV`
     */
    public static String toCSV(
        char separator,
        char escape,
        String nullValue,
        String... elements
    ){
        final StringBuilder buf = new StringBuilder();
        if(elements != null){
            for(int i = 0; i < elements.length; i++){
                buf.append(
                    escape(elements[i], separator, escape, nullValue)
                );
                if(i != elements.length - 1){
                    buf.append(separator);
                }
            }
        }
        return buf.toString();
    }
    
    /**
     * zw肳ꂽCSV`ɕϊB<p>
     *
     * @param separator Zp[^
     * @param enclosure ͂ݕ
     * @param nullValue nullCSVvfƂďƂꍇɁAo͂镶
     * @param isEscapeLineSeparator sGXP[v邩ǂBGXP[vꍇtrue
     * @param elements CSV̗vfƂȂ镶z
     * @return CSV`
     */
    public static String toEnclosedCSV(
        char separator,
        char enclosure,
        String nullValue,
        boolean isEscapeLineSeparator,
        String... elements
    ){
        final StringBuilder buf = new StringBuilder();
        if(elements != null){
            for(int i = 0; i < elements.length; i++){
                buf.append(
                    escape(elements[i], enclosure, nullValue, isEscapeLineSeparator)
                );
                if(i != elements.length - 1){
                    buf.append(separator);
                }
            }
        }
        return buf.toString();
    }
    
    /**
     * 񃊃Xgw肳ꂽCSV`ɕϊB<p>
     *
     * @param elements CSV̗vfƂȂ镶񃊃Xg
     * @param separator Zp[^
     * @param separatorEscape GXP[v
     * @param nullValue nullCSVvfƂďƂꍇɁAo͂镶
     * @return CSV`
     */
    public static String toCSV(
        List<String> elements,
        char separator,
        char separatorEscape,
        String nullValue
    ){
        final StringBuilder buf = new StringBuilder();
        if(elements != null){
            for(int i = 0, imax = elements.size(); i < imax; i++){
                buf.append(
                    escape(
                        elements.get(i),
                        separator,
                        separatorEscape,
                        nullValue
                    )
                );
                if(i != imax - 1){
                    buf.append(separator);
                }
            }
        }
        return buf.toString();
    }
    
    /**
     * 񃊃Xgw肳ꂽCSV`ɕϊB<p>
     *
     * @param elements CSV̗vfƂȂ镶񃊃Xg
     * @param separator Zp[^
     * @param enclosure ͂ݕ
     * @param nullValue nullCSVvfƂďƂꍇɁAo͂镶
     * @return CSV`
     */
    public static String toEnclosedCSV(
        List<String> elements,
        char separator,
        char enclosure,
        String nullValue,
        boolean isEscapeLineSeparator
    ){
        final StringBuilder buf = new StringBuilder();
        if(elements != null){
            for(int i = 0, imax = elements.size(); i < imax; i++){
                buf.append(
                    escape(elements.get(i), enclosure, nullValue, isEscapeLineSeparator)
                );
                if(i != imax - 1){
                    buf.append(separator);
                }
            }
        }
        return buf.toString();
    }
    
    /**
     * ڑ̕𐶐B<p>
     *
     * @return ڑ̕
     */
    public CSVWriter cloneWriter(){
        return cloneWriter(new CSVWriter());
    }
    
    /**
     * ڑ̕𐶐B<p>
     *
     * @param clone ڑ̃CX^X
     * @return ڑ̕
     */
    protected CSVWriter cloneWriter(CSVWriter clone){
        clone.separator = separator;
        clone.separatorEscape = separatorEscape;
        clone.enclosure = enclosure;
        clone.isEnclose = isEnclose;
        clone.lineSeparator = lineSeparator;
        clone.nullValue = nullValue;
        return clone;
    }
    
    @Override
    public Object convert(Object obj) throws ConvertException{
        return convert((String)(obj == null ? null : obj.toString()));
    }
    
    @Override
    public String convert(String str) throws ConvertException{
        return escape(str);
    }
    
    private static class WriterWrapper extends Writer{
        
        private Writer realWriter;
        
        public WriterWrapper(){
        }
        
        public WriterWrapper(Writer writer){
            realWriter = writer;
        }
        
        @SuppressWarnings("unused")
        public Writer getWriter(){
            return realWriter;
        }
        
        public void setWriter(Writer writer) throws IOException{
            if(realWriter != null){
                throw new IOException("Writer is already commited.");
            }
            realWriter = writer;
        }
        
        @Override
        public void write(int c) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            realWriter.write(c);
        }
        
        @Override
        public void write(char[] cbuf) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            realWriter.write(cbuf);
        }
        
        @Override
        public void write(char[] cbuf, int off, int len) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            realWriter.write(cbuf, off, len);
        }
        
        @Override
        public void write(String str) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            realWriter.write(str);
        }
        
        @Override
        public void write(String str, int off, int len) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            realWriter.write(str, off, len);
        }
        
        @Override
        public Writer append(CharSequence csq) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            return realWriter.append(csq);
        }
        
        @Override
        public Writer append(CharSequence csq, int off, int len) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            return realWriter.append(csq, off, len);
        }
        
        @Override
        public Writer append(char c) throws IOException{
            if(realWriter == null){
                throw new IOException("Writer is null.");
            }
            return realWriter.append(c);
        }
        
        @Override
        public void flush() throws IOException{
            if(realWriter != null){
                realWriter.flush();
            }
        }
        
        @Override
        public void close() throws IOException{
            if(realWriter != null){
                realWriter.close();
            }
        }
    }
}