/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link ServiceNameMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class ServiceNameMetaDataTest{

    @DataProvider(name = "ServiceNameMetaData")
    public Object[][] dataServiceNameMetaData(){
        return new Object[][]{
            {null},
            {new ManagerMetaData()},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#ServiceNameMetaData(jp.ossc.nimbus.core.MetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "ServiceNameMetaData")
    public void testServiceNameMetaData(MetaData parent){
        ServiceNameMetaData metaData = new ServiceNameMetaData(parent);
        assertEquals(metaData.getParent(), parent);
        assertEquals(metaData.getManagerName(), ServiceManager.DEFAULT_NAME);
        assertNull(metaData.getServiceName());
    }

    @DataProvider(name = "ServiceNameMetaData2")
    public Object[][] dataServiceNameMetaData2(){
        return new Object[][]{
            {null, null, ServiceManager.DEFAULT_NAME},
            {new ManagerMetaData(), null, ServiceManager.DEFAULT_NAME},
            {new ManagerMetaData(), "manager1", "manager1"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#ServiceNameMetaData(jp.ossc.nimbus.core.MetaData,java.lang.String)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "ServiceNameMetaData2")
    public void testServiceNameMetaData2(MetaData parent, String manager, String expected){
        ServiceNameMetaData metaData = new ServiceNameMetaData(parent, manager);
        assertEquals(metaData.getParent(), parent);
        assertEquals(metaData.getManagerName(), expected);
        assertNull(metaData.getServiceName());
    }

    @DataProvider(name = "ServiceNameMetaData3")
    public Object[][] dataServiceNameMetaData3(){
        return new Object[][]{
            {null, null, null, new String[]{ServiceManager.DEFAULT_NAME, null}},
            {new ManagerMetaData(), null, null, new String[]{ServiceManager.DEFAULT_NAME, null}},
            {new ManagerMetaData(), "manager1", null, new String[]{"manager1", null}},
            {new ManagerMetaData(), "manager1", "service1", new String[]{"manager1", "service1"}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#ServiceNameMetaData(jp.ossc.nimbus.core.MetaData,java.lang.String,java.lang.String)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ManagerMetaData#ManagerMetaData\\(\\)"},
          dataProvider = "ServiceNameMetaData3")
    public void testServiceNameMetaData3(MetaData parent, String manager, String service, String[] expected){
        ServiceNameMetaData metaData = new ServiceNameMetaData(parent, manager, service);
        assertEquals(metaData.getParent(), parent);
        assertEquals(metaData.getManagerName(), expected[0]);
        assertEquals(metaData.getServiceName(), expected[1]);
    }

    @DataProvider(name = "SetManagerName")
    public Object[][] dataSetManagerName(){
        return new Object[][]{
            {null, ServiceManager.DEFAULT_NAME},
            {"manager1", "manager1"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#setManagerName(java.lang.String)",
                    "jp.ossc.nimbus.core.ServiceNameMetaData#getManagerName()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#ServiceNameMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetManagerName")
    public void testSetManagerName(String name, String expected){
        ServiceNameMetaData metaData = new ServiceNameMetaData(null);
        metaData.setManagerName(name);
        assertEquals(metaData.getManagerName(), expected);
    }

    @DataProvider(name = "SetServiceName")
    public Object[][] dataSetServiceName(){
        return new Object[][]{
            {null},
            {"service1"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#setServiceName(java.lang.String)",
                    "jp.ossc.nimbus.core.ServiceNameMetaData#getServiceName()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#ServiceNameMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetServiceName")
    public void testSetServiceName(String name){
        ServiceNameMetaData metaData = new ServiceNameMetaData(null);
        metaData.setServiceName(name);
        assertEquals(metaData.getServiceName(), name);
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<serviceName>Service</serviceName>"), new Object[]{ServiceManager.DEFAULT_NAME, "Service"}},
            {createElement("<serviceName>#Service</serviceName>"), new Object[]{ServiceManager.DEFAULT_NAME, "Service"}},
            {createElement("<serviceName>manager#Service</serviceName>"), new Object[]{"manager", "Service"}},
            {createElement("<serviceName unknown=\"hoge\">Service</serviceName>"), new Object[]{ServiceManager.DEFAULT_NAME, "Service"}},
            {createElement("<serviceName>Service<unknown/></serviceName>"), new Object[]{ServiceManager.DEFAULT_NAME, "Service"}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#ServiceNameMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#getManagerName\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#getServiceName\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        ServiceNameMetaData metaData = new ServiceNameMetaData(null);
        metaData.importXML(element);
        assertEquals(metaData.getManagerName(), assertValues[0]);
        assertEquals(metaData.getServiceName(), assertValues[1]);
    }

    @DataProvider(name = "ImportXML2")
    public Object[][] dataImportXML2() throws Exception{
        return new Object[][]{
            {null, createElement("<serviceName>Service</serviceName>"), new Object[]{ServiceManager.DEFAULT_NAME, "Service"}},
            {null, createElement("<serviceName>#Service</serviceName>"), new Object[]{ServiceManager.DEFAULT_NAME, "Service"}},
            {null, createElement("<serviceName>manager#Service</serviceName>"), new Object[]{"manager", "Service"}},
            {"manager", createElement("<serviceName>Service</serviceName>"), new Object[]{"manager", "Service"}},
            {"manager", createElement("<serviceName>#Service</serviceName>"), new Object[]{"manager", "Service"}},
            {"manager", createElement("<serviceName>manager2#Service</serviceName>"), new Object[]{"manager2", "Service"}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#ServiceNameMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData,java\\.lang\\.String\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#getManagerName\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#getServiceName\\(\\)"},
          dataProvider = "ImportXML2")
    public void testImportXML2(
        String manager,
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        ServiceNameMetaData metaData = new ServiceNameMetaData(null, manager);
        metaData.importXML(element);
        assertEquals(metaData.getManagerName(), assertValues[0]);
        assertEquals(metaData.getServiceName(), assertValues[1]);
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<serviceName/>")},
            {createElement("<serviceName>manager#</serviceName>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#ServiceNameMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        ServiceNameMetaData metaData = new ServiceNameMetaData(null);
        metaData.importXML(element);
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {null, createElement("<serviceName>Service</serviceName>"), "<serviceName>" + ServiceManager.DEFAULT_NAME + "#Service</serviceName>"},
            {null, createElement("<serviceName>#Service</serviceName>"), "<serviceName>" + ServiceManager.DEFAULT_NAME + "#Service</serviceName>"},
            {"manager", createElement("<serviceName>Service</serviceName>"), "<serviceName>manager#Service</serviceName>"},
            {"manager", createElement("<serviceName>#Service</serviceName>"), "<serviceName>manager#Service</serviceName>"},
            {"manager", createElement("<serviceName>manager1#Service</serviceName>"), "<serviceName>manager1#Service</serviceName>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#ServiceNameMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(String manager, Element element, String expected) throws DeploymentException{
        ServiceNameMetaData metaData = new ServiceNameMetaData(null, manager);
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }

    @DataProvider(name = "Equals")
    public Object[][] dataEquals() throws Exception{
        return new Object[][]{
            {new ServiceNameMetaData(null, "manager", "service"), new ServiceNameMetaData(null, "manager", "service"), true},
            {new ServiceNameMetaData(null, "manager", "service1"), new ServiceNameMetaData(null, "manager", "service2"), false},
            {new ServiceNameMetaData(null, "manager1", "service"), new ServiceNameMetaData(null, "manager2", "service"), false},
            {new ServiceNameMetaData(null, "manager", "service"), new ServiceNameMetaData(null, "manager", null), false},
            {new ServiceNameMetaData(null, "manager", "service"), new ServiceNameMetaData(null, null, "service"), false},
            {new ServiceNameMetaData(null, "manager", "service"), new ServiceNameMetaData(null, null, null), false},
            {new ServiceNameMetaData(null, "manager", null), new ServiceNameMetaData(null, "manager", "service"), false},
            {new ServiceNameMetaData(null, null, "service"), new ServiceNameMetaData(null, "manager", "service"), false},
            {new ServiceNameMetaData(null, null, null), new ServiceNameMetaData(null, "manager", "service"), false},
            {new ServiceNameMetaData(null, null, null), null, false},
            {new ServiceNameMetaData(null, null, null), new Object(), false},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.ServiceNameMetaData#equals(java.lang.Object)",
            "jp.ossc.nimbus.core.ServiceNameMetaData#hashCode()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ServiceNameMetaData#ServiceNameMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData,java\\.lang\\.String,java\\.lang\\.String\\)"},
          dataProvider = "Equals")
    public void testEquals(
        ServiceNameMetaData comp1,
        Object comp2,
        boolean expected
    ) throws DeploymentException{
        assertEquals(comp1.equals(comp2), expected);
        if(comp2 != null){
            assertEquals(comp1.hashCode() == comp2.hashCode(), expected);
        }
    }
}
