/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.util.List;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link InvokeMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class InvokeMetaDataTest{

    @DataProvider(name = "InvokeMetaData")
    public Object[][] dataInvokeMetaData(){
        return new Object[][]{
            {null},
            {new ObjectMetaData()},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.InvokeMetaData#InvokeMetaData(jp.ossc.nimbus.core.MetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "InvokeMetaData")
    public void testInvokeMetaData(MetaData parent){
        InvokeMetaData metaData = new InvokeMetaData(parent);
        assertEquals(metaData.getParent(), parent);
        assertNull(metaData.getName());
        assertEquals(metaData.getCallState(), Service.State.CREATED.toString());
        assertEquals(metaData.getCallStateValue(), Service.State.CREATED);
        assertEquals(metaData.getArguments().size(), 0);
        assertNull(metaData.getTarget());
    }

    @DataProvider(name = "SetName")
    public Object[][] dataSetName(){
        return new Object[][]{
            {null},
            {"toString"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.InvokeMetaData#setName(java.lang.String)",
                    "jp.ossc.nimbus.core.InvokeMetaData#getName()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetName")
    public void testSetName(String name){
        InvokeMetaData metaData = new InvokeMetaData(null);
        metaData.setName(name);
        assertEquals(metaData.getName(), name);
    }

    @DataProvider(name = "SetTarget")
    public Object[][] dataSetTarget(){
        return new Object[][]{
            {null},
            {new ObjectMetaData()}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.InvokeMetaData#setTarget(jp.ossc.nimbus.core.MetaData)",
                    "jp.ossc.nimbus.core.InvokeMetaData#getTarget()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetTarget")
    public void testSetTarget(MetaData target){
        InvokeMetaData metaData = new InvokeMetaData(null);
        metaData.setTarget(target);
        assertEquals(metaData.getTarget(), target);
    }

    @DataProvider(name = "AddArgument")
    public Object[][] dataAddArgument(){
        return new Object[][]{
            {null},
            {new ArgumentMetaData[]{new ArgumentMetaData(null, null)}},
            {new ArgumentMetaData[]{new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null), new ArgumentMetaData(null, null)}},
            {new ArgumentMetaData[]{null, new ArgumentMetaData(null, null)}}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.InvokeMetaData#addArgument(jp.ossc.nimbus.core.ArgumentMetaData)",
                    "jp.ossc.nimbus.core.InvokeMetaData#getArguments()",
                    "jp.ossc.nimbus.core.InvokeMetaData#removeArgument(jp.ossc.nimbus.core.ArgumentMetaData)",
                    "jp.ossc.nimbus.core.InvokeMetaData#clearArguments()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(.*\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "AddArgument")
    public void testAddArgument(ArgumentMetaData[] arguments){
        InvokeMetaData metaData = new InvokeMetaData(null);
        if(arguments == null || arguments.length == 0){
            assertEquals(metaData.getArguments().size(), 0);
        }else{
            for(ArgumentMetaData argument : arguments){
                metaData.addArgument(argument);
            }
            List<ArgumentMetaData> list = metaData.getArguments();
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), arguments[i]);
            }
            for(int i = 0, imax = list.size(); i < imax; i+=2){
                metaData.removeArgument(arguments[i]);
            }
            list = metaData.getArguments();
            int index = 1;
            for(int i = 0; i < list.size(); i++){
                assertEquals(list.get(i), arguments[index]);
                index += 2;
            }
            metaData.clearArguments();
            assertEquals(metaData.getArguments().size(), 0);
        }
    }

    @DataProvider(name = "SetCallState")
    public Object[][] dataSetCallState(){
        return new Object[][]{
            {null, null, Service.State.UNKNOWN},
            {Service.State.CREATING.toString(), Service.State.CREATING.toString(), Service.State.CREATING},
            {Service.State.CREATED.toString(), Service.State.CREATED.toString(), Service.State.CREATED},
            {Service.State.STARTING.toString(), Service.State.STARTING.toString(), Service.State.STARTING},
            {Service.State.STARTED.toString(), Service.State.STARTED.toString(), Service.State.STARTED},
            {Service.State.STOPPING.toString(), Service.State.STOPPING.toString(), Service.State.STOPPING},
            {Service.State.STOPPED.toString(), Service.State.STOPPED.toString(), Service.State.STOPPED},
            {Service.State.DESTROYING.toString(), Service.State.DESTROYING.toString(), Service.State.DESTROYING},
            {Service.State.DESTROYED.toString(), Service.State.DESTROYED.toString(), Service.State.DESTROYED},
            {Service.State.FAILED.toString(), Service.State.FAILED.toString(), Service.State.FAILED},
            {Service.State.UNKNOWN.toString(), Service.State.UNKNOWN.toString(), Service.State.UNKNOWN},
            {"hoge", "hoge", Service.State.UNKNOWN},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.InvokeMetaData#setCallState(java.lang.String)",
                    "jp.ossc.nimbus.core.InvokeMetaData#getCallState()",
                    "jp.ossc.nimbus.core.InvokeMetaData#getCallStateValue()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetCallState")
    public void testSetCallState(String state, String expectedStr, Service.State expectedVal){
        InvokeMetaData metaData = new InvokeMetaData(null);
        metaData.setCallState(state);
        assertEquals(metaData.getCallState(), expectedStr);
        assertEquals(metaData.getCallStateValue(), expectedVal);
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<invoke name=\"toString\"/>"), new Object[]{"toString", Service.State.CREATED.toString(), Service.State.CREATED, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\"></invoke>"), new Object[]{"toString", Service.State.CREATED.toString(), Service.State.CREATED, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\" callState=\"" + Service.State.CREATING.toString() + "\"/>"), new Object[]{"toString", Service.State.CREATING.toString(), Service.State.CREATING, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\" callState=\"" + Service.State.CREATED.toString() + "\"/>"), new Object[]{"toString", Service.State.CREATED.toString(), Service.State.CREATED, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\" callState=\"" + Service.State.STARTING.toString() + "\"/>"), new Object[]{"toString", Service.State.STARTING.toString(), Service.State.STARTING, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\" callState=\"" + Service.State.STARTED.toString() + "\"/>"), new Object[]{"toString", Service.State.STARTED.toString(), Service.State.STARTED, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\" callState=\"" + Service.State.STOPPING.toString() + "\"/>"), new Object[]{"toString", Service.State.STOPPING.toString(), Service.State.STOPPING, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\" callState=\"" + Service.State.STOPPED.toString() + "\"/>"), new Object[]{"toString", Service.State.STOPPED.toString(), Service.State.STOPPED, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\" callState=\"" + Service.State.DESTROYING.toString() + "\"/>"), new Object[]{"toString", Service.State.DESTROYING.toString(), Service.State.DESTROYING, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\" callState=\"" + Service.State.DESTROYED.toString() + "\"/>"), new Object[]{"toString", Service.State.DESTROYED.toString(), Service.State.DESTROYED, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\" callState=\"" + Service.State.FAILED.toString() + "\"/>"), new Object[]{"toString", Service.State.FAILED.toString(), Service.State.FAILED, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\"><target><object code=\"java.lang.StringBuilder\"/></target></invoke>"), new Object[]{"toString", Service.State.CREATED.toString(), Service.State.CREATED, importXML(new ObjectMetaData(), "<object code=\"java.lang.StringBuilder\"/>"), new MetaData[0]}},
            {createElement("<invoke name=\"toString\"><target><service-ref>Service1</service-ref></target></invoke>"), new Object[]{"toString", Service.State.CREATED.toString(), Service.State.CREATED, importXML(new ServiceRefMetaData(null), "<service-ref>Service1</service-ref>"), new MetaData[0]}},
            {createElement("<invoke name=\"println\"><target><static-field-ref code=\"java.lang.System\" name=\"out\"/></target></invoke>"), new Object[]{"println", Service.State.CREATED.toString(), Service.State.CREATED, importXML(new StaticFieldRefMetaData(null), "<static-field-ref code=\"java.lang.System\" name=\"out\"/>"), new MetaData[0]}},
            {createElement("<invoke name=\"toString\"><target><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/></target></invoke>"), new Object[]{"toString", Service.State.CREATED.toString(), Service.State.CREATED, importXML(new StaticInvokeMetaData(null), "<static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/>"), new MetaData[0]}},
            {createElement("<invoke name=\"toString\"><target><invoke code=\"java.util.Calendar\" name=\"getTime\"><target><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/></target></invoke></target></invoke>"), new Object[]{"toString", Service.State.CREATED.toString(), Service.State.CREATED, importXML(new InvokeMetaData(null), "<invoke code=\"java.util.Calendar\" name=\"getTime\"><target><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/></target></invoke>"), new MetaData[0]}},
            {createElement("<invoke name=\"getProperty\"><argument>hoge</argument></invoke>"), new Object[]{"getProperty", Service.State.CREATED.toString(), Service.State.CREATED, null, new MetaData[]{importXML(new ArgumentMetaData(null, null), "<argument>hoge</argument>")}}},
            {createElement("<invoke name=\"substring\"><argument type=\"int\">0</argument><argument type=\"int\">10</argument></invoke>"), new Object[]{"substring", Service.State.CREATED.toString(), Service.State.CREATED, null, new MetaData[]{importXML(new ArgumentMetaData(null, null), "<argument type=\"int\">0</argument>"), importXML(new ArgumentMetaData(null, null), "<argument type=\"int\">10</argument>")}}},
            {createElement("<invoke name=\"toString\" unknown=\"hoge\"/>"), new Object[]{"toString", Service.State.CREATED.toString(), Service.State.CREATED, null, new MetaData[0]}},
            {createElement("<invoke name=\"toString\"><unknown/></invoke>"), new Object[]{"toString", Service.State.CREATED.toString(), Service.State.CREATED, null, new MetaData[0]}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.InvokeMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(.*\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#toXML\\(java\\.lang\\.StringBuilder\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#getName\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#getCallState\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#getCallStateValue\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#getArguments\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#getTarget\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        InvokeMetaData metaData = new InvokeMetaData(null);
        metaData.importXML(element);
        assertEquals(metaData.getName(), assertValues[0]);
        assertEquals(metaData.getCallState(), assertValues[1]);
        assertEquals(metaData.getCallStateValue(), assertValues[2]);
        assertMetaDataEquals(metaData.getTarget(), (MetaData)assertValues[3]);
        List<ArgumentMetaData> arguments = metaData.getArguments();
        MetaData[] args = (MetaData[])assertValues[4];
        if(args.length == 0){
            assertTrue(arguments.size() == 0);
        }else{
            assertEquals(arguments.size(), args.length);
            for(int i = 0; i < arguments.size(); i++){
                assertMetaDataEquals(arguments.get(i), args[i]);
            }
        }
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<invoke/>")},
            {createElement("<invoke name=\"toString\" callState=\"hoge\"/>")},
            {createElement("<invoke name=\"toString\"><target><invoke name=\"hoge\"/></target></invoke>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.InvokeMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        InvokeMetaData metaData = new InvokeMetaData(null);
        metaData.importXML(element);
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {null, createElement("<invoke name=\"toString\"/>"), "<invoke name=\"toString\"/>"},
            {new ServiceMetaData(), createElement("<invoke name=\"toString\"/>"), "<invoke name=\"toString\" callState=\"" + Service.State.CREATED.toString() + "\"/>"},
            {new ObjectMetaData(), createElement("<invoke name=\"getProperty\"><argument>hoge</argument></invoke>"), "<invoke name=\"getProperty\">" + System.getProperty("line.separator") + "    <argument>hoge</argument>" + System.getProperty("line.separator") + "</invoke>"},
            {new ObjectMetaData(), createElement("<invoke name=\"substring\"><argument type=\"int\">0</argument><argument type=\"int\">10</argument></invoke>"), "<invoke name=\"substring\">" + System.getProperty("line.separator") + "    <argument type=\"int\">0</argument>" + System.getProperty("line.separator") + "    <argument type=\"int\">10</argument>" + System.getProperty("line.separator") + "</invoke>"},
            {new ObjectMetaData(), createElement("<invoke name=\"toString\"><target><object code=\"java.lang.StringBuilder\"/></target></invoke>"), "<invoke name=\"toString\">" + System.getProperty("line.separator") + "    <target>" + System.getProperty("line.separator") + "        <object code=\"java.lang.StringBuilder\"/>" + System.getProperty("line.separator") + "    </target>" + System.getProperty("line.separator") + "</invoke>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.InvokeMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#InvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(MetaData parent, Element element, String expected) throws DeploymentException{
        InvokeMetaData metaData = new InvokeMetaData(parent);
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
