/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.util.ArrayList;
import java.util.List;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;

/**
 * {@link FactoryServiceBase}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class FactoryServiceBaseTest{
    
    @Test(groups = {"jp.ossc.nimbus.core.FactoryServiceBase#FactoryServiceBase()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#.*"})
    public void testFactoryServiceBase(){
        MyFactoryService service = new MyFactoryService();
        assertNotNull(service.getLogger());
        assertNotNull(service.getMessageRecordFactory());
        assertNull(service.getServiceManager());
        assertNull(service.getServiceManagerName());
        assertNull(service.getServiceName());
        assertNull(service.getServiceNameObject());
        assertEquals(service.getState(), Service.State.DESTROYED);
        assertNull(service.getSystemLoggerServiceName());
        assertNull(service.getSystemMessageRecordFactoryServiceName());
        assertEquals(service.getTarget(), service);
        assertFalse(service.isManagement());
        assertFalse(service.isThreadLocal());
        assertTrue(service.isCreateTemplateOnStart());
    }
    
    @Test(groups = {"jp.ossc.nimbus.core.FactoryServiceBase#newInstance()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#.*"})
    public void testNewInstance() throws Exception{
        MyFactoryService service = new MyFactoryService();
        service.create();
        service.start();
        assertEquals(service.instances.size(), 1);
        Object obj = service.newInstance();
        assertEquals(service.instances.size(), 2);
        assertEquals(obj, service.instances.get(1));
        obj = service.newInstance();
        assertEquals(service.instances.size(), 3);
        assertEquals(obj, service.instances.get(2));
        assertFalse(service.instances.get(1) == service.instances.get(2));
    }

    @DataProvider(name = "SetManagement")
    public Object[][] dataSetManagement(){
        return new Object[][]{
            {true},
            {false},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.FactoryServiceBase#setManagement(booleant)",
                    "jp.ossc.nimbus.core.FactoryServiceBase#isManagement()",
                    "jp.ossc.nimbus.core.FactoryServiceBase#release(java.lang.Object)",
                    "jp.ossc.nimbus.core.FactoryServiceBase#release()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.FactoryServiceBase#newInstance\\(\\)"},
          dataProvider = "SetManagement")
    public void testSetManagement(boolean isManagement) throws Exception{
        MyFactoryService service = new MyFactoryService();
        service.create();
        service.setManagement(isManagement);
        service.start();
        assertEquals(service.managedInstances.size(), 0);
        Object obj = service.newInstance();
        assertEquals(service.managedInstances.size(), service.isManagement() ? 1 : 0);
        assertEquals(service.managedInstances.contains(obj), service.isManagement());
        obj = service.newInstance();
        assertEquals(service.managedInstances.size(), service.isManagement() ? 2 : 0);
        assertEquals(service.managedInstances.contains(obj), service.isManagement());
        obj = service.newInstance();
        assertEquals(service.managedInstances.size(), service.isManagement() ? 3 : 0);
        assertEquals(service.managedInstances.contains(obj), service.isManagement());
        service.release(obj);
        assertEquals(service.managedInstances.size(), service.isManagement() ? 2 : 0);
        assertFalse(service.managedInstances.contains(obj));
        service.release();
        assertEquals(service.managedInstances.size(), 0);
        obj = service.newInstance();
        assertEquals(service.managedInstances.size(), service.isManagement() ? 1 : 0);
        service.destroy();
        assertEquals(service.managedInstances.size(), 0);
    }

    @DataProvider(name = "SetCreateTemplateOnStart")
    public Object[][] dataSetCreateTemplateOnStart(){
        return new Object[][]{
            {true},
            {false},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.FactoryServiceBase#setCreateTemplateOnStart(booleant)",
                    "jp.ossc.nimbus.core.FactoryServiceBase#isCreateTemplateOnStart()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#.*"},
          dataProvider = "SetCreateTemplateOnStart")
    public void testSetCreateTemplateOnStart(boolean isCreateTemplateOnStart) throws Exception{
        MyFactoryService service = new MyFactoryService();
        service.create();
        service.setCreateTemplateOnStart(isCreateTemplateOnStart);
        service.start();
        if(service.isCreateTemplateOnStart()){
            assertEquals(service.instances.size(), 1);
        }else{
            assertEquals(service.instances.size(), 0);
        }
    }

    @DataProvider(name = "SetThreadLocal")
    public Object[][] dataSetThreadLocal(){
        return new Object[][]{
            {true},
            {false},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.FactoryServiceBase#setThreadLocal(booleant)",
                    "jp.ossc.nimbus.core.FactoryServiceBase#isThreadLocal()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.ServiceBase#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.FactoryServiceBase#newInstance\\(\\)"},
          dataProvider = "SetThreadLocal")
    public void testSetThreadLocal(boolean isThreadLocal) throws Exception{
        final MyFactoryService service = new MyFactoryService();
        service.create();
        service.setThreadLocal(isThreadLocal);
        service.start();
        class MyRunnable implements Runnable{
            Object instance1;
            Object instance2;
            public void run(){
               instance1 = service.newInstance();
               instance2 = service.newInstance(); 
            }
        };
        MyRunnable runnable1 = new MyRunnable();
        MyRunnable runnable2 = new MyRunnable();
        Thread thread1 = new Thread(runnable1);
        Thread thread2 = new Thread(runnable2);
        thread1.start();
        thread2.start();
        thread1.join();
        thread2.join();
        if(service.isThreadLocal()){
            assertEquals(runnable1.instance1, runnable1.instance2);
            assertEquals(runnable2.instance1, runnable2.instance2);
            assertFalse(runnable1.instance1 == runnable2.instance1);
        }else{
            assertFalse(runnable1.instance1 == runnable1.instance2);
            assertFalse(runnable2.instance1 == runnable2.instance2);
            assertFalse(runnable1.instance1 == runnable2.instance1);
            assertFalse(runnable1.instance2 == runnable2.instance2);
        }
    }
    
    private static class MyFactoryService extends FactoryServiceBase<Object>{

        private static final long serialVersionUID = -4922503704344113521L;
        
        public List<Object> instances = new ArrayList<Object>();

        @Override
        protected Object createInstance() throws Exception{
            Object obj = new Object();
            instances.add(obj);
            return obj;
        }
    }
}
