/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.util.converter;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;

import javax.mail.MessagingException;
import javax.mail.internet.MimeUtility;

/**
 * BASE64Ro[^B<p>
 * javax.mail.internet.MimeUtilitygBASE64GR[hsB<br>
 *
 * @author M.Takata
 */
public class BASE64StringConverter extends StringStreamConverter implements StringConverter, ReversibleConverter, Serializable{
    
    private static final long serialVersionUID = 1193510073044683299L;

    /**
     * BASE64ւ̃GR[hϊ\ϊʒ萔B<p>
     */
    public static final int ENCODE = POSITIVE_CONVERT;
    
    /**
     * BASE64̃fR[hϊ\ϊʒ萔B<p>
     */
    public static final int DECODE = REVERSE_CONVERT;
    
    /**
     * 񁨃GR[hXg[\ϊʒ萔B<p>
     */
    public static final int ENCODE_STRING_TO_STREAM = 3;
    
    /**
     * 񁨃fR[hXg[\ϊʒ萔B<p>
     */
    public static final int DECODE_STRING_TO_STREAM = 4;
    
    /**
     * Xg[GR[h\ϊʒ萔B<p>
     */
    public static final int ENCODE_STREAM_TO_STRING = 5;
    
    /**
     * Xg[fR[h\ϊʒ萔B<p>
     */
    public static final int DECODE_STREAM_TO_STRING = 6;
    
    private int convertType = ENCODE;
    
    /**
     * ϊɎgp镶GR[fBOB<p>
     */
    private String characterEncoding;
    
    /**
     * 񁨃Xg[ϊɎgp镶GR[fBOB<p>
     */
    private String characterEncodingToStream;
    
    /**
     * Xg[ϊɎgp镶GR[fBOB<p>
     */
    private String characterEncodingToObject;
    
    /**
     * ϊʂݒ肷B<p>
     *
     * @param type ϊ
     * @see #ENCODE
     * @see #DECODE
     * @see #ENCODE_STRING_TO_STREAM
     * @see #DECODE_STRING_TO_STREAM
     * @see #ENCODE_STREAM_TO_STRING
     * @see #DECODE_STREAM_TO_STRING
     */
    public void setConvertType(int type){
        convertType = type;
    }
    
    /**
     * ϊʂ擾B<p>
     *
     * @return ϊ
     * @see #ENCODE
     * @see #DECODE
     * @see #ENCODE_STRING_TO_STREAM
     * @see #DECODE_STRING_TO_STREAM
     * @see #ENCODE_STREAM_TO_STRING
     * @see #DECODE_STREAM_TO_STRING
     */
    public int getConvertType(){
        return convertType;
    }
    
    /**
     * ϊɎgp镶GR[fBOݒ肷B<p>
     * 
     * @param encoding GR[fBO
     */
    public void setCharacterEncoding(String encoding){
        characterEncoding = encoding;
    }
    
    /**
     * ϊɎgp镶GR[fBO擾B<p>
     * 
     * @return GR[fBO
     */
    public String getCharacterEncoding(){
        return characterEncoding;
    }
    
    /**
     * 񁨃Xg[ϊɎgp镶GR[fBOݒ肷B<p>
     * 
     * @param encoding GR[fBO
     */
    public void setCharacterEncodingToStream(String encoding){
        characterEncodingToStream = encoding;
    }
    
    /**
     * 񁨃Xg[ϊɎgp镶GR[fBO擾B<p>
     * 
     * @return GR[fBO
     */
    public String getCharacterEncodingToStream(){
        return characterEncodingToStream;
    }
    
    /**
     * Xg[ϊɎgp镶GR[fBOݒ肷B<p>
     * 
     * @param encoding GR[fBO
     */
    public void setCharacterEncodingToObject(String encoding){
        characterEncodingToObject = encoding;
    }
    
    /**
     * Xg[ϊɎgp镶GR[fBO擾B<p>
     * 
     * @return GR[fBO
     */
    public String getCharacterEncodingToObject(){
        return characterEncodingToObject;
    }
    
    public Object convert(Object obj) throws ConvertException{
        if(obj == null){
            return null;
        }
        switch(convertType){
        case ENCODE:
        case DECODE:
            return convert(obj.toString());
        case ENCODE_STRING_TO_STREAM:
        case DECODE_STRING_TO_STREAM:
            return convertToStream(obj);
        case ENCODE_STREAM_TO_STRING:
        case DECODE_STREAM_TO_STRING:
            return convertToObject((InputStream)obj);
        default:
            throw new ConvertException("Illegal convert type : " + convertType);
        }
    }
    
    public String convert(String str) throws ConvertException{
        if(str == null){
            return null;
        }
        switch(convertType){
        case ENCODE:
            return encode(str);
        case DECODE:
            return decode(str);
        default:
            throw new ConvertException("Illegal convert type : " + convertType);
        }
    }
    
    protected byte[] convertToByteArray(Object obj) throws ConvertException{
        switch(convertType){
        case ENCODE:
        case ENCODE_STRING_TO_STREAM:
            return encodeBytes(super.convertToByteArray(obj));
        case DECODE:
        case DECODE_STRING_TO_STREAM:
            return decodeBytes(super.convertToByteArray(obj));
        default:
            throw new ConvertException("Illegal convert type : " + convertType);
        }
    }
    
    public Object convertToObject(InputStream is) throws ConvertException{
        
        byte[] bytes = toBytes(is);
        switch(convertType){
        case ENCODE:
        case ENCODE_STREAM_TO_STRING:
            return toString(encodeBytes(bytes));
        case DECODE:
        case DECODE_STREAM_TO_STRING:
            return toString(decodeBytes(bytes));
        default:
            throw new ConvertException("Illegal convert type : " + convertType);
        }
    }
    
    /**
     * w肳ꂽBASE64GR[hB<p>
     *
     * @param str 
     * @return BASE64GR[hꂽ
     * @exception ConvertException GR[hɎsꍇ
     */
    public String encode(String str) throws ConvertException{
        return encode(str, characterEncoding);
    }
    
    /**
     * w肳ꂽAw肳ꂽGR[fBOŃoCgzɕϊBASE64GR[hB<p>
     *
     * @param str 
     * @param encoding GR[fBO
     * @return BASE64GR[hꂽ
     * @exception ConvertException GR[hɎsꍇ
     */
    public static String encode(String str, String encoding) throws ConvertException{
        byte[] bytes = null;
        if(encoding == null){
            bytes = str.getBytes();
        }else{
            try{
                bytes = str.getBytes(encoding);
            }catch(IOException e){
                throw new ConvertException(e);
            }
        }
        return new String(encodeBytes(bytes));
    }
    
    /**
     * w肳ꂽoCgzBASE64GR[hB<p>
     *
     * @param bytes oCgz
     * @return BASE64GR[hꂽoCgz
     * @exception ConvertException GR[hɎsꍇ
     */
    public static byte[] encodeBytes(byte[] bytes) throws ConvertException{
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        OutputStream os = null;
        try{
            os = MimeUtility.encode(baos, "base64");
            os.write(bytes);
        }catch(MessagingException e){
            throw new ConvertException(e);
        }catch(IOException e){
            throw new ConvertException(e);
        }finally{
            if(os != null){
                try{
                    os.close();
                }catch(IOException e){}
            }
        }
        return baos.toByteArray();
    }
    
    /**
     * w肳ꂽBASE64fR[hB<p>
     *
     * @param str BASE64GR[hꂽ
     * @return fR[hꂽ
     * @exception ConvertException fR[hɎsꍇ
     */
    public String decode(String str) throws ConvertException{
        return decode(str, characterEncoding);
    }
    
    /**
     * w肳ꂽBASE64fR[hAw肳ꂽGR[fBO̕ɕϊB<p>
     *
     * @param str BASE64
     * @param encoding GR[fBO
     * @return fR[hꂽ
     * @exception ConvertException fR[hɎsꍇ
     */
    public static String decode(String str, String encoding) throws ConvertException{
        byte[] bytes = decodeBytes(str.getBytes());
        if(encoding == null){
            return new String(bytes);
        }else{
            try{
                return new String(bytes, encoding);
            }catch(IOException e){
                throw new ConvertException(e);
            }
        }
    }
    
    /**
     * w肳ꂽBASE64oCgzfR[hB<p>
     *
     * @param bytes BASE64oCgz
     * @return fR[hꂽoCgz
     * @exception ConvertException fR[hɎsꍇ
     */
    public static byte[] decodeBytes(byte[] bytes) throws ConvertException{
        final ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        byte[] outBytes = null;
        InputStream is = null;
        try{
            is = MimeUtility.decode(bais, "base64");
            byte[] tmpBytes = new byte[bytes.length];
            int length = is.read(tmpBytes);
            outBytes = new byte[length];
            System.arraycopy(tmpBytes, 0, outBytes, 0, length);
        }catch(MessagingException e){
            throw new ConvertException(e);
        }catch(IOException e){
            throw new ConvertException(e);
        }finally{
            if(is != null){
                try{
                    is.close();
                }catch(IOException e){}
            }
        }
        return outBytes;
    }
}
